#include <QLabel>
#include <QPushButton>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QTextBrowser>
#include <QMenu>
#include <QAction>
#include <QtGui>
#include <QMaemo5ValueButton>
#include <QMaemo5ListPickSelector>
#include <QMaemo5InformationBox>
#include <QMessageBox>
#include <QSizePolicy>
#include <QtMaemo5>
#include <QListWidget>
#include <QListWidgetItem>
#include <QScrollArea>
#include <QPixmap>
#include <QDBusConnection>
#include <QDBusMessage>
#include <QLineEdit>
#include <QComboBox>
#include <QBrush>
#include <QProgressBar>

#include <mce/mode-names.h>
#include <mce/dbus-names.h>

#include "mainwindow.h"
#include "misc.h"
#include "region.h"
#include "favourite.h"
#include "reminder.h"

//          	LAYOUT DIAGRAM
// --------------------------------------
// |           h1layout         |       |
// | 	     <--------->	    |action |
// | 	 Selection Panel	    |panel  |
// |----------------------------|       |
// | 							|       |
// | 	^						|       |
// | 	|             			|  v2   |
// | v1 | prog detail panel     |       | 
// | 	v						|       |
// | 							|       |
// | 							|       |
// |___________________________ |_______|
//   <---------- layout--------------->
//
//
const int NAVIGATION_BUTTON_SIZE = 70;
const int ACTION_BUTTON_SIZE = 70;

int i = 0;

MainWindow::MainWindow()
    : QMainWindow(), _channelIndex(0), _dayCount(0), _count(0), _landscapeOrientation(true), _searchMode(false), _currentListingMode(false)
{
	setWindowTitle("SHOWTIME   v1.2");

	//////////////////////////////////////////////////////////////////	
	// this is the class that inits and updates the settings file
	// and also provides getters for the settings
	//////////////////////////////////////////////////////////////////	
	_settingsWindow = new SettingsWindow(this);

	//////////////////////////////////////////////////////////////////	
	//#### inits the APP settings
	//////////////////////////////////////////////////////////////////	
	initAppSettings();
	//////////////////////////////////////////////////////////////////	

	// create different widget for different orientations
	_portraitMainWidget = new QWidget(this);
	_landscapeMainWidget = new QWidget(this);

	// this is the class that provides the channel information from XMLTV downloads
	_infoapi = new InfoAccess(this,_settingsWindow->getRegion());

	/////////////////////////////////////////////
	/////     CALENDER & ALARM OBJECT      //////
	/////////////////////////////////////////////
	initCalendar();
	//////////////// END OF CALENDER INIT ////////////////////////////////////
			
	/////////////////////////////////////////////
	// the list of widgets for the selection panel ( see layout diagram above )
	/////////////////////////////////////////////

	// date selector
	_dateListModel = new QStandardItemModel(this);
	_dateSelectorList = new QMaemo5ListPickSelector(this);
	_dateSelectorList->setModel(_dateListModel);
	_dateSelector = new QMaemo5ValueButton("PICK DATE", this);
	_dateSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
	_dateSelector->setPickSelector(_dateSelectorList);
	_dateSelector->setValueText(QDate::currentDate().toString());

	// channel selector
	_channelListModel = new QStandardItemModel(this);
	_channelSelectorList = new QMaemo5ListPickSelector(this);
	_channelSelectorList->setModel(_channelListModel);
	_channelSelector = new QMaemo5ValueButton("PICK CHANNELS", this);
	_channelSelector->setPickSelector(_channelSelectorList);
	_channelSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);

	// record and remind action buttons
	_recordButton = new QPushButton(this);
	_remindButton = new QPushButton(this);
	_recordButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_remindButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_recordButton->setIcon(QIcon(_settingsWindow->getIconPath() + "record.png"));
	_remindButton->setIcon(QIcon(_settingsWindow->getIconPath() + "remind.png"));

	// search programme title widgets
	_searchInput = new QComboBox(this);
	_searchInput->setEditable(true);
	_searchInput->setInsertPolicy(QComboBox::NoInsert);
	_searchInput->setCurrentIndex(-1);

	//_searchInput->setPlaceholderText("Type programme search string");
	// show them only when search is selected from the menu
	_searchInput->hide();

	/////////////////////////////////////////////
	// the list of widgets for the prog detail panel ( see diagram above for layout )
	/////////////////////////////////////////////

	// to list the programmes and their time
	_programmeList = new QListWidget(this);
	//_programmeList->setSelectionMode(QAbstractItemView::MultiSelection);
	_infoLabel = new QLabel(QString("<b>HIT GO !<\b>") + QString("\n\n*note data may be unavailable for some channels for some/all dates"), this);
	_infoLabel->setStyleSheet("QLabel { color : red }");
	_infoLabel->setWordWrap(true);
	_infoLabel->hide();

	// to show details of selected programmes
	_programmeDetail = new QTextBrowser(this);
	_programmeDetailLabel = new QLabel("YOUR CHOSEN PROGRAMMES", this);
	_programmeDetailLabel->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	_programmeDetailLabel->setStyleSheet("QLabel { background-color : lightgreen;  border-width: 5px; border-style: solid; border-color: green; color : black }");
	_programmeDetail->hide();
	_programmeDetailLabel->hide();

	// to get back to programme list
	_backToProgrammeListButton = new QPushButton(this);
	_backToProgrammeListButton->hide();
	_backToProgrammeListButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_backToProgrammeListButton->setIcon(QIcon(_settingsWindow->getIconPath() + "back.png"));

	// to clear selected programmes
	_clearProgButton = new QPushButton(this);
	_clearProgButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_clearProgButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));
	_clearProgButton->hide();

	/////////////////////////////////////////////
	// the list of widgets for the action panel ( see above diagram for layout detail )
	/////////////////////////////////////////////

	// button to fetch data if not available in local memory
	_goButton = new QPushButton(this);
	_goButton->setMinimumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_goButton->setIcon(QIcon(_settingsWindow->getIconPath() + "go.png"));

	// button to clear the selected channels and all local data
	_clearButton = new QPushButton(this);
	_clearButton->setMaximumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_clearButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));

	// button to display selected programme details window
	_programmeDetailButton = new QPushButton(this);
	_programmeDetailButton->setMaximumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_programmeDetailButton->setIcon(QIcon(_settingsWindow->getIconPath() + "detail.png"));

	// buttons to navigate between selected channels for the chosen date
	_nextChannelButton = new QPushButton(this);
	_prevChannelButton = new QPushButton(this);
	_nextChannelButton->setMinimumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_prevChannelButton->setMinimumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_nextChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "next.png"));
	_prevChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "prev.png"));
	// only display when multi channel selection
	_nextChannelButton->setDisabled(true);
	_prevChannelButton->setDisabled(true);

	///////////////////////////////////////////////////////////
	///// a progress bar widget to show animation//////////////
	_progressLabel = new QLabel("<font color=#00ff00>% of data in memory...</font>");
	_progress = new QProgressBar(this);
	_progress->hide();
	_progressLabel->hide();
	_progress->setValue(0);
	///////////////////////////////////////////////////////////

	/////////////////////////////////////////////////////////////////
	// init the layouts for both orientation modes
	initLandscape();
	initPortrait();
	/////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////	
	//#### inits the models used by the channel and date selector ####
	//////////////////////////////////////////////////////////////////	
	initListModels();
	//////////////////////////////////////////////////////////////////	
   
	//////////////////////////////////////////////////////////////////	
	//#### create the menus
	//////////////////////////////////////////////////////////////////	
	createMenus();
	//////////////////////////////////////////////////////////////////	

	//////////////////////////////////////////////////////////////////	
	//#### init(listen to) and set the default orientation 
	//////////////////////////////////////////////////////////////////	
	initOrientation();
   //////////////////////////////////////////////////////////////////	

    // CONNECT ALL BUTTONS AND WIDGET TO THEIR SLOTS
	connect(_infoapi, SIGNAL(finished(bool)), this, SLOT(finished(bool)));
	connect(_dateSelectorList, SIGNAL(selected(QString)), this, SLOT(selectDate(QString)));
	connect(_channelSelectorList, SIGNAL(selected(QString)), this, SLOT(selectChannel(QString)));
	connect(_goButton, SIGNAL(clicked()), this, SLOT(go()));
	connect(_backToProgrammeListButton, SIGNAL(clicked()), this, SLOT(showProgrammeListWidget()));
	connect(_clearProgButton, SIGNAL(clicked()), this, SLOT(clearProg()));
	connect(_clearButton, SIGNAL(clicked()), this, SLOT(clearAll()));
	connect(_nextChannelButton, SIGNAL(clicked()), this, SLOT(showNextChannel()));
	connect(_prevChannelButton, SIGNAL(clicked()), this, SLOT(showPrevChannel()));
	connect(_programmeList, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(addProgrammeActionList(QListWidgetItem*)));
	connect(_programmeDetailButton, SIGNAL(clicked()), this, SLOT(showProgrammeDetailWidget()));
	connect(_recordButton, SIGNAL(clicked()), this, SLOT(showRecordMessage()));
	connect(_remindButton, SIGNAL(clicked()), this, SLOT(showRemindMessage()));
	connect(this, SIGNAL(downloadIndex(int)), _progress, SLOT(setValue(int)));

	//////////////////////////////////////////////////////////////////	
	//#### show the default screen from the settings, if nothing is
	//#### set then show daily mode
	//////////////////////////////////////////////////////////////////	
	showDefaultScreen();
	//////////////////////////////////////////////////////////////////	
}

// LOADS THE APP SETTINGS TO MEMORY
void MainWindow::initAppSettings()
{
	_settingsWindow->loadSettings();
}

// INITS THE SHOWTIME CALENDAR
void MainWindow::initCalendar()
{
	_reminderManager = new Reminder;
	bool calexists = _reminderManager->checkCalendarNameExists("SHOWTIME");

	if ( !calexists )
	{
		//qDebug() << "SHOWTIME calendar does not exist, creating one";
		_reminderManager->addCalendar(QString("SHOWTIME"));
	}
	else
	{
		//qDebug() << "SHOWTIME calendar exists";
	}
}

// INITS THE DATE AND CHANNEL LISTS
void MainWindow::initListModels()
{
	//////////////////////////////////////////////////////
	// CHANNEL SELECTOR LIST 
	//////////////////////////////////////////////////////

	_dateListModel->clear();
	_channelListModel->clear();

	QStringList channelList = _infoapi->getChannelList();

	QStandardItem *item = new QStandardItem("Favourites");
	item->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
	item->setEditable(false); // prevent editing of the item
	_channelListModel->appendRow(item);

	foreach(QString channel, channelList)
	{
		QStandardItem *item = new QStandardItem(channel);
		item->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
		item->setEditable(false); // prevent editing of the item
		_channelListModel->appendRow(item);
	}

	//////////////////////////////////////////////////////
	// DATE SELECTOR LIST 
	//////////////////////////////////////////////////////
	for(int i = -1; i < 7; ++i )
	{
		QStandardItem *dateitem = new QStandardItem(QDate::currentDate().addDays(i).toString());
		dateitem->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
		dateitem->setEditable(false); // prevent editing of the item
		_dateListModel->appendRow(dateitem);
	}
}

void MainWindow::colorDateSelector()
{
	//////////////////////////////////////////////////////
	// COLOR DATE SELECTOR LIST 
	// Red - means data not in memory, can be tried to download
	// Green - means data is present in memory , no need to click GO
	//////////////////////////////////////////////////////

	_dateListModel->clear();

    ChannelMetaData metadata = (_infoapi->metadata())[_currentChannel];
	QDate minDate = QDate::fromString(metadata.minDate, "yyyyMMdd");
	QDate maxDate = QDate::fromString(metadata.maxDate, "yyyyMMdd");
	QSet<QString> availableDateList = metadata.availableDates;

	for(int i = -1; i < 7; ++i )
	{
		QStandardItem *dateitem = new QStandardItem(QDate::currentDate().addDays(i).toString());
		dateitem->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
		dateitem->setEditable(false); // prevent editing of the item

		QBrush brush(Qt::red);
		// IF the region is a SINGLEXMLTV file region and if the date falls between the min and max then show it as green
		// OR
		// IF the region is a per day channel file region and if the date is present in the availableDates then show as green
		if ( (_infoapi->isSingleXMLTVFile() && QDate::currentDate().addDays(i) >= minDate && QDate::currentDate().addDays(i) <= maxDate  ) ||
				(!_infoapi->isSingleXMLTVFile()	&& availableDateList.contains(QDate::currentDate().addDays(i).toString("yyyyMMdd")) )
		   )
			brush.setColor(Qt::green);
		else
			brush.setColor(Qt::red);
		dateitem->setForeground(brush);

		_dateListModel->appendRow(dateitem);
	}
}

// CREATES THE MENU
void MainWindow::createMenus()
{
	regionAct = new QAction(tr("&REGION"), this);
	favAct = new QAction(tr("&FAVOURITES"), this);
	setAct = new QAction(tr("&SETTINGS"), this);
	dailyAct = new QAction(tr("&DAILY MODE"), this);
	searchAct = new QAction(tr("&SEARCH MODE"), this);
	currentListingsAct = new QAction(tr("&CURRENT MODE"), this);
	clearCacheAct = new QAction(tr("&CLEAR CACHE"), this);
	helpAct = new QAction(tr("&HELP"), this);
	aboutAct = new QAction(tr("&ABOUT"), this);
	donateAct = new QAction(tr("&DONATE"), this);

	connect(regionAct, SIGNAL(triggered()), this , SLOT(showRegion()));
	connect(favAct, SIGNAL(triggered()), this , SLOT(showFavourite()));
	connect(setAct, SIGNAL(triggered()), this , SLOT(showSettings()));
	connect(dailyAct, SIGNAL(triggered()), this , SLOT(switchDaily()));
	connect(searchAct, SIGNAL(triggered()), this , SLOT(switchSearch()));
	connect(currentListingsAct, SIGNAL(triggered()), this , SLOT(switchCurrentListings()));
	connect(clearCacheAct, SIGNAL(triggered()), this , SLOT(clearCacheData()));
	connect(helpAct, SIGNAL(triggered()), this , SLOT(showHelp()));
	connect(aboutAct, SIGNAL(triggered()), this , SLOT(showAbout()));
	connect(donateAct, SIGNAL(triggered()), this , SLOT(showDonate()));

	menu = menuBar()->addMenu(tr(""));
	menu->addAction(regionAct);
	menu->addAction(favAct);
	menu->addAction(setAct);
	menu->addAction(dailyAct);
	menu->addAction(searchAct);
	menu->addAction(currentListingsAct);
	menu->addAction(clearCacheAct);
	menu->addAction(helpAct);
	menu->addAction(aboutAct);
	menu->addAction(donateAct);
}

//SHOW the default screen as set by the user, if nothing is set show daily mode
void MainWindow::showDefaultScreen()
{
	if ( _settingsWindow->getDefaultMode() == "DAILYMODE" )
	{
		switchDaily();
	}
	else if ( _settingsWindow->getDefaultMode() == "SEARCHMODE" )
	{
		switchSearch();
	}
	else if ( _settingsWindow->getDefaultMode() == "CURRENTLISTINGMODE" )
	{
		loadFavourites();
		switchCurrentListings();
	}
}

// CONNECTS THE ACCELEROMETER USING DBUS to listen to orientation changes in AUTO mode only
void MainWindow::initOrientation()
{
	if ( _settingsWindow->getOrientation() == "AUTO" )
	{
		QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
							MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ));

		QDBusConnection::systemBus().connect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			                MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString)));
		setLandscape(); // default
	}
    else if (_settingsWindow->getOrientation() == "LANDSCAPE") 
	{
		QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
							MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ));
		QDBusConnection::systemBus().disconnect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			                MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString)));
		setLandscape();
	}
    else if (_settingsWindow->getOrientation() == "PORTRAIT")
	{
		QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
							MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ));
		QDBusConnection::systemBus().disconnect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			                MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString)));
		setPortrait();
	}
}

// SLOT when orientation change is detected
void MainWindow::orientationChanged(const QString &newOrientation)
{
	if (newOrientation == QLatin1String(MCE_ORIENTATION_PORTRAIT))
		setPortrait();
	else
		setLandscape();
}

// The widgets that need to be hidden/show when the programme list is displayed
void MainWindow::showProgrammeListWidgetButtons()
{
	_goButton->show();  
	_recordButton->show();
	_remindButton->show();
	_programmeDetail->hide();
	_programmeDetailLabel->hide();
	_clearProgButton->hide();
	_programmeDetailButton->show();
	_backToProgrammeListButton->hide();
	_programmeList->show();
	_channelSelector->show();

	if ( _searchMode )
	{
		_dateSelector->hide();
		_searchInput->show();
		_searchInput->clear();
		_searchInput->addItems(_settingsWindow->getSearchTitles(_settingsWindow->getRegion(), _currentChannel));
		_searchInput->lineEdit()->setText(_searchStringPerChannelMap[_currentChannel]);
	}
	else
	{
		_dateSelector->show();
		_searchInput->hide();
	}

	_clearButton->show();
	_nextChannelButton->show();
	_prevChannelButton->show();

	showNextPrevChannelButtons();
}

// The method that calls the infoaccess class to download/read xmltv info into local memory
void MainWindow::go()
{
	/////////////////////////////////////////////////////////////////
	// if its weekly download go to the goSearch, dont do anything else
	if ( _searchMode )
	{
		goSearch();
		return;
	}
	/////////////////////////////////////////////////////////////////

	if (_channelSelector->valueText() == "" || _dateSelector->valueText() == "")
		return;

	/////////////////////////////////////////////////////////
	// display the progress of the download
	_progressLabel->show();
	_progress->show();
	_progress->setMaximum(_channelsSelected.size());
	/////////////////////////////////////////////////////////

	/////////////////////////////////////////////////////////////////
	////if its currentListingMode then fix the date to today/////////
	/////////and show the buttons for currentListings ///////////////
	if ( _currentListingMode )
	{
		showCurrentListingButtons();
		_dateSelector->setValueText(QDate::currentDate().toString());
	}
	/////////////////////////////////////////////////////////////////

	_channelSelector->setValueText(_channelsSelected[_channelIndex]);

	// set the current date to what is being displayed in the dateselector
	_currentDate = QDate::fromString(_dateSelector->valueText(), Qt::TextDate);

	// we will use _currentChannel to iterate through the selected channels
	// channelIndex we will not use as it can be changed by the user using the prev/next buttons
	// while the read/download of selected channels is going on, hence a different count ( _count)
	_currentChannel = _channelsSelected[_count];

	// set the input for infoapi
	DownloadInput d;
	d.requestDate = QDate::fromString(_dateSelector->valueText(), Qt::TextDate);
	d.requestChannel = _currentChannel;

	// if the data is not contained locally in the map then download/read from file, that means data was already read earlier
	if ( ! _channelDataAvailableMap[_currentChannel].contains(_currentDate.toString("yyyyMMdd")) )
	{
		toggleBusy(true);
		_infoapi->getChannelInfo(d);
	}
	else
	{
		//qDebug() << "channel data already in memory ";
		finished();
	}
}

// simple message to guide the user to hit the go button when data doesnt exist in local memory
// for selected channel and date
void MainWindow::showInfoLabel(QString info)
{
	_programmeList->clear();
	_infoLabel->setText(info);
	if(_channelSelector->valueText() != "")
		_infoLabel->show();
}

// SLOT when date is selected
void MainWindow::selectDate(QString date)
{
	//qDebug() << "Current Date " << _dateSelector->valueText();
	if ( _dateSelector->valueText() != "" )
	{
		_currentDate = QDate::fromString(_dateSelector->valueText(), Qt::TextDate);
	}
    
    // SHOW the prog list if in memory already
	showProgrammeListWidget();
}

// SLOT when channel is selected
void MainWindow::selectChannel(QString channel)
{
	// IF YOU HAVE SELECTED FAVOURITES SIMPLY LOAD THE FAVOURITES AND RETURN
	if ( channel == "Favourites" )
	{
		loadFavourites();
		return;
	}

	//qDebug() << "Current channel " << _channelSelector->valueText();
	if ( _channelSelector->valueText() != "" )
		_currentChannel = _channelSelector->valueText();

	// if the channel does not exist in the selected list then add it 
	// and set the index to current channel
	if ( ! _channelsSelected.contains(_currentChannel) )
	{
		_channelsSelected << _currentChannel;
		_channelIndex = _channelsSelected.count() - 1;
	}
	// set the index to the selected channel that exists
	else
	{
		_channelIndex = _channelsSelected.indexOf(_currentChannel);
	}

    // SHOW the prog list if in memory already
	showProgrammeListWidget();
}


// CLEAR all info and start afresh
void MainWindow::clearAll()
{
	_channelsSelected.clear();
	_channelIndex = 0;
	_channelDataMap.clear();
	_channelDataAvailableMap.clear();
	_searchStringPerChannelMap.clear();
	_currentChannel.clear();
	_programmeList->clear();
	_programmeDetail->clear();
	_programmeSetForAction.clear();
	_titleDisplayMap.clear();
	_nextChannelButton->setDisabled(true);
	_prevChannelButton->setDisabled(true);
	_infoapi->clear();

	_channelSelector->setValueText("");
	_dateSelector->setValueText("");
	_channelSelector->setText("PICK CHANNELS");
	_dateSelector->setValueText(QDate::currentDate().toString());
	_searchInput->clear();
}

// CLEAR selected programmes list
void MainWindow::clearProg()
{
	_programmeDetail->clear();
	_programmeSetForAction.clear();
}

// Callback SLOT when the XMLTV info required is downloaded/read
void MainWindow::finished(bool dataRequested)
{
	//qDebug() << "In finished";

	_infoLabel->hide();

	_channelDataAvailableMap[_currentChannel] = _infoapi->metadata()[_currentChannel].availableDates;

	// if weekly mode we need to iterate through day1 to day7 of the week so call finishedSearch
	if ( _searchMode )
	{
		// call the weekly download finished to check if we are finished 
		finishedSearch();
	}
	// if it is currentListingMode or the dailyMode just iterate through selected channels 1 by 1
	else 
	{
		//qDebug() << "recvd current listings for " << _channelsSelected[_count] << " channelIndex=" << _count;
		// if we have finished downloaded the files for the week	

		if ( ++_count == _channelsSelected.size() )
		{
			_count = 0;
			_progress->setValue(100);
			//qDebug() << "all current listings in memory now";
			toggleBusy(false);
			_progressLabel->hide();
			_progress->hide();
			_progress->setValue(0);
			// set the current channel back to what is being pointed by the index and whats displayed in the PICK CHANNEL button
			_currentChannel  = _channelsSelected[_channelIndex];
			showProgrammeListWidget(true);
		}
		else // keep downloading the files
		{
			emit downloadIndex(_count);
			go();
		}
	}
}


// dataRequested indicates that this is being called by finished via a callback after requesting data
// if this is false it means that its being called by usual navigation buttons 
// and in that case if there is no data we display HitGo
void MainWindow::showProgrammeListWidget(bool dataRequested)
{

	// If in current listings mode then just go and display the current listing and return from here
	if ( _currentListingMode )
	{
		showCurrentProgrammeListWidget(dataRequested);
		return;
	}

	_programmeList->clear();
	_titleDisplayMap.clear();
	_infoLabel->hide();
	showProgrammeListWidgetButtons();
	
    QList<ChannelData> completeData = (_infoapi->results())[_currentChannel];

	// marks available dates as green and others as red
	colorDateSelector();

	//qDebug() << _infoapi->results()[_currentChannel];

	bool dataAvailable = false;

	/////////////////////////////////////////////////////////////
	/////////////////// CHECK THE SETTINGS //////////////////////
	
	// hide listings < currentdatetime ?
	bool hidePastListings = (_settingsWindow->getShowPastInd() == "N") ? true : false;

	// see if prime time is set , then only show listings b/w those times
	bool primeTimeSet = false;
	QTime startPrimeTime = QTime::fromString(_settingsWindow->getPrimeTimeStart(), "hh:mm");
	QTime endPrimeTime = QTime::fromString(_settingsWindow->getPrimeTimeEnd(), "hh:mm");
	if ( _settingsWindow->getPrimeTimeStart() == "00:00" && _settingsWindow->getPrimeTimeEnd() == "00:00" )
		primeTimeSet = false;
	else
		primeTimeSet = true;

	// if weekly mode , see if the title matches the search input
	bool titleMatched = true;
	/////////////// END OF CHECK THE SETTINGS ///////////////////
	/////////////////////////////////////////////////////////////
	
	foreach(ChannelData data, completeData)
	{
		QDateTime progSDateTime = QDateTime::fromString(data.sdate + data.stime, "ddd MMM d yyyyHH:mm:ss");
		QDateTime progEDateTime = QDateTime::fromString(data.edate + data.etime, "ddd MMM d yyyyHH:mm:ss");
		QString title = data.title;
		QString sdate = data.sdate;
		QString stime = data.stime;
		stime.truncate(5);
		QString etime = data.etime;
		etime.truncate(5);
		QString time = stime + " - " + etime;

		// have 2 keys here as in the display we dont want to display date
		// but on clicking the item in the list we want the key to have the date also just in case the user selects the same program 
		// for multiple days
		QString keyForDisplay;
		QString keyForInsert;
		keyForInsert = title + sdate + time;
		breakTitle(title);
		if ( _landscapeOrientation )
		{
			if ( _searchMode ) // less tabs for weekly as we have date also
			{
				keyForDisplay = title + "\n\t\t\t\t" + sdate + " : " + time;
			}
			else
			{
				keyForDisplay = title + "\n\t\t\t\t\t\t" + time;
			}
		}
		else // in portrait mode we need less tabs to prevent horizontal scrolling
		{
			if ( _searchMode ) // less tabs for weekly as we have date also
			{
				keyForDisplay = title + "\n" + sdate + " : " + time;
			}
			else
			{
				keyForDisplay = title + "\n\t\t\t" + time;
			}
		}

		// see if title matches the search string in weekly search mode
		if ( _searchMode )
		{
			titleMatched = false;

			QStringList searchList = _searchInput->currentText().split(" + ", QString::SkipEmptyParts);
			foreach(QString searchString, searchList)
			{
				if ( data.title.contains(searchString, Qt::CaseInsensitive))
				{
					titleMatched = true;
					break;
				}
			}
		}
		else
		{
			titleMatched = true;
		}

		// if its weekly mode then check if title matches else check if programme start date matches the user chosen date
		if ( (!_searchMode && data.sdate == _currentDate.toString(Qt::TextDate)) || (_searchMode && titleMatched) )
		{
			if ( hidePastListings && progSDateTime < QDateTime::currentDateTime() && progEDateTime <= QDateTime::currentDateTime())
			{
				continue;
			}

			if ( primeTimeSet && (QTime::fromString(stime, "hh:mm") < startPrimeTime || QTime::fromString(stime, "hh:mm") > endPrimeTime) ) 
			{
				continue;
			}
		
			_titleDisplayMap[keyForDisplay]=keyForInsert;

			QListWidgetItem *item = new QListWidgetItem(keyForDisplay, _programmeList);	
			QFont itemFont;
			itemFont.setPixelSize(20);
			item->setFont(itemFont);
			QBrush brush(Qt::cyan);
			if ( (i++ % 2 ) ) item->setForeground(brush);
			_programmeList->addItem(item);
			// if its a selected programme put a tick next to it
			if ( _programmeSetForAction.find(keyForInsert) != _programmeSetForAction.end() )
				item->setIcon(QIcon(_settingsWindow->getIconPath() + "tick.png"));
			_channelDataMap[keyForInsert] = data;
			dataAvailable = true;
		}
	}
	// this means that data is not available locally in channelDataMap 
	if ( !dataAvailable )
	{
		if ( _searchMode )	showInfoLabel( QString("Have you clicked on Search already ? If yes then\nRequested programme title does not exist in the available data for the channel.\nIn case the data is old you could clear the data cache from the menu and try again.") );
		else				showInfoLabel( QString("Have you clicked on GO already ? If yes then\nNo data available for this channel !\nIn case the data is old you could clear the data cache from the menu and try again.") );
	}

}

// SLOT to show the selected programme details like description etc
void MainWindow::showProgrammeDetailWidget()
{
	_clearButton->hide();
	_programmeDetailButton->hide();
	_goButton->hide();
	_channelSelector->hide();
	_dateSelector->hide();
	// clear previous programme selection data
	_programmeDetail->clear();
	_prevChannelButton->hide();
	_nextChannelButton->hide();

	_programmeList->hide();  // hide list

	_programmeDetailLabel->show();
	_clearProgButton->show();
	_programmeDetail->show();  // show detail
	_backToProgrammeListButton->show();

	foreach(QString key, _programmeSetForAction)
	{
		_programmeDetail->setTextColor(QColor(Qt::darkGreen));
		_programmeDetail->append(_channelDataMap[key].channelId.toUpper());
		_programmeDetail->setTextColor(QColor(Qt::blue));
		_programmeDetail->append(_channelDataMap[key].title);
		_programmeDetail->setTextColor(QColor(Qt::red));
		_programmeDetail->append(_channelDataMap[key].sdate + "  ::  " + _channelDataMap[key].stime.left(5) +  "-" + _channelDataMap[key].etime.left(5));
		_programmeDetail->setTextColor(QColor(Qt::black));
		_programmeDetail->append(_channelDataMap[key].desc);
		_programmeDetail->append("");
		_programmeDetail->moveCursor(QTextCursor::Start);
	}
}

// show next selected channel
void MainWindow::showNextChannel()
{
	_prevChannelButton->show();

	++_channelIndex;
	showNextPrevChannelButtons();

	// shows the data for the channel if available for selected date
	showProgrammeListWidget(false);
}

// show previous selected channel
void MainWindow::showPrevChannel()
{

	_nextChannelButton->show();

	--_channelIndex;
	showNextPrevChannelButtons();

	// shows the data for the channel if available for selected date
	showProgrammeListWidget(false);
}

// make sure we are showing the next/prev arrows correctly
// also set the channel drop down button text to current selected channel
void MainWindow::showNextPrevChannelButtons()
{
	// change the text on the selector depending on the programme list you are viewing
	if ( _channelsSelected.count() != 0 )
	{
		_channelSelector->setValueText(_channelsSelected[_channelIndex]);
		_currentChannel = _channelSelector->valueText();
	}

    // show the arrows only when we have more than 1 channel
	if ( _channelsSelected.count() > 1 )
	{
		_prevChannelButton->setDisabled(false);
		_nextChannelButton->setDisabled(false);
	}

    // disable next when we have reached the last channel
	if( _channelIndex == _channelsSelected.count()-1 )
	{
		_nextChannelButton->setDisabled(true);
	}

    // disable prev when we have reached the first channel
	if( _channelIndex == 0 )
	{
		_prevChannelButton->setDisabled(true);
	}
}

// this adds the selected programs to the action list
// action list is used when applying action ( remind/record )
void MainWindow::addProgrammeActionList(QListWidgetItem* programme)
{
	QString key = _titleDisplayMap[programme->text()];

	// on clicking this slot is called so
	// if the program exists then remove it ( as user deselected) else add it(user selected)
	if(_programmeSetForAction.find(key) == _programmeSetForAction.end()) 
	{
		//qDebug() << "Selecting " << key;
		_programmeSetForAction.insert(key);
		// if its a selected programme put a tick next to it
		programme->setIcon(QIcon(_settingsWindow->getIconPath() + "tick.png"));
	}
	else
	{
		//qDebug() << "De Selecting " << key;
		_programmeSetForAction.remove(key);
		programme->setIcon(QIcon());
	}
}

// show the record action we are doing to the user
// currently only available to UK
void MainWindow::showRecordMessage()
{
    if ( _settingsWindow->getRegion() != "UK" && _settingsWindow->getRegion() != "IE")
    {
		QMaemo5InformationBox::information(this, "<font color=#000000>Sorry this functionality is only supported by UK/IE at the moment, if you are aware of any support for your region, please let me know either in my thread talk.maemo or send me an email</font>" , QMaemo5InformationBox::DefaultTimeout);

        return;
    }

	QString info1 = "Sent below message to the SMS app for recipient SKY to remote record\n";
	QString info2 = "You will be charged as per SKY UK rates, Please check their website for details\n";
	QString info3 = "\nEnsure you send that SMS and then click DONE here, else the next message(if any) will overwrite it in the SMS app\n\n";

	QMessageBox msgBox;
	foreach(QString progKey, _programmeSetForAction)
	{
		QString sdate = QDate::fromString(_channelDataMap[progKey].sdate, Qt::TextDate).toString("dd/MM");
		//qDebug() << _channelDataMap[progKey].stime;
		QString stime = QTime::fromString(_channelDataMap[progKey].stime, "HH:mm:ss").toString("HH:mm");
		QString recordMsg = _channelDataMap[progKey].title + "." +_channelDataMap[progKey].channelId.replace('_'," ") + "." + sdate + "." + stime;

		QStringList dataToSender;
		dataToSender << recordMsg << "61759";

		//qDebug() << "no. of programmes = " << _programmeSetForAction.count();
		
		_infoapi->sendRecordMsg(dataToSender);

		QMessageBox::StandardButton reply;
		reply = QMessageBox::information(this, "PLEASE READ THIS WARNING", info1+info2+info3+recordMsg);
	}

	clearProg();  // clear selected programmes
}

// SLOT to update the calendar DB and set the reminder
void MainWindow::showRemindMessage()
{
	initCalendar();

	foreach(QString progKey, _programmeSetForAction)
	{
		QDateTime progSTime = QDateTime::fromString(_channelDataMap[progKey].sdate + _channelDataMap[progKey].stime, "ddd MMM d yyyyHH:mm:ss");
		QDateTime progETime = QDateTime::fromString(_channelDataMap[progKey].edate + _channelDataMap[progKey].etime, "ddd MMM d yyyyHH:mm:ss");

		//qDebug() << "Alarm Start Time : " << progSTime << " Alarm End time : " << progETime;

		QString remindTitle = _channelDataMap[progKey].title + " on channel " + _channelDataMap[progKey].channelId;
		QString remindMessage = "This is a reminder of your favourite show from the SHOWTIME APP on the Nokia N900";

		if ( _settingsWindow->getRemindType() == "EVENT" )	
		{
			_reminderManager->addEvent("SHOWTIME", progSTime, progETime, remindTitle, remindMessage, 1);
		}
        else if ( _settingsWindow->getRemindType() == "ALARM" )	
		{
			_reminderManager->addAlarmEvent("SHOWTIME", progSTime, remindTitle, progSTime, progETime, remindMessage);
		}

	}

	if(_programmeSetForAction.count() != 0)
		QMaemo5InformationBox::information(this, "<font color=#000000><b>Your SHOWTIME calendar has been updated with your selected programmes</b></font>" , QMaemo5InformationBox::DefaultTimeout);

	clearProg();  // clear selected programmes
}

// set the app widgets to landscape mode
void MainWindow::setLandscape()
{
	initLandscape();
	setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
	setCentralWidget(_landscapeMainWidget);

	_landscapeOrientation = true;
	_programmeList->clear(); // have to do this as we are doing some manual formatting to display the programmes and times

	showProgrammeListWidget();
}

// set the app widgets to portrait mode
void MainWindow::setPortrait()
{
	initPortrait();
	setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
	setCentralWidget(_portraitMainWidget);

	_landscapeOrientation = false;
	_programmeList->clear(); // have to do this as we are doing some manual formatting to display the programmes and times

	showProgrammeListWidget();
}

// init the landscape layout
void MainWindow::initLandscape()
{
	/////////////////////////////////////////////
	// the layout definitions
	/////////////////////////////////////////////
	_lh1layout = new QHBoxLayout;
	_lv1layout = new QVBoxLayout;
	_lv2layout = new QVBoxLayout;
	_landscapeLayout = new QHBoxLayout;
	_landscapeMainWidget = new QWidget(this);
	
	// define the selection panel layout	
	_lh1layout->addWidget(_searchInput);
	_lh1layout->addWidget(_programmeDetailLabel);
	_lh1layout->addWidget(_dateSelector);
	_lh1layout->addWidget(_recordButton);
	_lh1layout->addWidget(_remindButton);
	_lh1layout->addWidget(_channelSelector);
	_lh1layout->addWidget(_clearProgButton);
	_lh1layout->addWidget(_backToProgrammeListButton);

	// define the action panel layout	
	_lv2layout->addWidget(_goButton);
	_lv2layout->addWidget(_programmeDetailButton);
	_lv2layout->addWidget(_clearButton);
	_lv2layout->addWidget(_nextChannelButton);
	_lv2layout->addWidget(_prevChannelButton);

	// define the data panel layout	
	_lv1layout->addLayout(_lh1layout);
	_lv1layout->addWidget(_programmeDetail);
	_lv1layout->addWidget(_programmeList);
	_lv1layout->addWidget(_progressLabel);
	_lv1layout->addWidget(_progress);
	_lv1layout->addWidget(_infoLabel);

	// define the main layout	
	_landscapeLayout->addLayout(_lv1layout);
	_landscapeLayout->addLayout(_lv2layout);

	_landscapeMainWidget->setLayout(_landscapeLayout);
}

// init the portrait layout
void MainWindow::initPortrait()
{
	/////////////////////////////////////////////
	// the layout definitions
	/////////////////////////////////////////////
	_ph1layout = new QHBoxLayout;
	_ph2layout = new QHBoxLayout;
	_ph3layout = new QHBoxLayout;
	_ph4layout = new QHBoxLayout;
	_pv1layout = new QVBoxLayout;
	_portraitLayout = new QHBoxLayout;
	_portraitMainWidget = new QWidget(this);
	
	// channel selector with record
	_ph1layout->addWidget(_programmeDetailLabel); // this is available in programme detail mode only
	_ph1layout->addWidget(_channelSelector);
	_ph1layout->addWidget(_recordButton);

	// date selector with remind
	_ph2layout->addWidget(_searchInput); // this is available only in the search option
	_ph2layout->addWidget(_dateSelector);
	_ph2layout->addWidget(_remindButton);
	_ph2layout->addWidget(_clearProgButton); // this is available in programme detail mode only
	_ph2layout->addWidget(_backToProgrammeListButton);// this is available in programme detail mode only


	// define the data panel layout	
	_ph3layout->addWidget(_programmeDetail);// this is available in programme detail mode only
	_ph3layout->addWidget(_programmeList);

	// define the action panel layout	
	_ph4layout->addWidget(_clearButton);
	_ph4layout->addWidget(_programmeDetailButton);
	_ph4layout->addWidget(_goButton);
	_ph4layout->addWidget(_prevChannelButton);
	_ph4layout->addWidget(_nextChannelButton);

	_pv1layout->addLayout(_ph1layout);
	_pv1layout->addLayout(_ph2layout);
	_pv1layout->addLayout(_ph3layout);
	_pv1layout->addWidget(_progressLabel);
	_pv1layout->addWidget(_progress);
	_pv1layout->addWidget(_infoLabel);
	_pv1layout->addLayout(_ph4layout);

	// define the main layout	
	_portraitLayout->addLayout(_pv1layout);

	_portraitMainWidget->setLayout(_portraitLayout);
}


MainWindow::~MainWindow()
{
}

// slot to show info about showtime
void MainWindow::showAbout()
{
	QString about = QString("<font color=#0000ff>A global TV channel list viewer with calendar reminder integration. </font>") +
 "This app downloads xmltv data depending on the region selected and displays " +
 "the tv channel listings , in most cases for an entire week. The user can then " +
 "view details of the programme and select them to be added to the N900 calendar.";

	QString title = QString("SHOWTIME v1.2");
	QMessageBox::about(this, title, about);
}

// slot to encourage me more ;-)
void MainWindow::showDonate()
{
	 QDesktopServices::openUrl(QUrl("https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=rkmaemo@gmx.com&lc=GB&item_name=SHOWTIME Maemo&currency_code=GBP&bn=PP%2dDonationsBF%3abtn_donateCC_LG%2egif%3aNonHosted"));
}

// slot to set/display your favourite channel selection
void MainWindow::showFavourite()
{
	_favouriteWindow = new FavouriteWindow(this);
	_favouriteWindow->setAttribute(Qt::WA_Maemo5StackedWindow);

	connect(_favouriteWindow, SIGNAL(clicked()), this, SLOT(reInitFavouriteSettings()));

	_favouriteWindow->show();
}

// slot to set/display your region
// resets all data to start
void MainWindow::showRegion()
{
	_regionWindow = new RegionWindow(this);
	_regionWindow->setAttribute(Qt::WA_Maemo5StackedWindow);

	connect(_regionWindow, SIGNAL(clicked()), this, SLOT(reInitInfoAccess()));

	_regionWindow->show();

}

// slot to set/display settings 
void MainWindow::showSettings()
{
	_miscWindow = new MiscSettingsWindow(this);
	_miscWindow->setAttribute(Qt::WA_Maemo5StackedWindow);

	connect(_miscWindow, SIGNAL(clicked()), this, SLOT(reInitMiscSettings()));

	_miscWindow->show();
}

// slot to search for the title string in the selected channel for the week
void MainWindow::switchDaily()
{
	_programmeList->clear();
	clearProg(); // keys are different in different modes
	_searchMode = false;
	_currentListingMode = false;
	showProgrammeListWidget();
	
	_goButton->setIcon(QIcon(_settingsWindow->getIconPath() + "go.png"));
}

void MainWindow::switchSearch()
{
	_programmeList->clear();
	clearProg(); // keys are different in different modes
	_searchMode = true;
	_currentListingMode = false;
	showProgrammeListWidget();

	_goButton->setIcon(QIcon(_settingsWindow->getIconPath() + "search.png"));
}

void MainWindow::switchCurrentListings()
{
	showCurrentListingButtons();
	_programmeList->clear();
	clearProg(); // keys are different in different modes
	_searchMode = false;
	_currentListingMode = true;
	if ( _channelsSelected.size() == 0 )
		QMaemo5InformationBox::information(this, "<font color=#000000>Go back to Daily Mode and select some channels to see the current listings.</font>" 
																												, QMaemo5InformationBox::DefaultTimeout);
	go();
}

// SLOT::reloads the favourite settings
void MainWindow::reInitFavouriteSettings()
{
	initAppSettings();

	_searchInput->clear();
	_searchInput->addItems(_settingsWindow->getSearchTitles(_settingsWindow->getRegion(), _currentChannel));
	_searchInput->lineEdit()->setText(_searchStringPerChannelMap[_currentChannel]);

	_favouriteWindow->close();
}

// SLOT::reloads the settings , clears all data
// and starts afresh with newly selected region
void MainWindow::reInitInfoAccess()
{
	initAppSettings();

	//qDebug() << "Region has been set to " << _settingsWindow->getRegion();
	clearAll();
	// disconnect the old slot
	disconnect(_infoapi, SIGNAL(finished(bool)), this, SLOT(finished(bool)));
	_infoapi = new InfoAccess(this, _settingsWindow->getRegion());
	// connect the new slot
	connect(_infoapi, SIGNAL(finished(bool)), this, SLOT(finished(bool)));
	initListModels();

	_regionWindow->close();
}

// SLOT::reloads the misc settings
void MainWindow::reInitMiscSettings()
{
	initAppSettings(); // in case ALARM type is changed
	initOrientation(); // in case Orientation is changed

	// IN CASE ICON SET IS UPDATED
	_recordButton->setIcon(QIcon(_settingsWindow->getIconPath() + "record.png"));
	_remindButton->setIcon(QIcon(_settingsWindow->getIconPath() + "remind.png"));
	_backToProgrammeListButton->setIcon(QIcon(_settingsWindow->getIconPath() + "back.png"));
	_clearProgButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));
	if ( _searchMode )
		_goButton->setIcon(QIcon(_settingsWindow->getIconPath() + "search.png"));
	else
		_goButton->setIcon(QIcon(_settingsWindow->getIconPath() + "go.png"));
	_clearButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));
	_programmeDetailButton->setIcon(QIcon(_settingsWindow->getIconPath() + "detail.png"));
	_nextChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "next.png"));
	_prevChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "prev.png"));

	_miscWindow->close();
}

// SLOT::this function is called by the selectChannel slot when selected channel is FAVOURITES
void MainWindow::loadFavourites()
{
	QStringList favourites = _settingsWindow->getFavourites(_settingsWindow->getRegion());

	if (favourites.isEmpty())
	{
		//qDebug() << "No favourites present for the selected region";
		_channelSelector->setValueText(_currentChannel);
		return;
	}

	foreach(_currentChannel, favourites)
	{
		// if the channel does not exist in the selected list then add it 
		// and set the index to current channel
		if ( ! _channelsSelected.contains(_currentChannel) )
		{
			//qDebug() << "Adding Favourite Channel to list" << _currentChannel;
			_channelsSelected << _currentChannel;
			_channelSelector->setValueText(_currentChannel);
			_channelIndex = _channelsSelected.count() - 1;
		}
	}

    // SHOW the prog list if in memory already
	showProgrammeListWidget();
}

void MainWindow::toggleBusy(bool isBusy)
{
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, isBusy);
}

void MainWindow::showHelp()
{
	QLabel *imageLabel = new QLabel;
	QImage image("/opt/showtime/data/helpfiles/showtime_help.gif");
	imageLabel->setPixmap(QPixmap::fromImage(image));

	QMainWindow *_helpWindow = new QMainWindow ( this );
	_helpWindow->setAttribute(Qt::WA_Maemo5StackedWindow);
	_helpWindow->setAttribute(Qt::WA_DeleteOnClose);
	_helpWindow->setWindowTitle("HELP WINDOW (scroll to view)");

	QScrollArea *_scroll= new QScrollArea;
	_scroll->setWidget(imageLabel);

	_helpWindow->setCentralWidget(_scroll);
	_helpWindow->show();
}

void MainWindow::clearCacheData()
{
	bool result = true;
	QDate downloadedDate = _infoapi->getDownloadDate();

	if ( downloadedDate.daysTo(QDate::currentDate()) < 3 )
	{
		QMessageBox::StandardButton ret;
		QString question = QString("You are about to clear your data cache and memory completely and start all fresh again.\n") +
						"The data cache you have is only " + QString::number(downloadedDate.daysTo(QDate::currentDate())) + " day(s) old.\n" +
						"Maybe you should wait till its atleast 3 days old so the chances that the source has the data is higher.\n" +
						"If you still wish to proceed click YES else click NO to use available data";
		ret = QMessageBox::question(this, tr("Early Download Warning !"), question ,QMessageBox::Yes | QMessageBox::No);
		if (ret == QMessageBox::Yes)
			result = true;
		else if (ret == QMessageBox::No)
			result = false;
	}
	else
	{
		result = true; // if its more than 3 days old no need to ask the user just download :-)
	}

	if ( result )
	{
		_infoapi->clearRegionData();
		clearAll();
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////// SEARCH TITLE FUNCTIONALITY //////////////////////////////////////
///////////////////////////////////////// introduced in version 1.0-8 //////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// The method that calls the infoaccess class to download/read xmltv weekly data and calls the search function
void MainWindow::goSearch(QDate date)
{
	if (_channelSelector->valueText() == "" || _searchInput->currentText() == "")
		return;

	//qDebug() << "In GoSearch : " << _channelSelector->valueText() << " " << date;

	// store the current search string for the channel so when you press the arrow keys your
	// search is also remembered :-)	
	_searchStringPerChannelMap[_currentChannel] = _searchInput->currentText();
	_currentSearchString = _searchInput->currentText();

	DownloadInput d;
	d.requestChannel = _currentChannel;
	d.requestDate = date;

	if ( ! _channelDataAvailableMap[_currentChannel].contains(date.toString("yyyyMMdd")) )
	{
		// in case of single XMLTV files we dont want to download files greater than max date avaialble for the channel
		// we simply search in the avaialable data, the user can switch mode to see what data is available
		if ( !_infoapi->isSingleXMLTVFile() || (date == QDate::currentDate() || date.toString("yyyyMMdd") <= _infoapi->metadata()[_currentChannel].maxDate ) )
		{
			toggleBusy(true);
			_infoapi->getChannelInfo(d);
		}
		else
		{
			//qDebug() << "the Date requested was more than the max date of channel";
			finished();
		}
	}
	else
	{
		//qDebug() << "channel data already in memory";
		finished();
	}
}

// called from a callback (finished) when the XMLTV info required is downloaded/read
void MainWindow::finishedSearch()
{
	_infoLabel->hide();

	// if we have finished downloaded the files for the week	
	if ( _dayCount == 6 )
	{
		_dayCount = 0;
		toggleBusy(false);
		showProgrammeListWidget(true);
	}
	else // keep downloading the files
	{
		++_dayCount;
		goSearch(QDate::currentDate().addDays(_dayCount));
	}

	return;
}

// SLOT : to append the favourite search strings to the current
void MainWindow::addSearchString(QString text)
{
	if ( _currentSearchString.simplified() == "" ) 
		_currentSearchString += text;
	else
		_currentSearchString += " + "  + text;

	_searchInput->lineEdit()->setText(_currentSearchString);
}

// SLOT : to set the current search string
void MainWindow::setCurrentSearchString()
{
	_currentSearchString = _searchInput->currentText();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
///////////////////////////////// CURRENT LISTINGS CHANGES after version 1.1-4 /////////////////////////////

void MainWindow::showCurrentListingButtons()
{
	_searchInput->hide();
	_dateSelector->hide();
	_recordButton->hide();
	_remindButton->hide();
	_channelSelector->hide();
	_goButton->hide();
	_prevChannelButton->hide();
	_nextChannelButton->hide();
	_programmeDetail->hide();
	_programmeDetailLabel->hide();
	_clearProgButton->hide();
	_backToProgrammeListButton->hide();
	_programmeList->show();
	_clearButton->show();	
	_programmeDetailButton->show();
}

void MainWindow::finishedCurrent()
{
	_infoLabel->hide();

	//qDebug() << "recvd current listings for " << _channelsSelected[_channelIndex] << " channelIndex=" << _channelIndex;
	// if we have finished downloaded the files for the week	

	if ( _channelIndex-- == 0 )
	{
		_channelIndex = 0;
		//qDebug() << "all current listings in memory now";
		toggleBusy(false);
		_progressLabel->hide();
		_progress->hide();
		showProgrammeListWidget(true);
	}
	else // keep downloading the files
	{
		emit downloadIndex(_channelsSelected.size() - _channelIndex);

		_channelSelector->setValueText(_channelsSelected[_channelIndex]);
		go();
	}

	return;
}

void MainWindow::showCurrentProgrammeListWidget(bool dataRequested)
{
	_programmeList->clear();
	_titleDisplayMap.clear();
	_infoLabel->hide();

	showCurrentListingButtons();

	// store the data in memory for all the favourite channels
   	QList<ChannelData> completeData;
	foreach(QString channel , _channelsSelected)
		completeData.append( (_infoapi->results())[channel] );

	bool dataAvailable = false;

	foreach(ChannelData data, completeData)
	{
		QDateTime progSDateTime = QDateTime::fromString(data.sdate + data.stime, "ddd MMM d yyyyHH:mm:ss");
		QDateTime progEDateTime = QDateTime::fromString(data.edate + data.etime, "ddd MMM d yyyyHH:mm:ss");
		QString title = data.title;
		QString sdate = data.sdate;
		QString stime = data.stime;
		stime.truncate(5);
		QString etime = data.etime;
		etime.truncate(5);
		QString time = stime + " - " + etime;
		QString channelname = data.channelId;

		QString keyForInsert = title + sdate + time;
		breakTitle(title);
		QString displayLine;
		if ( _landscapeOrientation )
		{
			displayLine = title + "\n" + "Channel: " + channelname + "\n\t\t\t\t\t\t" + time;
		}
		else
		{
			displayLine = title + "\n" + "Channel: " + channelname + "\n\t\t\t" + time;
		}

		// if th programme is today and started 1hour ago and is still on 
		// and the programme start time is in the next 1 hour
		// and the programme end time is greater than current time then show the programme as its currently ON.
		QFont itemFont;
		itemFont.setPixelSize(17);
		if ( data.sdate == QDate::currentDate().toString(Qt::TextDate) && 
			progSDateTime > QDateTime::currentDateTime().addSecs(-3601) && progSDateTime < QDateTime::currentDateTime().addSecs(3601) && 
			progEDateTime > QDateTime::currentDateTime())
		{
			_titleDisplayMap[displayLine]=keyForInsert;

			QListWidgetItem *item = new QListWidgetItem(displayLine, _programmeList);	
			item->setFont(itemFont);
			QBrush brush(Qt::cyan);
			if ( (i++ % 2 ) ) item->setForeground(brush);
			_programmeList->addItem(item);
			// if its a selected programme put a tick next to it
			if ( _programmeSetForAction.find(keyForInsert) != _programmeSetForAction.end() )
				item->setIcon(QIcon(_settingsWindow->getIconPath() + "tick.png"));
			_channelDataMap[keyForInsert] = data;
			dataAvailable = true;
		}
	}
	// this means that data is not available locally in channelDataMap and also it hasnt been
	// requested so it means you have to hit GO :-) !
	if ( !dataAvailable )
	{
		showInfoLabel( QString("No data available for this channel !\nYou could try to clear the Data Cache from the menu if its old and try again") );
	}

}

void MainWindow::breakTitle(QString& title)
{
	int position;
	if ( _landscapeOrientation )
		position = 50;
	else
		position = 40;

	// break the huge title in order to prevent horizontal scrolling that makes scrolling slow and difficult
	if ( title.length() > position )
	{
		QString s1 =  title.left(position);
		QString s2 =  title.right(title.length() - position);
		title = s1 + "\n" + s2;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////############## END OF SHOWTIM ###############////////////////////////////////////////////////
