#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QStringList>
#include <QXmlStreamReader>
#include <QXmlStreamAttributes>
#include <QDebug>
#include <QWidget>
#include <QMaemo5InformationBox>
#include <QFile>
#include <QProcess>
#include <QDateTime>
#include <QTime>
#include <QMap>
#include <QDate>
#include <QDir>

#include "ukfiles.h"
#include "aufiles.h"
#include "nzfiles.h"
#include "sefiles.h"
#include "defiles.h"
#include "frfiles.h"
#include "esfiles.h"
#include "itfiles.h"
#include "hrfiles.h"
#include "sifiles.h"
#include "dkfiles.h"
#include "nlfiles.h"
#include "befiles.h"
#include "nofiles.h"
#include "rufiles.h"
#include "myfiles.h"
#include "zafiles.h"
#include "plfiles.h"
#include "fifiles.h"
#include "iefiles.h"
#include "infoaccess.h"

struct InfoAccessPrivate {
	QMap<QString, QList<ChannelData> > results;  // result per channel
	QMap<QString, ChannelMetaData> metadata;  // channel meta data
	QUrl apiUrl;
	QNetworkAccessManager *manager;
};

InfoAccess::InfoAccess( QObject *parent , QString region)
        : QObject( parent ) 
{
	_region = region;

	if ( _region == "UK" ) { dataDownloader = new UKFiles(this); }
	else if ( _region == "AU" ) { dataDownloader = new AUFiles(this); }
	else if ( _region == "NZ" ) { dataDownloader = new NZFiles(this); }
	else if ( _region == "SE" ) { dataDownloader = new SEFiles(this); }
	else if ( _region == "DE" ) { dataDownloader = new DEFiles(this); }
	else if ( _region == "FR" ) { dataDownloader = new FRFiles(this); }
	else if ( _region == "ES" ) { dataDownloader = new ESFiles(this); }
	else if ( _region == "IT" ) { dataDownloader = new ITFiles(this); }
	else if ( _region == "HR" ) { dataDownloader = new HRFiles(this); }
	else if ( _region == "SI" ) { dataDownloader = new SIFiles(this); }
	else if ( _region == "DK" ) { dataDownloader = new DKFiles(this); }
	else if ( _region == "NL" ) { dataDownloader = new NLFiles(this); }
	else if ( _region == "BE" ) { dataDownloader = new BEFiles(this); }
	else if ( _region == "NO" ) { dataDownloader = new NOFiles(this); }
	else if ( _region == "RU" ) { dataDownloader = new RUFiles(this); }
	else if ( _region == "MY" ) { dataDownloader = new MYFiles(this); }
	else if ( _region == "ZA" ) { dataDownloader = new ZAFiles(this); }
	else if ( _region == "PL" ) { dataDownloader = new PLFiles(this); }
	else if ( _region == "FI" ) { dataDownloader = new FIFiles(this); }
	else if ( _region == "IE" ) { dataDownloader = new IEFiles(this); }
	else { dataDownloader = new UKFiles(this); }  // default is UK

	d = new InfoAccessPrivate;
	d->manager = new QNetworkAccessManager( this );

	connect( d->manager, SIGNAL(finished(QNetworkReply*)), SLOT(finishedDownload(QNetworkReply*)) );
}

InfoAccess::~InfoAccess()
{
	delete d;
}

void InfoAccess::setUrl(QUrl url) 
{
	d->apiUrl = url;
}

QMap<QString, QList<ChannelData> > InfoAccess::results() const
{
	return d->results;
}

QMap<QString, ChannelMetaData>  InfoAccess::metadata() const
{
	return d->metadata;
}

QStringList InfoAccess::getChannelList() 
{
	return dataDownloader->getChannelList();
}

bool InfoAccess::getChannelInfo(DownloadInput& downloadInput)
{
	_currentChannel = downloadInput.requestChannel;
	bool didDownload = false;

	//qDebug() << "Requesting data for " << downloadInput.requestChannel << " on " << downloadInput.requestDate;

	// call the uk downloader for each channel and its requested dates
	dataDownloader->setInput(downloadInput);
	dataDownloader->initSettings();
	dataDownloader->checkDataStatus();
	//qDebug() << "The download URL is " << dataDownloader->getUrl();

	if( ! dataDownloader->getUrl().isEmpty() )
	{
		didDownload = true;
		setUrl(dataDownloader->getUrl());
		downloadXMLTV();
	}

	// write the final settings data to a file
	// so if the same channel is requested again then we have the latest settings
	// to indicate that we have the data already
	if (!didDownload)
		processXMLTV();

	return didDownload;
}

void InfoAccess::downloadXMLTV()
{
	//qDebug() << "Downloading Data : " << d->apiUrl;
	d->manager->get( QNetworkRequest(d->apiUrl) );
}

void InfoAccess::finishedDownload( QNetworkReply *reply )
{
	if ( reply->error() != QNetworkReply::NoError ) {
		//qDebug() << "finishedDownload() : Request failed, " << reply->errorString();
		emit finished(false);
		return;
	}

	dataDownloader->finishedDownload( _currentChannel,  reply );

	//qDebug() << "Status request succeeded";
	processXMLTV();
}

bool InfoAccess::processXMLTV()
{
	// sometimes there is no file generated for a channel so no files will be read, in that 
	bool filesRead = false;
	foreach(QString filename, dataDownloader->getFilelist())
	{
		filesRead = true;
		processXMLTV(filename);
	}

	// if there are no files or no programmes in the file then maxDate will be ""
	// for per day channel files this max date is ignored the settings are updated with _requestDate
	dataDownloader->updateSettingsFile(d->metadata[_currentChannel].maxDate);

	emit finished( true );
	return true;
}

bool InfoAccess::processXMLTV( QString filename )
{
	//qDebug() << "Processing " << filename;

	QFile source(filename);
	source.open((QIODevice::ReadOnly));

	QXmlStreamReader reader( &source );
	QString key;
	ChannelMetaData metadata;
	metadata.downloadedDate = QDate::currentDate().toString("yyyyMMdd");
	metadata.minDate = "99999999";
	metadata.maxDate = "";
	QString startDate = "UNSET";
	bool dataPresent = false; // to indicate if any programme data was present

	while ( !reader.atEnd() )
	{
		ChannelData data;

		QXmlStreamReader::TokenType tokenType = reader.readNext();
		//qDebug() << "Token: " << reader.tokenString() << "Name : " << reader.name();

		if ( tokenType == QXmlStreamReader::StartElement && reader.name() == "channel" )
		{
			QXmlStreamAttributes attributes = reader.attributes();
			key = _currentChannel; // the channel id in the xml is usually named differently
			tokenType = reader.readNext();
		}
		else if ( tokenType == QXmlStreamReader::StartElement  && reader.name() == "programme" )
		{

			QXmlStreamAttributes attributes = reader.attributes();

			key = _currentChannel; // the channel id in the xml is usually named differently
			data.channelId = _currentChannel; // the channel id in the xml is usually named differently

			QString tmpStr = (attributes.value("start")).toString();
			tmpStr.truncate(12);
			QDateTime datetime = QDateTime::fromString(tmpStr , "yyyyMMddHHmm");

			// sometimes the data is generated in another country so to display time properly we need the 
			// time zone difference between the regions
			// example for UK we get data from a belgium site which is 1hr ahead so we need to subtract 1 hr
			if ( dataDownloader->timeZoneOffset() != 0 )
				datetime = datetime.addSecs(3600*dataDownloader->timeZoneOffset());

			data.sdate = (datetime.date()).toString();
			data.stime = datetime.time().toString();

			if ( datetime.toString("yyyyMMdd") < metadata.minDate ) 
			{
				metadata.minDate = datetime.toString("yyyyMMdd");
			}
			if ( datetime.toString("yyyyMMdd") > metadata.maxDate ) 
			{
				metadata.maxDate = datetime.toString("yyyyMMdd");
				dataPresent = true;
			}
			// need only the first date in the file that represents the start date
			if ( startDate == "UNSET" ) 
				startDate = datetime.toString("yyyyMMdd");

			tmpStr = (attributes.value("stop")).toString();
			tmpStr.truncate(12);
			datetime = QDateTime::fromString(tmpStr, "yyyyMMddHHmm");

			// sometimes the data is generated in another country so to display time properly we need the 
			// time zone difference between the regions
			// example for UK we get data from a belgium site which is 1hr ahead so we need to subtract 1 hr
			if ( dataDownloader->timeZoneOffset() != 0 )
				datetime = datetime.addSecs(3600*dataDownloader->timeZoneOffset());

			data.edate = datetime.date().toString();
			data.etime = datetime.time().toString();

			tokenType = reader.readNext();

			while ( !reader.atEnd() && !(tokenType == QXmlStreamReader::EndElement  && reader.name() == "programme") )
			{
				//qDebug() << "Token" << int(tokenType);
				//qDebug() << "Token: " << reader.tokenString() << "Name : " << reader.name();
				if ( reader.name() == "title" && tokenType == QXmlStreamReader::StartElement )
				{
					reader.readNext();
					data.title = QString::fromUtf8(reader.text().toString().toUtf8().constData());
				}
				if ( reader.name() == "desc" && tokenType == QXmlStreamReader::StartElement )
				{
					reader.readNext();
					data.desc = QString::fromUtf8(reader.text().toString().toUtf8().constData());
				}
				if ( reader.name() == "category" && tokenType == QXmlStreamReader::StartElement )
				{
					reader.readNext();
					data.type = QString::fromUtf8(reader.text().toString().toUtf8().constData());
				}
	
				tokenType = reader.readNext();
			}

			bool progAlreadyPresent = false;
			foreach(ChannelData existingData, d->results[key])
			{
				bool channelPresent = (existingData.channelId == data.channelId);
				bool titlePresent = (existingData.title== data.title); 
				bool sdatePresent = (existingData.sdate== data.sdate);
				bool stimePresent = (existingData.stime== data.stime);
				progAlreadyPresent = channelPresent && titlePresent && sdatePresent && stimePresent;

				// add the programme only which is not already present as we are reading from multiple files.
				// if present then continue and dont proceed down to add the programme
				if ( progAlreadyPresent ) 
				{
					//qDebug() << "Prog present" << existingData.channelId << existingData.title << existingData.sdate << existingData.stime;
					break;
				}
			}	
			// add the programme only which is not already present as we are reading from multiple files.
			// add only data which was inside programme tags
			if ( !progAlreadyPresent ) 
			{
				(d->results)[key].append(data);
				//qDebug() << "ADDED" << key << data.channelId << data.title << data.sdate << data.stime << data.edate << data.etime;
			}
		}
	}

	(d->metadata)[key].minDate = metadata.minDate;
	(d->metadata)[key].maxDate = metadata.maxDate;
	(d->metadata)[key].downloadedDate = metadata.downloadedDate;
	if ( isSingleXMLTVFile() )
	{
		// if its a single XMLTV file then the available dates are everythign between min and max
		QDate minDate = QDate::fromString(metadata.minDate, "yyyyMMdd");
		QDate maxDate = QDate::fromString(metadata.maxDate, "yyyyMMdd");
		for(int i = 0; minDate.addDays(i) <= maxDate; ++i)
			(d->metadata)[key].availableDates << minDate.addDays(i).toString("yyyyMMdd"); 
	}
	else
	{
		// the start date per file for the per day channel files will be added to the list of avaialable dates
		// this is because we read all available per day files at once 
		(d->metadata)[key].availableDates << startDate;
	}

	source.close();
	return dataPresent;
}

bool InfoAccess::isSingleXMLTVFile()
{
	return dataDownloader->isSingleXMLTVFile();
}

void InfoAccess::sendRecordMsg(QStringList input)
{
	dataDownloader->sendRecordMsg(input);
}

QDate InfoAccess::getDownloadDate()
{
	    return dataDownloader->getDownloadDate();
}

void InfoAccess::clear()
{
	// clear the previous info
	(d->results).clear();
	(d->metadata).clear();
}
