#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QStringList>
#include <QXmlStreamReader>
#include <QXmlStreamAttributes>
#include <QDebug>
#include <QWidget>
#include <QMaemo5InformationBox>
#include <QFile>
#include <QProcess>
#include <QDateTime>
#include <QTime>
#include <QMap>
#include <QDate>
#include <QDir>

#include "ukfiles.h"
#include "aufiles.h"
#include "nzfiles.h"
#include "sefiles.h"
#include "defiles.h"
#include "esfiles.h"
#include "itfiles.h"
#include "hrfiles.h"
#include "sifiles.h"
#include "infoaccess.h"

struct InfoAccessPrivate {
	QMap<QString, QList<ChannelData> > results;  // result per channel
	QUrl apiUrl;
	QNetworkAccessManager *manager;
};

InfoAccess::InfoAccess( QObject *parent , QString region)
        : QObject( parent ) 
{
	_region = region;

	if ( _region == "UK" ) { dataDownloader = new UKFiles(this); }
	else if ( _region == "AU" ) { dataDownloader = new AUFiles(this); }
	else if ( _region == "NZ" ) { dataDownloader = new NZFiles(this); }
	else if ( _region == "SE" ) { dataDownloader = new SEFiles(this); }
	else if ( _region == "DE" ) { dataDownloader = new DEFiles(this); }
	else if ( _region == "ES" ) { dataDownloader = new ESFiles(this); }
	else if ( _region == "IT" ) { dataDownloader = new ITFiles(this); }
	else if ( _region == "HR" ) { dataDownloader = new HRFiles(this); }
	else if ( _region == "SI" ) { dataDownloader = new SIFiles(this); }
	else { dataDownloader = new UKFiles(this); }  // default is UK

	d = new InfoAccessPrivate;
	d->manager = new QNetworkAccessManager( this );

	connect( d->manager, SIGNAL(finished(QNetworkReply*)), SLOT(finishedDownload(QNetworkReply*)) );
}

InfoAccess::~InfoAccess()
{
	delete d;
}

void InfoAccess::setUrl(QUrl url) 
{
	d->apiUrl = url;
}

QMap<QString, QList<ChannelData> > InfoAccess::results() const
{
	return d->results;
}

QStringList InfoAccess::getChannelList() 
{
	return dataDownloader->getChannelList();
}

bool InfoAccess::getChannelInfo(DownloadInput& downloadInput)
{
	_currentChannel = downloadInput.requestChannel;
	bool didDownload = false;

	//qDebug() << "Requesting data for " << downloadInput.requestChannel << " on " << downloadInput.requestDate;

	// call the uk downloader for each channel and its requested dates
	dataDownloader->setInput(downloadInput);
	dataDownloader->initSettings();
	dataDownloader->checkDataStatus();
	//qDebug() << "The download URL is " << dataDownloader->getUrl();

	if( ! dataDownloader->getUrl().isEmpty() )
	{
		didDownload = true;
		setUrl(dataDownloader->getUrl());
		downloadXMLTV();
	}

	// write the final settings data to a file
	// so if the same channel is requested again then we have the latest settings
	// to indicate that we have the data already
	if (!didDownload)
		processXMLTV();

	return didDownload;
}

void InfoAccess::downloadXMLTV()
{
	//qDebug() << "Downloading Data : " << d->apiUrl;
	d->manager->get( QNetworkRequest(d->apiUrl) );
}

void InfoAccess::finishedDownload( QNetworkReply *reply )
{
	if ( reply->error() != QNetworkReply::NoError ) {
		//qDebug() << "finishedDownload() : Request failed, " << reply->errorString();
		emit finished(false);
		return;
	}

	QString filename = dataDownloader->finishedDownload( _currentChannel,  reply );

	//qDebug() << "Status request succeeded";
	dataDownloader->updateSettingsFile();

	processXMLTV();
}

bool InfoAccess::processXMLTV()
{
	foreach(QString filename, dataDownloader->getFilelist(_currentChannel))
	{
		//////////////////////////////////////////////////////////
		// DELETE ALL OLD FILES, the ones that are 7 days or older
		//////////////////////////////////////////////////////////
		QString filedatestr = filename.section("--",1,1);
		//qDebug() << filedatestr;
		QDate filedate = QDate::fromString(filedatestr, "ddMMyyyy");
		if ( QDate::currentDate() > filedate.addDays(2) )
		{
			//qDebug() << "Deleting old file " << filename;
			QFile::remove(filename);
		}
		else
		{
			processXMLTV(filename);
		}
	}

	emit finished( true );
	return true;
}

bool InfoAccess::processXMLTV( QString filename )
{
	//qDebug() << "Processing " << filename;

	QFile source(filename);
	source.open((QIODevice::ReadOnly));

	QXmlStreamReader reader( &source );
	QString key;

	while ( !reader.atEnd() )
	{
		ChannelData data;

		QXmlStreamReader::TokenType tokenType = reader.readNext();
		//qDebug() << "Token: " << reader.tokenString() << "Name : " << reader.name();

		if ( tokenType == QXmlStreamReader::StartElement && reader.name() == "channel" )
		{
			QXmlStreamAttributes attributes = reader.attributes();
			key = _currentChannel; // the channel id in the xml is usually named differently
			tokenType = reader.readNext();
		}
		else if ( tokenType == QXmlStreamReader::StartElement  && reader.name() == "programme" )
		{

			key = _currentChannel; // the channel id in the xml is usually named differently
			data.channelId = _currentChannel; // the channel id in the xml is usually named differently
			QXmlStreamAttributes attributes = reader.attributes();

			QString tmpStr = (attributes.value("start")).toString();
			tmpStr.truncate(12);
			QDateTime datetime = QDateTime::fromString(tmpStr , "yyyyMMddHHmm");
			data.sdate = (datetime.date()).toString();
			data.stime = datetime.time().toString();

			tmpStr = (attributes.value("stop")).toString();
			tmpStr.truncate(12);
			datetime = QDateTime::fromString(tmpStr, "yyyyMMddHHmm");
			data.edate = datetime.date().toString();
			data.etime = datetime.time().toString();

			tokenType = reader.readNext();

			while ( !reader.atEnd() && !(tokenType == QXmlStreamReader::EndElement  && reader.name() == "programme") )
			{
				//qDebug() << "Token" << int(tokenType);
				//qDebug() << "Token: " << reader.tokenString() << "Name : " << reader.name();
				if ( reader.name() == "title" && tokenType == QXmlStreamReader::StartElement )
				{
					reader.readNext();
					data.title = reader.text().toString();	
				}
				if ( reader.name() == "desc" && tokenType == QXmlStreamReader::StartElement )
				{
					reader.readNext();
					data.desc = reader.text().toString();	
				}
				if ( reader.name() == "category" && tokenType == QXmlStreamReader::StartElement )
				{
					reader.readNext();
					data.type = reader.text().toString();	
				}
	
	
				tokenType = reader.readNext();
			}

			bool progAlreadyPresent = false;
			foreach(ChannelData existingData, d->results[key])
			{
				bool channelPresent = (existingData.channelId == data.channelId);
				bool titlePresent = (existingData.title== data.title); 
				bool sdatePresent = (existingData.sdate== data.sdate);
				bool stimePresent = (existingData.stime== data.stime);
				progAlreadyPresent = channelPresent && titlePresent && sdatePresent && stimePresent;

				// add the programme only which is not already present as we are reading from multiple files.
				// if present then continue and dont proceed down to add the programme
				if ( progAlreadyPresent ) 
				{
					//qDebug() << "Prog present" << existingData.channelId << existingData.title << existingData.sdate << existingData.stime;
					break;
				}
			}	
			// add the programme only which is not already present as we are reading from multiple files.
			// add only data which was inside programme tags
			if ( !progAlreadyPresent ) 
			{
				(d->results)[key].append(data);
				//qDebug() << "ADDED" << key << data.channelId << data.title << data.sdate << data.stime << data.edate << data.etime;
			}
		}
	}
	
	source.close();
	return true;
}

void InfoAccess::sendRecordMsg(QStringList input)
{
	dataDownloader->sendRecordMsg(input);
}

void InfoAccess::clear()
{
	// clear the previous info
	(d->results).clear();
}
