/* Shopper
 * Copyright (C) 2008 David Greaves <david@dgreaves.com>
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

//#define DEBUG_SHOPPER 1
#include "shopper.h"           // automake, i8n, gettext
#include <QXmlAttributes>
#include <QXmlParseException>
#include <QXmlSimpleReader>
#include <QXmlInputSource>
#include <QXmlStreamWriter>
#include <iostream>
#include <sstream>
#include <string>
#include <stdexcept>
#include <list>
#include <set>
#include <signal.h>
#include <cstdlib>
#include "shopperList.h"

using namespace std;

namespace Shopper
{
	void abort (QString msg){
		std::cerr << msg;
		exit(1); // FIXME
	}

	ListParser::ListParser(List *list):
		l(list), c(0), i(0),
		active_cat_id(0)
	{
	}
	
	void ListParser::start_list_el(const QXmlAttributes & attrs)
	{
		// zero the id counters
		Category::id_master = 0;
		Item::id_master = 0;
		QString tmp(QLatin1String("name"));
		for (int at = 0; at != attrs.count(); ++at) {
			QString at_name = attrs.localName(at);
			QString at_value = attrs.value(at);
			if (at_name.toLower() == QLatin1String("name")) {
				l->name = at_value;
			} else if (at_name.toLower() == QLatin1String("state")) {   // save as int? or human readable?
				bool ok;
				l->state = sState(at_value.toInt(&ok));
				if (!ok) {
					abort(QLatin1String("Converting state failed\n"));
				}
			} else {
				DEBUG("<list> Unknown attribute : " << at_name << "='" <<at_value<<"'\n");
			}
		}
	}
	void ListParser::start_cat_el(const QXmlAttributes & attrs)
	{
		if (l == 0) {
			std::cerr << "<category> Not inside <list>\n";
			exit(1); // FIXME throw?
		}
		c = new Category();
		for (int at = 0; at != attrs.count(); ++at) {
			QString at_name = attrs.localName(at);
			QString at_value = attrs.value(at);
			if (at_name.toLower() == QLatin1String("name")) {
				c->name = at_value;
			} else if (at_name.toLower() == QLatin1String("active")) { // is this the category active?
				if (at_value.toLower() == QLatin1String("true") or at_value == QLatin1String("1"))
					l->active_category = c;
			} else {
				DEBUG("<category> Unknown attribute : " << at_name << "='" <<at_value<<"'\n");
			}
		}
		l->add(*c);
	}
	void ListParser::start_item_el(const QXmlAttributes & attrs)
	{
		if (c == 0) {
			std::cerr << "<item> Not inside <category>\n";
			exit(1); // FIXME
		}

		i = new Item();
		i->category=c; // current category
		for (int at = 0; at != attrs.count(); ++at) {
			QString at_name = attrs.localName(at);
			QString at_value = attrs.value(at);
			if (at_name.toLower() == QLatin1String("desc")) {
				i->desc = at_value;
			} else if (at_name.toLower() == QLatin1String("note")) {
				i->note = at_value;
			} else if (at_name.toLower() == QLatin1String("wanted")) {
				i->set_wanted(at_value.toLower() == QLatin1String("true") or at_value == QLatin1String("1"));
			} else if (at_name.toLower() == QLatin1String("bought")) {
				i->set_bought(at_value.toLower() == QLatin1String("true") or at_value == QLatin1String("1"));
			} else {
				DEBUG("<item> Unknown attribute : " << at_name << "='" <<at_value<<"'\n");
			}
		}
		l->add(*i);
	}

	bool ListParser::startElement (const QString & namespaceURI,
								   const QString & el,
								   const QString & qName,
								   const QXmlAttributes & attrs)
	{
		Q_UNUSED(namespaceURI)
		Q_UNUSED(qName)
		DEBUG("adding " << el << "\n");
		if (el.toLower() == QLatin1String("list")) {
			start_list_el(attrs);
		} else if (el.toLower() == QLatin1String("category")) {
			start_cat_el(attrs);
		} else if (el.toLower() == QLatin1String("item")) {
			start_item_el(attrs);
		} else {
			throw ;
		}
		return true;
	}

	bool ListParser::endElement (const QString & namespaceURI,
								 const QString & el,
								 const QString & qName)
	{
		Q_UNUSED(namespaceURI)
		Q_UNUSED(qName)
		DEBUG("done " << el << "\n");
		if (el.toLower() == QLatin1String("list")) {
			l->resequence(); // Ensure the id's are sensible FIXME : not needed?
			l = 0; // No current list
		} else if (el.toLower() == QLatin1String("category")) {
			// add the created cat to list
			c->dbg();
			c->items.sort(cmpItem);
			c=0; // No current category
		} else if (el.toLower() == QLatin1String("item")) {
			// add the created item to list
			i->dbg();
			DEBUG("Assigned " << i->desc << " to category " << i->category->name << "\n");
			i = 0; // No current item
		} else {
			return false;
		}
		return true;
	}
	bool ListParser::fatalError ( const QXmlParseException & exception )
	{
		Q_UNUSED(exception)
		DEBUG("Markup error\n");
		return true;
	}

	bool ListParser::from_string(QString xml)
	{
		QXmlSimpleReader xmlReader;
		xmlReader.setContentHandler(this);
		xmlReader.setErrorHandler(this);
		QXmlInputSource source;
		source.setData(xml);
		bool ok = xmlReader.parse(&source);
		if (!ok)
		{
			std::cout << "Parsing failed:" << std::endl << xml << std::endl;
			return false;
		}
		DEBUG("Parsing EXIT\n");
		return true;
		}

////////////////////////////////////////////////////////////////
	XMLWriter::XMLWriter(const Shopper::List *list) :
		l(list)
	{}
		

	QString XMLWriter::write()
	{
		QString xml;
		QXmlStreamWriter out(&xml);
		out.setAutoFormatting(true);

		DEBUG("in write\n");
		out.writeStartDocument();
		out.writeStartElement(QLatin1String("list"));
		out.writeAttribute(QLatin1String("name"), l->name);
		out.writeAttribute(QLatin1String("state"), QString::number(l->state));
		for (List::pCategoryIter c = l->categories.begin(); c != l->categories.end(); ++c) {
			out.writeStartElement(QLatin1String("category"));
			if (l->active_category == *c)
				out.writeAttribute(QLatin1String("active"), QLatin1String("1"));
			out.writeAttribute(QLatin1String("name"), (*c)->name);
			out.writeAttribute(QLatin1String("id"), QString::number((*c)->id));
			for (Category::pItemIter i = (*c)->items.begin(); i != (*c)->items.end(); ++i) {
				out.writeStartElement(QLatin1String("item"));
				out.writeAttribute(QLatin1String("wanted"), (*i)->wanted?QLatin1String("1"):QLatin1String("0"));
				out.writeAttribute(QLatin1String("bought"), (*i)->bought?QLatin1String("1"):QLatin1String("0"));
				out.writeAttribute(QLatin1String("desc"), (*i)->desc);
				out.writeAttribute(QLatin1String("note"), (*i)->note);
				out.writeEndElement(); // /item
			}
		out.writeEndElement(); // /category
		}
		out.writeEndElement(); // /list
		out.writeEndDocument();
		DEBUG("done write\n");
		return xml;
	}
}
