/* Shopper
 * Copyright (C) 2008 David Greaves <david@dgreaves.com>
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

//#define DEBUG_SHOPPER 1
#include "mainwindow.h"
#include "shopper.h"           // automake, i8n, gettext
#include "ListImport.h"

#include "CatListModel.h"
#include "CategoryDialog.h"
#include "ItemDialog.h"
#include "ListDialog.h"
#include "SaveDialog.h"
#include "PreferencesDialog.h"
#include <QSettings>

// class MainWindow : public QMainWindow
MainWindow::MainWindow(QString file) :
	mylist(0),
	catCombo(0),
	lv(0),
	filename(file),
	fontsize(4),
	full_screen(false)
{
	// The constructor handles only 'one off' stuff - main menu,
	// buttons etc. Anything List specific is separated off to
	// handle_list() so it can be reused when opening new lists
	connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(stateChanged()));
	this->setAttribute(Qt::WA_Maemo5AutoOrientation, true);

	// File Actions
	QAction *fileAct = new QAction(tr("Save/Load/New"), this);
    connect(fileAct, SIGNAL(triggered()), this, SLOT(on_action_file()));

	// List Edit Actions
	QAction *editListAct = new QAction(tr("Make List"), this);
    connect(editListAct, SIGNAL(triggered()), this, SLOT(on_action_edit_list()));

	QAction *addItemAct = new QAction(QIcon("/opt/usr/share/icons/hicolor/64x64/apps/shopper/add.png"),
									  tr("Add Item"), this);
    addItemAct->setStatusTip(tr("Add an item to the list"));
    connect(addItemAct, SIGNAL(triggered()), this, SLOT(on_action_add_item()));

	QAction *manageCatAct = new QAction(QIcon("/opt/usr/share/icons/hicolor/64x64/apps/shopper/categories.png"),
										tr("Manage Categories"), this);
    manageCatAct->setStatusTip(tr("Manage the list of categories"));
    connect(manageCatAct, SIGNAL(triggered()), this, SLOT(on_action_manage_category()));

	// Mode switch Actions
	QAction *fullListAct = new QAction(QIcon("/opt/usr/share/icons/hicolor/64x64/apps/shopper/full-list.png"),
									   tr("Out Shopping: Full List"), this);
    fullListAct->setStatusTip(tr("Show the full list (items in the basket will be visible and ticked)"));
    connect(fullListAct, SIGNAL(triggered()), this, SLOT(on_action_fullList()));

	QAction *whatsLeftAct = new QAction(QIcon("/opt/usr/share/icons/hicolor/64x64/apps/shopper/whats-left.png"),
										tr("Out Shopping: What's Left"), this);
    whatsLeftAct->setStatusTip(tr("Show what's left to get"));
    connect(whatsLeftAct, SIGNAL(triggered()), this, SLOT(on_action_whatsLeft()));

	QAction *makingListAct = new QAction(QIcon("/opt/usr/share/icons/hicolor/64x64/apps/shopper/making-list.png"),
										 tr("Making List"), this);
    makingListAct->setStatusTip(tr("Pick items to get when you go shopping"));
    connect(makingListAct, SIGNAL(triggered()), this, SLOT(on_action_makingList()));

	// Information / admin Actions
	QAction *prefsAct = new QAction(qApp->style()->standardIcon(QStyle::SP_DialogSaveButton),
									tr("Preferences..."), this);
    prefsAct->setStatusTip(tr("Application preferences"));
    connect(prefsAct, SIGNAL(triggered()), this, SLOT(on_action_preferences()));

	QAction *aboutAct = new QAction(qApp->style()->standardIcon(QStyle::SP_MessageBoxInformation),
									tr("About / Help"), this);
    aboutAct->setStatusTip(tr("About Shopper..."));
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(on_action_about()));

	// Navigation Actions
	nextAct = new QAction(QIcon("/opt/usr/share/icons/hicolor/26x26/apps/shopper/right-arrow-red.png"),
						  tr("Next Category"), this);
    nextAct->setStatusTip(tr("View the next category"));
    connect(nextAct, SIGNAL(triggered()), this, SLOT(on_action_next()));
	
	QAction *prevAct = new QAction(QIcon("/opt/usr/share/icons/hicolor/26x26/apps/shopper/left-arrow-red.png"),
								   tr("Prev Category"), this);
    prevAct->setStatusTip(tr("View the previous category"));
    connect(prevAct, SIGNAL(triggered()), this, SLOT(on_action_prev()));

	// Menus
    menuBar()->addAction(fullListAct);
    menuBar()->addAction(whatsLeftAct);
	
    menuBar()->addAction(fileAct);
    menuBar()->addAction(editListAct);

    menuBar()->addAction(prefsAct);
    menuBar()->addAction(aboutAct);
	
	// Toolbars - shown/hidden according to view
    editBar = new QToolBar(tr("Edit"));
	editBar->setFloatable(false);
	editBar->setMovable(false);
	editBar->setIconSize(QSize(64,64));
	editBar->addAction(addItemAct);
    editBar->addAction(manageCatAct);
    modeBar = new QToolBar(tr("Mode"));
	modeBar->setFloatable(false);
	modeBar->setMovable(false);
	modeBar->setIconSize(QSize(64,64));
    modeBar->addAction(fullListAct);
    modeBar->addAction(whatsLeftAct);
    modeBar->addAction(makingListAct);

	navBar = new QToolBar(tr("Navigation"));
	navBar->setFloatable(false);
	navBar->setMovable(false);
	navBar->setIconSize(QSize(64,64));
	navBar->addAction(prevAct);
//    editBar->addAction(toggleWhatsLeftAct); // ??? idea
	navBar->addAction(nextAct);

	addToolBar(Qt::BottomToolBarArea, modeBar);
    addToolBar(Qt::BottomToolBarArea, navBar);
	addToolBar(Qt::BottomToolBarArea, editBar);

	QIcon shopper_icon("/opt/usr/share/icons/hicolor/64x64/apps/shopper/shopper.png");
	setWindowIcon(shopper_icon);

	// Prepare our Settings
	QCoreApplication::setOrganizationName("dgreaves.com");
	QCoreApplication::setOrganizationDomain("dgreaves.com");
	QCoreApplication::setApplicationName("Shopper");

	// size and position
    readSettings();
	timerId = startTimer(0);

	QLabel *notice = new QLabel("List Loading. Wait just a moment...", this);
	notice->show();
	setCentralWidget(notice);
	_LEAVE;
}

void MainWindow::timerEvent(QTimerEvent *event)
{
	killTimer(event->timerId());
	qApp->processEvents(); // ensure we're quiet
	qApp->processEvents(); // ensure we're quiet
	loadList();
}

void MainWindow::loadList()
{
	Shopper::List *l;
	if (mylist == 0) {
		l = Shopper::List::from_file(filename);
		if (l) { // If filename valid
			create_list_view(l);
		} else { // If filename wasn't read
			on_action_file_new();
			while (!mylist) { // mylist and list_view are done in a file_open()
				DEBUG("Couldn't open " << filename << " for reading\n");
				on_action_file_open();
				// FIXME :: Popup dialog box and allow new/open
			}
		}
	}
}

void MainWindow::create_list_view(Shopper::List *list)
{
	// Remove the old listview... and any data
	if (mylist) mylist->deleteLater();
	if (lv) lv->deleteLater();
	if (catCombo)
	{
		if (catCombo->model()) delete catCombo->model();
		delete catCombo;
	}
	// and use a list and a new main shopperList widget
	mylist = list;
	lv = new Shopper::ListView(*list, this);
	lv->setProperty("FingerScrollBars", true);
	lv->setProperty("FingerScrollable", true);

	QWidget *central = new QWidget();
	QHBoxLayout *centralBox = new QHBoxLayout(central);
	centralBox->addWidget(lv);
	setCentralWidget(central);

	// GUI: set title, add the view, update the menu
	setWindowTitle("Shopper:"+mylist->modeText());

	// Make sure there's a menu to show to enable the popup functionality	
	catCombo = new QComboBox();
	Shopper::CatListModel *clmodel = new Shopper::CatListModel(*mylist);
	catCombo->setModel(clmodel);
	catCombo->setModelColumn(1);
	catCombo->setCurrentIndex(clmodel->currentIndex());
	catCombo->setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::Fixed);
	connect(catCombo, SIGNAL(activated(int)),
			this, SLOT(cat_selected(int)));
	connect(clmodel, SIGNAL(currentChanged(int)),
			catCombo, SLOT(setCurrentIndex(int)));
	navBar->insertWidget(nextAct, catCombo);

	// Update UI when the state changes
	connect(mylist, SIGNAL(state_changed()),
			this, SLOT(stateChanged()));

	_LEAVE;
}

void MainWindow::cat_selected(int i)
{
	Q_UNUSED(i)
	_ENTER;
	QVariant v = catCombo->itemData(catCombo->currentIndex());
	Shopper::Category *c = v.value<Shopper::Category*>();
	mylist->make_category_active(*c);
}


void MainWindow::closeEvent(QCloseEvent *event)
{
	_ENTER;
	QSettings settings;
	if (settings.value("data/SaveOnExit").toBool())
		on_action_file_save();
	writeSettings();
	event->accept();
}

#define _QUOTEME(x) #x
#define QUOTEME(x) _QUOTEME(x)
void MainWindow::on_action_about()
{
	QMessageBox::about(this, tr("About Shopper"),
					   tr("<b>Shopper</b> V." QUOTEME(SHOPPER_VERSION)
						  " was written by David Greaves (lbt). "
						  "It is Free Software and licensed under the GPL.<br>"
						  "I hope you enjoy using it.<br>"
						  "<a href='http://shopper.garage.maemo.org/help.html'>Click here for online help</a><br>"
						  "Please let me know of any problems or suggestions at <a href='http://tinyurl.com/shopper-bugs'>the bug page</a>"));
}

void MainWindow::readSettings()
{
    QSettings settings;
    QPoint pos = settings.value("pos", QPoint(200, 200)).toPoint();
    QSize size = settings.value("size", QSize(400, 400)).toSize();
    filename = settings.value("filename", filename).toString();
    resize(size);
    move(pos);
}

void MainWindow::writeSettings()
{
    QSettings settings;
    settings.setValue("pos", pos());
    settings.setValue("size", size());
    settings.setValue("filename", filename);
	DEBUG("Wrote Settings");
}

// Action handlers
// Add item
void MainWindow::on_action_add_item()
{
	Shopper::ItemAdd itemD(this, mylist);
	if (itemD.exec() == QDialog::Accepted) {
		DEBUG("\nADDED:\n");
	}
}

void MainWindow::on_action_edit_list()
{
	Shopper::ListDialog listD(this, mylist);
	if (listD.exec() == QDialog::Accepted) {
		DEBUG("\nOK:\n");
	}
}

void MainWindow::on_action_make_list()
{
	Shopper::ListDialog listD(this, mylist);
	if (listD.exec() == QDialog::Accepted) {
		DEBUG("\nOK:\n");
	}
}

void MainWindow::on_action_manage_category()
{
	Shopper::CategoryManage catD(this, mylist);
	catD.exec();
}

// Category cycle
void MainWindow::on_action_next(){ 	mylist->make_next_category_active(); }
void MainWindow::on_action_prev(){ 	mylist->make_prev_category_active(); }

// Clear List
void MainWindow::on_action_clear_wanted() { mylist->clear(Shopper::CLEAR_WANTED); }
void MainWindow::on_action_clear_bought() { mylist->clear(Shopper::CLEAR_BOUGHT); }

// Shopping mode
void MainWindow::on_action_fullList() { mylist->set_state(Shopper::OUT_SHOPPING); }
void MainWindow::on_action_whatsLeft() { mylist->set_state(Shopper::WHATS_LEFT); }
void MainWindow::on_action_makingList() { mylist->set_state(Shopper::MAKING_LIST); }

void MainWindow::stateChanged()
{
	setWindowTitle("Shopper:" + mylist->modeText());
	// No, it can't tell if the screen's resized...
	QRect screenGeometry = QApplication::desktop()->screenGeometry();
	if (screenGeometry.width() > screenGeometry.height()) {
		modeBar->show();
		if (mylist->get_state() == Shopper::MAKING_LIST)
			editBar->show();
		else
			editBar->hide();
	} else {
		modeBar->hide();
		editBar->hide();
	}
}
////////////////////////////////////////////////////////////////
// File Handling
void MainWindow::on_action_file()
{
	Shopper::SaveDialog saveD(this, this);
	if (saveD.exec() == QDialog::Accepted) {
		DEBUG("\nOK:\n");
	}
}

void MainWindow::on_action_file_new()
{
	if (dialog_active == true) return;
	// FIXME: This really ought to check if the user *WANTS* to save...
	if (mylist)
		on_action_file_save();
	filename="";
	QSettings settings;
	if (settings.contains("data/DoneFirstNew")) {
		create_list_view(new Shopper::List(
			 QString(
				 "<list name='Supermarket' state='0'>"
				 "<category name='Aisle 1'>"
				 "<item desc='First Item' note='' wanted='1' bought='0'/>"
				 "</category>"
				 "</list>"
				 )));
	} else {
		settings.setValue("data/DoneFirstNew", true);
		create_list_view(new Shopper::List(
			 QString(
				 "<list name='Supermarket' state='0'>"
				 "<category name='Aisle 1 - Dairy'>"
				 "<item desc='A left -> right stroke on an item will mark it.' note='' wanted='1' bought='0'/>"
				 "<item desc='Or right -> left. A bit like crossing it off.' note='' wanted='1' bought='0'/>"
				 "<item desc='Stroke left -> right -> left (a scribble) to edit items.' note='' wanted='1' bought='0'/>"
				 "<item desc='To delete this help edit each of these items' note='' wanted='1' bought='0'/>"
				 "<item desc='You can also ... ' note='have notes which can be cleared' wanted='1' bought='0'/>"
				 "</category>"
				 "<category name='Frozen'>"
				 "<item desc='Peas' note='' wanted='1' bought='0'/>"
				 "<item desc='Prawns' note='' wanted='1' bought='0'/>"
				 "</category>"
				 "<category name='Veg'>"
				 "<item desc='Potatoes' note='' wanted='0' bought='0'/>"
				 "<item desc='Brocolli' note='' wanted='1' bought='0'/>"
				 "</category>"
				 "<category name='Shoes'>"
				 "<item desc='Versace' note='For Tracy' wanted='0' bought='0'/>"
				 "</category>"
				 "</list>"
				 )));
	}
}
void MainWindow::on_action_file_open()
{
	if (dialog_active == true) return;
	// Get a filename
	QString dir = filename.isEmpty() ? QString(HOME_DIR)+ QString(DEFAULT_LIST) : filename ;
	//Handle the response:
	dialog_active = true; // not modal
	filename = QFileDialog::getOpenFileName(this,
											"Select a list",
											dir,
											tr("Lists (*.xml)"));
	dialog_active = false;
	if (! filename.isEmpty()) {
		// Open file and create a view from it.
		create_list_view(Shopper::List::from_file(filename));
		writeSettings(); // FIXME this should be on success
	} else {
		if (mylist == 0) on_action_file_new();  // If CANCEL pressed and there's no list, make a new one
	}

}
void MainWindow::on_action_file_import()
{
	if (dialog_active == true) return;
	// Get a filename
	QString dir = filename.isEmpty() ? QString(HOME_DIR)+ QString(DEFAULT_LIST) : filename ;
	//Handle the response:
	dialog_active = true;
	filename = QFileDialog::getOpenFileName(this,
										   "Select a list",
										   dir,
										   tr("Lists (*.xml)"));
	dialog_active = false;
	if (! filename.isEmpty()) {
		// Open file and create a view from it.
		create_list_view(Shopper::ListImport::from_plaintext(filename));
		// treat it like a new list ?
		filename = "";
		//FIXME handle errors
		//on_action_file_saveas();
	} else {
		if (mylist == 0) on_action_file_new();  // If CANCEL pressed and there's no list, make a new one
	}

}
////////////////////////////////////////////////////////////////
void MainWindow::on_action_file_save()
{
	if (dialog_active == true) return;
	DEBUG("Saving....\n");
	if (filename.isEmpty()) {
		on_action_file_saveas();
	} else {
		mylist->to_file(filename);
	}
}
void MainWindow::on_action_file_saveas()
{
	if (dialog_active == true) return;
	// Get a filename
	QString dir = HOME_DIR+(filename==""?DEFAULT_LIST:filename);
	dialog_active = true;
	filename = QFileDialog::getSaveFileName(this,
									  "Save list as",
									  dir,
									  tr("Lists (*.xml)"),
									  NULL,
									  QFileDialog::HideNameFilterDetails
		);
	dialog_active = false;
	if (! filename.isEmpty()) {
		// Open file and create a view from it.
		writeSettings();
		mylist->to_file(filename);
	}
}
void MainWindow::on_action_preferences()
{
	Shopper::PreferencesDialog prefsD(this);
	prefsD.exec();
	lv->update();
}

void MainWindow::keyPressEvent ( QKeyEvent * event )
{
	switch (event->key())
	{
    case Qt::Key_F7: // Zoom in
		lv->zoomIn();
		break;
    case Qt::Key_F8: // Zoom out
		lv->zoomOut();
		break;
    default:
		QMainWindow::keyPressEvent(event);
		return;
		break;
	}
	return;
}

void MainWindow::on_action_rotate()
{
}

void MainWindow::on_action_help()
{

}
