#include <glib.h>
#include <glib/gprintf.h>
#include <osso-log.h>
#include <string.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <fcntl.h>

#include <sharing-http.h>
#include <sharing-plugin-interface.h>
#include <sharing-entry.h>
#include <sharing-tag.h>

#include "twitpic_post.h"

#define TWITPIC_URL "http://twitpic.com/api/uploadAndPost"

struct share_data
{
  gboolean *dead_mans_switch;
  SharingTransfer *transfer;
  guint64 total_size;
  guint64 total_sent;
};

static gboolean
twitpic_progress_func(SharingHTTP *http, guint64 bytes_send, gpointer user_data)
{
  struct share_data *data = (struct share_data *)user_data;
  SharingTransfer *transfer = data->transfer;

  *(data->dead_mans_switch) = FALSE;
  if (transfer != NULL) {
    if (sharing_transfer_continue(transfer) == FALSE) {
      ULOG_DEBUG_L ("user cancelled transfer");
      return FALSE;
    }
  }

  /* The sum of all other media sent already + the current entry */
  guint64 total_sent = data->total_sent + bytes_send;
  sharing_transfer_set_progress(transfer, (gdouble)total_sent/(gdouble)data->total_size);

  return TRUE;
}



int
twitpic_upload_media(const char *username,
                     const char *password,
                     SharingEntryMedia *media,
                     struct share_data *data)
{
  SharingHTTP *http = sharing_http_new ();

  gchar *filename = sharing_entry_media_get_filename(media);
  gchar *mime = sharing_entry_media_get_mime(media);
  const gchar *message = sharing_entry_media_get_desc(media);

  sharing_http_add_req_multipart_data(http, "username", username, -1, "text/plain");
  sharing_http_add_req_multipart_data(http, "password", password, -1, "text/plain");
  sharing_http_add_req_multipart_data(http, "message", message ? message : "", -1, "text/plain");
  sharing_http_add_req_multipart_file_with_filename(http, "media",
                                                          sharing_entry_media_get_localpath(media),
                                                          mime ? mime : "image/jpeg", filename ? filename : "image.jpg");

  sharing_http_set_progress_callback(http, twitpic_progress_func, data);
  *(data->dead_mans_switch) = FALSE;

  ULOG_DEBUG_L("WILL SEND");
  gchar *url = g_strdup_printf(TWITPIC_URL);
  SharingHTTPRunResponse res = sharing_http_run (http, url);
  ULOG_DEBUG_L("WILL SEND: OK");

  int ret = SHARING_SEND_SUCCESS;
  if (res == SHARING_HTTP_RUNRES_SUCCESS) {
  	ret = SHARING_SEND_SUCCESS;
  } else if (res == SHARING_HTTP_RUNRES_CANCELLED) {
    ULOG_DEBUG_L ("USER CANCEL");
    ret = SHARING_SEND_CANCELLED;
  } else {
#if DEBUG
      guint res_len;
      ULOG_DEBUG_L ("SEND ERROR: %s", sharing_http_get_res_body(http, &res_len));
#endif
      ret = SHARING_SEND_ERROR_UNKNOWN;
  }

  sharing_entry_media_set_sent(media, TRUE);
  sharing_http_unref (http);
  *(data->dead_mans_switch) = FALSE;

  ULOG_DEBUG_L ("CLEAN RESPONSE: RETURN:", ret);

  if (filename)
      g_free(filename);

  if (mime)
      g_free(mime);

  g_free(url);
  return ret;
}

SharingPluginInterfaceAccountValidateResult
twitpic_post_validate_account (SharingAccount *account,
                               ConIcConnection *con,
                               gboolean *cont,
                               gboolean *dead_mans_switch)
{
  *dead_mans_switch = FALSE;
  ULOG_DEBUG_L ("Entered %s", __FUNCTION__);

  SharingPluginInterfaceAccountValidateResult ret =
      SHARING_ACCOUNT_VALIDATE_SUCCESS;

  gchar *user = sharing_account_get_username(account);
  gchar *password = sharing_account_get_password(account);
  gchar buffer[1204];

  /* Sanity check user/pass lengths */
  if ((strlen(user) + strlen(password)) > 1024)
          return SHARING_ACCOUNT_VALIDATE_NOT_STARTED;

  /* Create b64 encoded user/pass into buffer */
  snprintf(buffer, sizeof(buffer), "%s:%s", user, password);
  gchar *encoded = g_base64_encode((guchar *)buffer, (gsize)strlen(buffer));
  if (!encoded)
          return SHARING_ACCOUNT_VALIDATE_ERROR_UNKNOWN;
  snprintf(buffer, sizeof(buffer), "Basic %s", encoded);

  g_free(encoded);
  g_free(user);
  g_free(password);

  SharingHTTP * http = sharing_http_new ();

  sharing_http_add_req_header(http, "Authorization", buffer);
  sharing_http_add_req_header(http, "Accept-Encoding", "gzip,deflate");

  SharingHTTPRunResponse res;
  res = sharing_http_run (http, "http://api.twitter.com/1/account/verify_credentials.xml");

  if (res == SHARING_HTTP_RUNRES_SUCCESS) {
    if (sharing_http_get_res_code(http) != 200)
      ret = SHARING_ACCOUNT_VALIDATE_FAILED;
  } else {
    ret = SHARING_ACCOUNT_VALIDATE_FAILED;
  }

  sharing_http_unref (http);

  return ret;
}

SharingPluginInterfaceSendResult
twitpic_post_upload_to_service (SharingTransfer *transfer,
                               ConIcConnection *con,
                               gboolean *dead_mans_switch)
{
  GSList *m = NULL;
  int result = SHARING_SEND_SUCCESS;

  *dead_mans_switch = FALSE;

  struct share_data *data = g_new0(struct share_data, 1);
  data->dead_mans_switch = dead_mans_switch;
  data->transfer = transfer;

  ULOG_DEBUG_L ("Entered %s DEAD_MANS_SWITCH[%p]", __FUNCTION__, data->dead_mans_switch);

  SharingEntry* entry = sharing_transfer_get_entry(transfer);
  SharingAccount* account = sharing_entry_get_account(entry);
  data->total_size = sharing_entry_get_size(entry);

  gchar *username = sharing_account_get_username(account);
  gchar *password = sharing_account_get_password(account);

  for(m = sharing_entry_get_media(entry); m != NULL; m = g_slist_next(m)) {
  	if (sharing_entry_media_get_sent(m->data))
      continue;

    result = twitpic_upload_media(username,
                                  password,
                                  m->data, data);

  	/* Keep track of the total bytes sent... used in progress callback */
  	data->total_sent += sharing_entry_media_get_size(m->data);

    if (result != SHARING_SEND_SUCCESS) {
      break;
    }
  }

  g_free(username);
  g_free(password);
  g_free(data);
  return result;
}

