/**
    File: picasa_post.c
    
    Author: Tero Niemel

*/


#include <glib.h>
#include <glib/gprintf.h>
#include <osso-log.h>
#include <string.h>
#include <stdlib.h>

#include <sharing-http.h>
#include <sharing-plugin-interface.h>
#include <sharing-entry.h>
#include <sharing-tag.h>

#include "picasa_post.h"
#include "picasa_http.h"
#include "picasa_xml.h"

/**
 * g_response_buffer:
 *
 * Buffer where CURL response is stored.
*/
GString *g_response_buffer = NULL;
/**
 * g_auth:
 *
 * Global variable for login auth string.
*/
gchar *g_auth = NULL;
/**
 * g_photo_url:
 *
 * Url where the photo is uploaded.
*/
gchar *g_photo_url = NULL;



/**
 * picasa_post_progress_data_t:
 * @dead_mans_switch: -
 * @cont: Continue flag if user cancels
 * @transfer: Sharing transfer
 * @entry: Sharing entry
 * @account: Sharing account
 * @request: HTTP request
 * @total_size: total size to upload
 * @uploaded_size: uploaded size
 *
 * Data structure for CURL progress callback.
*/
typedef struct {
  gboolean *dead_mans_switch;
  gboolean *cont;
  SharingTransfer *transfer;
  SharingEntry *entry;
  SharingAccount *account;
  http_request_t *request;
  guint64 total_size;
  guint64 uploaded_size;
} picasa_post_progress_data_t;

/**
 * picasa_post_response_type_t:
 *
 * Response types based on made request.
*/
typedef enum {
  RES_LOGIN = 0x01,
  RES_GET_ALBUMS = 0x02,
  RES_UPLOAD = 0x03,
  RES_UPDATE = 0x04,
  RES_GET_COORDINATES = 0x05
} picasa_post_response_type_t;
  
/**
 * picasa_post_response_t:
 * @code: HTTP response code
 * @msg: Error message or other information
 *
 * HTTP post response structure.
*/
typedef struct {
  gint code;
  gchar *msg;
} picasa_post_response_t;


/*
  PRIVATE functions 
*/


/**
 * picasa_post_init_response_buffer:
 *
 * Creates new string buffer for responses.
*/
static void
picasa_post_init_response_buffer () {

  g_response_buffer = g_string_new (NULL);
}

/**
 * picasa_post_clear_response_buffer:
 *
 * Frees response buffer.
*/
static void
picasa_post_clear_response_buffer () {

  g_string_free (g_response_buffer, TRUE);
  g_response_buffer = NULL;
}

/**
 * picasa_post_free_globals:
 *
 * Frees all global variables.
*/
static void
picasa_post_free_globals () {

  if (g_response_buffer) {
    picasa_post_clear_response_buffer ();
  }

  if (g_photo_url) {
    g_free (g_photo_url);
    g_photo_url = NULL;
  }

  if (g_auth) {
    g_free (g_auth);
    g_auth = NULL;
  }
}

/**
 * picasa_post_convert_http_error:
 * @state: HTTP request state
 *
 * Converts request state for error code.
 *
 * Returns: Error code
*/
static picasa_response_codes_t
picasa_post_convert_http_error (http_request_state_t state) {

  picasa_response_codes_t status = NETWORK_SEND_FAILED;

  switch (state) {
    
  case DISCONNECTED:
    status = CONNECTION_DISCONNECTED;
    break;
  case CANCELLED:
    status = CANCELLED_BY_USER;
    break;
  default:
    status = NETWORK_SEND_FAILED;
    break;
  }

  return status;
}

/**
 * picasa_post_convert_update_options:
 * @status: Update options status
 *
 * Converts request status to result code.
 *
 * Returns: Result code
*/
static SharingPluginInterfaceUpdateOptionsResult
picasa_post_convert_update_options (gint status) {

  SharingPluginInterfaceUpdateOptionsResult result = 
    SHARING_UPDATE_OPTIONS_ERROR_UNKNOWN;

  switch (status) {

  case HTTP_OK:
    result = SHARING_UPDATE_OPTIONS_SUCCESS;
    break;
  case CANCELLED_BY_USER:
    result = SHARING_UPDATE_OPTIONS_CANCELLED;
    break;
    
  default:
    result = SHARING_UPDATE_OPTIONS_ERROR_UNKNOWN;
    break;

  }

  return result;
}

/**
 * picasa_post_convert_validate_response:
 * @res: HTTP post response
 *
 * Converts validate response to result code.
 *
 * Returns: Result code
*/
static SharingPluginInterfaceAccountValidateResult
picasa_post_convert_validate_response (picasa_post_response_t res) {

  SharingPluginInterfaceAccountValidateResult result = 
    SHARING_ACCOUNT_VALIDATE_ERROR_UNKNOWN;

  switch (res.code) {
    
  case HTTP_OK:
    result = SHARING_ACCOUNT_VALIDATE_SUCCESS;
    break;
  case HTTP_FAILED:
    result = SHARING_ACCOUNT_VALIDATE_FAILED;
    break;
  case NETWORK_SEND_FAILED:
  case CONNECTION_DISCONNECTED:
    result = SHARING_ACCOUNT_VALIDATE_ERROR_CONNECTION;
    break;
  case CANCELLED_BY_USER:
    result = SHARING_ACCOUNT_VALIDATE_CANCELLED;
    break;
    /* rest is general error */
  default:
    result = SHARING_ACCOUNT_VALIDATE_ERROR_UNKNOWN;
    break;

  }

  /* clear response msg if any */
  if (res.msg != NULL) { 
    g_free (res.msg); 
    res.msg = NULL;
  }

  return result;
}

/**
 * picasa_post_convert_send_response:
 * @res: HTTP post response 
 *
 * Converts send response to result code
 *
 * Returns: Result code
*/
static SharingPluginInterfaceSendResult
picasa_post_convert_send_response (picasa_post_response_t res) {

  SharingPluginInterfaceSendResult result = 
    SHARING_SEND_ERROR_UNKNOWN;

  switch (res.code) {
  case HTTP_OK:
  case HTTP_CREATED:
    result = SHARING_SEND_SUCCESS;
    break;
  case HTTP_BAD_REQ:
    if (g_strcmp0 (res.msg, PICASA_POST_FILE_TOO_LARGE_C) == 0) {
      result = SHARING_SEND_ERROR_UPLOAD_LIMIT;
    }
    else if (g_strcmp0 (res.msg, PICASA_POST_NOT_AN_IMAGE_C) == 0) {
      result = SHARING_SEND_ERROR_FILE_FORMAT;
    }
    else {
      result = SHARING_SEND_ERROR_UNKNOWN;
    }
    break;
  case HTTP_FAILED:
    if (g_strcmp0 (res.msg, PICASA_POST_BAD_AUTHENTICATION_C) == 0) {
      result = SHARING_SEND_ERROR_AUTH;
    }
    else {
      result = SHARING_SEND_ERROR_UNKNOWN;
    }
    break;
  case HTTP_UNSUPPORTED_MEDIA:
    result = SHARING_SEND_ERROR_FILE_FORMAT;
    break;
  case NETWORK_SEND_FAILED:
  case CONNECTION_DISCONNECTED:
    result = SHARING_SEND_ERROR_CONNECTION;
    break;
  case CANCELLED_BY_USER:
    result = SHARING_SEND_CANCELLED;
    break;
    /* rest is general error */
  default:
    result = SHARING_SEND_ERROR_UNKNOWN;
    break;
  }

  /* clear response msg if any */
  if (res.msg != NULL) { 
    g_free (res.msg); 
    res.msg = NULL;
  }

  return result;
}

/**
 * picasa_post_parse_response:
 * @data: Progress data
 * @type: Response type
 *
 * Parses HTTP response data.
 *
 * Returns: Response struct with code and message
*/
static picasa_post_response_t
picasa_post_parse_response (picasa_post_progress_data_t *data,
			    picasa_post_response_type_t type) {

#ifdef DEBUG
  gchar *fn = "picasa_post_parse_response()";
#endif

  ULOG_DEBUG_L ("%s: Entered", fn);

  ULOG_DEBUG_L ("%s: Response: \n%s", fn, g_response_buffer->str);
  
  gchar **tokens = NULL;
  gchar *iter = NULL;
  gchar *tmp = (gchar*)g_response_buffer->str;
  gint count = 0;
  guint offset = 0;
  GSList *albums = NULL;
  gboolean success = FALSE;
  picasa_post_response_t res = { 0, NULL };

  tokens = g_strsplit_set ((gchar*)g_response_buffer->str,
                           "\n", /* split based on linefeed */
                           0);

  iter = tokens[0];
  while (iter) {

    if (g_str_has_prefix (iter, "HTTP/1.1 ")) {
      res.code = atoi (g_strndup (iter + HTTP_RES_CODE_OFFSET_C, 3));
    }

    if (g_str_has_prefix (iter, "Error=")) {
      res.msg = g_strdup (iter + HTTP_RES_ERR_OFFSET_C);
      ULOG_DEBUG_L ("%s: %s", fn, res.msg);
      break;
    }

    if (g_str_has_prefix (iter, "Auth=")) {
      g_auth = g_strdup (iter + 5);
      ULOG_DEBUG_L ("%s: Auth = %s", fn, g_auth);
      break;
    }

    if (g_str_has_prefix (iter, "<?xml")) {
      ULOG_DEBUG_L ("%s: Body found", fn);
      if (type == RES_UPLOAD) {
	g_photo_url = picasa_xml_parse_upload_response (iter);
	ULOG_DEBUG_L ("%s: uploaded photo id parsed (%s)", 
		      fn, g_photo_url);
      }
      else if (type == RES_GET_COORDINATES) {
	res.msg = picasa_xml_parse_gps_response ((tmp+(offset+count)));
	ULOG_DEBUG_L ("%s: coordinates parsed = %s", 
		      fn, res.msg);
      }
      break;
    }

    /* calculate offset from beginning */
    offset = offset + strlen (iter);

    count++;
    iter = tokens[count];

    /* if last line and status failed */
    if (res.code == HTTP_BAD_REQ && iter == NULL) {
      res.msg = g_strdup (tokens[count-1]);
      ULOG_DEBUG_L ("%s: last line = %s", fn, res.msg);
    }

  }
  
  
  g_strfreev (tokens);

  ULOG_DEBUG_L ("%s: Leaving", fn);
  
  return res;
}

/**
 * picasa_post_progress:
 * @clientp: Pointer to client data
 * @dltotal: Total DL size
 * @dlnow: DL size now
 * @ultotal: Total UL size
 * @ulnow: UL size now
 *
 * Progress function which CURL calls once a sec.
 *
 * Returns: 0 for continue and 1 if CURL needs to exit
*/
static int
picasa_post_progress (void *clientp,
		      double dltotal,
		      double dlnow,
		      double ultotal,
		      double ulnow) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_progress()";
#endif

  picasa_post_progress_data_t *data = (picasa_post_progress_data_t*)clientp;
  gdouble progress = 0.0;

  ULOG_DEBUG_L ("%s: data = %0X ; data->transfer = %0X ; data->cont = %0X ; data->request = %0X",
		fn, (guint)data, (guint)data->transfer, (guint)data->cont, (guint)data->request);


  /* Check has user cancelled the transfer */
  if (data->transfer != NULL) {
    if (sharing_transfer_continue (data->transfer) == FALSE) {
      ULOG_DEBUG_L ("%s: user cancelled transfer",fn);
      data->request->state = CANCELLED;
      return 1;
    }
  }

  /* dialog cancelled */
  if (data->cont != NULL) {
    if (*(data->cont) == FALSE) {
      ULOG_DEBUG_L ("%s: upload cancelled",fn);
      return 1;
    }
  }

  /* check for conic disconnect */
  if (data->request->state == DISCONNECTED) {
    ULOG_DEBUG_L ("%s: connection disconnected", fn);
    return 1;
  }

  /* if total_size defined. Not defined for example account validation */
  if (data->total_size > 0) {
    progress = (data->uploaded_size + ulnow) / data->total_size;
    if (progress > 1.0) {
      ULOG_DEBUG_L ("%s: FULL PROGRESS", fn);
      progress = 1.0;
    }

    ULOG_DEBUG_L ("%s: progress = %f", fn, progress);
    
    sharing_transfer_set_progress (data->transfer, progress);
  }

  /* pull dead_mans_switch */
  *(data->dead_mans_switch) = FALSE;
  
  return 0;
}


/**
 * picasa_post_login:
 * @con: Conic connection
 * @daya: Progress data
 *
 * Creates and sends login request to service.
 *
 * Returns: Request response
*/
static picasa_post_response_t
picasa_post_login (ConIcConnection *con,
		   picasa_post_progress_data_t *data) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_login()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };

  gchar *uname = sharing_account_get_param (data->account, "username");
  gchar *passwd = sharing_account_get_param (data->account, "password");

  http_request_t *http = picasa_http_new (LOGIN_URL_C);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload",fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_post_data (http, "accountType", "GOOGLE");
  picasa_http_add_post_data (http, "Email", uname);
  picasa_http_add_post_data (http, "Passwd", passwd);
  picasa_http_add_post_data (http, "service", SERVICE_C);
  picasa_http_add_post_data (http, "source", "remote uploader");

  picasa_post_init_response_buffer ();

  picasa_http_set_response_buffer (http, 
				   PICASA_POST_BUF_LIMIT_C ,
				   g_response_buffer);
  
  picasa_http_set_connection (http, con);
  picasa_http_set_progress_cb (http, picasa_post_progress, data);

  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post to %s",fn,LOGIN_URL_C);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (http, POST);

  *data->dead_mans_switch = FALSE;
  
  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  /* if sending post was success parse response */
  if (status == 0) {
    res = picasa_post_parse_response (data, RES_LOGIN);
  }
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
  }

  /* free memory and buffers */
  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (passwd);
  g_free (uname);

  return res;
}


/**
 * picasa_post_get_albums:
 * @con: Conic connection
 * @data: Progress data
 *
 * Creates album update request and sends to service.
 *
 * Returns: Request response
*/
static picasa_post_response_t 
picasa_post_get_albums (ConIcConnection *con,
			picasa_post_progress_data_t *data) {

#ifdef DEBUG
  gchar *fn = "picasa_post_get_albums()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };

  gchar *auth = g_strconcat ("GoogleLogin auth=", g_auth, NULL);
  http_request_t *http = http = picasa_http_new (FEED_URL_C);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_header_data (http, "GData-Version", "2");
  picasa_http_add_header_data (http, "Authorization", auth);

  picasa_post_init_response_buffer ();

  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);

  picasa_http_set_connection (http, con);
  picasa_http_set_progress_cb (http, picasa_post_progress, data);

  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post to %s", fn, FEED_URL_C);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (http, GET);

  *data->dead_mans_switch = FALSE;

  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  if (status == 0) {
    res = picasa_post_parse_response (data, RES_GET_ALBUMS);
  } 
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
  }

  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (auth);
  
  return res;
}

/**
 * picasa_post_upload_item:
 * @media: Media to upload
 * @con: Conic connection
 * @data: Progress data
 *
 * Creates item upload request and send it to service.
 *
 * Returns: Request response
*/
static picasa_post_response_t
picasa_post_upload_item (SharingEntryMedia *media,
			 ConIcConnection *con,
			 picasa_post_progress_data_t *data) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_upload_item()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };
  gchar *metadata = NULL;

  gchar *mime = sharing_entry_media_get_mime (media);
  gchar *title = sharing_entry_media_get_title (media);
  gchar *desc = (gchar*)sharing_entry_media_get_desc (media);

  gchar *album_id = "default";
  
  gchar *url = g_strconcat (FEED_URL_C, "/albumid/", album_id, NULL);
  gchar *auth = g_strconcat ("GoogleLogin auth=", g_auth, NULL);

  ULOG_DEBUG_L ("%s: uploading to url %s", fn, url);

  http_request_t *http = picasa_http_new (url);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_header_data (http, "GData-Version", "2");
  picasa_http_add_header_data (http, "Authorization", auth);

  if (title == NULL) { 
    title = sharing_entry_media_get_filename (media); 
  }

  metadata = picasa_xml_create_metadata_entry (title, desc);
  
  if (metadata == NULL) {
    ULOG_ERR_L ("%s: failed to create metadata entry", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  picasa_http_add_post_data_with_type (http, metadata, "application/atom+xml");
  g_free (metadata); 
  
  picasa_http_add_bin_file (http, sharing_entry_media_get_localpath (media),
			    NULL, mime);

  picasa_http_add_header_data (http, "MIME-version", "1.0");
  picasa_http_add_header_data (http, "Content-Type", "multipart/related"); 
  
  picasa_post_init_response_buffer ();
  
  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);
  
  picasa_http_set_connection (http, con);
  
  picasa_http_set_progress_cb (http, picasa_post_progress, data);
  
  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post to %s", fn, url);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (http, POST);

  *data->dead_mans_switch = FALSE;

  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  /* send successfully, check picasa response */
  if (status == 0) {
    res = picasa_post_parse_response (data, RES_UPLOAD);
  }
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
    ULOG_ERR_L ("%s: transfer error %d", fn, res.code);
  }

  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (url);
  g_free (auth);

  return res;
}


/**
 * picasa_post_update_media:
 * @tags: Tags
 * @gps: GPS location
 * @con: Conic connection
 * @data: Progress data
 *
 * Creates media update reqeust and sends it to service.
 *
 * Returns: Request response
*/
static picasa_post_response_t
picasa_post_update_media (gchar *tags,
			  gchar *gps,
			  ConIcConnection *con,
			  picasa_post_progress_data_t *data) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_update_media()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };
  gchar *length = NULL;
  gchar *media = NULL;

  gchar *auth = g_strconcat ("GoogleLogin auth=", g_auth, NULL);

  if (g_photo_url == NULL) {
    ULOG_WARN_L ("%s: photo url NULL -> tags will not be updated", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  ULOG_DEBUG_L ("%s: uploading to url %s", fn, g_photo_url);

  http_request_t *http = picasa_http_new (g_photo_url);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_header_data (http, "GData-Version", "2");
  picasa_http_add_header_data (http, "Authorization", auth);
  picasa_http_add_header_data (http, "Content-Type", "application/atom+xml");
  picasa_http_add_header_data (http, "If-Match", "*");
  picasa_http_add_header_data (http, "Expect", "Expect: 100-continue");

  media = picasa_xml_create_media_entry (tags, gps);

  picasa_http_append_read_data (http, media);
  g_free (media);
  
  length = g_strdup_printf("%d", http->read_data.data->len);
  picasa_http_add_header_data (http, "Content-Length", length);
  g_free (length);

  picasa_post_init_response_buffer ();
  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);
  
  picasa_http_set_connection (http, con);

  ULOG_DEBUG_L ("%s: Sending post to %s", fn, g_photo_url);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (http, PUT);

  *data->dead_mans_switch = FALSE;

  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  /* send successfully, check picasa response */
  if (status == 0) {
    res = picasa_post_parse_response (data, RES_UPDATE);
  } else {
    ULOG_WARN_L ("%s: failed to update media", fn);
  }

  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (auth);

  return res;
}

/**
 * picasa_post_get_gps_coordinates:
 * @con: Conic connection
 * @data: Progress data
 * @country: Country
 * @city: City
 * @suburb: Suburb aka district
 *
 * Creates geocoding request and sends it to service.
 *
 * Return: Request response.
*/
static picasa_post_response_t 
picasa_post_get_gps_coordinates (ConIcConnection *con,
				 picasa_post_progress_data_t *data,
				 gchar *country,
				 gchar *city,
				 gchar *suburb) {

#ifdef DEBUG
  gchar *fn = "picasa_post_get_gps_coordinates()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };

  GString *url = g_string_new (MAPS_URL_C);

  g_string_append (url, "q=");
  if (country != NULL) {
    g_string_append (url, "+");
    g_string_append (url, country);
  }
  if (city != NULL) {
    g_string_append (url, "+");
    g_string_append (url, city);
  }
  if (suburb != NULL) {
    g_string_append (url, "+");
    g_string_append (url, suburb);
  }
  g_string_append (url, "&output=xml");
  g_string_append (url, "&sensor=false");

  ULOG_DEBUG_L ("%s: created url = %s", fn, url->str);

  http_request_t *http = picasa_http_new (url->str);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_post_init_response_buffer ();

  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);

  picasa_http_set_connection (http, con);
  
  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post", fn);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (http, GET);

  *data->dead_mans_switch = FALSE;

  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  if (status == 0) {
    res = picasa_post_parse_response (data, RES_GET_COORDINATES);
  } 
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
  }

  picasa_post_clear_response_buffer ();
  g_string_free (url, TRUE);
  
  return res;
}

/**
 * picasa_post_add_tags:
 * @media: Sharing media
 * @con: Conic connection
 * @data: Progress data
 *
 * Checks tags and geotags from media and updates to service.
 *
 * Returns: Response code
*/
static gint
picasa_post_add_tags (SharingEntryMedia *media,
		      ConIcConnection *con,
		      picasa_post_progress_data_t *data) {
#ifdef DEBUG
  gchar *fn = "picasa_post_add_tags()";
#endif

  ULOG_DEBUG_L ("%s: entered", fn);
  SharingTag *tag = NULL;
  GString *tag_str = NULL;
  gboolean first = TRUE;
  picasa_post_response_t res = { 0, NULL };
  GSList *t = NULL;
  const gchar *geo_country = NULL;
  const gchar *geo_city = NULL; 
  const gchar *geo_suburb = NULL;
  gchar *tags = NULL;

  GSList *media_tags = (GSList*)sharing_entry_media_get_tags (media);

  for (t = media_tags; t != NULL; t = g_slist_next(t)) {

    tag = (SharingTag*)t->data;
    
    ULOG_DEBUG_L ("%s: tag with type %d", fn, sharing_tag_get_type (tag));

    switch (sharing_tag_get_type (tag)) {

      /* normal tags */
    case SHARING_TAG_SHARE:
      {
	if(first == TRUE) {
	  tag_str = g_string_new (sharing_tag_get_word (tag));
	  first = FALSE;
	  tags = tag_str->str;
	} else {
	  g_string_append (tag_str, ",");
	  g_string_append (tag_str, sharing_tag_get_word (tag));
	}
	
	ULOG_DEBUG_L ("%s: found new tag = %s", fn, tag_str->str);

	break;
      }

    case SHARING_TAG_GEO_COUNTRY:
      {
	geo_country = sharing_tag_get_word (tag);
	ULOG_DEBUG_L ("%s: found geo country = %s", fn, geo_country);
	break;
      }

    case SHARING_TAG_GEO_CITY:
      {
	geo_city = sharing_tag_get_word (tag);
	ULOG_DEBUG_L ("%s: found geo city = %s", fn, geo_city);
	break;
      }

    case SHARING_TAG_GEO_SUBURB:
      {
	geo_suburb = sharing_tag_get_word (tag);
	ULOG_DEBUG_L ("%s: found geo suburb = %s", fn, geo_suburb);
	break;
      }

    default:
      {
	ULOG_DEBUG_L ("%s: unknown tag type", fn);
      }
      
    } /* switch */
  } /* for */ 

  /* in case some of geostrings found try to get coordinates */
  if (geo_country != NULL || geo_city != NULL || geo_suburb != NULL) {

    ULOG_DEBUG_L ("%s: trying to get coordinates", fn);
    
    res = picasa_post_get_gps_coordinates (con, data,
					   (gchar*)geo_country,
					   (gchar*)geo_city,
					   (gchar*)geo_suburb);

    if (res.code == HTTP_OK &&
	res.msg != NULL) {
      ULOG_DEBUG_L ("%s: found coordinate = %s", fn, res.msg);
      
    }
    
  }

  if (tag_str != NULL) { tags = tag_str->str; }

  /* if tags or coordinates to update */
  if (tags != NULL || res.msg != NULL) {
    ULOG_DEBUG_L ("%s: trying to update media entry", fn);
    res = picasa_post_update_media (tags, res.msg, con, data);

    ULOG_DEBUG_L ("%s: media entry updated with status = %d", fn, res.code);
  }

  ULOG_DEBUG_L ("%s: leaving", fn);

  /* cleanup */
  if (tag_str != NULL) {
    g_string_free (tag_str, TRUE);
  }
  if (res.msg != NULL) { 
    g_free (res.msg); 
    res.msg = NULL;
  }

  return res.code;
}




/*
  PUBLIC functions
*/


/**
 * picasa_post_validate_account:
 * @account: Account to validate
 * @con: Conic connection
 * @cont: Continue flag is user cancels
 * @dead_mans_switch: -
 *
 * Tries to validate account by logging in to service.
 *
 * Returns: Validation result
*/
SharingPluginInterfaceAccountValidateResult
picasa_post_validate_account (SharingAccount *account,
			      ConIcConnection *con,
			      gboolean *cont,
			      gboolean *dead_mans_switch) {

#ifdef DEBUG
  gchar *fn = "picasa_post_validate_account()";
#endif

  ULOG_DEBUG_L ("Entered %s",fn);

  SharingPluginInterfaceAccountValidateResult
    status = SHARING_ACCOUNT_VALIDATE_FAILED;

  picasa_post_response_t res = { 0, NULL };

  picasa_post_progress_data_t data;

  *dead_mans_switch = FALSE;

  /* fill data structure */
  
  data.dead_mans_switch = dead_mans_switch;
  data.cont = cont;
  data.account = account;
  data.transfer = NULL;
  data.entry = NULL;
  data.request = NULL;
  data.total_size = 0;
  data.uploaded_size = 0;

  ULOG_DEBUG_L ("%s: LOGIN",fn);  

  /* login to validate */
  res = picasa_post_login (con, &data);

  status = picasa_post_convert_validate_response (res);

  picasa_post_free_globals ();

  return status;
}


/**
 * picasa_post_send_to_service:
 * @transfer: Sharing transfer data
 * @con: Conic connection
 * @dead_mans_switch: -
 * 
 * Uploads all media under one transfer into service.
 *
 * Returns: Send result
*/
SharingPluginInterfaceSendResult 
picasa_post_upload_to_service (SharingTransfer *transfer,
			       ConIcConnection *con,
			       gboolean *dead_mans_switch) {

#ifdef DEBUG
  gchar *fn = "picasa_post_upload_to_service()";
#endif

  ULOG_DEBUG_L ("Entered %s",fn);

  gint status = 0;
  picasa_post_response_t res;
  picasa_post_progress_data_t data;
  GSList *m = NULL;

  /* fill data structure */
  data.dead_mans_switch = dead_mans_switch;
  data.transfer = transfer;
  data.entry = sharing_transfer_get_entry (transfer);
  data.account = sharing_entry_get_account (data.entry);
  data.request = NULL;
  data.total_size = sharing_entry_get_size (data.entry);
  data.uploaded_size = 0;
  data.cont = NULL;

  if (!data.account) {
    ULOG_DEBUG_L ("%s: no account found",fn);
    return SHARING_SEND_ERROR_ACCOUNT_NOT_FOUND;
  }

  *dead_mans_switch = FALSE;
  
  ULOG_DEBUG_L ("%s: LOGIN",fn);

  res = picasa_post_login (con, &data);
  
  if (res.code != HTTP_OK) {
    ULOG_DEBUG_L ("%s: login failed %d", fn, res.code);
    return picasa_post_convert_send_response (res);
  }

  /* post all items in loop */
  for (m = sharing_entry_get_media(data.entry); 
       m != NULL; m = g_slist_next(m)) {
    SharingEntryMedia *media = m->data;

    /* if not sent yet */
    if (!sharing_entry_media_get_sent( media )) {

      ULOG_DEBUG_L ("%s: UPLOAD ITEM",fn);

      res = picasa_post_upload_item (media, con, &data);
      if (res.code == HTTP_OK || res.code == HTTP_CREATED) {
	data.uploaded_size += sharing_entry_media_get_size (media);
	sharing_entry_media_set_sent (media, TRUE);
	/* try to update the tags */
	ULOG_DEBUG_L ("%s: UPDATE TAGS", fn);
	picasa_post_add_tags (media, con, &data);
	status = SHARING_SEND_SUCCESS;
      } else {
	ULOG_DEBUG_L ("%s: sending to service failed %d", fn, res.code);
	status = picasa_post_convert_send_response (res);
	break;
      }
    } else {
      /* item already sent previously */
      /* only update uploaded size for progress bar */
      data.uploaded_size += sharing_entry_media_get_size (media);
    }

  }

  picasa_post_free_globals ();

  return status;
}


/* EOF */

