/**
    File: picasa_http.c
        
    Author: Tero Niemel

*/


#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <osso-log.h>
#include <conicconnection.h>
#include <conicconnectionevent.h>

#include <sharing-plugin-interface.h>

#include "picasa_http.h"


/*
   PRIVATE functions
*/


/**
   picasa_http_read_cb
*/
static size_t 
picasa_http_read_cb (void *buffer,
		     size_t size,
		     size_t nmemb,
		     void *userp) {
  
#ifdef DEBUG
  gchar *fn = "picasa_http_read_cb()";
#endif

  size_t req_size = size * nmemb;
  size_t read_size = 0;
  size_t left_size = 0;
  size_t *sent_size = 0;


  ULOG_DEBUG_L ("%s: entered", fn);
  ULOG_DEBUG_L ("%s: curl requesting %d bytes", fn, (size*nmemb));

  http_request_t *req = (http_request_t*)userp;
  
  /* if prefix send first */
  if ((req->read_data).data != NULL &&
      (req->read_data).sent != (req->read_data).data->len) {
    
    sent_size = &(req->read_data).sent;
    left_size = (req->read_data).data->len - *sent_size;
   
    if (left_size > req_size) {
      ULOG_DEBUG_L ("%s: sending only part ( %d < %d)", fn, 
		    req_size, left_size);
      memcpy (buffer, (req->read_data).data->str + *sent_size, req_size);
      *sent_size += req_size;
      read_size = req_size;
    } else {
      ULOG_DEBUG_L ("%s: sending whole ( %d >= %d)", fn,
		    req_size, left_size);
      memcpy (buffer, (req->read_data).data->str + *sent_size, left_size);
      read_size = left_size;
      *sent_size += left_size;
      
    }
    
  }

  return read_size;
}


/**
   picasa_http_write_cb
*/
static size_t 
picasa_http_write_cb (void *buffer,
		      size_t size,
		      size_t nmemb,
		      void *userp) {

#ifdef DEBUG  
  gchar *fn = "picasa_http_curl_callback()";
#endif

  http_request_t *request = (http_request_t*)userp;
  gint total_size = 0;
  gint copy_size = 0;
  gint copy_offset = 0;

  ULOG_DEBUG_L ("%s: buffer %0X ; size %d ; nmemb %d ; userp %0X",
		fn,
		(int)buffer,
		size,
		nmemb,
		(int)userp);

  if (request->resp_buf == NULL) {
    ULOG_ERR_L ("%s: no response buffer set",fn);
    return 0;
  }

  /* TODO: check for buffer max size */

  total_size = (size * nmemb);
  copy_size = total_size;
  
  g_string_append_len (request->resp_buf,
		       (((gchar*)buffer)+copy_offset),copy_size);
  
  /* all data received and handled */
  return copy_size;
}


#ifndef USE_SBOX
/**
   picasa_http_connection_event_cb
*/
static void
picasa_http_connection_event_cb (ConIcConnection *con,
				  ConIcConnectionEvent *event,
				  http_request_t *req) {

#ifdef DEBUG
  gchar *fn = "galery_http_connection_event_cb()";
#endif

  if (event == NULL ||
      req == NULL) {
    ULOG_ERR_L ("%s: NULL pointer to callback",fn);
    return;
  }

  ConIcConnectionStatus status = 
    con_ic_connection_event_get_status (event);

  ULOG_DEBUG_L ("%s: conic event %d received", fn, status);

  if (status == CON_IC_STATUS_DISCONNECTED) {
    /* change only if needed */
    if (req->state != DISCONNECTED) {
      ULOG_DEBUG_L ("%s: disconnected", fn );
      req->state = DISCONNECTED;
    }
  }
  
}
#endif


/*
  PUBLIC functions
*/


/**
  picasa_http_new
*/
http_request_t*
picasa_http_new (gchar *url) {

#ifdef DEBUG
  gchar *fn = "picasa_http_new()";
#endif

  if (url == NULL) {
    ULOG_ERR_L ("%s: NULL url",fn);
    return NULL;
  }

  http_request_t *new_request = g_malloc (sizeof (http_request_t));

  new_request->url = g_malloc (strlen (url)+1);
  g_stpcpy (new_request->url,url);

  new_request->hdr = NULL;
  new_request->post = NULL;
  new_request->last = NULL;
  new_request->buf_size_max = 0;
  new_request->resp_buf = NULL;
  new_request->progress_cb = NULL;
  new_request->progress_data = NULL;
  new_request->con = NULL;
  new_request->read_data.data = NULL;
  new_request->read_data.sent = 0;
  new_request->body_size = 0;

  ULOG_DEBUG_L ("%s: new request created",fn);
  
  return new_request;
}

/**
  picasa_http_append_read_data
*/
http_result_t
picasa_http_append_read_data (http_request_t *request,
			      gchar *content) {

  if ((request->read_data).data == NULL) {
    (request->read_data).data = g_string_new (NULL);
  }

  g_string_append ((request->read_data).data, content);

  return HTTP_SUCCESS;
}


/**
  picasa_http_add_header_data
*/
http_result_t
picasa_http_add_header_data (http_request_t *request, 
			     gchar *name, gchar *value) {
  
  gchar *line = NULL;

  line = g_strconcat (name, ": ", value, NULL);

  request->hdr = curl_slist_append (request->hdr, line);

  g_free (line);

  return HTTP_SUCCESS;
}


/**
  picasa_http_add_post_data
*/
http_result_t
picasa_http_add_post_data (http_request_t *request, 
			   gchar *key, gchar *value) {

#ifdef DEBUG
  gchar *fn = "picasa_http_add_post_data()";
#endif
  
  if (request == NULL || key == NULL ||  value == NULL) {
    ULOG_ERR_L ("%s: NULL pointer in parameters",fn);
    return HTTP_ERROR;
  }
  
  curl_formadd (&request->post,
		&request->last,
		CURLFORM_COPYNAME, key,
		CURLFORM_COPYCONTENTS, value,
		CURLFORM_END);
  
  return HTTP_SUCCESS;
}

/**
  picasa_http_add_post_data_with_type
*/
http_result_t
picasa_http_add_post_data_with_type (http_request_t *request, 
				     gchar *value, gchar *type) {
  
#ifdef DEBUG
  gchar *fn = "picasa_http_add_post_data()";
#endif
  
  if (request == NULL || value == NULL || type == NULL) {
    ULOG_ERR_L ("%s: NULL pointer in parameters",fn);
    return HTTP_ERROR;
  }
  
  curl_formadd (&request->post,
		&request->last,
		CURLFORM_NAMELENGTH, 0,
		CURLFORM_COPYNAME, "",
		CURLFORM_COPYCONTENTS, value,
		CURLFORM_CONTENTTYPE, type,
		CURLFORM_END);
  
  return HTTP_SUCCESS;
}

/**
  picasa_http_add_bin_file
*/
http_result_t
picasa_http_add_bin_file (http_request_t *request, 
			  const gchar *file, gchar *filename,
			  gchar *type) {
  
#ifdef DEBUG
  gchar *fn = "picasa_http_add_bin_file()";
#endif

  if (request == NULL ||
      file == NULL) {
    ULOG_ERR_L ("%s: NULL pointer in parameters",fn);
    return HTTP_ERROR;
  }

  curl_formadd (&request->post,
		&request->last,	
		CURLFORM_NAMELENGTH, 0,
		CURLFORM_COPYNAME, "",
		CURLFORM_FILE, file, 
		/* CURLFORM_FILENAME, filename, */
		CURLFORM_CONTENTTYPE, type,
		CURLFORM_END);
  
  return HTTP_SUCCESS;
}

/**
  picasa_http_set_response_buffer
*/
http_result_t
picasa_http_set_response_buffer (http_request_t *request, 
				 gint max_buf_size, 
				 GString *buf) {

#ifdef DEBUG
  gchar *fn = "picasa_http_set_response_buffer()";
#endif

  if (request == NULL || buf == NULL) {
    ULOG_ERR_L ("%s: NULL pointer in parameters",fn);
    return HTTP_ERROR;
  }
  
  request->buf_size_max = max_buf_size;
  request->resp_buf = buf;
  
  return HTTP_SUCCESS;
}


/**
   picasa_http_set_progress_cb
*/
http_result_t
picasa_http_set_progress_cb (http_request_t *request, 
			     curl_progress_callback callback,
			     gpointer data) {

#ifdef DEBUG
  gchar *fn = "picasa_http_set_progress_cb()";
#endif

  if (request == NULL) {
    ULOG_ERR_L ("%s: NULL pointer in parameters",fn);
    return HTTP_ERROR;
  }

  if (callback) {
    request->progress_cb = callback;
  }
  if (data) {
    request->progress_data = data;
  }

  return HTTP_SUCCESS;
}


/**
   picasa_http_set_connection
*/
http_result_t
picasa_http_set_connection (http_request_t *request, 
			    ConIcConnection *con) {

#ifdef DEBUG
  gchar *fn = "picasa_http_set_connection()";
#endif

  if (request == NULL) {
    ULOG_ERR_L ("%s: NULL pointer in parameters",fn);
    return HTTP_ERROR;
  }

  if (con) {
    request->con = con;
  }

  return HTTP_SUCCESS;
}

/**
  picasa_http_send
*/
http_result_t
picasa_http_send (http_request_t *request,
		  http_type_t type) {
  
#ifdef DEBUG
  gchar *fn = "picasa_http_send()";
#endif

  if (request == NULL) {
    ULOG_ERR_L ("%s: empty request",fn);
    return HTTP_ERROR;
  }

  const gchar *proxy_str = NULL;
#ifndef USE_SBOX
  const gchar *proxy_host = NULL;
  gint proxy_port = 0;
  gint conn_handler = 0;
#endif
  
  guint status = 0;

  CURL *handle = NULL;

  handle = curl_easy_init ();
  if (handle == NULL) {
    ULOG_ERR_L ("%s: failed to create new CURL handle", fn);
    return HTTP_ERROR;
  }

  /* Options */
#ifdef DEBUG_CURL
  curl_easy_setopt (handle, CURLOPT_VERBOSE, 1);
  curl_easy_setopt (handle, CURLOPT_STDERR, stdout);
#else
  curl_easy_setopt (handle, CURLOPT_VERBOSE, 0);
#endif

  curl_easy_setopt (handle, CURLOPT_HEADER, 1);

  if (request->hdr) {
    curl_easy_setopt(handle, CURLOPT_HTTPHEADER, request->hdr);
  }

  /* HTTPS settings */
  if (g_str_has_prefix (request->url, "https://")) {
    curl_easy_setopt (handle, CURLOPT_SSL_VERIFYHOST, 0);
    curl_easy_setopt (handle, CURLOPT_SSL_VERIFYPEER, 0);
  }

  /* timeout for connection phase */ /* 10 sec */
  curl_easy_setopt (handle, CURLOPT_CONNECTTIMEOUT, 10);

  /* set url */
  curl_easy_setopt (handle, CURLOPT_URL, request->url);
  
  /* set callback writefunction */
  curl_easy_setopt (handle, CURLOPT_WRITEFUNCTION, picasa_http_write_cb);
  curl_easy_setopt (handle, CURLOPT_WRITEDATA, request);

  if (type == POST && (request->read_data.data != NULL)) {
    ULOG_DEBUG_L ("%s: setting as POST with read data", fn);
    curl_easy_setopt (handle, CURLOPT_READFUNCTION, picasa_http_read_cb);
    curl_easy_setopt (handle, CURLOPT_READDATA, request);
    curl_easy_setopt (handle, CURLOPT_HTTPPOST, NULL);
    curl_easy_setopt (handle, CURLOPT_POSTFIELDS, NULL);
    curl_easy_setopt (handle, CURLOPT_POSTFIELDSIZE,request->read_data.data->len);
  }
  else if (type == POST && request->post != NULL) {
    ULOG_DEBUG_L ("%s: setting as POST", fn);
    curl_easy_setopt (handle, CURLOPT_HTTPPOST, request->post);
    curl_easy_setopt (handle, CURLOPT_READFUNCTION, NULL);
    curl_easy_setopt (handle, CURLOPT_READDATA, NULL);
  }

  if (type == GET) {
    ULOG_DEBUG_L ("%s: setting as GET", fn);
    curl_easy_setopt (handle, CURLOPT_HTTPGET, 1);
  }

  if (type == PUT && request->read_data.data != NULL) {
    ULOG_DEBUG_L ("%s: setting as PUT with read data", fn);
    curl_easy_setopt (handle, CURLOPT_UPLOAD, 1);
    curl_easy_setopt (handle, CURLOPT_READFUNCTION, picasa_http_read_cb);
    curl_easy_setopt (handle, CURLOPT_READDATA, request);
    curl_easy_setopt (handle, CURLOPT_HTTPPOST, NULL);
    curl_easy_setopt (handle, CURLOPT_POSTFIELDS, NULL);
    curl_easy_setopt (handle, CURLOPT_POSTFIELDSIZE,request->read_data.data->len);
  }

  /* progress function for send */
  if (request->progress_cb != NULL) {
    ULOG_DEBUG_L ("%s: setting progress", fn);
    curl_easy_setopt (handle, CURLOPT_NOPROGRESS, 0);
    curl_easy_setopt (handle, CURLOPT_PROGRESSFUNCTION, request->progress_cb);
  } else { /* clear progress function */
    ULOG_DEBUG_L ("%s: clearing progress", fn);
    curl_easy_setopt (handle, CURLOPT_NOPROGRESS, 1);
    curl_easy_setopt (handle, CURLOPT_PROGRESSFUNCTION, NULL);
  }

  /* progress function data */
  if (request->progress_data != NULL) {
    ULOG_DEBUG_L ("%s: setting progress data", fn);
    curl_easy_setopt (handle, CURLOPT_PROGRESSDATA, request->progress_data);
  } 

#ifndef USE_SBOX
  /* Connect conic signals */
  if (request->con != NULL) {
    ULOG_DEBUG_L ("%s: setting conic", fn);
    conn_handler = g_signal_connect (request->con, "connection-event",
				     G_CALLBACK (picasa_http_connection_event_cb),
				     request);
  } else {
    ULOG_ERR_L ("%s, no connection available",fn);
    /* return HTTP_ERROR; */
  }
#endif

  /* if SBOX used then check env variable http_proxy for proxy config */
#ifdef USE_SBOX
  proxy_str = g_getenv ("http_proxy");
#else

  /* proxy in use */
  if (request->con != NULL) {
    if (con_ic_connection_get_proxy_mode (request->con) != CON_IC_PROXY_MODE_NONE) {
      proxy_host = con_ic_connection_get_proxy_host (request->con, 
						     CON_IC_PROXY_PROTOCOL_HTTP);
      proxy_port = con_ic_connection_get_proxy_port (request->con, 
						     CON_IC_PROXY_PROTOCOL_HTTP);
      proxy_str = g_strdup_printf ("http://%s:%d", proxy_host, proxy_port);
    }
  }
#endif
  
  if (proxy_str && (g_utf8_strlen (proxy_str,-1) > 1)) {
    ULOG_DEBUG_L ("%s: setting proxy to %s",fn,proxy_str);
    curl_easy_setopt (handle, CURLOPT_PROXY, proxy_str);
  }

  ULOG_DEBUG_L ("%s: setting up the request done",fn);

  request->state = SENDING;

  status = curl_easy_perform (handle);

  ULOG_DEBUG_L ("%s, curl returned %d",fn,status);  

  /* disconnect the conic signal */
#ifndef USE_SBOX
  if (request->con != NULL) {
    g_signal_handler_disconnect (request->con, conn_handler);
  }
#endif

  /* cleanup */
  curl_easy_cleanup (handle);  

  if (status != 0) {
    /* error while sending */
    status = HTTP_ERROR;
  }
  
  return status;
}


/**
  picasa_http_free
*/
void 
picasa_http_free (http_request_t *request) {

  if ((request->read_data).data != NULL) {
    g_string_free ((request->read_data).data, TRUE);
  }
  g_free (request->url);
  curl_formfree (request->post);
  curl_slist_free_all (request->hdr);
  g_free (request);
  request = NULL;
  
}









