/**
    File: picasa_post.c
    
    Author: Tero Niemel

*/



#include <glib.h>
#include <glib/gprintf.h>
#include <osso-log.h>
#include <string.h>
#include <stdlib.h>

#include <sharing-http.h>
#include <sharing-plugin-interface.h>
/* #include <sharing-service-option.h> */
#include <sharing-entry.h>
#include <sharing-tag.h>

#include "picasa_post.h"
#include "picasa_http.h"
#include "picasa_xml.h"


/**
   HTTP_RES_CODE_OFFSET_C
*/
#define HTTP_RES_CODE_OFFSET_C 9
/**
   HTTP_RES_ERR_OFFSET_C
*/
#define HTTP_RES_ERR_OFFSET_C 6
/**
   LOGIN_URL_C
*/
#define LOGIN_URL_C "https://www.google.com/accounts/ClientLogin"
/**
   ALBUM_URL_C
*/
#define FEED_URL_C "http://picasaweb.google.com/data/feed/api/user/default"
/**
   SERVICE_C
*/
#define SERVICE_C "lh2"
/**
   MAPS_URL_C
*/
#define MAPS_URL_C "http://maps.google.com/maps/geo?"


/** Globals **/

/**
   g_response_buffer
*/
GString *g_response_buffer = NULL;
/**
   g_auth
*/
gchar *g_auth = NULL;
/**
   g_photo_id
*/
gchar *g_photo_url = NULL;



/**
   picasa_post_progress_data_t
*/
typedef struct {
  gboolean *dead_mans_switch;
  gboolean *cont;
  SharingTransfer *transfer;
  SharingEntry *entry;
  SharingAccount *account;
  http_request_t *request;
  guint64 total_size;
  guint64 uploaded_size;
} picasa_post_progress_data_t;

/**
   picasa_post_response_type_t
*/
typedef enum {
  RES_LOGIN = 0x01,
  RES_GET_ALBUMS = 0x02,
  RES_UPLOAD = 0x03,
  RES_UPDATE = 0x04,
  RES_GET_COORDINATES = 0x05
} picasa_post_response_type_t;
  
/**
   picasa_post_tag_type_t
*/
typedef enum {
  NORMAL_TAG = 0x01,
  GPS_TAG = 0x02,
  GEO_TAG = 0x03,
  FAV_TAG = 0x04
} picasa_post_tag_type_t;

/**
   picasa_post_response_t
*/
typedef struct {
  gint code;
  gchar *msg;
} picasa_post_response_t;


/*
  PRIVATE functions 
*/


/**
   picasa_post_init_response_buffer
*/
static void
picasa_post_init_response_buffer () {

  g_response_buffer = g_string_new (NULL);
}



/**
   picasa_post_clear_response_buffer
*/
static void
picasa_post_clear_response_buffer () {

  g_string_free (g_response_buffer, TRUE);
  g_response_buffer = NULL;
}

/**
  picasa_post_free_globals
*/
static void
picasa_post_free_globals () {

  if (g_response_buffer) {
    picasa_post_clear_response_buffer ();
  }

  if (g_photo_url) {
    g_free (g_photo_url);
    g_photo_url = NULL;
  }

  if (g_auth) {
    g_free (g_auth);
    g_auth = NULL;
  }
}

/**
   picasa_post_convert_http_error
*/
static picasa_response_codes_t
picasa_post_convert_http_error (http_request_state_t state) {

  picasa_response_codes_t status = NETWORK_SEND_FAILED;

  switch (state) {
    
  case DISCONNECTED:
    status = CONNECTION_DISCONNECTED;
    break;
  case CANCELLED:
    status = CANCELLED_BY_USER;
    break;
  default:
    status = NETWORK_SEND_FAILED;
    break;
  }

  return status;
}

#if 0
/**
   picasa_post_convert_update_options
*/
static SharingPluginInterfaceUpdateOptionsResult
picasa_post_convert_update_options (gint status) {

  SharingPluginInterfaceUpdateOptionsResult result = 
    SHARING_UPDATE_OPTIONS_ERROR_UNKNOWN;

  switch (status) {

  case HTTP_OK:
    result = SHARING_UPDATE_OPTIONS_SUCCESS;
    break;
  case CANCELLED_BY_USER:
    result = SHARING_UPDATE_OPTIONS_CANCELLED;
    break;
    
  default:
    result = SHARING_UPDATE_OPTIONS_ERROR_UNKNOWN;
    break;

  }

  return result;
}
#endif


/**
   picasa_post_convert_validate_response
*/
static SharingPluginInterfaceAccountValidateResult
picasa_post_convert_validate_response (picasa_post_response_t res) {

  SharingPluginInterfaceAccountValidateResult result = 
    SHARING_ACCOUNT_VALIDATE_ERROR_UNKNOWN;

  switch (res.code) {
    
  case HTTP_OK:
    result = SHARING_ACCOUNT_VALIDATE_SUCCESS;
    break;
  case HTTP_FAILED:
    result = SHARING_ACCOUNT_VALIDATE_FAILED;
    break;
  case NETWORK_SEND_FAILED:
  case CONNECTION_DISCONNECTED:
    result = SHARING_ACCOUNT_VALIDATE_ERROR_CONNECTION;
    break;
  case CANCELLED_BY_USER:
    result = SHARING_ACCOUNT_VALIDATE_CANCELLED;
    break;
    /* rest is general error */
  default:
    result = SHARING_ACCOUNT_VALIDATE_ERROR_UNKNOWN;
    break;

  }

  /* clear response msg if any */
  if (res.msg != NULL) { 
    g_free (res.msg); 
    res.msg = NULL;
  }

  return result;
}


/**
   picasa_post_convert_send_response
*/
static SharingPluginInterfaceSendResult
picasa_post_convert_send_response (picasa_post_response_t res) {

  SharingPluginInterfaceSendResult result = 
    SHARING_SEND_ERROR_UNKNOWN;

  switch (res.code) {
  case HTTP_OK:
  case HTTP_CREATED:
    result = SHARING_SEND_SUCCESS;
    break;
  case HTTP_BAD_REQ:
    if (g_strcmp0 (res.msg, PICASA_POST_FILE_TOO_LARGE_C) == 0) {
      result = SHARING_SEND_ERROR_UPLOAD_LIMIT;
    }
    else {
      result = SHARING_SEND_ERROR_UNKNOWN;
    }
    break;

  case HTTP_FAILED:
    if (g_strcmp0 (res.msg, PICASA_POST_BAD_AUTHENTICATION_C) == 0) {
      result = SHARING_SEND_ERROR_AUTH;
    }
    else {
      result = SHARING_SEND_ERROR_UNKNOWN;
    }
    break;
  case HTTP_UNSUPPORTED_MEDIA:
    result = SHARING_SEND_ERROR_FILE_FORMAT;
    break;
  case NETWORK_SEND_FAILED:
  case CONNECTION_DISCONNECTED:
    result = SHARING_SEND_ERROR_CONNECTION;
    break;
  case CANCELLED_BY_USER:
    result = SHARING_SEND_CANCELLED;
    break;
    /* rest is general error */
  default:
    result = SHARING_SEND_ERROR_UNKNOWN;
    break;
  }

  /* clear response msg if any */
  if (res.msg != NULL) { 
    g_free (res.msg); 
    res.msg = NULL;
  }

  return result;
}

/**
  picasa_post_parse_login_response
*/
static picasa_post_response_t
picasa_post_parse_response (picasa_post_progress_data_t *data,
			    picasa_post_response_type_t type) {

#ifdef DEBUG
  gchar *fn = "picasa_post_parse_response()";
#endif

  ULOG_DEBUG_L ("%s: Entered", fn);

  ULOG_DEBUG_L ("%s: Response: \n%s", fn, g_response_buffer->str);
  
  gchar **tokens = NULL;
  gchar *iter = NULL;
  gchar *tmp = (gchar*)g_response_buffer->str;
  gint count = 0;
  guint offset = 0;
  GSList *albums = NULL;
  gboolean success = FALSE;
  picasa_post_response_t res = { 0, NULL };

  tokens = g_strsplit_set ((gchar*)g_response_buffer->str,
                           "\n", /* split based on linefeed */
                           0);

  iter = tokens[0];
  while (iter) {

    if (g_str_has_prefix (iter, "HTTP/1.1 ")) {
      res.code = atoi (g_strndup (iter + HTTP_RES_CODE_OFFSET_C, 3));
    }

    if (g_str_has_prefix (iter, "Error=")) {
      res.msg = g_strdup (iter + HTTP_RES_ERR_OFFSET_C);
      ULOG_DEBUG_L ("%s: %s", fn, res.msg);
      break;
    }

    if (g_str_has_prefix (iter, "Auth=")) {
      g_auth = g_strdup (iter + 5);
      ULOG_DEBUG_L ("%s: Auth = %s", fn, g_auth);
      break;
    }

    if (g_str_has_prefix (iter, "<?xml")) {
      ULOG_DEBUG_L ("%s: Body found", fn);
      if (type == RES_GET_ALBUMS) {
	albums = picasa_xml_parse_album_response (iter);
	ULOG_DEBUG_L ("%s: albums parsed (%0X)", 
		      fn, (guint)albums);
      }
      else if (type == RES_UPLOAD) {
	g_photo_url = picasa_xml_parse_upload_response (iter);
	ULOG_DEBUG_L ("%s: uploaded photo id parsed (%s)", 
		      fn, g_photo_url);
      }
      else if (type == RES_GET_COORDINATES) {
	res.msg = picasa_xml_parse_gps_response ((tmp+(offset+count)));
	ULOG_DEBUG_L ("%s: coordinates parsed = %s", 
		      fn, res.msg);
      }
      break;
    }

    /* calculate offset from beginning */
    offset = offset + strlen (iter);

    count++;
    iter = tokens[count];

    /* if last line and status failed */
    if (res.code == HTTP_BAD_REQ && iter == NULL) {
      res.msg = g_strdup (tokens[count-1]);
      ULOG_DEBUG_L ("%s: last line = %s", fn, res.msg);
    }

  }
  
#if 0
  if (albums != NULL && res.code == HTTP_OK) {
    ULOG_DEBUG_L ("%s: setting up account option values (%d) to account %s", 
		  fn, g_slist_length (albums), 
		  sharing_account_get_param (data->account, "username"));
    
    success = sharing_account_set_option_values (data->account,
						 "albums",
						 albums);
    if (success == FALSE) {
      ULOG_DEBUG_L ("%s: failed to update albums",fn);
      res.code = GENERAL_ERROR;
    }

    sharing_service_option_values_free (albums);
    
  }
#endif
  
  g_strfreev (tokens);

  ULOG_DEBUG_L ("%s: Leaving", fn);
  
  return res;
}

/**
  picasa_post_progress
*/
static int
picasa_post_progress (void *clientp,
		      double dltotal,
		      double dlnow,
		      double ultotal,
		      double ulnow) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_progress()";
#endif

  picasa_post_progress_data_t *data = (picasa_post_progress_data_t*)clientp;
  gdouble progress = 0.0;

  ULOG_DEBUG_L ("%s: data = %0X ; data->transfer = %0X ; data->cont = %0X ; data->request = %0X",
		fn, (guint)data, (guint)data->transfer, (guint)data->cont, (guint)data->request);


  /* Check has user cancelled the transfer */
  if (data->transfer != NULL) {
    if (sharing_transfer_continue (data->transfer) == FALSE) {
      ULOG_DEBUG_L ("%s: user cancelled transfer",fn);
      data->request->state = CANCELLED;
      return 1;
    }
  }

  /* dialog cancelled */
  if (data->cont != NULL) {
    if (*(data->cont) == FALSE) {
      ULOG_DEBUG_L ("%s: upload cancelled",fn);
      return 1;
    }
  }

  /* check for conic disconnect */
  if (data->request->state == DISCONNECTED) {
    ULOG_DEBUG_L ("%s: connection disconnected", fn);
    return 1;
  }

  /* if all fine update progress */
  progress = (data->uploaded_size + ulnow) / data->total_size;
  if (progress > 1.0) {
    ULOG_DEBUG_L ("%s: FULL PROGRESS", fn);
    progress = 1.0;
  }

  ULOG_DEBUG_L ("%s: progress = %f", fn,progress);

  sharing_transfer_set_progress (data->transfer, progress);

  /* pull dead_mans_switch */
  *(data->dead_mans_switch) = FALSE;

  return 0;

}


/**
   picasa_post_login
*/
static picasa_post_response_t
picasa_post_login (http_session_t session,
		   ConIcConnection *con,
		   picasa_post_progress_data_t *data) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_login()";
#endif

  gint status = 0;
  picasa_post_response_t res = { GENERAL_ERROR, NULL };

  gchar *uname = sharing_account_get_param (data->account, "username");
  gchar *passwd = sharing_account_get_param (data->account, "password");

  http_request_t *http = picasa_http_new (LOGIN_URL_C);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload",fn);
    return res;
  }

  data->request = http;

  picasa_http_add_post_data (http, "accountType", "HOSTED_OR_GOOGLE");
  picasa_http_add_post_data (http, "Email", uname);
  picasa_http_add_post_data (http, "Passwd", passwd);
  picasa_http_add_post_data (http, "service", SERVICE_C);
  picasa_http_add_post_data (http, "source", "remote uploader");

  picasa_post_init_response_buffer ();

  picasa_http_set_response_buffer (http, 
				   PICASA_POST_BUF_LIMIT_C ,
				   g_response_buffer);
  
  picasa_http_set_connection (http, con);
  /* picasa_http_set_progress_cb (http, picasa_post_progress, data); */

  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post to %s",fn,LOGIN_URL_C);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (session, http, POST);

  *data->dead_mans_switch = FALSE;
  
  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  /* if sending post was success parse response */
  if (status == 0) {
    res = picasa_post_parse_response (data, RES_LOGIN);
  }
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
  }

  /* free memory and buffers */
  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (passwd);
  g_free (uname);

  return res;
}


#if 0
/**
  picasa_post_get_albums
*/
static picasa_post_response_t 
picasa_post_get_albums (http_session_t session,
			ConIcConnection *con,
			picasa_post_progress_data_t *data) {

#ifdef DEBUG
  gchar *fn = "picasa_post_get_albums()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };

  gchar *auth = g_strconcat ("GoogleLogin auth=", g_auth, NULL);
  http_request_t *http = http = picasa_http_new (FEED_URL_C);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_header_data (http, "GData-Version", "2");
  picasa_http_add_header_data (http, "Authorization", auth);

  picasa_post_init_response_buffer ();

  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);

  picasa_http_set_connection (http, con);
  
  picasa_http_set_progress_cb (http, picasa_post_progress, data);

  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post to %s", fn, FEED_URL_C);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (session, http, GET);

  *data->dead_mans_switch = FALSE;

  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  if (status == 0) {
    res = picasa_post_parse_response (data, RES_GET_ALBUMS);
  } 
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
  }

  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (auth);
  
  return res;
}
#endif

/**
  picasa_post_upload_item
*/
static picasa_post_response_t
picasa_post_upload_item (http_session_t session, 
			 SharingEntryMedia *media,
			 ConIcConnection *con,
			 picasa_post_progress_data_t *data) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_upload_item()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };
  gchar *metadata, *tag_entry = NULL;
  gchar *length = NULL;

  /* GSList *tags = (GSList*)sharing_entry_media_get_tags (media); */
  gchar *mime = sharing_entry_media_get_mime (media);
  guint64 size = sharing_entry_media_get_size (media);
  gchar *filename = sharing_entry_media_get_filename (media);
  gchar *title = sharing_entry_media_get_title (media);
  gchar *desc = (gchar*)sharing_entry_media_get_desc (media);

  gchar *album_id = "default"; /* (gchar*)sharing_entry_get_option (data->entry, "albums"); */
  
  gchar *url = g_strconcat (FEED_URL_C, "/albumid/", album_id, NULL);
  gchar *auth = g_strconcat ("GoogleLogin auth=", g_auth, NULL);

  ULOG_DEBUG_L ("%s: uploading to url %s", fn, url);

  http_request_t *http = picasa_http_new (url);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_header_data (http, "GData-Version", "2");
  picasa_http_add_header_data (http, "Authorization", auth);

  if (title == NULL) { title = filename; }

  metadata = picasa_xml_create_metadata_entry (title, desc);
  
#if 0 /* testing how to upload image with all metadata at once */
  metadata = g_strdup_printf ("<?xml version=\"1.0\"?>\n"
			      "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n"
			      "xmlns:media=\"http://search.yahoo.com/mrss/\"\n"
			      "xmlns:georss=\"http://www.georss.org/georss\"\n"
			      "xmlns:gphoto=\"http://schemas.google.com/photos/2007\"\n"
			      "xmlns:gml=\"http://www.opengis.net/gml\">\n"
			      "<title>plz-to-love-realcat.jpg</title>\n"
			      "<summary>Real cat wants attention too.</summary>\n"
			      "<gphoto:location>Finland, Helsinki, Ruoholahti</gphoto:location>"
			      "<media:group>\n"
			      /*
			      "<media:title type=\"plain\">new_title</media:title>\n"
			      "<media:description type=\"plain\">new_description</media:description>\n"
			      */
			      "<media:keywords>tag, tag3</media:keywords>\n"
			      "</media:group>\n"
			      "<georss:where>\n"
			      "<gml:Point>\n"
			      "<gml:pos>35.669998 139.770004</gml:pos>\n"
			      "</gml:Point>\n"
			      "</georss:where>\n"
			      "<category scheme=\"http://schemas.google.com/g/2005#kind\"\n"
			                "term=\"http://schemas.google.com/photos/2007#photo\"/>"
			      "</entry>\n");
#endif

  if (metadata != NULL) {

    picasa_http_add_post_data_with_type (http, metadata, "application/atom+xml");

    picasa_http_add_bin_file (http, sharing_entry_media_get_localpath (media),
			      NULL, mime);
    g_free (metadata); 
    g_free (tag_entry);

    /*
    picasa_http_add_header_data (http, "Content-Type", 
                                 "multipart/related; boundary=\"RANDOM_BOUNDRY\"");
    picasa_http_append_prefix (http, "\n--RANDOM_BOUNDRY\n"); 
    picasa_http_append_prefix (http, "Content-Type: application/atom+xml\n\n"); 
    picasa_http_append_prefix (http, metadata); 
    picasa_http_append_prefix (http, "--RANDOM_BOUNDRY\n"); 
    picasa_http_append_prefix (http, "Content-Type: "); 
    picasa_http_append_prefix (http, mime); 
    picasa_http_append_prefix (http, "\n\n"); 
    picasa_http_set_bin_file (http,  
			      sharing_entry_media_get_localpath (media), 
			      size); 
    picasa_http_append_suffix (http, "\n--RANDOM_BOUNDRY--\n"); 
    length = g_strdup_printf("%d", picasa_http_get_body_length (http)); 
    ULOG_DEBUG_L ("%s: total length = %d", fn, picasa_http_get_body_length (http)); 
    picasa_http_add_header_data (http, "Content-Length", length);
    */

    picasa_http_add_header_data (http, "MIME-version", "1.0");
    picasa_http_add_header_data (http, "Content-Type", "multipart/related");

    /* ULOG_DEBUG_L ("%s: multipart prefix = %s", fn, (http->prefix).data->str); */

  } else {
    picasa_http_add_header_data (http, "Content-Type", mime);
    length = g_strdup_printf("%lu", (unsigned long int)size);
    picasa_http_add_header_data (http, "Content-Length", length);
    picasa_http_add_header_data (http, "Slug", filename);

    picasa_http_set_bin_file (http, 
			      sharing_entry_media_get_localpath (media),
			      size);
  }
  
  picasa_post_init_response_buffer ();
  
  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);
  
  picasa_http_set_connection (http, con);
  
  picasa_http_set_progress_cb (http, picasa_post_progress, data);
  
  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post to %s", fn, url);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (session, http, POST);

  *data->dead_mans_switch = FALSE;

  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  /* send successfully, check picasa response */
  if (status == 0) {
    res = picasa_post_parse_response (data, RES_UPLOAD);
  }
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
    ULOG_ERR_L ("%s: transfer error %d", fn, res.code);
  }

  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (url);
  g_free (auth);
  g_free (length);

  return res;
}


/**
  picasa_post_update_media
*/
static picasa_post_response_t
picasa_post_update_media (http_session_t session,
			  gchar *tags,
			  gchar *gps,
			  ConIcConnection *con,
			  picasa_post_progress_data_t *data) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_update_media()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };
  gchar *length = NULL;
  gchar *media = NULL;

  gchar *auth = g_strconcat ("GoogleLogin auth=", g_auth, NULL);

  if (g_photo_url == NULL) {
    ULOG_WARN_L ("%s: photo url NULL -> tags will not be updated", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  ULOG_DEBUG_L ("%s: uploading to url %s", fn, g_photo_url);

  http_request_t *http = picasa_http_new (g_photo_url);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_header_data (http, "GData-Version", "2");
  picasa_http_add_header_data (http, "Authorization", auth);
  picasa_http_add_header_data (http, "Content-Type", "application/atom+xml");
  picasa_http_add_header_data (http, "If-Match", "*");
  picasa_http_add_header_data (http, "Expect", "Expect: 100-continue");

  media = picasa_xml_create_media_entry (tags, gps);

#if 0
  media = g_strdup_printf ("<?xml version=\"1.0\"?>\n"
			   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n"
			   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n"
			   "xmlns:georss=\"http://www.georss.org/georss\"\n"
			   "xmlns:gphoto=\"http://schemas.google.com/photos/2007\"\n"
			   "xmlns:gml=\"http://www.opengis.net/gml\">\n"
			   "<media:group>\n"
			   "<media:keywords>tag, tag3</media:keywords>\n"
			   "</media:group>\n"
			   "<georss:where>\n"
			   "<gml:Point>\n"
			   "<gml:pos>35.669998 139.770004</gml:pos>\n"
			   "</gml:Point>\n"
			   "</georss:where>\n"
			   "<category scheme=\"http://schemas.google.com/g/2005#kind\"\n"
			   "term=\"http://schemas.google.com/photos/2007#photo\"/>"
			   "</entry>\n");
#endif

  picasa_http_append_prefix (http, media);
  length = g_strdup_printf("%d", http->prefix.data->len);
  picasa_http_add_header_data (http, "Content-Length", length);
  g_free (media);

  picasa_post_init_response_buffer ();
  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);
  
  picasa_http_set_connection (http, con);

  ULOG_DEBUG_L ("%s: Sending post to %s", fn, g_photo_url);
  *data->dead_mans_switch = FALSE;
  status = picasa_http_send (session, http, PUT);
  *data->dead_mans_switch = FALSE;
  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  /* send successfully, check picasa response */
  if (status == 0) {
    res = picasa_post_parse_response (data, RES_UPDATE);
  } else {
    ULOG_WARN_L ("%s: failed to update media", fn);
  }

  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (auth);

  return res;
}

/**
  picasa_post_get_gps_coordinates
*/
static picasa_post_response_t 
picasa_post_get_gps_coordinates (ConIcConnection *con,
				 picasa_post_progress_data_t *data,
				 gchar *country,
				 gchar *city,
				 gchar *suburb) {

#ifdef DEBUG
  gchar *fn = "picasa_post_get_gps_coordinates()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };

  GString *url = g_string_new (MAPS_URL_C);

  g_string_append (url, "q=");
  if (country != NULL) {
    g_string_append (url, "+");
    g_string_append (url, country);
  }
  if (city != NULL) {
    g_string_append (url, "+");
    g_string_append (url, city);
  }
  if (suburb != NULL) {
    g_string_append (url, "+");
    g_string_append (url, suburb);
  }
  g_string_append (url, "&output=xml");
  g_string_append (url, "&sensor=false");

  ULOG_DEBUG_L ("%s: created url = %s", fn, url->str);

  http_request_t *http = picasa_http_new (url->str);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_post_init_response_buffer ();

  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);

  picasa_http_set_connection (http, con);
  
  /* picasa_http_set_progress_cb (http, picasa_post_progress, data); */

  /* send http post */
  ULOG_DEBUG_L ("%s: Sending post", fn);

  *data->dead_mans_switch = FALSE;

  status = picasa_http_send (NULL, http, GET);

  *data->dead_mans_switch = FALSE;

  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  if (status == 0) {
    res = picasa_post_parse_response (data, RES_GET_COORDINATES);
  } 
  /* curl send problem or connection disconnected */
  else {
    res.code = picasa_post_convert_http_error (http->state);
  }

  picasa_post_clear_response_buffer ();
  g_string_free (url, TRUE);
  
  return res;
}

#if 0
/**
  picasa_post_add_tags
*/
static picasa_post_response_t
picasa_post_add_tag (http_session_t session, 
		     gchar *tag,
		     gint type,
		     ConIcConnection *con,
		     picasa_post_progress_data_t *data) {
  
#ifdef DEBUG
  gchar *fn = "picasa_post_add_tag()";
#endif

  gint status = 0;
  picasa_post_response_t res = { 0, NULL };
  gchar *tag_entry = NULL;
  gchar *length = NULL;

  gchar *auth = g_strconcat ("GoogleLogin auth=", g_auth, NULL);

  if (g_photo_url == NULL) {
    ULOG_WARN_L ("%s: photo url NULL -> tags will not be updated", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  ULOG_DEBUG_L ("%s: uploading to url %s", fn, g_photo_url);

  http_request_t *http = picasa_http_new (g_photo_url);

  if (!http) {
    ULOG_ERR_L ("%s: failed to allocate new http request, aborting upload", fn);
    res.code = GENERAL_ERROR;
    return res;
  }

  data->request = http;

  picasa_http_add_header_data (http, "GData-Version", "2");
  picasa_http_add_header_data (http, "Authorization", auth);
  picasa_http_add_header_data (http, "Content-Type", "application/atom+xml");
  picasa_http_add_header_data (http, "Expect", "Expect: 100-continue");

  if (type == NORMAL_TAG) {
    tag_entry = picasa_xml_create_tag_entry (tag);
  }
  else if (type == GPS_TAG) {
    tag_entry = picasa_xml_create_gpstag_entry (tag);
  } 
  else {
    ULOG_DEBUG_L ("%s: unknown type", fn);
    picasa_http_free (http);
    res.code = GENERAL_ERROR;
    return res;
  }

  picasa_http_append_prefix (http, tag_entry);
  length = g_strdup_printf("%d", http->prefix.data->len);
  picasa_http_add_header_data (http, "Content-Length", length);
  g_free (tag_entry);

  picasa_post_init_response_buffer ();
  picasa_http_set_response_buffer (http,
				   PICASA_POST_BUF_LIMIT_C,
				   g_response_buffer);
  
  picasa_http_set_connection (http, con);

  ULOG_DEBUG_L ("%s: Sending post to %s", fn, g_photo_url);
  *data->dead_mans_switch = FALSE;
  status = picasa_http_send (session, http, POST);
  *data->dead_mans_switch = FALSE;
  ULOG_DEBUG_L ("%s: sharing_http_request_send returned %d",fn,status);

  /* send successfully, check picasa response */
  if (status == 0) {
    res = picasa_post_parse_response (data, RES_UPDATE);
  } else {
    ULOG_WARN_L ("%s: failed to update tag %s", fn, tag);
  }

  picasa_http_free (http);
  picasa_post_clear_response_buffer ();
  g_free (auth);

  return res;
}
#endif


/**
   picasa_post_add_tags
*/
static gint
picasa_post_add_tags (http_session_t session,
		      SharingEntryMedia *media,
		      ConIcConnection *con,
		      picasa_post_progress_data_t *data) {
#ifdef DEBUG
  gchar *fn = "picasa_post_add_tags()";
#endif

  ULOG_DEBUG_L ("%s: entered", fn);
  SharingTag *tag = NULL;
  GString *tag_str = NULL;
  gboolean first = TRUE;
  picasa_post_response_t res = { 0, NULL };
  GSList *t = NULL;
  const gchar *geo_country = NULL;
  const gchar *geo_city = NULL; 
  const gchar *geo_suburb = NULL;
  gchar *tags = NULL;

  GSList *media_tags = (GSList*)sharing_entry_media_get_tags (media);

  for (t = media_tags; t != NULL; t = g_slist_next(t)) {

    tag = (SharingTag*)t->data;
    
    ULOG_DEBUG_L ("%s: tag with type %d", fn, sharing_tag_get_type (tag));

    switch (sharing_tag_get_type (tag)) {

      /* normal tags */
    case SHARING_TAG_SHARE:
      {
	if(first == TRUE) {
	  tag_str = g_string_new (sharing_tag_get_word (tag));
	  first = FALSE;
	  tags = tag_str->str;
	} else {
	  g_string_append (tag_str, ",");
	  g_string_append (tag_str, sharing_tag_get_word (tag));
	}
	
	ULOG_DEBUG_L ("%s: found new tag = %s", fn, tag_str->str);
	/*
	  res = picasa_post_add_tag (session, (gchar*)tag_name, 
	  NORMAL_TAG, con, data);
	*/
	break;
      }

    case SHARING_TAG_GEO_COUNTRY:
      {
	geo_country = sharing_tag_get_word (tag);
	ULOG_DEBUG_L ("%s: found geo country = %s", fn, geo_country);
	break;
      }

    case SHARING_TAG_GEO_CITY:
      {
	geo_city = sharing_tag_get_word (tag);
	ULOG_DEBUG_L ("%s: found geo city = %s", fn, geo_city);
	break;
      }

    case SHARING_TAG_GEO_SUBURB:
      {
	geo_suburb = sharing_tag_get_word (tag);
	ULOG_DEBUG_L ("%s: found geo suburb = %s", fn, geo_suburb);
	break;
      }

    default:
      {
	ULOG_DEBUG_L ("%s: unknown tag type", fn);
      }
      
    } /* switch */
  } /* for */ 

  /* in case some of geostrings found try to get coordinates */
  if (geo_country != NULL || geo_city != NULL || geo_suburb != NULL) {

    ULOG_DEBUG_L ("%s: trying to get coordinates", fn);
    
    res = picasa_post_get_gps_coordinates (con, data,
					   (gchar*)geo_country,
					   (gchar*)geo_city,
					   (gchar*)geo_suburb);

    if (res.code == HTTP_OK &&
	res.msg != NULL) {
      ULOG_DEBUG_L ("%s: found coordinate = %s", fn, res.msg);
      
      /*
	res = picasa_post_add_tag (session, res.msg, 
	GPS_TAG, con, data);
      */
    }
    
  }

  /* if tags or coordinates to update */
  if (tags != NULL || res.msg != NULL) {
    ULOG_DEBUG_L ("%s: trying to update media entry", fn);
    res = picasa_post_update_media (session, tags, res.msg,
				    con, data);

    ULOG_DEBUG_L ("%s: media entry updated with status = %d", fn, res.code);
  }

  ULOG_DEBUG_L ("%s: leaving", fn);

  /* cleanup */
  g_string_free (tag_str, TRUE);
  if (res.msg != NULL) { 
    g_free (res.msg); 
    res.msg = NULL;
  }

  return res.code;
}


#if 0
/**
   picasa_post_update_albums
*/
SharingPluginInterfaceUpdateOptionsResult
picasa_post_update_albums (SharingAccount *account,
			   ConIcConnection *con,
			   gboolean *cont,
			   gboolean *dead_mans_switch) {

#ifdef DEBUG
  gchar *fn = "picasa_post_update_albums()";
#endif

  ULOG_DEBUG_L ("Entered %s",fn);

  SharingPluginInterfaceUpdateOptionsResult
    status = 0;
  picasa_post_response_t res = { 0, NULL };

  picasa_post_progress_data_t data;

  *dead_mans_switch = FALSE;

  /* fill data structure */
  data.dead_mans_switch = dead_mans_switch;
  data.cont = cont;
  data.account = account;
  data.transfer = NULL;
  data.entry = NULL;
  data.request = NULL;

  http_session_t session = picasa_http_init ();

  ULOG_DEBUG_L ("%s: LOGIN",fn);  

  /* login to validate */
  res = picasa_post_login (session, con, &data);

  /* fetch albums for updating them */
  ULOG_DEBUG_L ("%s: GET ALBUMS",fn);

  res = picasa_post_get_albums (session, con, &data);

  status = picasa_post_convert_update_options (res.code);

  picasa_http_uninit (session);
  picasa_post_free_globals ();
  if (res.msg != NULL) {
    g_free (res.msg);
    res.msg = NULL;
  }

  return status;
}
#endif


/*
  PUBLIC functions
*/


#if 0
/**
   picasa_post_update_options
*/
gpointer
picasa_post_update_options (gpointer data) {

#ifdef DEBUG
  gchar *fn = "picasa_post_update_options()";
#endif

  picasa_options_data_t *opt_data = (picasa_options_data_t*)data;
  SharingPluginInterfaceUpdateOptionsResult status = 
    SHARING_UPDATE_OPTIONS_ERROR_UNKNOWN;

  ULOG_DEBUG_L ("%s: entered",fn);
  
  status = picasa_post_update_albums (opt_data->account,
				      opt_data->con,
				      opt_data->cont,
				      opt_data->dead_mans_switch);
  
  ULOG_DEBUG_L ("%s: updated options with status %d",fn,status);
 
  if (opt_data->cb_func != NULL) {
    /* callback */
    void (*fp) (SharingPluginInterfaceUpdateOptionsResult, gpointer);
    /* UpdateOptionsCallback*/ 
    fp = opt_data->cb_func;
    fp (status, opt_data->cb_data);
  } else {
    ULOG_CRIT_L ("%s: callback pointer NULL", fn);
  }

  /* free data */
  /* g_free (opt_data); */

  opt_data->active = FALSE;

  ULOG_DEBUG_L ("%s: update_options DONE", fn);

  return NULL;  
}
#endif

/**
   picasa_post_validate_account
*/
SharingPluginInterfaceAccountValidateResult
picasa_post_validate_account (SharingAccount *account,
			       ConIcConnection *con,
			       gboolean *cont,
			       gboolean *dead_mans_switch) {

#ifdef DEBUG
  gchar *fn = "picasa_post_validate_account()";
#endif

  ULOG_DEBUG_L ("Entered %s",fn);

  SharingPluginInterfaceAccountValidateResult
    status = SHARING_ACCOUNT_VALIDATE_FAILED;
  picasa_post_response_t res = { 0, NULL };

  picasa_post_progress_data_t data;

  *dead_mans_switch = FALSE;

  /* fill data structure */
  
  data.dead_mans_switch = dead_mans_switch;
  data.cont = cont;
  data.account = account;
  data.transfer = NULL;
  data.entry = NULL;
  data.request = NULL;

  http_session_t session = picasa_http_init ();

  ULOG_DEBUG_L ("%s: LOGIN",fn);  

  /* login to validate */
  res = picasa_post_login (session, con, &data);

  status = picasa_post_convert_validate_response (res);

  picasa_http_uninit (session);
  picasa_post_free_globals ();

  return status;
}


/**
   picasa_post_send_to_service
*/
SharingPluginInterfaceSendResult 
picasa_post_upload_to_service (SharingTransfer *transfer,
			      ConIcConnection *con,
			      gboolean *dead_mans_switch) {

#ifdef DEBUG
  gchar *fn = "picasa_post_upload_to_service()";
#endif

  ULOG_DEBUG_L ("Entered %s",fn);

  gint status = 0;
  picasa_post_response_t res;
  picasa_post_progress_data_t data;
  GSList *m = NULL;

  /* fill data structure */
  data.dead_mans_switch = dead_mans_switch;
  data.transfer = transfer;
  data.entry = sharing_transfer_get_entry (transfer);
  data.account = sharing_entry_get_account (data.entry);
  data.request = NULL;
  data.total_size = sharing_entry_get_size (data.entry);
  data.uploaded_size = 0;
  data.cont = NULL;

  if (!data.account) {
    ULOG_DEBUG_L ("%s: no account found",fn);
    return SHARING_SEND_ERROR_ACCOUNT_NOT_FOUND;
  }

  *dead_mans_switch = FALSE;
  
  http_session_t session = NULL; /* picasa_http_init (); */

  ULOG_DEBUG_L ("%s: LOGIN",fn);

  res = picasa_post_login (session, con, &data);
  
  if (res.code != HTTP_OK) {
    ULOG_DEBUG_L ("%s: login failed %d", fn, res.code);
    return picasa_post_convert_send_response (res);
  }

  /* do we want to update album list every time new upload
     is started?? */
#if 0
  ULOG_DEBUG_L ("%s: FETCH ALBUMS",fn);

  res = picasa_post_get_albums (session, con, &data);

  if (status != 0) {
    ULOG_DEBUG_L ("%s: fetch albums failed %d",fn,status);
    return picasa_post_convert_send_response (res);
  } 
#endif

  /* post all items in loop */
  for (m = sharing_entry_get_media(data.entry); 
       m != NULL; m = g_slist_next(m)) {
    SharingEntryMedia *media = m->data;

    /* if not sent yet */
    if (!sharing_entry_media_get_sent( media )) {

      ULOG_DEBUG_L ("%s: UPLOAD ITEM",fn);

      res = picasa_post_upload_item (session, media, con, &data);
      if (res.code == HTTP_OK || res.code == HTTP_CREATED) {
	data.uploaded_size += sharing_entry_media_get_size (media);
	sharing_entry_media_set_sent (media, TRUE);
	/* try to update the tags */
	ULOG_DEBUG_L ("%s: UPDATE TAGS", fn);
	picasa_post_add_tags (session, media, con, &data);
	/* ULOG_DEBUG_L ("%s: tags updated with status = %d", fn, status); */
	status = SHARING_SEND_SUCCESS;
      } else {
	ULOG_DEBUG_L ("%s: sending to service failed %d",fn,status);
	status = picasa_post_convert_send_response (res);
	break;
      }
    }
    
  }

  /* picasa_http_uninit (session); */
  picasa_post_free_globals ();

  return status;
}


/* EOF */

