/*
 * vim:ts=4:sw=4:expandtab
 * This file is part of sharing-plugin-template
 *
 * Copyright (C) 2008-2009 Nokia Corporation. All rights reserved.
 *
 * This maemo code example is licensed under a MIT-style license,
 * that can be found in the file called "COPYING" in the root
 * directory.
 *
 */
#define DEBUG

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <glib.h>
#include <osso-log.h>
#include <sharing-http.h>
#include "send.h"
#include <neon/ne_session.h>
#include <neon/ne_request.h>
#include <neon/ne_auth.h>

#include "common.h"

struct webdav_transfer {
    int fd;
    ne_off_t total_sent;
    ne_off_t total_bytes;
    SharingTransfer *transfer;
    gboolean *dead_mans_switch;
};

/*
 * Just a wrapper around read(), but sets dead_mans_switch to FALSE (to
 * indicate that we are still doing something) and updates the progress (of the
 * whole transfer, not only this specific file).
 *
 */
static ssize_t prov(void *userdata, char *buffer, size_t buflen) {
    struct webdav_transfer *s = userdata;
    if (buflen == 0)
        return 0;
    s->dead_mans_switch = FALSE;
    sharing_transfer_set_progress(s->transfer, (float)s->total_sent / (float)s->total_bytes);
    ssize_t r = read(s->fd, buffer, buflen);
    s->total_sent += r;
    return r;
}


SharingPluginInterfaceSendResult send_foo(SharingTransfer *transfer,
    ConIcConnection *con, gboolean *dead_mans_switch)
{
    ULOG_DEBUG_L("Sending file via WebDAV");
    SharingEntry *entry = sharing_transfer_get_entry(transfer);
    SharingAccount *account = sharing_entry_get_account(entry);

    *dead_mans_switch = FALSE;
    struct webdav_transfer dtransfer;
    dtransfer.total_sent = 0;
    dtransfer.total_bytes = 0;
    dtransfer.transfer = transfer;
    dtransfer.dead_mans_switch = dead_mans_switch;

    /* get the total amount of bytes for this transfer (it might consist of
     * multiple media files) */
    for (GSList* p = sharing_entry_get_media (entry); p != NULL; p = g_slist_next(p)) {
        SharingEntryMedia *media = p->data;
        if (sharing_entry_media_get_sent(media))
            continue;

        dtransfer.total_bytes += sharing_entry_media_get_size(media);
    }

    /* initialize the WebDAV session */
    gchar *davpath = NULL;
    ne_session *sess = new_webdav_session(account, &davpath);

    /* upload all media files */
    for (GSList* p = sharing_entry_get_media (entry); p != NULL; p = g_slist_next(p)) {
        SharingEntryMedia *media = p->data;

        if (sharing_entry_media_get_sent(media)) {
            ULOG_DEBUG_L("media already sent, skipping");
            continue;
        }

        ULOG_DEBUG_L("Opening file %s", sharing_entry_media_get_localpath(media));
        int fd = open(sharing_entry_media_get_localpath(media), O_RDONLY | O_LARGEFILE);
        if (fd == -1) {
            ULOG_ERR_L("could not open local file %s", sharing_entry_media_get_localpath(media));
            g_free(davpath);
            ne_session_destroy(sess);
            return SHARING_SEND_ERROR_UNKNOWN;
        }

        gchar *path = g_strdup_printf("%s/%s", davpath, sharing_entry_media_get_filename(media));
        ULOG_DEBUG_L("Uploading to %s", path);
        ne_request *req = ne_request_create(sess, "PUT", path);
        g_free(path);

        dtransfer.fd = fd;
        ne_set_request_body_provider(req, sharing_entry_media_get_size(media), prov, &dtransfer);

        if (ne_request_dispatch(req) != 0)  {
            ULOG_ERR_L("PUT request failed: %s", ne_get_error(sess));
            g_free(davpath);
            ne_request_destroy(req);
            ne_session_destroy(sess);
            close(fd);
            return SHARING_SEND_ERROR_CONNECTION;
        }
        close(fd);

        ne_request_destroy(req);

        sharing_entry_media_set_sent(media, TRUE);
        ULOG_DEBUG_L("media marked as sent");
    }

    ne_session_destroy(sess);
    g_free(davpath);

    sharing_transfer_set_progress(transfer, 1.0);
    *dead_mans_switch = FALSE;
    return SHARING_SEND_SUCCESS;
}
