#include <glib.h>
#include <sharing-http.h>
#include <osso-log.h>
#include <openssl/ssl.h>
#include <string.h>

#include "cJSON.h"

#include "mobilblogg.h"

static gboolean mobilblogg_request_data (mobilblogg_t *mobblogg, gchar *url, gchar **data, gsize *size);
static gchar *mobilblogg_sha1_hash (const gchar *salt, const gchar *password);
static gchar *mobilblogg_call_login (mobilblogg_t *mobblogg, const gchar *passwordhash);
static gchar *mobilblogg_get_salt (mobilblogg_t *mobblogg);
static guint parse_send_content (const gchar *content);

struct mobilblogg_St {
    ConIcConnection *con;
    gchar *cookie;
    gchar *username;
    gchar *password;
    gchar *secretword;
    mobilblogg_error_code error;
};

static gboolean
mobilblogg_request_data_and_header (mobilblogg_t *mobblogg, gchar *url, gchar **data, gsize *size,
                                    gchar **header, gsize *header_size)
{
    SharingHTTP *http = sharing_http_new ();
    SharingHTTPRunResponse res = 0;
    sharing_http_set_connection (http, mobblogg->con);
    
    ULOG_INFO_F("Connecting to %s", url);
    
    res = sharing_http_run (http, url);
    
    if (res == SHARING_HTTP_RUNRES_SUCCESS && sharing_http_get_res_code (http) == 200)
    {
        gsize content_len = 0;
        gsize header_len = 0;
        
        const gchar *body = sharing_http_get_res_body (http, &content_len);
        *data = g_malloc0 (content_len);
        memcpy (*data, body, content_len);
        *size = content_len;
        
        ULOG_INFO_F("body = %s", body);
        
        if (header) {
            const gchar *content = sharing_http_get_res_content (http, &header_len);
            gchar **lines = g_strsplit (content, "\n", 0);
            int i = 0;
            while (lines[i]) {
                if (g_ascii_strncasecmp (lines[i], "Set-Cookie:", 11) == 0) {
                    gchar **cvalue = g_strsplit (lines[i], ": ", 2);
                    char *comma = strchr (cvalue[1], ';');
                    if (!comma) {
                        ULOG_INFO_F("no comma?");
                        /* Bad error handling? */
                        return FALSE;
                    }
                    *header = g_malloc0 (comma - cvalue[1] + 1);
                    memcpy (*header, cvalue[1], comma - cvalue[1]);
                    *header_size = (gsize)(comma - cvalue[1])+1;
                }
                i++;
            }
        }
        
        sharing_http_unref (http);
        
        return TRUE;
    }
    
    ULOG_INFO_F ("No dice, we failed. http code: %d", sharing_http_get_res_code (http));
    
    mobblogg->error = MobilBloggConnectionError;
    return FALSE;
}

static gboolean
mobilblogg_request_data (mobilblogg_t *mob, gchar *url, gchar **data, gsize *size)
{
    return mobilblogg_request_data_and_header (mob, url, data, size, NULL, NULL);
}

static gchar *mobilblogg_get_salt (mobilblogg_t *mob)
{
    gchar *url = g_strdup_printf ("http://api.mobilblogg.nu/o.o.i.s?template=api_v1.0.t&func=getSalt&user=%s", mob->username);
    gchar *data;
    gsize data_len;
    
    if (!mobilblogg_request_data (mob, url, &data, &data_len))
    {
        ULOG_WARN_L("Failed to get data");
        return NULL;
    }
    g_free (url);

    cJSON *root = cJSON_Parse (data);
    if (!root)
    {
        ULOG_WARN_L("Failed to parse data!");
        mob->error = MobilBloggServerError;
        return NULL;
    }
    
    g_free (data);

    cJSON *node = root->child;
    if (!node) {
        ULOG_WARN_L ("Failed to get child node");
        cJSON_Delete (root);
        mob->error = MobilBloggServerError;
        return NULL;
    }

    cJSON *saltnode = cJSON_GetObjectItem (node, "salt");
    if (!saltnode) {
        ULOG_WARN_L("Missing salt member");
        cJSON_Delete (root);
        mob->error = MobilBloggServerError;
        return NULL;
    }

    gchar *salt = g_strdup (saltnode->valuestring);
    if (!salt) {
        ULOG_WARN_L("Missing salt member");
        cJSON_Delete (root);
        mob->error = MobilBloggServerError;
        return NULL;
    }

    cJSON_Delete (root);

    ULOG_INFO_F("SALT: %s", salt);
    return salt;
}

static gchar *mobilblogg_call_login (mobilblogg_t *mob, const gchar *passwordhash)
{
    gchar *url = g_strdup_printf ("http://api.mobilblogg.nu/o.o.i.s?template=api_v1.0.t&func=login&username=%s&password=%s", mob->username, passwordhash);
    gchar *data, *header;
    gsize data_len, header_len;
    
    if (!mobilblogg_request_data_and_header (mob, url, &data, &data_len, &header, &header_len))
    {
        ULOG_WARN_L("Failed to get data");
        return NULL;
    }
    g_free (url);

    cJSON *root = cJSON_Parse (data);
    if (!root) {
        ULOG_WARN_L ("Couldn't parse data!");
        mob->error = MobilBloggServerError;
        return NULL;
    }

    g_free (data);

    cJSON *node = root->child;
    if (!node) {
        ULOG_WARN_L ("Couldn't get child node");
        cJSON_Delete (root);
        mob->error = MobilBloggServerError;
        return NULL;
    }

    cJSON *status = cJSON_GetObjectItem (node, "status");
    if (!status) {
        ULOG_WARN_L ("No status item?");
        cJSON_Delete (root);
        mob->error = MobilBloggServerError;
        return NULL;
    }

    if (status->valueint != 1) {
        ULOG_WARN_L ("Status failed, so we didn't login");
        cJSON_Delete (root);
        mob->error = MobilBloggCredentialsError;
        return NULL;
    }

    cJSON_Delete (root);
    
    if (header) {
        return header;
    }
    
    ULOG_WARN_L ("No header, returning null");
    mob->error = MobilBloggServerError;
    return NULL;
}


mobilblogg_t *
mobilblogg_init (ConIcConnection *con, const gchar *username, const gchar *password)
{
    mobilblogg_t *mobblogg = g_new0 (mobilblogg_t, 1);
    mobblogg->con = con;
    mobblogg->username = g_strdup (username);
    mobblogg->password = g_strdup (password);
    mobblogg->error = MobilBloggNoError;
    return mobblogg;
}

void
mobilblogg_set_secretword (mobilblogg_t* mob, const gchar *secretword)
{
    mob->secretword = g_strdup (secretword);
}

void
mobilblogg_destroy (mobilblogg_t *mobblogg)
{
    g_free (mobblogg->username);
    g_free (mobblogg->password);
    
    if (mobblogg->cookie)
        g_free (mobblogg->cookie);
        
    if (mobblogg->secretword)
        g_free (mobblogg->secretword);
    
    g_free (mobblogg);
    
}

mobilblogg_error_code
mobilblogg_login(mobilblogg_t *mobblogg)
{
    gchar *salt = mobilblogg_get_salt(mobblogg);
    if (!salt) {
        return mobblogg->error;
    }
    gchar *hash = mobilblogg_sha1_hash(salt, mobblogg->password);
    g_free (salt);
    
    ULOG_INFO_F("hash = %s", hash);
    
    gchar *cookie = mobilblogg_call_login(mobblogg, hash);
    g_free (hash);

    if (!cookie) {
        return mobblogg->error;
    }
    
    ULOG_INFO_F("Cookie = %s", cookie);
    
    mobblogg->cookie = cookie;
    return MobilBloggNoError;
}

static gchar *
mobilblogg_sha1_hash (const gchar *salt, const gchar *password)
{
    gchar *hashandpassword = g_strdup_printf ("%s%s", salt, password);
    SHA_CTX s;
    unsigned char hash[20];
    SHA1_Init (&s);
    SHA1_Update (&s, hashandpassword, strlen (hashandpassword));
    SHA1_Final (hash, &s);
    
    g_free (hashandpassword);
    
    gchar *rethash = g_malloc0 (40+1);
    for (int i = 0; i < 20; i++)
    {
        sprintf(rethash+i*2, "%02x", (int)hash[i]);
    }
    return rethash;
}

typedef struct mobilblogg_entry_St
{
    SharingTransfer *transfer;
    guint64 progress_start;
    guint64 progress_end;
    guint64 media_bytes;
    gboolean *dead_mans_switch;
} mobilblogg_entry_t;


static gboolean
mobilblogg_send_callback (SharingHTTP *http, guint64 bytes_sent, gpointer user_data)
{
    mobilblogg_entry_t *entry = user_data;
    if (!sharing_transfer_continue (entry->transfer))
        return FALSE;
    
    *entry->dead_mans_switch = FALSE;
    gdouble progress = (entry->progress_start + entry->progress_end) / 2.0;
    
    if (entry->media_bytes) {
        if (bytes_sent >= entry->media_bytes) {
            progress = entry->progress_end;
        } else {
            progress = entry->progress_start + (bytes_sent / (gdouble)entry->media_bytes) * (entry->progress_end - entry->progress_start);
        }
    }
    
    sharing_transfer_set_progress (entry->transfer, progress);
    return TRUE;
}

mobilblogg_error_code
mobilblogg_send (mobilblogg_t *mob, SharingTransfer *transfer, guint64 progress_start, 
                 guint64 progress_end, gboolean *dead_mans_switch, const gchar *right, 
                 SharingEntryMedia *media)
{
    SharingHTTP *http = sharing_http_new ();
    gchar *caption = sharing_entry_media_get_title (media);
    gchar *filename = sharing_entry_media_get_filename (media);
    const gchar *body = sharing_entry_media_get_desc (media);
    mobilblogg_entry_t mobentry;
    
    mobentry.transfer = transfer;
    mobentry.progress_start = progress_start;
    mobentry.progress_end = progress_end;
    mobentry.dead_mans_switch = dead_mans_switch;
    mobentry.media_bytes = 0;
        
    ULOG_INFO_F("Send %s (%s)", caption ? caption : "", filename);    
    
    gchar *url = g_strdup_printf ("http://api.mobilblogg.nu/o.o.i.s");
    if (mob->cookie) {
        gchar *cookie = g_strdup_printf ("Cookie: %s", mob->cookie);
        sharing_http_add_req_header_line (http, cookie);
        g_free (cookie);
    }
    
    sharing_http_add_req_multipart_data (http, "template", "api_v1.0.t", -1, "text/plain");
    sharing_http_add_req_multipart_data (http, "func", "upload", -1, "text/plain");
    sharing_http_add_req_multipart_data (http, "wtd", "ladda_upp", -1, "text/plain");
    sharing_http_add_req_multipart_data (http, "header", caption ? caption : "", -1, "text/plain");
    sharing_http_add_req_multipart_data (http, "text", body ? body : "", -1, "text/plain");
    sharing_http_add_req_multipart_data (http, "secretword", mob->secretword, -1, "text/plain");
    /* change me */
    sharing_http_add_req_multipart_data (http, "rights", right, -1, "text/plain");
    gchar *path = g_strdup_printf ("/files/%s", mob->username);
    sharing_http_add_req_multipart_data (http, "path", path, -1, "text/plain");
    g_free (path);
    
    /* and the picture */
    sharing_http_add_req_multipart_file_with_filename (http,
        "image", sharing_entry_media_get_localpath (media), "image/jpeg", filename ? filename : "image.jpg");
    
    if (caption)
        g_free (caption);
    if (filename)
        g_free (filename);
    
    mobentry.media_bytes = sharing_entry_media_get_size (media);
    sharing_http_set_progress_callback (http, mobilblogg_send_callback, &mobentry);
    *dead_mans_switch = FALSE;
    
    sharing_http_set_connection (http, mob->con);
    SharingHTTPRunResponse res = sharing_http_run (http, url);
    g_free (url);
    
    mob->error = MobilBloggNoError;
    
    if (res == SHARING_HTTP_RUNRES_SUCCESS && sharing_http_get_res_code (http) == 200) {
        gsize content_len = 0;
        const gchar *content = sharing_http_get_res_body (http, &content_len);
        ULOG_INFO_F("content = %s", content);
        guint imgid = parse_send_content (content);
        if (imgid == 0) {
            mob->error = MobilBloggServerError;
        }
    } else if (res == SHARING_HTTP_RUNRES_CANCELLED) {
        mob->error = MobilBloggErrorCancelled;
    } else {
        mob->error = MobilBloggServerError;
    }
    
    sharing_http_unref (http);
    *dead_mans_switch = FALSE;
    
    sharing_transfer_set_progress (transfer, progress_end);
    return mob->error;
}

static guint
parse_send_content (const gchar *content)
{
    cJSON *root = cJSON_Parse (content);
    if (!root) {
        ULOG_WARN_L ("Couldn't parse data!");
        return 0;
    }

    cJSON *node = root->child;
    if (!node) {
        ULOG_WARN_L ("Couldn't get child node");
        cJSON_Delete (root);
        return 0;
    }

    cJSON *imgnode = cJSON_GetObjectItem (node, "imgid");
    if (!imgnode) {
        ULOG_WARN_L ("No imgid item?");
        cJSON_Delete (root);
        return 0;
    }
    
    ULOG_INFO_F("imgid = %d", imgnode->valueint);
    
    guint imgid = imgnode->valueint;
    
    cJSON_Delete (root);
    
    return imgid;
}
