#include "blip_xml.h"

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <string.h>

struct BlipXml_s
{
    xmlDocPtr doc;
    xmlXPathContextPtr xpathCtx;
};

BlipXml blip_xml_create(const gchar* xml,
                        BlipError * oError)
{
    BlipXml blipXml;

    if (xml == NULL)
    {
        if (oError)
            *oError= blip_error_create(BLIP_DOMAIN_BLIP_XML,
                                       BLIP_XML_ERROR_NULL_RESPONSE,
                                       "Blipfoto XML response was null");
        return NULL;
    }

    blipXml= (BlipXml) g_try_malloc0(sizeof(struct BlipXml_s));
    if (blipXml == NULL)
    {
        if (oError)
            // Is there any point in doing this? No doubt there'll be
            // no memory to create the BlipError either.
            *oError= blip_error_create(BLIP_DOMAIN_BLIP_XML,
                                       BLIP_XML_ERROR_OUT_OF_MEMORY,
                                       "No memory to create XML response");
        return NULL;
    }

    blipXml->doc= xmlReadMemory(xml,
                                strlen(xml),
                                "response.xml",
                                NULL,
                                0);
    if (blipXml->doc == NULL)
    {
        if (oError)
            *oError= blip_error_create(BLIP_DOMAIN_BLIP_XML,
                                       BLIP_XML_ERROR_PARSE_FAILED,
                                       "Failed to parse Blipfoto XML response");
        blip_xml_free(blipXml);
        return NULL;
    }

    blipXml->xpathCtx= xmlXPathNewContext(blipXml->doc);
    if (blipXml->xpathCtx == NULL)
    {
        if (oError)
            *oError= blip_error_create(BLIP_DOMAIN_BLIP_XML,
                                       BLIP_XML_ERROR_NO_XPATH_CONTEXT,
                                       "Could not create XPath context");
        blip_xml_free(blipXml);
        return NULL;
    }

    return blipXml;
}

void blip_xml_free(BlipXml blipXml)
{
    if (blipXml)
    {
        if (blipXml->xpathCtx)
            xmlXPathFreeContext(blipXml->xpathCtx);
        if (blipXml->doc)
            xmlFreeDoc(blipXml->doc);
        g_free(blipXml);
    }
}

gchar* blip_xml_get_text_node_contents(BlipXml blipXml,
                                       gchar * xpathExpression,
                                       BlipError * oError)
{
    gchar* result= NULL;
    if (blipXml)
    {
        xmlXPathObjectPtr xpathObj;
        xpathObj= xmlXPathEvalExpression(BAD_CAST xpathExpression,
                                         blipXml->xpathCtx);
        if(xpathObj != NULL)
        {
            if (xpathObj->nodesetval
                && xpathObj->nodesetval->nodeNr > 0
                && xpathObj->nodesetval->nodeTab[0]
                && xpathObj->nodesetval->nodeTab[0]->type == XML_TEXT_NODE)
            {
                result= g_strdup((gchar*)xpathObj->nodesetval->nodeTab[0]->content);
            }
            else
            {
                if (oError)
                    *oError= blip_error_create(BLIP_DOMAIN_BLIP_XML,
                                               BLIP_XML_ERROR_XPATH_EXPRESSION_NOT_A_TEXT_NODE,
                                               "Xpath expression doesn't evaluate to a a text node");
            }
            xmlXPathFreeObject(xpathObj);
        }
        else
        {
            if (oError)
                *oError= blip_error_create(BLIP_DOMAIN_XML,
                                           blipXml->xpathCtx->lastError.code,
                                           blipXml->xpathCtx->lastError.message);
        }
    }
    return result;
}

BlipError blip_xml_extract_error(BlipXml blipXml)
{
    BlipError error= NULL;
    if (blipXml)
    {
        gchar* errorCode= blip_xml_get_text_node_contents(blipXml,
                                                          "/blipapi/error/code/text()",
                                                          NULL);
        if (errorCode)
        {
            error= blip_error_create_no_copy(BLIP_DOMAIN_BLIP_API,
                                             atoi(errorCode),
                                             blip_xml_get_text_node_contents(blipXml,
                                                                             "/blipapi/error/message/text()",
                                                                             NULL));
            g_free(errorCode);
        }
    }
    return error;
}
