/*
 * This file is part of sharing-plugin-blipfoto
 *
 * Copyright (C) 2010 Dave Elcock. All rights reserved.
 *
 * This code is licensed under a MIT-style license, that can be
 * found in the file called "COPYING" in the root directory.
 *
 */

#ifndef __BLIP_ERROR_H__
#define __BLIP_ERROR_H__

#include <glib.h>

G_BEGIN_DECLS

        struct BlipError_s;
typedef struct BlipError_s* BlipError;

/**
 * IMPORTANT: When adding new domains, remember to update blip_error_domain_name().
 */
enum BlipErrorDomain_e
{
    /// Error is internal to the Blipfoto library.
    BLIP_DOMAIN_BLIP_LIB = 0,
    /// Error is from the response from calling the Blipfoto API.
    BLIP_DOMAIN_BLIP_API = 1,
    /// Error is from this library's use of the underlying XML impelementation.
    BLIP_DOMAIN_BLIP_XML = 2,
    /// Error is from the underlying XML implementation. I.e. raw unhandled error
    /// from the xml subsystem.
    BLIP_DOMAIN_XML = 3,
    /// Error is from the underlying transport implementation. E.g. raw error
    /// code from Curl.
    BLIP_DOMAIN_TRANSPORT = 4,
    // This should always be last in the list. Please ensure this
    // number is the highest or blip_error_domain_name() will break.
    BLIP_DOMAIN_NONE = 5,
};
typedef enum BlipErrorDomain_e BlipErrorDomain;

enum BlipLibError_e
{
    BLIP_LIB_NULL_API_KEY,
    BLIP_LIB_NULL_API_SECRET,
    BLIP_LIB_ERROR_OUT_OF_MEMORY,
    BLIP_LIB_CONNECTION_CANCELLED_BY_USER,
};
typedef enum BlipLibError_e BlipLibError;

/**
 * @param [in] message
 *   This will be copied. If you want the error to free the message you pass
 *   in, use blip_error_create_no_copy() instead.
 */
BlipError blip_error_create(BlipErrorDomain domain,
                            gint code,
                            const gchar* message);

/**
 * @param [in] message
 *   Must be allocated with g_malloc(), g_strdup() etc as it will be freed
 *   during blip_error_free().
 */
BlipError blip_error_create_no_copy(BlipErrorDomain domain,
                                    gint code,
                                    gchar* message);

/**
 * @brief Frees resources allocated by blip_error_create().
 */
void blip_error_free(BlipError error);

/**
 * @brief Retrieve the error domain.
 * @return See BlipErrorDomain definitions for possible results.
 */
BlipErrorDomain blip_error_domain(BlipError error);

/**
 * @brief Retrieve the error code.
 * @return See BLIP_ERROR_xyz definitions for possible results.
 */
gint blip_error_code(BlipError error);

/**
 * @brief A string representation of the error.
 * @return
 *   Must NOT be freed.
 */
const gchar* blip_error_message(BlipError error);

/**
 * @brief Tests to see if the error matches the given domain and error code.
 * @param error
 *   The error instance to consult.
 * @param domain
 *   The sole acceptable domain.
 * @param ...
 *   A list of error codes within the selected domain terminated with
 *   BLIP_ERROR_MATCH_END. If any one of them match, TRUE is returned.
 * @return
 *   TRUE if an error exists, its domain matches and any code matches;
 *   FALSE otherwise.
 */
gboolean blip_error_match(BlipError       error,
                          BlipErrorDomain domain,
                          ...);

/**
 * @brief Gives a string representation of the error domain.
 */
const gchar* blip_error_domain_name(BlipError error);

G_END_DECLS

#endif // ! __BLIP_ERROR_H__
