/* Shariks
 *
 * Copyright (C) 2010 Dmitriev V.V.     <vdmitrie@cs.karelia.ru>
 * Copyright (C) 2010 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2010 Sotnikov A.A.     <sotnikov@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include <cstdlib>
#include <SDL_image.h>
#include <SDL_ttf.h>
#include <SDL_gfxPrimitives.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sstream>
#include <time.h>
#include <math.h>
#include <unistd.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-gtype-specialized.h>
#include <dbus/dbus.h>

#include "main.h"
#include "Vector2D.h"
#include "Vector3D.h"
#include "Ball.h"
#include "TBList.h"
#include "DevStates.h"
#include "Timer.h"
#include "Ui.h"
#include "ScoreSystem.h"

#include "config.h"

using namespace std;

DBusError error; // display state error
DBusConnection *dbus_conn;

float CRAZY_TIME = 10000; // min value of crazy time in msec
int AGRESSION = NORMAL;

const float kAccFilter = 0.1;

const char* font_file = "/usr/share/shariks/data/courier.ttf";
const char* font_file2 = "/usr/share/shariks/data/teen_bold.ttf";

const char* gameover_file = "/usr/share/shariks/data/gameover.png";

float t_frame; // length of time frame
int ms_frame;  // length of time frame (milliseconds)
int ms_end;    // end of frame

float game_time;

int crazy_start;
int crazy_end;
int weakness_start = 0;

Timer crazy_timer;   // timer for managing "crazy time" of crazy balls
Timer played_timer;  // timer, which keeps player's played time

float gamespeed = 1.0;

int SCORE = 0;

bool paused = false;
bool gameover = false;
bool newhighscore = false;
bool weakness = false; // weakness after loosing life

int num_linked_tb;
int lives = LIVES;
int blink_counter = 0;

SDL_Surface* screen; 		  // container for all sprites
SDL_Surface* back;   		  // background image

TBList* tb_list;
DevStates* dev_states;
Ball* ball;                      // player's ball
Ball* tail_balls[N_TAILBALLS];   // tail balls
Ball* crazy_balls[N_CRAZYBALLS]; // crazy balls
Ball* stars[N_STARS];            // stars
Ui* ui;

SDL_Surface* tb_icon;    // icon for not linked tail ball
SDL_Surface* tbl_icon;   // icon for linked tail ball

SDL_Surface* tb_sprite;  // sprite for not linked tail ball
SDL_Surface* tbl_sprite; // sprite for linked tail ball

Vector3D* acc_vector;    // vector of current acceleration
Vector2D* prev_acc = new Vector2D(0, 0);

char* display_state = NULL;

// Draw all game objects
void DrawScene() {
    int i;
    SDL_Rect dest;    

    if (!gameover) {
        // draw background
        dest.x = 0;
        dest.y = 0;
        SDL_BlitSurface(back, NULL, screen, &dest);

        // Crazy Balls
        for (i = 0; i < N_CRAZYBALLS; i++) {
            dest.x = crazy_balls[i]->x;
            dest.y = crazy_balls[i]->y;
            SDL_BlitSurface(crazy_balls[i]->sprite, NULL, screen, &dest);
        }

        // Stars
        for (i = 0; i < N_STARS; i++) {
            dest.x = stars[i]->x;
            dest.y = stars[i]->y;
            SDL_BlitSurface(stars[i]->sprite, NULL, screen, &dest);
        }
   
        // Links between Ball and Tail Balls
        element *p_temp = tb_list->p_head;
        while (p_temp != NULL) {
            if (p_temp->ball->fixed) {
                p_temp->ball->sprite = tbl_sprite;
                DrawLine(ball, p_temp->ball);
            } else {
                p_temp->ball->sprite = tb_sprite;
                if (p_temp->ball->linked)                  
                    DrawLine(ball, p_temp->ball);
            }
            p_temp = p_temp->next;
        }
 
        // Ball
        dest.x = ball->x;
        dest.y = ball->y;

        if (weakness) {
            if (blink_counter >= 4) {
                if (blink_counter == 6)
                    blink_counter = 0;
				else
                    blink_counter++;
            } else {
                blink_counter++;
				SDL_BlitSurface(ball->sprite, NULL, screen, &dest);
            }
	} else {
            SDL_BlitSurface(ball->sprite, NULL, screen, &dest);
	}

        // Tail Balls
        p_temp = tb_list->p_head;
        while (p_temp != NULL) {
            dest.x = p_temp->ball->x;
            dest.y = p_temp->ball->y;
            SDL_BlitSurface(p_temp->ball->sprite, NULL, screen, &dest);
            p_temp = p_temp->next;
        }
		delete p_temp;

        ShowInfo();
    } else {
	if (!newhighscore)
            // Game Over
            GameOver();
	else {
            ui->NewHighscore(screen, SCORE);
            played_timer.stop();
	}
    }    

    // Update screen
    SDL_Flip(screen);
}

// Draws link beetween two Balls
void DrawLine(Ball *s1, Ball *s2) {
    Uint32 color = SDL_MapRGB(screen->format, 255, 255, 255);
    lineColor(screen,
                      s1->x + s1->radius,
                      s1->y + s1->radius,
                      s2->x + s2->radius,
                      s2->y + s2->radius,
                      color);
}

// Load image for sprite (colorkey for transparent color)
SDL_Surface* LoadImage(const char *file) {
    SDL_Surface *temp1, *temp2;
    temp1 = IMG_Load(file);
    if(temp1 == NULL) {
        printf("Can't load image file %s\n",file);
        return NULL;
    }
    temp2 = SDL_DisplayFormat(temp1);
    if (temp2 != NULL) {
        // Set transparent color
        Uint32 colorKey = SDL_MapRGB(temp2->format, 255, 0, 255);
        SDL_SetColorKey(temp2, SDL_SRCCOLORKEY, colorKey);
    }

    SDL_FreeSurface(temp1);
    return temp2;
}

// Load image for sprite (using alpha channel)
SDL_Surface* LoadImageAlpha(const char *file) {
    SDL_Surface *temp1, *temp2;
    temp1 = IMG_Load(file);
    if(temp1 == NULL) {
        printf("Can't load image file %s\n",file);
        return NULL;
    }

    temp2 = SDL_DisplayFormatAlpha(temp1);
    SDL_FreeSurface(temp1);
    return temp2;
}

// Locates all balls on game field randomly
void LocateBalls() {
    int i,j;

    // amount of all balls
    int num_balls = N_CRAZYBALLS + N_STARS + N_TAILBALLS + 1;

    // array of all balls
    Ball* balls_array[num_balls];

    // fill array
    for (i = 0; i < N_CRAZYBALLS; i++)
        balls_array[i] = crazy_balls[i];

    for (i = N_CRAZYBALLS; i < N_STARS + N_CRAZYBALLS; i++)
        balls_array[i] = stars[i - N_CRAZYBALLS];

    element *p_temp = tb_list->p_head;
    i = 0;
    while (p_temp != NULL) {
        balls_array[i + N_CRAZYBALLS + N_STARS] = p_temp->ball;
        p_temp = p_temp->next;
        i++;
    }
    delete p_temp;

    balls_array[num_balls - 1] = ball;

    // random location of balls
    for (i = 0; i < num_balls; i++) {
        bool free;
        Ball* s1 = balls_array[i];

        // try to locate ball to random place
        // if it isn't free try again
        do {
            s1->x = (SCREEN_WIDTH - BORDER_WIDTH - s1->radius * 2) *
                    0.01 * (rand() % 100) + s1->radius;
            s1->y = (SCREEN_HEIGHT - BORDER_WIDTH - s1->radius * 2) *
                    0.01 * (rand() % 100) + s1->radius;

            free = true;
            for (j = 0; j < i; j++) {
                Ball* s2 = balls_array[j];

                // check if it is free
                if ((s2->x - s1->x) * (s2->x - s1->x) +

                    (s2->y - s1->y) * (s2->y - s1->y) <=
                    (s2->radius + s1->radius) * (s2->radius + s1->radius) + 32 * 32) {
                        free = false;
                        break;
                    }
            }
        } while (!free);
    }
}

// Init SDL and sprites
void Init() {
    int i, j;

    const char* ball_file = "/usr/share/shariks/data/ball.png";
    const char* crazy_file = "/usr/share/shariks/data/crazy.png";
    const char* star_file = "/usr/share/shariks/data/star.png";
    const char* bg_file = "/usr/share/shariks/data/bg.png";
    const char* tail_file = "/usr/share/shariks/data/tailball.png";

    const char* tbl_file = "/usr/share/shariks/data/tb_linked.png";
    const char* tb_file = "/usr/share/shariks/data/tailball.png";

    acc_vector = new Vector3D(0, 0, 0);

    if (SDL_Init(SDL_INIT_VIDEO| SDL_INIT_TIMER) < 0) {
        printf("Error while loading SDL! %s \n", SDL_GetError());
        exit(1);
    }

#ifdef SDL_DISABLE
    // Remove mouse cursor on a tablet
    // ACC: 
	SDL_ShowCursor(SDL_DISABLE);
#endif

    // Init TrueType Fonts
    TTF_Init();

    // Init random generator
    srand(time(0));

    atexit(SDL_Quit);
    atexit(TTF_Quit);

    SDL_WM_SetCaption("Shariks", "Shariks");

    // screen settings
    screen = SDL_SetVideoMode(SCREEN_WIDTH,
                              SCREEN_HEIGHT,
                              SCREEN_BPP,
                              SDL_HWSURFACE|SDL_FULLSCREEN|SDL_DOUBLEBUF);

    SDL_EnableUNICODE(SDL_ENABLE);

    // ACC: 
	dev_states = new DevStates();
    tb_list = new TBList;

    ui = new Ui();

    // Create all objects
    ball = new Ball(B_MASS, B_RADIUS);

    for (i = 0; i < N_CRAZYBALLS; i++)
        crazy_balls[i] = new Ball(CB_MASS, CB_RADIUS);

    for (i = 0; i < N_STARS; i++)
        stars[i] = new Ball(STAR_MASS, STAR_RADIUS);

    while (tb_list->count_elem != N_TAILBALLS) {
        tb_list->AddToList(new Ball(TB_MASS, TB_RADIUS));
    }

    LocateBalls();

    // Load images for all objects
    back = LoadImageAlpha(bg_file);
    
    ball->sprite = LoadImageAlpha(ball_file);

    for (i = 0; i < N_CRAZYBALLS; i++)
        crazy_balls[i]->sprite = LoadImageAlpha(crazy_file);

    for (i = 0; i < N_STARS; i++)
        stars[i]->sprite = LoadImageAlpha(star_file);

    element *p_temp = tb_list->p_head;
    while (p_temp != NULL) {
        p_temp->ball->sprite = LoadImageAlpha(tail_file);
        p_temp = p_temp->next;
    }
    delete p_temp;

    ms_end = SDL_GetTicks();
	
    tb_sprite = LoadImageAlpha(tb_file);
    tbl_sprite = LoadImageAlpha(tbl_file);

	dbus_error_init (&error);
	dbus_conn = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
}

void TinySleep() {
    struct timespec t;
    t.tv_nsec = 10000;
    t.tv_sec = 0;
    nanosleep(&t, 0);
}

// Calculates next position of Ball
void MoveBall() {
    ball->x += ball->dx * t_frame;
    ball->y += ball->dy * t_frame;
}

/* // Calculates next position of Ball (accelerometer mode, old method)
void AccMoveBall() {
	values_x[acc_index] = acc_vector->x * t_frame;
	values_y[acc_index] = acc_vector->y * t_frame;

	acc_index++;
	if (acc_index >= 10)
    	acc_index = 0;
	
	ball->dx = ball->dx *10;
	ball->dy = ball->dy *10;
	
	for (int i = 0; i < 10; i++) {
    	ball->dx += values_x[i];
    	ball->dy += values_y[i];
	}
	ball->dx = ball->dx / 10 ;
	ball->dy = ball->dy / 10 ;
	
	//ball->dx += acc_vector->x * t_frame;
    //ball->dy += acc_vector->y * t_frame;
    ball->x += ball->dx * t_frame;
    ball->y += ball->dy * t_frame;
} */

// Experemental ball movement
void AccMoveBall2() {
    float length = sqrt(acc_vector->x * acc_vector->x + acc_vector->y * acc_vector->y);
    float pitch;
    float roll;
    float vx, vy;

    if (length != 0) {
        pitch = acc_vector->y / length;
        roll = acc_vector->x / length;        
    } else {        
        pitch = 1;
        roll = 0;
    }

    float sinA = roll * G * t_frame;
    float cosA = pitch * G * t_frame;

    vx = ball->dx * t_frame + sinA * 0.5 * t_frame;
    vy = ball->dy * t_frame + cosA * 0.5 * t_frame;

    ball->x += vx;
    ball->y += vy;

    ball->dx += sinA;
    ball->dy += cosA;
}

// Calculates next position of Ball (accelerometer mode)
void AccMoveBall() {
    acc_vector->x = 0.1 * acc_vector->x + 0.9 * prev_acc->x;
    acc_vector->y = 0.1 * acc_vector->y + 0.9 * prev_acc->y;

    ball->dx += acc_vector->x * t_frame;
    ball->dy += acc_vector->y * t_frame;

    prev_acc->x = acc_vector->x;
    prev_acc->y = acc_vector->y;
}

void AccMoveBall1() {
    Vector2D direction;
    Vector2D acc2D;

    acc2D.x = acc_vector->x;
    acc2D.y = acc_vector->y;
    float len = acc2D.length();

    // define direction
    if (len == 0) {
    	direction.x = 0;
    	direction.y = 0;
    } else {
        // normalize vector
	direction = acc2D / len;
    }

    // acceleration
    ball->dx += direction.x * 1.6 * t_frame;
    ball->dy += direction.y * 1.6 * t_frame;
}

// Random movement of crazy balls
void MoveCrazyBalls() {
    float vx, vy;
    for (int i = 0; i < N_CRAZYBALLS; i++) {

        // velocity is random beetween -1.5 and 1.5
        vx = AGRESSION * (0.03 * (rand() % 100) - 1.5);
        vy = AGRESSION * (0.03 * (rand() % 100) - 1.5);

        // new position of crazy ball
        crazy_balls[i]->x += crazy_balls[i]->dx * t_frame;
        crazy_balls[i]->y += crazy_balls[i]->dy * t_frame;

        // new velocity
        crazy_balls[i]->dx += vx * t_frame;
        crazy_balls[i]->dy += vy * t_frame;
    }
}

// Tail Balls movement
void MoveTailBalls() {
    element *p_temp = tb_list->p_head;
    while (p_temp != NULL) {
        p_temp->ball->x += p_temp->ball->dx * t_frame;
        p_temp->ball->y += p_temp->ball->dy * t_frame;
        p_temp = p_temp->next;
    }
    delete p_temp;
}

// Random movement of stars
void MoveStars() {
    float vx, vy;

    for (int i = 0; i < N_STARS; i++) {
        // velocity is random beetween -1.5 and 1.5
        vx = AGRESSION * (0.03 * (rand() % 100) - 1.5);
        vy = AGRESSION * (0.03 * (rand() % 100) - 1.5);

        // new position of stars
        stars[i]->x += stars[i]->dx * t_frame;
        stars[i]->y += stars[i]->dy * t_frame;

        // new velocity
        stars[i]->dx += vx * t_frame;
        stars[i]->dy += vy * t_frame;
    }
}

// Prints message on screen using SDL_ttf
void PrintTTF(SDL_Surface* sDest, char* message, const char* font, int size,
              SDL_Color color, SDL_Rect dest) {
    TTF_Font *fnt = TTF_OpenFont(font, size);
    SDL_Surface *sText = TTF_RenderText_Blended(fnt, message, color);
    SDL_BlitSurface(sText, NULL, sDest, &dest);
    SDL_FreeSurface(sText);
    TTF_CloseFont(fnt);
}

// Handle collision objects with field boundaries
void Bounce() {
    int i,j;
    
    /**** Ball ****/
    // Bouncing off right or left border
    if (ball->x < BORDER_WIDTH ||
        ball->x + B_RADIUS * 2 > SCREEN_WIDTH - BORDER_WIDTH) {
		// Play the bounce sound effect
        //Mix_PlayChannel( -1, bounce_border, 0 );

        ball->x -= ball->dx * t_frame;
        ball->dx = -ball->dx * BOUNCINESS;
        ball->x = fconstrain(ball->x,
                             SCREEN_WIDTH - ball->radius*2 - BORDER_WIDTH);
    }

    // Bouncing off down or upper border
    if (ball->y < BORDER_WIDTH ||
        ball->y + B_RADIUS * 2 > SCREEN_HEIGHT - BORDER_WIDTH) {
		// Play the bounce sound effect
        //Mix_PlayChannel( -1, bounce_border, 0 );

        ball->y -= ball->dy * t_frame;
        ball->dy = -ball->dy * BOUNCINESS;
        ball->y = fconstrain(ball->y,
                             SCREEN_HEIGHT - ball->radius*2 - BORDER_WIDTH);
    }

    /**** Crazy Balls ****/
    for (i = 0; i < N_CRAZYBALLS; i++) {
        // Bouncing off right or left border
        if (crazy_balls[i]->x < BORDER_WIDTH || crazy_balls[i]->x +
                CB_RADIUS * 2 > SCREEN_WIDTH - BORDER_WIDTH) {
	   		// Play the bounce sound effect
            //Mix_PlayChannel( -1, bounce_crazy_border, 0 );

            crazy_balls[i]->x -= crazy_balls[i]->dx * t_frame;
            crazy_balls[i]->dx = -crazy_balls[i]->dx * BOUNCINESS;
            crazy_balls[i]->x = fconstrain(crazy_balls[i]->x, SCREEN_WIDTH -
                    crazy_balls[i]->radius*2 - BORDER_WIDTH);
        }

        // Bouncing off down or upper border
        if (crazy_balls[i]->y < BORDER_WIDTH ||
            crazy_balls[i]->y + crazy_balls[i]->radius * 2 > SCREEN_HEIGHT - BORDER_WIDTH) {
	    	// Play the bounce sound effect
            //Mix_PlayChannel( -1, bounce_crazy_border, 0 );

            crazy_balls[i]->y -= crazy_balls[i]->dy * t_frame;
            crazy_balls[i]->dy = -crazy_balls[i]->dy * BOUNCINESS;
            crazy_balls[i]->y = fconstrain(crazy_balls[i]->y, SCREEN_HEIGHT -
                    CB_RADIUS * 2 - BORDER_WIDTH);
        }
    }

    /**** Star ****/
    for (i = 0; i < N_STARS; i++) {
        // Bouncing off right or left border
        if (stars[i]->x < BORDER_WIDTH ||
            stars[i]->x + stars[i]->radius*2 > SCREEN_WIDTH - BORDER_WIDTH) {

            stars[i]->x -= stars[i]->dx * t_frame;
            stars[i]->dx = -stars[i]->dx * BOUNCINESS;
            stars[i]->x = fconstrain(stars[i]->x,
                                     SCREEN_WIDTH - stars[i]->radius*2 - BORDER_WIDTH);
        }

        // Bouncing off down or upper border
        if (stars[i]->y < BORDER_WIDTH ||
            stars[i]->y + STAR_RADIUS * 2 > SCREEN_HEIGHT - BORDER_WIDTH) {

            stars[i]->y -= stars[i]->dy * t_frame;
            stars[i]->dy = -stars[i]->dy * BOUNCINESS;
            stars[i]->y = fconstrain(stars[i]->y,
                                     SCREEN_HEIGHT - stars[i]->radius*2 - BORDER_WIDTH);
        }
    }

    /**** TailBalls ****/
    element *p_temp = tb_list->p_head;
    while (p_temp != NULL) {
        // Bouncing off right or left border
        if (p_temp->ball->x < BORDER_WIDTH ||
            p_temp->ball->x + TB_RADIUS * 2 > SCREEN_WIDTH - BORDER_WIDTH) {

            p_temp->ball->x -= p_temp->ball->dx * t_frame;
            p_temp->ball->dx = -p_temp->ball->dx * BOUNCINESS;
            p_temp->ball->x = fconstrain(p_temp->ball->x,
                                     SCREEN_WIDTH - p_temp->ball->radius*2 - BORDER_WIDTH);
        }

        // Bouncing off down or upper border
        if (p_temp->ball->y < BORDER_WIDTH ||
            p_temp->ball->y + TB_RADIUS * 2 > SCREEN_HEIGHT - BORDER_WIDTH) {

            p_temp->ball->y -= p_temp->ball->dy * t_frame;
            p_temp->ball->dy = -p_temp->ball->dy * BOUNCINESS;
            p_temp->ball->y = fconstrain(p_temp->ball->y,
                                     SCREEN_HEIGHT - p_temp->ball->radius*2 - BORDER_WIDTH);
        }
        p_temp = p_temp->next;
    }
    delete p_temp;
}

// Prints GAME OVER
void GameOver() {
    SDL_Surface* back_gameover;   // background image for gameover
    back_gameover = LoadImageAlpha(gameover_file);
    SDL_Rect dest;    
    char msg[100];

    dest.x = 0;
    dest.y = 0;
    SDL_BlitSurface(back_gameover, NULL, screen, &dest);


    // print player's score and time
    sprintf(msg, "%d (%4.3f sec.)", SCORE, game_time);

    dest.x = 314;
    dest.y = 372;

    SDL_Color clr1 = {255, 255, 255};
    PrintTTF(screen, msg, font_file2, 20, clr1, dest);

    // input player's name
    char name [255];
    snprintf(name,254,"%s", (char*) ui->str.c_str());

    dest.x = 416;
    dest.y = 340;

    SDL_Color clr2 = {0, 0, 0, 0};
    PrintTTF(screen, name, font_file2, 24, clr2, dest);
	
    played_timer.stop();
    SDL_FreeSurface(back_gameover);
}

// Solve collisions beetween any two balls
void Collision() {
    float length;
    float distance;
    int i, j;

    // Ball vs Crazy Balls
    for (i = 0; i < N_CRAZYBALLS; i++) {
        length = (ball->x - crazy_balls[i]->x) * (ball->x - crazy_balls[i]->x) +
                 (ball->y - crazy_balls[i]->y) * (ball->y - crazy_balls[i]->y);

        distance = B_RADIUS + CB_RADIUS;
        if(length < distance * distance && !weakness) {
            //Mix_PlayChannel( -1, bounce_crazy, 0 ); // Play the bounce sound effect
            CollisionBalls(ball, crazy_balls[i]);
		}
    }

    // Crazy Ball vs Crazy Ball
    for (i = 0; i < N_CRAZYBALLS - 1; i++) {
        for (j = i + 1; j < N_CRAZYBALLS; j++) {
            length = (crazy_balls[i]->x - crazy_balls[j]->x) *
                     (crazy_balls[i]->x - crazy_balls[j]->x) +
                     (crazy_balls[i]->y - crazy_balls[j]->y) *
                     (crazy_balls[i]->y - crazy_balls[j]->y);

            distance = CB_RADIUS + CB_RADIUS;

            if(length < distance * distance) {
                //Mix_PlayChannel( -1, bounce_crazy_crazy, 0 ); // Play the bounce sound effect
                CollisionBalls(crazy_balls[i], crazy_balls[j]);
			}
        }
    }

    // Ball vs Star
    for (i = 0; i < N_STARS; i++) {
        length = (ball->x - stars[i]->x) * (ball->x - stars[i]->x) +
                 (ball->y - stars[i]->y) * (ball->y - stars[i]->y);

        distance = B_RADIUS + STAR_RADIUS;

        // If Ball faced with Star, then loosing life
        if(length < distance * distance && !weakness) {
            if (lives == 0) {
                int highscore;               

                gameover = true;
                highscore = ui->score_system->HandleHighscore();

                // Comparison of highscore with current score
                if (SCORE > highscore)
                    newhighscore = true;                
            } else {
                CollisionBalls(ball, stars[i]);
		weakness_start = SDL_GetTicks();
		lives--;
		tb_list->DelLinks();
		weakness = true;
            }
        }
    }

    element *p_temp;
    // Crazy Balls vs Tail Balls
    for (i = 0; i < N_CRAZYBALLS; i++) {
        p_temp = tb_list->p_head;
        while (p_temp != NULL) {
            length = (crazy_balls[i]->x - p_temp->ball->x) *
                     (crazy_balls[i]->x - p_temp->ball->x) +
                     (crazy_balls[i]->y - p_temp->ball->y) *
                     (crazy_balls[i]->y - p_temp->ball->y);

            distance = CB_RADIUS + TB_RADIUS;

            if(length < distance * distance)
                CollisionBalls(crazy_balls[i], p_temp->ball);
            p_temp = p_temp->next;
        }
    }

    // Tail Balls vs Tail Balls
    p_temp = tb_list->p_head;
    element *p_temp2;
    while (p_temp != NULL) {
        p_temp2 = p_temp;
        while (p_temp2 != NULL) {
            length = (p_temp->ball->x - p_temp2->ball->x) *
                     (p_temp->ball->x - p_temp2->ball->x) +
                     (p_temp->ball->y - p_temp2->ball->y) *
                     (p_temp->ball->y - p_temp2->ball->y);

            distance = TB_RADIUS + TB_RADIUS;

            if(length < distance * distance && length > 0) {
                //Mix_PlayChannel( -1, bounce_tail_tail, 0 ); // Play the bounce sound effect
                CollisionBalls(p_temp->ball, p_temp2->ball);
			}
            p_temp2 = p_temp2->next;
        }
        p_temp = p_temp->next;
    }    

    // Ball vs Tail Ball
    p_temp = tb_list->p_head;
    while (p_temp != NULL) {
        length = (ball->x - p_temp->ball->x) * (ball->x - p_temp->ball->x) +
                 (ball->y - p_temp->ball->y) * (ball->y - p_temp->ball->y);

        if (length < SPRING_MAX_LENGTH * SPRING_MAX_LENGTH && !weakness) {
            if (!p_temp->ball->linked) {
                // link this tail ball to player's ball
                p_temp->ball->linked = true;

                if (!p_temp->ball->timer.is_started()) {
                    p_temp->ball->timer.start();
                }
            } else {
                if (p_temp->ball->timer.get_ticks() >= LINK_DELAY) {
                    p_temp->ball->fixed = true;
                    tb_list->count_linked++;
					//Mix_PlayChannel( -1, tail_adding, 0 ); // Play the bounce sound effect
                    p_temp->ball->timer.stop();
                }

                // counts force, attracting to ball and tail ball
                SolveLink(p_temp->ball);
            }
        } else {
            p_temp->ball->timer.stop();
            if (p_temp->ball->linked) {
                p_temp->ball->linked = false;                
            } 

            if (p_temp->ball->fixed) {
                p_temp->ball->fixed = false;
                tb_list->count_linked--;
            }
        }
        p_temp = p_temp->next;
    }
    delete p_temp;
}

// simple sign function
int sign(int v) {
    return v > 0 ? 1 : (v < 0) ? -1 : 0;
}

// Solve Hooke's law f = -k * x - v * b and apply this force to ball and current tail ball
void SolveLink(Ball* linked_ball) {
    const float SPRING_TOLERANCE = 0.0000000005;
    float distance;     // distance between ball and tail ball (x)
    float delta;        // x - d
    float intensity;    // k * (x - d)
    Vector2D force_dir; // force direction
    Vector2D force;     // force vector
    Vector2D tb_acc;    // acceleration of tail ball
    Vector2D ball_acc;  // acceleration of ball

    /**** Some theory... ****/
    // Hooke's law for spring-damper system: F = -k * (x - d) - v * b, where
    // k - constant of stiffnes of the spring
    // x - distance between two ends of the spring
    // d - rest distance of the spring
    // v - relative velocity
    // b - coefficient of damping

    float dx = ball->x - linked_ball->x;
    float dy = ball->y - linked_ball->y;

    // count x as length of vector between ball and tail ball
    distance = (float)sqrt(dx * dx + dy * dy);

    // direction of force
    force_dir.x = dx;
    force_dir.y = dy;

    // normalize vector
    force_dir.x /= distance;
    force_dir.y /= distance;

    force_dir *= SPRING_REST_LENGTH;
    dx -= force_dir.x;
    dy -= force_dir.y;

    // some magic
    if (distance < SPRING_TOLERANCE) {
        force.x = 0;
        force.y = 0;
    } else {
        // F = -(x - d) * k
        force.x += dx * SPRING_STRENGTH;
        force.y += dy * SPRING_STRENGTH;
    }

    // count relative velocity
    float vx = ball->dx - linked_ball->dx;
    float vy = ball->dy - linked_ball->dy;

    Vector2D r_vel = Vector2D(vx, vy);

    // sub damping force
    force += r_vel * SPRING_DAMPING;

    // extract acceleration from third Newton's Law F = m*a, so a = F / m
    tb_acc = force / linked_ball->mass;
    ball_acc = force / ball->mass;

    // update velocity of ball and tail ball
    ball->dx += ball_acc.x * t_frame;
    ball->dy += ball_acc.y * t_frame;
    linked_ball->dx += tb_acc.x * t_frame;
    linked_ball->dy += tb_acc.y * t_frame;
}

// Find free position for unlinked tail ball and place it
void FindFreePosition(Ball *b) {
    int i,j;
    b->reset_velocity();

    int num_balls = N_CRAZYBALLS + N_STARS + 1;   
    Ball* balls_array[num_balls]; // array of all balls, except tail balls

    // fill array
    for (i = 0; i < N_CRAZYBALLS; i++)
        balls_array[i] = crazy_balls[i];

    for (i = N_CRAZYBALLS; i < N_STARS + N_CRAZYBALLS; i++)
        balls_array[i] = stars[i - N_CRAZYBALLS];

    balls_array[num_balls - 1] = ball;

    // random location of balls
    bool free;
    Ball* s1 = b;

    // try to locate ball to random place
    // if it isn't free try again
    do {
        s1->x = (SCREEN_WIDTH - BORDER_WIDTH - s1->radius * 2) *
                0.01 * (rand() % 100) + s1->radius;
        s1->y = (SCREEN_HEIGHT - BORDER_WIDTH - s1->radius * 2) *
                0.01 * (rand() % 100) + s1->radius;

        free = true;
        for (j = 0; j < num_balls; j++) {
            Ball* s2 = balls_array[j];

            // check if it is free
            if ((s2->x - s1->x) * (s2->x - s1->x) +
                (s2->y - s1->y) * (s2->y - s1->y) <=
                (s2->radius + s1->radius) * (s2->radius + s1->radius)) {
                    free = false;
                    break;
            }
        }
    } while (!free);
}

// Counts linked tail balls and cut tail
void CountLinks() {
    element *p_temp = tb_list->p_head;
    CountScore(tb_list->count_linked);
    while (p_temp != NULL) {
        p_temp->ball->fixed = false;

        // if it is linked, then delete link and locate tail ball to new position
        if (p_temp->ball->linked) {
	    	// Play the sound effect
            //Mix_PlayChannel( -1, tail_breaking, 0 );

            p_temp->ball->linked = false;            
            FindFreePosition(p_temp->ball);
        }
        p_temp = p_temp->next;
    }
    delete p_temp;

    // reset counter of linked balls
    tb_list->count_linked = 0;
}

// Counts player score
void CountScore(int num_links) {
    SCORE += (int)(num_links * ONE_BALL_COST * exp(sqrt(num_links)));
    SCORE += (int)(game_time * sqrt(num_links));
}

// Shows lives in right top corner
void ShowInfo() {
    SDL_Color clr = {255, 255, 255, 0};
    SDL_Rect rect = {20, 20, 0, 0};
    char message[30];
    sprintf(message, "SCORE: %d", SCORE);
    PrintTTF(screen, message, font_file, 24, clr, rect);

	rect.x = 640;
	rect.y = 20;
	sprintf(message, "LIVES: %d", lives+1);
    PrintTTF(screen, message, font_file, 24, clr, rect);
}

// Count velocity after collision
void CollisionBalls(Ball *a, Ball *b) {
    float x, y, n; // (x, y) is unit vector from a to b.
    float va, vb;  // va, vb are balls' speeds along (x, y)
    float ma, mb;  // ma, mb are the balls' masses.
    float vc;      // vc is the "center of momentum"

    // (x, y) is unit vector pointing from A's center to B's center.
    x = (b->x + b->radius) - (a->x + a->radius);
    y = (b->y + b->radius) - (a->y + a->radius);
    n = sqrt(x * x + y * y);
    x /= n;
    y /= n;

    // velocities along (x, y)
    va = x * a->dx + y * a->dy;
    vb = x * b->dx + y * b->dy;
    // don't bounce if we're already moving away.
    if (vb - va > 0)
        return;

    // get masses and compute "center" speed
    ma = a->mass;
    mb = b->mass;
    vc = (va * ma + vb * mb) / (ma + mb);

    // bounce off the center speed.
    a->dx += 2 * x * (vc - va);
    a->dy += 2 * y * (vc - va);
    b->dx += 2 * x * (vc - vb);
    b->dy += 2 * y * (vc - vb);
}

// Start new game
void NewGame() {
    int i;
	paused = false;
	ui->flag = true;
	lives = LIVES;
	weakness = false;

    tb_list->DelLinks();

    // reset tail balls velocity
    element *p_temp = tb_list->p_head;
    while (p_temp != NULL) {
        p_temp->ball->reset_velocity();
        p_temp = p_temp->next;
    }
    delete p_temp;

    // Reset velocity of Ball and move balls to start locations
    ball->reset_velocity();
    LocateBalls();

    gameover = false;
	newhighscore = false;

    SCORE = 0;
    ms_end = SDL_GetTicks();    

    for (i = 0; i < N_CRAZYBALLS; i++) 
        crazy_balls[i]->reset_velocity();

    for (i = 0; i < N_STARS; i++) 
        stars[i]->reset_velocity();

    played_timer.start();
    crazy_timer.stop();
}

// Get acceleration from accelerometer
Uint32 DevCallback(Uint32 interval, void *param) {
    acc_vector = dev_states->get();
    return interval;
}

/*
 * Main function of program, contains game loop.
 */
int main(int argc, char** argv) {
    SDL_Event event;
    Uint8 *keys;
    float x, y;
    int done = 0; // flag for game loop
	//int display = 1;
    // Initialize SDL and creates all game objects
	bool help_var = false;
    Init();

	//dbus_threads_init_default();

    // Check if file scores.xml exist, then use it. Else create new.
    ui->score_system->CheckFile();

    ui->flag = false;
    ui->Init(screen);
    ui->menu = true;
	
    ui->flag_m = MENU;

    // timers
    // ACC: 
	SDL_TimerID accTimer = SDL_AddTimer(ACC_DELAY, DevCallback, callback_param);

	//acc_timer.start();
    played_timer.start();
    crazy_timer.start();

    // game loop
    while (done == 0) {
		// if display is off - pause and disable accelerometer
		if ((accTimer == NULL) && (dev_states->display == 1)) {
				accTimer = SDL_AddTimer(ACC_DELAY, DevCallback, callback_param);
		} else {
			if (dev_states->display == 0) {
				if ((accTimer != NULL)) {
					SDL_RemoveTimer(accTimer);
					accTimer = NULL;
				}
				dev_states->displayState();
			}
		}

        // weakness end
        if (SDL_GetTicks() - weakness_start > WK_DELAY && weakness) {
            weakness = false;
            blink_counter = 0;
        }

		if ((ui->menu) && (accTimer != NULL)) {
			paused = false;
			SDL_RemoveTimer(accTimer);
			accTimer = NULL;
		}

        ms_frame = SDL_GetTicks() - ms_end;
        ms_end += ms_frame;

        if(ms_frame > 50)
           	ms_frame = 50;
        t_frame = gamespeed * ms_frame / 50;
		
        // Handle keyboard events
        while (SDL_PollEvent(&event)) {
			if (event.type == SDL_ACTIVEEVENT) {
				if ((event.active.gain == 1) && (help_var)) {
					paused = false;
					help_var = false;
				} else {
					if ((event.active.gain == 0) && (accTimer != NULL)) {
						paused = true;
						SDL_RemoveTimer(accTimer);
						accTimer = NULL;
						help_var = true;
					}
				}
			}

            if (event.type == SDL_QUIT) {
                done = 1;
				break;
            }
            if (!ui->menu) {
				if (!gameover) { 
                    if (event.type == SDL_KEYDOWN) {
                        switch (event.key.keysym.sym) {
                            case SDLK_ESCAPE :    // exit
                                done = 1;
                                break;
                            case SDLK_KP_ENTER :  // pause
                                paused = !paused;
                                break;
                            case SDLK_SPACE :     // start new game
                                NewGame();
                                break;
                            case SDLK_a :         // cut tail
                                CountLinks();
                                break;
                            case SDLK_m :
                                ui->flag = true;
                                ui->Init(screen);
                                ui->menu = true;
                                break;
                    	}
					}
                } else {
                    // Input name to save result in highscores
                    done = ui->HandleInputName(screen, event, SCORE);
				}

				if (event.type == SDL_MOUSEBUTTONDOWN) {
                    if (event.button.button == SDL_BUTTON_LEFT) {
                        if (gameover) {
                            if (ui->flag_m != NEW_GAME) {
                                x = event.button.x;
								y = event.button.y;
				
								char string [255];
								char name [255];

								snprintf(string,254,"%d", SCORE);

                                if (ui->str != "") {
                                    snprintf(name,254,"%s", (char*) ui->str.c_str());
                                    ui->score_system->XmlConnect(true, name, string);
								}                               

                                ui->str = "";
								if (!newhighscore) {
                                    if (((x > 597) && (x < 773) && (y > 414) && (y < 454)) ) {
                                        ui->flag = false;
                                        ui->flag_m = HIGHSCORE;
                                        ui->Highscore(screen);
                                        ui->menu = true;
                                        break;
                                    } else {
                                        NewGame();
                                        ui->flag_m = MENU;
                                        ui->menu = false;
                                        break;
                                    }
								} else {
                                    ui->flag = false;
                                    ui->flag_m = HIGHSCORE;
                                    ui->Highscore(screen);
                                    ui->menu = true;
                                    break;
								}
                            } else {
								NewGame();
								ui->flag_m = MENU;
								ui->menu = false;	
								break;				
                            }
						} else {
                            CountLinks();
						}	
                    }
				}
            } else {
				done = ui->HandleInput(screen, event);
            }
        }

		if (ui->flag_m == NEW_GAME) {
            NewGame();
            ui->flag_m = MENU;
            ui->menu = false;
        } else {
            if (ui->flag_m == EXIT)
                done = 1;

		}

        if (!ui->menu) {
            keys = SDL_GetKeyState(NULL);
            if (keys[SDLK_UP])
             	ball->dy -= THRUSTER_STRENGTH * t_frame;
            if (keys[SDLK_DOWN])
               	ball->dy += THRUSTER_STRENGTH * t_frame;
            if (keys[SDLK_LEFT])
              	ball->dx -= THRUSTER_STRENGTH * t_frame;
            if (keys[SDLK_RIGHT])
           	    ball->dx += THRUSTER_STRENGTH * t_frame;

            if (!paused && dev_states->display == 1 && !gameover) {
                // move objects
                // ACC: 
				AccMoveBall();
                MoveBall();
                MoveCrazyBalls();
                MoveTailBalls();
                MoveStars();
                Bounce();
	
                // solve collisions
                Collision();
            }
	
            // Count played time
            if (paused || dev_states->display == 0) {
                played_timer.pause();
                crazy_timer.pause();
            } else {
                if (crazy_timer.is_paused()) {
                    crazy_timer.unpause();
                }
            }
		
            if (played_timer.is_started()) {
                game_time = (played_timer.get_ticks() / 1000);
            } else {
                if (played_timer.is_paused())
                    played_timer.unpause();
            }
	
            // activate/disable crazy time for crazy balls
            if (!crazy_timer.is_started()) {
                CRAZY_TIME = rand() % 20000;
                crazy_timer.start();
            } else {
                if (crazy_timer.get_ticks() >= CRAZY_DELAY &&
                    crazy_timer.get_ticks() < CRAZY_DELAY + CRAZY_TIME) {
                    AGRESSION = CRAZY;
        	} else {                    
                    AGRESSION = NORMAL;      
                    if (crazy_timer.get_ticks() >= CRAZY_DELAY + CRAZY_TIME)
                        crazy_timer.stop();
                }                
            }

            // update scene
            if (!paused && dev_states->display == 1) 
                DrawScene();

            TinySleep();
		} else {
            SDL_Flip(screen);
		}
    }

    // ACC: 
	SDL_RemoveTimer(accTimer);

    delete ui;

    return 0;
}
