/* This file is part of QtTvDB
 *
 * Copyright (C) 2009-2010 Corentin Chary <corentin.chary@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef QTTVDB_H
#define QTTVDB_H

#include "qttvdb_export.h"

#include <QDebug>
#include <QtCore/QObject>
#include <QtCore/QStringList>
#include <QtCore/QUrl>
#include <QtCore/QDateTime>
#include <QtXml/QDomNode>
#include <QtCore/QVariant>
#include <QtCore/QPair>

#  define TVDB_PROPERTY_(type_, name_, getter_, setter_)		\
private:                                                                \
 type_ name_##_;							\
public:									\
  inline type_ & getter_() { return name_##_; }				\
  inline type_ const & getter_() const { return name_##_; }


#  define TVDB_PROPERTY_SIMPLE(type_, name_, getter_, setter_)		\
  TVDB_PROPERTY_(type_, name_, getter_, setter_)			\
  inline void setter_(type_ const & name_##__) { name_##_ = name_##__; }


#  define TVDB_PROPERTY(type_, name_, getter_, setter_)			\
  TVDB_PROPERTY_(type_, name_, getter_, setter_)			\
  inline void setter_(type_ const & name_##__) {			\
    map_[#name_] = name_##__;						\
    name_##_ = name_##__;						\
  }

namespace QtTvDB {

    class QTTVDB_EXPORT Mirrors {
	TVDB_PROPERTY_SIMPLE(QString, key, key, setKey);
	TVDB_PROPERTY_SIMPLE(QList<QUrl>, xmlMirrors, xmlMirrors, setXmlMirrors);
	TVDB_PROPERTY_SIMPLE(QList<QUrl>, bannerMirrors, bannerMirrors, setBannerMirrors);
	TVDB_PROPERTY_SIMPLE(QList<QUrl>, zipMirrors, zipMirrors, setZipMirrors);
	TVDB_PROPERTY_SIMPLE(QStringList, languages, languages, setLanguages);

    public:
	Mirrors();
	Mirrors(const QString & key);
	~Mirrors();

	enum Mask {
	  XML = 1,
	  BANNER = 2,
	  ZIP = 4
	};

	QUrl mirrorsUrl();
	QUrl languagesUrl();
	QUrl serverTimeUrl();
	QUrl searchShowUrl(const QString & showName, const QString & language = QString());
	QUrl showUrl(qint64 showId);
	QUrl episodeUrl(qint64 episodeId);
	QUrl showAndEpisodesUrl(qint64 showId, const QString & language = QString());
	QUrl showAndEpisodesUrlZip(qint64 showId, const QString & language);
	QUrl updatesUrl(const QString & period);
	QUrl updatedShowsUrl(const QDateTime & period);
	QUrl updatedEpisodesUrl(const QDateTime & period);
	QUrl bannersUrl(qint64 showId);
	QUrl bannerUrl(const QString & path);
	int languageId(const QString & l);
	QString languageById(int id);
	QString languageNameById(int id);
	QString languageName(const QString & l);
	static QList< QPair < Mask, QUrl > > parseMirrors(const QString & xml);
	static QList< QPair < Mask, QUrl > > parseMirrors(const QDomNode & node);
	QUrl randomMirror(const QList<QUrl> & mirrors, bool key = false,
			  const QString & postfix = QString(), bool api = true);
    private:
	void init();
	QMap < QString, QPair < int, QString > > languagesDb_;
	QMap < int, QString > languagesById_;
    };

    class QTTVDB_EXPORT Episode {
    public:
	Episode();
	Episode(const QString & xml);
	Episode(const QDomNode & node);
	~Episode();

	bool isNull();
	QVariantMap map();
	static QList< Episode * > parseEpisodes(const QString & xml);
	static QList< Episode * > parseEpisodes(const QDomNode & node);

	TVDB_PROPERTY(qint64, id, id, setId);
	TVDB_PROPERTY(qint64, showId, showId, setShowId);
	TVDB_PROPERTY(qint64, seasonId, seasonId, setSeasonId);
	TVDB_PROPERTY(QString, name, name, setName);
	TVDB_PROPERTY(QString, overview, overview, setOverview);
	TVDB_PROPERTY(int, season, season, setSeason);
	TVDB_PROPERTY(int, episode, episode, setEpisode);
	TVDB_PROPERTY(QString, director, director, setDirector);
	TVDB_PROPERTY(QStringList, guestStars, guestStars, setGuestStars);
	TVDB_PROPERTY(QString, language, language, setLanguage);
	TVDB_PROPERTY(QString, productionCode, productionCode, setProductionCode);
	TVDB_PROPERTY(QString, rating, rating, setRating);
	TVDB_PROPERTY(QString, writer, writer, setWriter);

	TVDB_PROPERTY(QDateTime, firstAired, firstAired, setFirstAired);

	TVDB_PROPERTY(QString, dvdChapter, dvdChapter, setDvdChapter);
	TVDB_PROPERTY(QString, dvdDiscId, dvdDiscId, setDvdDiscId);
	TVDB_PROPERTY(QString, dvdEpisodeNumber, dvdEpisodeNumber, setDvdEpisodeNumber);
	TVDB_PROPERTY(QString, dvdSeason, dvdSeason, setDvdSeason);

	TVDB_PROPERTY(QString, image, image, setImage);

	TVDB_PROPERTY(QString, airsAfterSeason, airsAfterSeason, setAirsAfterSeason);
	TVDB_PROPERTY(QString, airsBeforeSeason, airsBeforeSeason, setAirsBeforeSeason);
	TVDB_PROPERTY(QString, airsBeforeEpisode, airsBeforeEpisode, setAirsBeforeEpisode);

	TVDB_PROPERTY(int, combinedEpisode, combinedEpisode, setCombinedEpisode);
	TVDB_PROPERTY(int, combinedSeason, combinedSeason, setCombinedSeason);
	TVDB_PROPERTY(int, absoluteNumber, absoluteNumber, setAbsoluteNumber);
	TVDB_PROPERTY(QString, epImgFlag, epImgFlag, setEpImgFlag);

	TVDB_PROPERTY(QString, imdbId, imdbId, setImdbId);

	TVDB_PROPERTY(QDateTime, lastUpdated, lastUpdated, setLastUpdated);
    private:
	void init();
	void init(const QDomNode &node);
	QVariantMap map_;
    };

    class QTTVDB_EXPORT Show {
    public:
	Show();
	Show(const QString & xml);
	Show(const QDomNode & node);
	~Show();

	bool isNull();
	QVariantMap map();

	static QList< Show * > parseShows(const QString & xml);
	static QList< Show * > parseShows(const QDomNode & node);

	TVDB_PROPERTY(qint64, id, id, setId);
	TVDB_PROPERTY(QString, name, name, setName);
	TVDB_PROPERTY(QString, overview, overview, setOverview);
	TVDB_PROPERTY(QStringList, genre, genre, setGenre);
	TVDB_PROPERTY(QStringList, actors, actors, setActors);
	TVDB_PROPERTY(QString, network, network, setNetwork);
	TVDB_PROPERTY(QString, contentRating, contentRating, setContentRating);
	TVDB_PROPERTY(QString, rating, rating, setRating);
	TVDB_PROPERTY(QTime, runtime, runtime, setRuntime);
	TVDB_PROPERTY(QString, status, status, setStatus);
	TVDB_PROPERTY(QString, language, language, setLanguage);

	TVDB_PROPERTY(QDateTime, firstAired, firstAired, setFirstAired);
	TVDB_PROPERTY(QString, airsDay, airsDay, setAirsDay);
	TVDB_PROPERTY(QTime, airsTime, airsTime, setAirsTime);

	TVDB_PROPERTY(QString, banner, banner, setBanner);
	TVDB_PROPERTY(QString, poster, poster, setPoster);
	TVDB_PROPERTY(QString, fanArt, fanArt, setFanArt);

	TVDB_PROPERTY(QString, imdbId, imdbId, setImdbId);
	TVDB_PROPERTY(QString, seriesId, seriesId, setSeriesId);
	TVDB_PROPERTY(QString, zap2itId, zap2itId, setZap2ItId);

	TVDB_PROPERTY(QDateTime, lastUpdated, lastUpdated, setLastUpdated);
    private:
	void init();
	void init(const QDomNode &node);
	QVariantMap map_;
    };

    class QTTVDB_EXPORT Banner {
    public:
	Banner();
	Banner(const QString & xml);
	Banner(const QDomNode & node);
	~Banner();

	bool isNull();
	QVariantMap map();

	static QList< Banner * > parseBanners(const QString & xml);
	static QList< Banner * > parseBanners(const QDomNode & node);

	TVDB_PROPERTY(qint64, id, id, setId);
	TVDB_PROPERTY(QString, path, path, setPath);
	TVDB_PROPERTY(QString, type, type, setType);
	TVDB_PROPERTY(QString, type2, type2, setType2);
	TVDB_PROPERTY(QString, language, language, setLanguage);
	TVDB_PROPERTY(QString, season, season, setSeason);
    private:
	void init();
	void init(const QDomNode &node);
	QVariantMap map_;
    };
};

#endif // QTTVDB_H
