/*
 * Copyright (C) 2009, Jamie McCracken (jamiecc at gnome org)
 * Copyright (C) 2010, Bundyo (bundyo at gmail.com)
 *
 * Initially written by Jamie McCracken in Genie. 
 * Converted to Vala and butchered (downgraded to Tracker 0.6, 
 * Hildonized and adapted to Maemo 5) by Bundyo.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA  02110-1301, USA.
 */     


using Gtk;
using Hildon;
using TrackerUtils;

enum ResultColumns {
    Uri,
    Icon,
    DisplayName,
    Mime,
    Category,
    Snippet,
    IsDirectory,
    Path,
    NumOfCols
}

string launch_uri;
bool launch_dir;
    
const TargetEntry[] targets = {{ "text/uri-list", 0, 1 },{ "text/plain",    0, 0 },{ "STRING",	   0, 0 }};

class TrackerResultGrid : PannableArea {
    public ListStore store;
    public IconView iconview;
    public TrackerQuery _query;
    public Gtk.Menu menu;
    public unowned Gdk.EventButton last_event;
    
    private string last_selection;
    private bool cancel_event;

    public signal void SelectionChanged (TreePath? path);

    public TrackerQuery Query { 
        get {
        	return _query;
       	}
        set {
	    	if (value != null) {
                _query = value;
                _query.SearchSettingsChanged += () => {
                    RefreshQuery ();
                };
                _query.ClearSearchResults += () => {
                    store.clear ();
                };
            }
        }
    }

    public TreePath? GetSelectedPath () {
        weak GLib.List<TreePath>? l;

        l = iconview.get_selected_items ();

        if ((l != null) && (l.data != null))
            return l.data;

        return null;
    }


    public string GetSelectedUri ( out bool is_dir ) {
        TreeIter iter;
        weak string uri;
        is_dir = false;

        var path = GetSelectedPath ();
        if (path != null) {
            store.get_iter (out iter, path);
            store.get (iter, ResultColumns.Uri, out uri);
	        store.get (iter, ResultColumns.IsDirectory, out is_dir);
            return uri;
        }

        return "";
	}

    construct {

        set( "mov_mode", MovementMode.VERT );
        initial_hint = true;
        low_friction_mode = false;
        cancel_event = false;
        
        hscrollbar_policy = PolicyType.NEVER;
        vscrollbar_policy = PolicyType.AUTOMATIC;

        store = new ListStore (ResultColumns.NumOfCols, typeof (string), typeof (Gdk.Pixbuf), typeof (string), 
                               typeof (int), typeof (string), typeof (string), typeof (bool), typeof (string));

        iconview = (IconView) gtk_icon_view_new_with_model ( UIMode.NORMAL, store);
        iconview.set_pixbuf_column (ResultColumns.Icon);
        iconview.set_text_column (ResultColumns.DisplayName);
        iconview.set_selection_mode (SelectionMode.SINGLE);
        iconview.set_orientation(Orientation.HORIZONTAL);
        iconview.set_item_width (189);
        iconview.set_row_spacing (10);
        
        iconview.button_press_event += iconview_pressed;
        iconview.button_release_event += iconview_released;
        iconview.item_activated += ActivateUri;

        iconview.selection_changed += () => {
            var path = GetSelectedPath ();
            SelectionChanged (path);
        };
        
        menu = (Gtk.Menu) gtk_menu_new();
        
        iconview.tap_and_hold_setup( menu, show_popup, TapAndHoldFlags.NONE );
        
        add_with_viewport (iconview);
        show_all ();
    }

	public bool iconview_pressed (Gtk.IconView self, Gdk.EventButton ev) {
        last_event = ev;
        
        return false;
    }
    
	public bool iconview_released (Gtk.IconView self, Gdk.EventButton ev) {
		if (!cancel_event) {
			string uri;
			bool is_dir;
	
			uri = GetSelectedUri(out is_dir);
	
	        if (uri == last_selection)
	        	OpenUri (uri, is_dir);
	        else
	        	last_selection = uri;
    	}
        return false;
    }

	public static void show_popup () {
		
		launch_uri = grid.GetSelectedUri( out launch_dir );
		grid.empty_menu(grid.menu);

		if (launch_uri == "") {
			var item = new MenuItem.with_label("Clear Search");
			item.activate += (s) => {
				entry.text = "";
				entry.activate();
				entry.grab_focus();
			};
			grid.menu.add(item);
		} else {
			
			var item = new MenuItem.with_label("Open");
			item.activate += (s) => {
				OpenUri (launch_uri, launch_dir);
			};
			grid.menu.add(item);
			
			item = new MenuItem.with_label("Open with...");
			item.activate += (s) => {
				OpenUriWith (launch_uri, launch_dir);
			};
			grid.menu.add(item);
			
//			item = new MenuItem.with_label("Open Containing Folder");
//			item.activate += (s) => {
//				OpenUri (uri, true);
//			};
//			grid.menu.add(item);
		}
		
		grid.menu.show_all();
		
//			var item = new MenuItem.with_label("Properties");
//			item.activate += (s) => {
//				OpenUri (uri, is_dir);
//			};
//			menu.add(item);
	}

	public void empty_menu (Gtk.Menu menu) {
        if (menu.children.length() > 0) {
	        for (int i = (int) menu.children.length()-1; i >= 0; i--) {
		        menu.children.nth_data(i).destroy();
        	}
  	  	}
	}

//	public string? GetSelectedUriFromEvent ( Gdk.EventButton e, out bool is_dir ) {
//		TreePath? path;
//		TreeIter iter;
//        is_dir = false;
//        weak string uri;
//        
//        iconview.get_item_at_pos((int) e.x, (int) e.y, out path, null);
////        if (path == null)
////        	return "";
//        	
//        store.get_iter (out iter, path);
//        store.get (iter, ResultColumns.Uri, out uri);
//        store.get (iter, ResultColumns.IsDirectory, out is_dir);
//        
//        return uri;
//	}

    public void RefreshQuery () {
        if (_query != null) {
            string[,] results = _query.NormalQuery ();
            
            var has_results = false;
 
            store.clear ();

            if (results == null) return;

			for (uint i = 0; i < results.length[0]; i++) {
	            TreeIter iter;
	            
	           	var uri = results[i, 0];
	           	
                has_results = true;

                var file = File.new_for_path (uri);

                try {
                    var info =  file.query_info ("standard::display-name,standard::icon,thumbnail::path", 
                                                 FileQueryInfoFlags.NONE, null);

                    var filetype =  info.get_file_type ();
                    store.append (out iter);
                    store.set (iter, ResultColumns.Uri, uri, ResultColumns.Icon, GetThumbNail (info, 32, 32, get_screen()), 
                              ResultColumns.DisplayName, info.get_display_name().strip(), ResultColumns.IsDirectory, 
                              ((results[i, 1] == "Folders") ? true : false), -1); 
                } catch (Error e) {
                    print ("Could not get file info for %s", uri);
                }
			}

            /* select first result */
//            if (has_results) {
//                var path = new TreePath.from_string ("0:0:0");
//                if (path != null)
//                    iconview.select_path (path);
//            }
        }
	}

    public void ActivateUri (TreePath path) {
        TreeIter iter;
        bool is_dir = false;

        store.get_iter (out iter, path);
        weak string uri;
        store.get (iter, ResultColumns.Uri, out uri);
        store.get (iter, ResultColumns.IsDirectory, out is_dir);

        OpenUri (uri, is_dir);
    }
}