#include <string.h>
#include <hildon/hildon.h>
#include "scv-file.h"
#include "charmap-dialog.h"
#include "maemo-gucharmap-chartable.h"
#include "suggestion-codepoint-list.h"

#define OSSO_BROWSER_SERVICE    "com.nokia.osso_browser"
#define OSSO_BROWSER_PATH       "/com/nokia/osso_browser/request"
#define OSSO_BROWSER_IFACE      "com.nokia.osso_browser"
#define OSSO_BROWSER_NEW_WINDOW "open_new_window"

#define N_PER_ROW 7

#define IS_DIRTY_KEY "is-dirty"
#define SCV_FILE_KEY "scv-file"
#define CHARMAP_KEY "charmap"

static void make_wnd(ScvFile *scv_file);

typedef struct {
	KBButton *kbb;
	GtkWidget *choice_details;
	GtkWidget *picker;
	GtkWidget *suggestions;
	gboolean is_dirty;
	GtkWidget *regular_char_suggestions;
	GtkWidget *show_suggestions_button;
	GtkWidget *suggestions_label;
} ChoiceParams;

static struct { char *c; char *usage ; } modifier_suggestions[] = {
	{ .c = "^", .usage = " -- â, Â, ê, Ê" },
	{ .c = "°", .usage = " -- å, Å, ů, Ů" },
	{ .c = "¸", .usage = " -- ç, Ç, ş, Ş" },
	{ .c = "¨", .usage = " -- ä, Ä, ö, Ö" },
	{ .c = "´", .usage = " -- á, Á, é, É" },
	{ .c = "`", .usage = " -- à, À, è, È" },
	{ .c = "~", .usage = " -- ñ, Ñ, õ, Õ" },
	{ .c = "˘", .usage = " -- ă, Ă, ĕ, Ĕ" },
	{ .c = "˙", .usage = " -- ė, Ė, ż, Ż" },
	{ .c = "˝", .usage = " -- ő, Ő, ű, Ű" },
	{ .c = "ˇ", .usage = " -- č, Č, š, Š" },
	{ .c = "¯", .usage = " -- ē, Ē, ō, Ō" },
};

static void
sync_visible(GObject *src, GParamSpec *pspec, GObject *dst)
{
	gboolean src_visible, dst_visible;

	g_object_get(src, "visible", &src_visible, NULL);
	g_object_get(dst, "visible", &dst_visible, NULL);

	if (src_visible != dst_visible)
		g_object_set(dst, "visible", src_visible, NULL);
}

static void
connect_visibility(GtkWidget *widget1, GtkWidget *widget2)
{
	g_signal_connect(G_OBJECT(widget1), "notify::visible", (GCallback)sync_visible, widget2);
	g_signal_connect(G_OBJECT(widget2), "notify::visible", (GCallback)sync_visible, widget1);
}

static void
set_suggestion_model(GtkTreeView *tv, ChoiceParams *cp)
{
	GtkListStore *ls = NULL;

	if (CHAR_TYPE_MODIFIER == KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb)) {
		int Nix;
		GtkTreeIter itr;

		ls = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);

		for (Nix = 0 ; Nix < G_N_ELEMENTS(modifier_suggestions) ; Nix++) {
			gtk_list_store_append(ls, &itr);
			gtk_list_store_set(ls, &itr,
				 0, modifier_suggestions[Nix].c, 
				 1, modifier_suggestions[Nix].usage,
				-1);
		}
	}

	g_object_set(G_OBJECT(tv), "model", ls, NULL);
	if (ls)
		g_object_unref(G_OBJECT(ls));

	g_object_set(G_OBJECT(cp->suggestions), "visible", ls != NULL, NULL);
	g_object_set(G_OBJECT(cp->regular_char_suggestions), "visible", ls == NULL && CHAR_TYPE_TAB != KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb), NULL);
	g_object_set(G_OBJECT(cp->show_suggestions_button), "visible", ls == NULL && CHAR_TYPE_TAB != KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb), NULL);
}

static void
scroll_to_iter(GtkTreeView *suggestions, HildonPannableArea *hpa, GtkTreeIter *itr)
{
	GtkTreePath *tp;
	GtkTreeModel *tm = gtk_tree_view_get_model(suggestions);

	tp = gtk_tree_model_get_path(tm, itr);
	if (tp) {
		gtk_tree_view_scroll_to_cell(suggestions, tp, gtk_tree_view_get_column(suggestions, 0), TRUE, 0.0, 0.5);
/*
		int x_tree, y_tree, x_widget, y_widget;
		GdkRectangle rc;

		gtk_tree_view_get_cell_area(suggestions, tp, gtk_tree_view_get_column(suggestions, 0), &rc);
		gtk_tree_view_convert_bin_window_to_tree_coords(suggestions, rc.x, rc.y, &x_tree, &y_tree);
		gtk_tree_view_convert_tree_to_widget_coords(suggestions, x_tree, y_tree, &x_widget, &y_widget);

		g_print("suggestions: %2sVISIBLE, hpa: %2sVISIBLE\n", GTK_WIDGET_VISIBLE(suggestions) ? "" : "IN", GTK_WIDGET_VISIBLE(hpa) ? "" : "IN");

		hildon_pannable_area_scroll_to(hpa, -1, y_tree);

		g_print("bin: (%3d, %3d), tree: (%3d, %3d), widget: (%3d, %3d)\n", rc.x, rc.y, x_tree, y_tree, x_widget, y_widget);
*/
		gtk_tree_path_free(tp);
	}
}

static void
find_suggestion(ChoiceParams *cp)
{
	if (KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb) == CHAR_TYPE_MODIFIER) {
		GtkTreeIter itr;
		GtkTreeSelection *ts;
		GtkTreeModel *tm;
		char *cmp;

		ts = gtk_tree_view_get_selection(GTK_TREE_VIEW(cp->suggestions));

		if (gtk_tree_selection_get_selected(ts, &tm, &itr)) {
			gtk_tree_model_get(tm, &itr, 0, &cmp, -1);
			if (!g_strcmp0(cmp, cp->kbb->key)) {
				g_free(cmp);
				return;
			}
		}

		gtk_tree_selection_unselect_all(ts);

		if (tm)
			if (gtk_tree_model_get_iter_first(tm, &itr))
				do {
					gtk_tree_model_get(tm, &itr, 0, &cmp, -1);
					if (!g_strcmp0(cmp, cp->kbb->key)) {
						gtk_tree_selection_select_iter(ts, &itr);
						scroll_to_iter(GTK_TREE_VIEW(cp->suggestions), HILDON_PANNABLE_AREA(gtk_widget_get_parent(cp->suggestions)), &itr);
						g_free(cmp);
						break;
					}
					g_free(cmp);
				} while(gtk_tree_model_iter_next(tm, &itr));
	}
	else
	if (KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb) == CHAR_TYPE_REGULAR) {
		gunichar c = g_utf8_get_char_validated(cp->kbb->key, -1);
		int idx = gucharmap_codepoint_list_get_index(gucharmap_chartable_get_codepoint_list(GUCHARMAP_CHARTABLE(cp->regular_char_suggestions)), c);
		gboolean should_be_visible = (idx >= 0 && (g_utf8_strlen(cp->kbb->key, -1) == 1));

		if (gucharmap_chartable_get_active_character(GUCHARMAP_CHARTABLE(cp->regular_char_suggestions)) != c && should_be_visible)
			gucharmap_chartable_set_active_character(GUCHARMAP_CHARTABLE(cp->regular_char_suggestions), c);

		g_object_set(G_OBJECT(cp->show_suggestions_button), "visible", !should_be_visible, NULL);
		g_object_set(G_OBJECT(cp->regular_char_suggestions), "visible", should_be_visible, NULL);
	}
}

static void
choice_sel_changed(GtkTreeSelection *sel, ChoiceParams *cp)
{
	GtkTreeModel *tm;
	GtkTreeIter itr;
	int char_type;

	gtk_tree_selection_get_selected(sel, &tm, &itr);

	gtk_tree_model_get(tm, &itr, 1, &char_type, -1);

	memcpy(cp->kbb->magic, char_type_magic[char_type], sizeof(cp->kbb->magic));

	cp->is_dirty = TRUE;

	set_suggestion_model(GTK_TREE_VIEW(cp->suggestions), cp);
	g_object_set(G_OBJECT(cp->choice_details), "visible", (KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb) != CHAR_TYPE_TAB), NULL);
	find_suggestion(cp);

	if (KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb) == CHAR_TYPE_TAB) {
		scv_file_assign_tab_key(cp->kbb);
		g_object_set(G_OBJECT(cp->picker), "label", cp->kbb->key, NULL);
	}
}

static GtkWidget *
make_choice(ChoiceParams *cp)
{
	GtkWidget *tv;
	GtkListStore *ls;
	GtkTreeIter itr;
	GtkTreeSelection *sel;
	int char_type = -1;

	ls = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_UINT);
	gtk_list_store_append(ls, &itr);
	gtk_list_store_set(ls, &itr, 0, "Regular Character",  1, CHAR_TYPE_REGULAR,  -1);
	gtk_list_store_append(ls, &itr);
	gtk_list_store_set(ls, &itr, 0, "Modifier Character", 1, CHAR_TYPE_MODIFIER, -1);
	gtk_list_store_append(ls, &itr);
	gtk_list_store_set(ls, &itr, 0, "Tab Character",      1, CHAR_TYPE_TAB,      -1);

	tv = g_object_new(GTK_TYPE_TREE_VIEW, "visible", TRUE, "model", ls, "hildon-ui-mode", HILDON_UI_MODE_EDIT, NULL);
	g_object_unref(ls);
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(tv));
	gtk_tree_selection_set_mode(sel, GTK_SELECTION_BROWSE);

	gtk_tree_view_append_column(GTK_TREE_VIEW(tv),
		gtk_tree_view_column_new_with_attributes(NULL, gtk_cell_renderer_text_new(),
			"text", 0, NULL));

	char_type = KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb);

	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(ls), &itr);
	if (CHAR_TYPE_MODIFIER == char_type)
		gtk_tree_model_iter_next(GTK_TREE_MODEL(ls), &itr);
	else
	if (CHAR_TYPE_TAB == char_type) {
		gtk_tree_model_iter_next(GTK_TREE_MODEL(ls), &itr);
		gtk_tree_model_iter_next(GTK_TREE_MODEL(ls), &itr);
	}

	gtk_tree_selection_select_iter(sel, &itr);

	g_signal_connect(G_OBJECT(sel), "changed", (GCallback)choice_sel_changed, cp);

	return g_object_new(HILDON_TYPE_PANNABLE_AREA, "visible", TRUE, "child", tv, NULL);
}

static void
choice_details_changed(GtkWidget *widget, GParamSpec *pspec, ChoiceParams *cp)
{
	const char *str = hildon_button_get_value(HILDON_BUTTON(widget));

	cp->kbb->key_length = strlen(str);
	g_free(cp->kbb->key);
	cp->kbb->key = g_strdup(str);
	g_object_set(G_OBJECT(cp->picker), "label", cp->kbb->key, NULL);

	find_suggestion(cp);

	cp->is_dirty = TRUE;
}

static void
choice_details_clicked(GtkWidget *btn, ChoiceParams *cp)
{
	GtkWidget *charmap = g_object_get_data(G_OBJECT(btn), CHARMAP_KEY);

	if (charmap) {
		g_object_set(G_OBJECT(charmap), "char-string", hildon_button_get_value(HILDON_BUTTON(btn)), NULL);
		gtk_window_set_transient_for(GTK_WINDOW(charmap), GTK_WINDOW(gtk_widget_get_toplevel(btn)));
		gtk_widget_show(charmap);
	}
}

static void
charmap_char_string_changed(GObject *charmap, GParamSpec *pspec, ChoiceParams *cp)
{
	char *str = NULL;

	g_object_get(G_OBJECT(charmap), "char-string", &str, NULL);
	if (g_strcmp0(hildon_button_get_value(HILDON_BUTTON(cp->choice_details)), str))
		g_object_set(G_OBJECT(cp->choice_details), "value", str, NULL);
	g_free(str);
}

static gboolean
charmap_delete_event(GtkWidget *charmap, GdkEvent *event, gpointer null)
{
	gtk_widget_hide(charmap);
	return TRUE;
}

static GtkWidget *
make_choice_details(GtkWidget *choice_tv, ChoiceParams *cp)
{
	GtkWidget *charmap = g_object_ref_sink(g_object_new(CHARMAP_DIALOG_TYPE, "modal", TRUE, NULL));
	GtkWidget *widget = g_object_new(HILDON_TYPE_BUTTON,
		"size",        HILDON_SIZE_FINGER_HEIGHT,
		"arrangement", HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
		"style",       HILDON_BUTTON_STYLE_PICKER,
		"title",       "Assigned Value:",
		"value",       cp->kbb->key,
		NULL);

	gtk_widget_set_size_request(charmap, -1, 420);
	if (KBB_CHAR_TYPE_FROM_MAGIC(cp->kbb) != CHAR_TYPE_TAB)
		gtk_widget_show(widget);

	g_object_set_data_full(G_OBJECT(widget), CHARMAP_KEY, charmap, (GDestroyNotify)g_object_unref);
	g_signal_connect(G_OBJECT(widget), "notify::value", (GCallback)choice_details_changed, cp);
	g_signal_connect(G_OBJECT(charmap), "notify::char-string", (GCallback)charmap_char_string_changed, cp);
	g_signal_connect(G_OBJECT(charmap), "delete-event", (GCallback)charmap_delete_event, NULL);
	g_signal_connect(G_OBJECT(widget), "clicked", (GCallback)choice_details_clicked, cp);

	return widget;
}

static gboolean
tv_button_release_event(GtkTreeView *tv, GdkEventButton *ev, ChoiceParams *cp)
{
	GtkTreeIter itr;
	GtkTreeModel *tm = gtk_tree_view_get_model(tv);
	GtkTreePath *tp;
	char *str;

	if (gtk_tree_view_get_path_at_pos(tv, ev->x, ev->y, &tp, NULL, NULL, NULL))
		if (tp) {
			if (gtk_tree_model_get_iter(tm, &itr, tp)) {
				GtkTreeSelection *ts = gtk_tree_view_get_selection(tv);

				gtk_tree_model_get(tm, &itr, 0, &str, -1);
				if (g_strcmp0(str, hildon_button_get_value(HILDON_BUTTON(cp->choice_details)))) {
//					g_signal_handlers_block_matched(G_OBJECT(cp->choice_details), G_SIGNAL_MATCH_FUNC, 0, 0, NULL, choice_details_changed, cp);
					g_object_set(G_OBJECT(cp->choice_details), "value", str, NULL);
//					g_signal_handlers_unblock_matched(G_OBJECT(cp->choice_details), G_SIGNAL_MATCH_FUNC, 0, 0, NULL, choice_details_changed, cp);
				}
				g_free(str);
				gtk_tree_selection_select_iter(ts, &itr);
			}
			gtk_tree_path_free(tp);
		}

	return FALSE;
}

static GtkWidget *
make_suggestions(KBButton *kbb, ChoiceParams *cp)
{
	GtkCellRenderer *cr = NULL;
	GtkWidget *tv = g_object_new(GTK_TYPE_TREE_VIEW, "visible", TRUE, "hildon-ui-mode", HILDON_UI_MODE_EDIT, NULL);
	GtkTreeViewColumn *col = g_object_new(GTK_TYPE_TREE_VIEW_COLUMN, NULL);

	cr = g_object_new(GTK_TYPE_CELL_RENDERER_TEXT, "visible", TRUE, NULL);
	gtk_tree_view_column_pack_start(col, cr, FALSE);
	gtk_tree_view_column_add_attribute(col, cr, "text", 0);

	cr = g_object_new(GTK_TYPE_CELL_RENDERER_TEXT, "visible", TRUE, NULL);
	gtk_tree_view_column_pack_start(col, cr, TRUE);
	gtk_tree_view_column_add_attribute(col, cr, "text", 1);

	gtk_tree_view_append_column(GTK_TREE_VIEW(tv), col);

	g_signal_connect(G_OBJECT(tv), "button-release-event", (GCallback)tv_button_release_event, cp);

	return tv;
}

static GtkWidget *
make_regular_char_suggestions()
{
	GtkWidget *widget = g_object_new(MAEMO_GUCHARMAP_CHARTABLE_TYPE, "visible", TRUE, NULL);

	gucharmap_chartable_set_codepoint_list(GUCHARMAP_CHARTABLE(widget),
		suggestion_codepoint_list_from_string("♥☹☺|<>[]{}&\\_=;«»~%‰¡¿²³½º©®€$£¥§ÆæáąćéęëєèѓìíіїłóúãõçμñøœŒßńśźż"));

	return widget;
}

static void
notify_active_character(GObject *chart, GParamSpec *pspec, ChoiceParams *cp)
{
	GString *str = g_string_new("");
	gunichar c;

	g_object_get(G_OBJECT(chart), "active-character", &c, NULL);

	g_string_append_unichar(str, c);

	if (g_strcmp0(str->str, hildon_button_get_value(HILDON_BUTTON(cp->choice_details))))
		g_object_set(G_OBJECT(cp->choice_details), "value", str->str, NULL);

	g_string_free(str, TRUE);
}

static void
decide_label_visible(GObject *src, GParamSpec *pspec, ChoiceParams *cp)
{
	gboolean suggestions_visible, regular_char_suggestions_visible, label_visible;

	g_object_get(G_OBJECT(cp->suggestions), "visible", &suggestions_visible, NULL);
	g_object_get(G_OBJECT(cp->regular_char_suggestions), "visible", &regular_char_suggestions_visible, NULL);
	g_object_get(G_OBJECT(cp->suggestions_label), "visible", &label_visible, NULL);

	suggestions_visible = (suggestions_visible || regular_char_suggestions_visible);

	if (label_visible != suggestions_visible)
		g_object_set(G_OBJECT(cp->suggestions_label), "visible", suggestions_visible, NULL);
}

static void
show_suggestions_clicked(GtkWidget *suggestions_button, ChoiceParams *cp)
{
	gtk_widget_hide(suggestions_button);
	gtk_widget_show(cp->regular_char_suggestions);
}

static void
choose_char(GtkWidget *picker, KBButton *kbb)
{
	GtkWidget *dlg = g_object_new(GTK_TYPE_DIALOG, "visible", TRUE, "modal", TRUE, NULL);
	GtkWidget *container;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *choice;

	ChoiceParams cp = { .kbb = kbb, .picker = picker, .is_dirty = FALSE };

	gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(gtk_widget_get_toplevel(picker)));

	gtk_widget_set_size_request(dlg, -1, 420);
	hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, "spacing", HILDON_MARGIN_DEFAULT, NULL);

	choice = make_choice(&cp);
	gtk_container_add(GTK_CONTAINER(hbox), choice);

	cp.choice_details = make_choice_details(choice, &cp);

	vbox = g_object_new(GTK_TYPE_VBOX, "visible", TRUE, NULL);
	gtk_container_add(GTK_CONTAINER(hbox), vbox);

	gtk_container_add_with_properties(GTK_CONTAINER(vbox), cp.choice_details, "expand", FALSE, "fill", TRUE, NULL);

	cp.suggestions_label = g_object_new(GTK_TYPE_LABEL,
		"label", "Suggestions:", "xalign", 0.0, "yalign", 0.0, "justify", GTK_JUSTIFY_LEFT, NULL);
	gtk_container_add_with_properties(GTK_CONTAINER(vbox), cp.suggestions_label, "expand", FALSE, "fill", TRUE, NULL);

	cp.regular_char_suggestions = make_regular_char_suggestions();
	container = g_object_new(HILDON_TYPE_PANNABLE_AREA,
		"visible",        TRUE,
		"child",          cp.regular_char_suggestions,
		"hovershoot-max", 0,
		"vovershoot-max", 0, 
		NULL);
	g_object_set_data(G_OBJECT(container), "debug-name", "regular_char_suggestions_hpa");
	g_signal_connect(G_OBJECT(cp.regular_char_suggestions), "notify::active-character", (GCallback)notify_active_character, &cp);
	gtk_container_add(GTK_CONTAINER(vbox), container);
	connect_visibility(cp.regular_char_suggestions, container);

	cp.show_suggestions_button = g_object_new(HILDON_TYPE_BUTTON,
		"visible",     TRUE,
		"size",        HILDON_SIZE_FINGER_HEIGHT,
		"arrangement", HILDON_BUTTON_ARRANGEMENT_VERTICAL,
		"label",       "Suggest",
		NULL);
	container =	g_object_new(GTK_TYPE_ALIGNMENT,
			"visible", TRUE,
			"child",   cp.show_suggestions_button,
			"xalign",  0.5,
			"yalign",  0.5,
			"xscale",  0.0,
			"yscale",  0.0,
			NULL);
	g_object_set_data(G_OBJECT(container), "debug-name", "show_suggestions_button_alignment");
	gtk_container_add(GTK_CONTAINER(vbox), container);
	connect_visibility(cp.show_suggestions_button, container);
	g_signal_connect(G_OBJECT(cp.show_suggestions_button), "clicked", (GCallback)show_suggestions_clicked, &cp);

	cp.suggestions = make_suggestions(kbb, &cp);
	container = g_object_new(HILDON_TYPE_PANNABLE_AREA,
		"visible", TRUE,
		"child",   cp.suggestions,
		NULL);
	g_object_set_data(G_OBJECT(container), "debug-name", "suggestions_hpa");
	gtk_container_add(GTK_CONTAINER(vbox), container);
	connect_visibility(cp.suggestions, container);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), hbox);

	g_object_set_data(G_OBJECT(cp.suggestions),              "debug-name", "cp.suggestions");
	g_object_set_data(G_OBJECT(cp.suggestions_label),        "debug-name", "label");
	g_object_set_data(G_OBJECT(cp.regular_char_suggestions), "debug-name", "cp.regular_char_suggestions");

	g_signal_connect(G_OBJECT(cp.suggestions),              "notify::visible", (GCallback)decide_label_visible, &cp);
	g_signal_connect(G_OBJECT(cp.regular_char_suggestions), "notify::visible", (GCallback)decide_label_visible, &cp);

	set_suggestion_model(GTK_TREE_VIEW(cp.suggestions), &cp);
	find_suggestion(&cp);

	gtk_dialog_run(GTK_DIALOG(dlg));
	gtk_widget_destroy(dlg);

	if (cp.is_dirty)
		g_object_set_data(G_OBJECT(gtk_widget_get_toplevel(picker)), IS_DIRTY_KEY, GINT_TO_POINTER(TRUE));
}

GtkWidget *
make_picker_button(KBButton *kbb)
{
	GtkWidget *picker = g_object_new(HILDON_TYPE_BUTTON,
		"arrangement", HILDON_BUTTON_ARRANGEMENT_VERTICAL,
		"size",        HILDON_SIZE_FINGER_HEIGHT,
		"visible",     TRUE,
		"label",       kbb->key,
		NULL);

	g_signal_connect(G_OBJECT(picker), "clicked", (GCallback)choose_char, kbb);

	return picker;
}

static void
kb_chosen(GtkTreeView *tv, GtkTreePath *tp, GtkTreeViewColumn *col, GtkWidget *dlg)
{
	GtkTreeModel *tm = gtk_tree_view_get_model(tv);
	GtkTreeIter itr;

	if (tm)
		if (gtk_tree_model_get_iter(tm, &itr, tp)) {
			char *fname;

			gtk_tree_model_get(tm, &itr, 1, &fname, -1);

			g_object_set_data_full(G_OBJECT(dlg), "new-fname", fname, (GDestroyNotify)g_free);
		}

	gtk_dialog_response(GTK_DIALOG(dlg), GTK_RESPONSE_OK);
}

static GtkWidget *
make_kb_list(GtkWidget *dlg)
{
	GtkWidget *ret = NULL;
	GDir *dir = NULL;

	dir = g_dir_open("/usr/share/scv_layouts", 0, NULL);
	if (dir) {
		GtkListStore *ls;
		char *fname = NULL;
		gboolean is_empty = TRUE;
		char *description = NULL;
		GtkTreeIter itr;

		ls = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);

		while (TRUE) {
			fname = (char *)g_dir_read_name(dir);
			if (fname) {
				fname = g_build_filename("/usr/share/scv_layouts", fname, NULL);
				description = scv_file_read_description(fname, NULL);

				if (fname && description) {
					is_empty = FALSE;
					gtk_list_store_append(ls, &itr);
					gtk_list_store_set(ls, &itr, 0, description, 1, fname, -1);
				}

				g_free(description);
				g_free(fname);
			}
			else
				break;
		}

		g_dir_close(dir);

		if (!is_empty) {
			GtkWidget *tv = g_object_new(GTK_TYPE_TREE_VIEW, "visible", TRUE, "model", ls, NULL);

			gtk_tree_view_append_column(GTK_TREE_VIEW(tv),
				gtk_tree_view_column_new_with_attributes(NULL, g_object_new(GTK_TYPE_CELL_RENDERER_TEXT, "visible", TRUE, NULL),
				"text", 0, NULL));

			g_signal_connect(G_OBJECT(tv), "row-activated", (GCallback)kb_chosen, dlg);

			ret = g_object_new(HILDON_TYPE_PANNABLE_AREA, "visible", TRUE, "child", tv, NULL);
		}

		g_object_unref(ls);
	}

	return ret;
}

static GtkWidget *
make_open_dlg(GtkWidget *parent)
{
	GtkWidget *dlg = g_object_new(GTK_TYPE_DIALOG, "visible", TRUE, "title", "Open Keyboard File", NULL);

	gtk_widget_set_size_request(dlg, -1, 420);

	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(parent));
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), make_kb_list(dlg));

	return dlg;
}

static void
btn_open_dialog_response(GtkWidget *dlg, int response, GtkWidget *wnd)
{
	char *fname = g_object_get_data(G_OBJECT(dlg), "new-fname");

	if (GTK_RESPONSE_CANCEL == response || GTK_RESPONSE_DELETE_EVENT == response || NULL == fname) {
		if (!wnd)
			gtk_main_quit();
		gtk_widget_destroy(dlg);
	}
	else {
		GError *error = NULL;
		ScvFile *scv_file = scv_file_new_from_file(fname, &error);

		if (scv_file) {
			if (wnd)
				gtk_widget_destroy(wnd);

			gtk_widget_destroy(dlg);
			make_wnd(scv_file);
		}
		else {
			GtkWidget *msg_dialog = gtk_message_dialog_new(GTK_WINDOW(dlg), 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "Failed to load file \"%s\": ",
				fname, error ? error->message : "Unknown error");

			gtk_dialog_run(GTK_DIALOG(msg_dialog));
			gtk_widget_destroy(msg_dialog);
		}
		if (error)
			g_error_free(error);
	}
}

static int
try_to_save_file(GtkWidget *wnd, int response, ScvFile *scv_file_old)
{
	if (GPOINTER_TO_INT(g_object_get_data(G_OBJECT(wnd), IS_DIRTY_KEY))) {
		GtkWidget *msg_dialog = gtk_message_dialog_new(GTK_WINDOW(wnd), 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE, "The current keyboard layout has changed. Save it first?");

		gtk_dialog_add_buttons(GTK_DIALOG(msg_dialog),
			GTK_STOCK_SAVE,    GTK_RESPONSE_YES,
			GTK_STOCK_DISCARD, GTK_RESPONSE_NO,
			GTK_STOCK_CANCEL,  GTK_RESPONSE_CANCEL,
			NULL);

		response = gtk_dialog_run(GTK_DIALOG(msg_dialog));
		if (response == GTK_RESPONSE_DELETE_EVENT)
			response = GTK_RESPONSE_CANCEL;
		gtk_widget_destroy(msg_dialog);

		if (GTK_RESPONSE_YES == response) {
			GError *error = NULL;
			if (!scv_file_save(scv_file_old, &error)) {
				GtkWidget *msg_dialog = gtk_message_dialog_new(GTK_WINDOW(wnd), 0, GTK_MESSAGE_WARNING, GTK_BUTTONS_NONE, "Failed to save the current keyboard layout: %s\nOpen the new one anyway?",
					error ? error->message : "Unknown error");

				gtk_dialog_add_buttons(GTK_DIALOG(msg_dialog),
					GTK_STOCK_OPEN,   GTK_RESPONSE_YES,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					NULL);

				response = gtk_dialog_run(GTK_DIALOG(msg_dialog));
				if (response == GTK_RESPONSE_DELETE_EVENT)
					response = GTK_RESPONSE_CANCEL;
				gtk_widget_destroy(msg_dialog);
			}
			else
				g_object_set_data(G_OBJECT(wnd), IS_DIRTY_KEY, GINT_TO_POINTER(FALSE));
			if (error)
				g_error_free(error);
		}
	}

	return response;
}

static void
btn_open_clicked(GtkWidget *btn, GtkWidget *wnd)
{
	int response = GTK_RESPONSE_NO;

	if (wnd) {
		ScvFile *scv_file_old = g_object_get_data(G_OBJECT(wnd), SCV_FILE_KEY);

		response = try_to_save_file(wnd, response, scv_file_old);
	}

	if (GTK_RESPONSE_CANCEL != response) {
		GtkWidget *dlg = make_open_dlg(wnd);

		g_signal_connect(G_OBJECT(dlg), "response", (GCallback)btn_open_dialog_response, wnd);

		gtk_widget_show(dlg);
	}
}

static gboolean
wnd_delete_event(GtkWidget *wnd, GdkEvent *event, gpointer null)
{
	ScvFile *scv_file = g_object_get_data(G_OBJECT(wnd), SCV_FILE_KEY);

	if (GTK_RESPONSE_CANCEL == try_to_save_file(wnd, GTK_RESPONSE_NO, scv_file))
		return TRUE;

	gtk_main_quit();
	return FALSE;
}

static void
btn_save_clicked(GtkWidget *btn, GtkWidget *wnd)
{
	ScvFile *scv_file_old = g_object_get_data(G_OBJECT(wnd), SCV_FILE_KEY);
	GError *error = NULL;

	if (!scv_file_save(scv_file_old, &error)) {
		GtkWidget *msg_dialog = gtk_message_dialog_new(GTK_WINDOW(wnd), 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "Failed to save the current keyboard layout: %s",
			error ? error->message : "Unknown error");

		gtk_dialog_run(GTK_DIALOG(msg_dialog));
		gtk_widget_destroy(msg_dialog);
	}
	else
		g_object_set_data(G_OBJECT(wnd), IS_DIRTY_KEY, GINT_TO_POINTER(FALSE));

	if (error)
		g_error_free(error);
}

static void
make_wnd(ScvFile *scv_file) {
	GtkWidget *hbox = NULL;
	GtkWidget *wnd = NULL;
	GtkWidget *tbl = NULL;
	GtkWidget *widget = NULL;
	GtkWidget *btn_open = NULL;
	GtkWidget *btn_save = NULL;
	GtkWidget *app_menu = NULL;
	int idx_row = -1, Nix;

	if (!scv_file)
		btn_open_clicked(NULL, NULL);
	else {
		wnd = g_object_new(HILDON_TYPE_WINDOW, "title", "Edit Keyboard", NULL);

		app_menu = g_object_new(HILDON_TYPE_APP_MENU, NULL);
		btn_open = g_object_new(HILDON_TYPE_BUTTON, "visible", TRUE, "label", GTK_STOCK_OPEN, "use-stock", TRUE, NULL);
		btn_save = g_object_new(HILDON_TYPE_BUTTON, "visible", TRUE, "label", GTK_STOCK_SAVE, "use-stock", TRUE, NULL);
		hildon_app_menu_append(HILDON_APP_MENU(app_menu), GTK_BUTTON(btn_open));
		hildon_app_menu_append(HILDON_APP_MENU(app_menu), GTK_BUTTON(btn_save));
		hildon_window_set_app_menu(HILDON_WINDOW(wnd), HILDON_APP_MENU(app_menu));

		g_signal_connect(G_OBJECT(btn_open), "clicked", (GCallback)btn_open_clicked, wnd);
		g_signal_connect(G_OBJECT(btn_save), "clicked", (GCallback)btn_save_clicked, wnd);

		g_signal_connect(G_OBJECT(wnd), "delete-event", (GCallback)wnd_delete_event, NULL);

		g_object_set_data_full(G_OBJECT(wnd), SCV_FILE_KEY, scv_file, (GDestroyNotify)scv_file_free);

		tbl = g_object_new(GTK_TYPE_TABLE, "visible", TRUE, NULL);

		hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, NULL);
		gtk_table_attach(GTK_TABLE(tbl), hbox, 0, N_PER_ROW, 0, 1,
			(GtkAttachOptions)(GTK_EXPAND | GTK_FILL), (GtkAttachOptions)(GTK_FILL), 0, 0);

		widget = g_object_new(HILDON_TYPE_ENTRY, "visible", TRUE, "text", scv_file->description1, "editable", FALSE, NULL);
		gtk_container_add(GTK_CONTAINER(hbox), widget);

		widget = g_object_new(HILDON_TYPE_ENTRY, "visible", TRUE, "text", scv_file->description2, "editable", FALSE, NULL);
		gtk_container_add(GTK_CONTAINER(hbox), widget);

		widget = g_object_new(HILDON_TYPE_ENTRY, "visible", TRUE, "text", scv_file->description3, "editable", FALSE, NULL);
		gtk_container_add(GTK_CONTAINER(hbox), widget);

		gtk_table_attach(GTK_TABLE(tbl), g_object_new(GTK_TYPE_LABEL,
			"visible", TRUE, "label", "Lowercase:", "xalign", 0.0, "justify", GTK_JUSTIFY_LEFT, NULL),
			0, N_PER_ROW, 1, 2,
			(GtkAttachOptions)(GTK_EXPAND | GTK_FILL), (GtkAttachOptions)(GTK_FILL), 0, 0);

		idx_row += 2;

		for (Nix = 0 ; Nix < G_N_ELEMENTS(scv_file->kb1) ; Nix++) {
			if (!(Nix % N_PER_ROW)) idx_row++;

			widget = make_picker_button(&(scv_file->kb1[Nix]));
			gtk_table_attach(GTK_TABLE(tbl), widget, Nix % N_PER_ROW, Nix % N_PER_ROW + 1, idx_row, idx_row + 1,
				(GtkAttachOptions)(GTK_EXPAND | GTK_FILL), (GtkAttachOptions)(GTK_EXPAND | GTK_FILL), 0, 0);
		}

		idx_row++;
		gtk_table_attach(GTK_TABLE(tbl), g_object_new(GTK_TYPE_LABEL,
			"visible", TRUE, "label", "Uppercase:", "xalign", 0.0, "justify", GTK_JUSTIFY_LEFT, NULL),
			0, N_PER_ROW, idx_row, idx_row + 1,
			(GtkAttachOptions)(GTK_EXPAND | GTK_FILL), (GtkAttachOptions)(GTK_FILL), 0, 0);
		idx_row++;

		for (Nix = 0 ; Nix < G_N_ELEMENTS(scv_file->kb2) ; Nix++) {
			if (!(Nix % N_PER_ROW)) idx_row++;

			widget = make_picker_button(&(scv_file->kb2[Nix]));
			gtk_table_attach(GTK_TABLE(tbl), widget, Nix % N_PER_ROW, Nix % N_PER_ROW + 1, idx_row, idx_row + 1,
				(GtkAttachOptions)(GTK_EXPAND | GTK_FILL), (GtkAttachOptions)(GTK_EXPAND | GTK_FILL), 0, 0);
		}

		gtk_container_add(GTK_CONTAINER(wnd), 
			g_object_new(HILDON_TYPE_PANNABLE_AREA,
				"visible", TRUE,
				"child", g_object_new(GTK_TYPE_VIEWPORT, "visible", TRUE, "child", tbl, NULL),
				NULL));

		gtk_widget_show(wnd);
	}
}

int
main(int argc, char **argv)
{
	ScvFile *scv_file = NULL;

	gtk_init(&argc, &argv);

	if (argc == 2) {
		GError *error = NULL;
		scv_file = scv_file_new_from_file(argv[1], &error);

		if (!scv_file) {
			GtkWidget *msg_dialog = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "Failed to load file \"%s\": ",
				argv[1], error ? error->message : "Unknown error");

			gtk_dialog_run(GTK_DIALOG(msg_dialog));
			gtk_widget_destroy(msg_dialog);
		}
		if (error)
			g_error_free(error);
	}

	make_wnd(scv_file);

	gtk_main();

	return 0;
}
