/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

// These tables are modified from their FFmpeg counterparts so that
// they work on both little and big endian systems.

#ifndef VIDEO_CODECS_SVQ1_CB_H
#define VIDEO_CODECS_SVQ1_CB_H

#include "common/scummsys.h"

namespace Video {

static const uint32 s_svq1InterCodebook4x2[192] = {
	0xf9fa0207, 0xfcfd0307, 0x0807fef9, 0x0403fcf8,
	0x03091113, 0xf8f4f0f2, 0xfdf8f0ee, 0x080c0e0b,
	0x14f6f007, 0x14f6ef07, 0xeb0812fa, 0xec0912f9,
	0xf2ec0319, 0xf3ee071d, 0x0e15fce3, 0x0e14fae1,
	0xe8e4e6ed, 0x0a16201f, 0x1c1f180f, 0xf3eae0e0,
	0xe6e9f802, 0x231b03f7, 0x15150b03, 0xdee5fc08,
	0x2f0ce1e2, 0x2f0de2e3, 0xd2ef1e26, 0xd2ed1a22,
	0xd5cdced6, 0x30373022, 0x2a333630, 0xd1cbccd4,
	0xfa000504, 0x0100fefe, 0xfefffaf5, 0x01090801,
	0x05fa0100, 0x02f40108, 0x08f9f207, 0x0800f805,
	0x080b0401, 0xfb00f8f4, 0x040001ff, 0x1003f8f1,
	0xfafc0811, 0xf8f3fc09, 0xee010602, 0xf40b0bff,
	0x00020006, 0xebf9060e, 0xecf3ff01, 0x150a0101,
	0x0d07fbea, 0x0c04fff5, 0x130e00f9, 0xedfb03fc,
	0x0f0af2e6, 0xfefa0412, 0xeefb1319, 0x0204f9ec,
	0xfcfffaf3, 0xddfe2519, 0x01010405, 0x2b02dceb,
	0x03fffe02, 0xfffafe08, 0x0c02fdfe, 0xfffefefb,
	0xfbfffffd, 0xfe0807ff, 0xfd050702, 0xf8fd0101,
	0xfefdfffd, 0x0d02fdfe, 0xfaf5000f, 0x00000003,
	0xfcfbf7fa, 0x03010412, 0x0400030c, 0xfd03fdf0,
	0x021203ef, 0xfffffdff, 0x00f810fa, 0x00f90ef7,
	0xfb0ef303, 0xfc0ef303, 0xe90e14f9, 0x04f8f908,
	0x10edf108, 0xfd0b0df6, 0x1a01ff09, 0x02e5f105,
	0xfc1007ec, 0x011f09d8, 0xf9e2f41a, 0x04edfe28,
	0x00000006, 0x0201fefa, 0xfa00ff00, 0xfffe0009,
	0xfd0208f9, 0x02fd02ff, 0x04fefc07, 0xfa000002,
	0x0209fefd, 0xfc00fffe, 0xfd00fdfd, 0x040a02fa,
	0x08f60003, 0x04fc0000, 0x020401ff, 0x07f7f903,
	0xfcf70102, 0x00000cff, 0xfc07ff03, 0x0204f203,
	0x0b01f7f4, 0x00010502, 0x02000103, 0xed060800,
	0xfcf9f6fa, 0x07050709, 0xfd031506, 0xfefbf7f5,
	0xfff0f7fc, 0x2401fbfe, 0x00130b08, 0xd7fc0502,
	0xfffeffff, 0x0601fefe, 0xf8010400, 0x00010101,
	0x0004fdfe, 0xfd03ff02, 0x01fc0301, 0xfb0003ff,
	0x03020403, 0xfffafdfe, 0x02fefdfe, 0x000108fc,
	0x060204f9, 0x0001fff9, 0x01fc02fe, 0xff02fa08,
	0x020002fa, 0xfff80405, 0x0900f5ff, 0x0202fe00,
	0xfffcfb11, 0xfefefcff, 0xfd09f300, 0x02f90cff,
	0x02fbfe00, 0xfd14fbf9, 0xe2ff0707, 0x01080503,
	0xfcff03fa, 0x12f5fe02, 0x0e03f900, 0xf7eefd14,
	0xff00fe07, 0xff0000fe, 0x0001fffc, 0x040002fe,
	0x0102fd01, 0x01fb0103, 0xfeff00fd, 0xfd000107,
	0xfe000502, 0x01fffb02, 0xff04feff, 0x0005fd00,
	0xfeff0300, 0xff0501fc, 0x09ff00ff, 0xfffffeff,
	0xff0505fe, 0xfefd02fe, 0x01f50201, 0x02030102,
	0xfefff602, 0x01040204, 0x01fb0504, 0x01f50600,
	0x06060001, 0xf1010200, 0x09050307, 0x020202e2,
	0x020901de, 0x02080805, 0x06060207, 0x0401e502
};

static const uint32 s_svq1InterCodebook4x4[384] = {
	0xf9fa0004, 0xf7f3f8fc, 0x06fff8f8, 0x1b1605fe,
	0x0a0bf9f0, 0x0a0df9ee, 0x080cfcf1, 0x0509fff7,
	0xf00f02fe, 0xed1302fd, 0xed1302fd, 0xf20f03fe,
	0x10161611, 0xfefbf9fa, 0xf4f0f0f4, 0xfdff0101,
	0x0800ef0b, 0x09ffeb0e, 0x08feeb0e, 0x06fef00b,
	0x0bf0fe07, 0x0eebfe09, 0x0eeaff0a, 0x0aeeff08,
	0xf70310f6, 0xf50414f3, 0xf60415f2, 0xf80310f5,
	0xf7f7040b, 0xf2f4060f, 0xf2f40811, 0xf5f90a10,
	0x0d0e0a04, 0x100f07ff, 0x0803f9f4, 0xf6eee9ec,
	0xe7e6eef6, 0xf5fa0104, 0x030b0f0d, 0x080d0f0c,
	0xf5f0edf0, 0x0b0f0c07, 0x0b10100b, 0xf6f5f7fa,
	0x050c1312, 0xfc051012, 0xf1f60006, 0xeae9eff7,
	0x15fff2f6, 0x1d00eff5, 0x1e01f0f5, 0x1700f2f6,
	0xfaf4eff0, 0xf9f2eded, 0x0201fffd, 0x131d231b,
	0x1717f8db, 0x1d1cf7d6, 0x1c1af6d5, 0x1613f5da,
	0xdff01020, 0xdbee1427, 0xdaed1326, 0xdeef0f20,
	0xfcfa0918, 0x03faf6ff, 0x03fff7f8, 0xfa020703,
	0x00fffdff, 0xf90204ff, 0xf0030bfd, 0xee091401,
	0x0c06f8fd, 0x0d07f6fb, 0x0705f7fa, 0xff02fbfb,
	0xfffd0cf8, 0x01fd0ff6, 0x01fc0df5, 0x02fd08f5,
	0xf4fb0609, 0xf3f80003, 0xfffffcfc, 0x120f01fc,
	0x0c0e0d09, 0xfeff0304, 0xfbf8fbfe, 0xfcf7f5f9,
	0xfcf9fb07, 0xfcf9fe0e, 0xfbf80011, 0xfbf9010f,
	0x0406fff6, 0x0402f7f1, 0x00fdff02, 0xf6f80d19,
	0xf0fd0b07, 0xf1fd0b07, 0xf7fe0706, 0xfbfd0204,
	0x00fffff9, 0x0602fef7, 0x0e06fcf4, 0x1308faf3,
	0xfbf5eeee, 0x040300fd, 0x06060806, 0x06060606,
	0xf60d03fb, 0xf70f01fa, 0xfa0ffdfa, 0xfd0afafa,
	0xf7f70109, 0x0506090b, 0x07080300, 0xfbfaf2f1,
	0x130bfaf5, 0xf8f7fbfe, 0xf6f70206, 0x05040506,
	0x0f08fdf9, 0x0f0a03ff, 0xfeff0505, 0xe7ebfe04,
	0x05fafa06, 0x09f9f708, 0x0df9f408, 0x0ef9f204,
	0x0101fdfc, 0xfdfefbfd, 0xfcfe0007, 0xfcfc0714,
	0x0afaecfd, 0x01000006, 0xff050805, 0xfe0000fd,
	0x02ff060d, 0x03020305, 0x000300fd, 0xfbfef8f0,
	0x00faf9fe, 0x01fdfafd, 0xff02fffb, 0x05100cff,
	0x080901f9, 0x0305fef6, 0x030702fa, 0xf9ff00fc,
	0xe8f70403, 0x03060200, 0x0704ffff, 0xfeff0305,
	0x02f70603, 0x01f30601, 0x02f60801, 0x01f90801,
	0x1602fdfd, 0x0cfbfdfe, 0x02f6fdfe, 0x02fcfffd,
	0x02080c0b, 0xf8fbfbfb, 0xfd00fcfa, 0x0303fffe,
	0xfffefa0c, 0xfefef80c, 0xfd00f909, 0xfe02fa04,
	0xfd0c01ed, 0xfc0504fc, 0xfffe0106, 0x07fefc04,
	0xf8f9fcfd, 0x00fefcfc, 0x100e02ff, 0x0404fefc,
	0xfb0207ff, 0x01ff00fe, 0x0dfffd04, 0x08f2f406,
	0xfb0405ff, 0xf70305fe, 0xf40407fe, 0xf70407ff,
	0x0101fdfa, 0xfa000b0b, 0xf9fe0406, 0x0a03f6f4,
	0xfefdfdfe, 0x0a0e0b06, 0xf6f6f5f7, 0x02030202,
	0xfff9fbf9, 0x070002ff, 0x090001ff, 0xfffb0403,
	0xfff1ff0a, 0x02fb0104, 0x01ff01fd, 0x040401fd,
	0x0a04ff02, 0x00ff0206, 0xf4f90202, 0xfd0002fc,
	0xf8fffcff, 0xf702ff03, 0xfb050004, 0x03080002,
	0x01010203, 0x0300fe04, 0x0104ff02, 0xe7ff0600,
	0xfcfefeff, 0xfcff00fd, 0x02fcffff, 0x1902fa00,
	0x0005fff5, 0x02fe0007, 0x04fdff0a, 0xfffefbfb,
	0xff030600, 0x01fffffe, 0xfbf4f9ff, 0x04020608,
	0xfaff0602, 0xfcff0a09, 0x00fc0001, 0xfbf7fe03,
	0x000403fc, 0x000303fc, 0x020300f5, 0x020703f5,
	0x0307fc02, 0x0107f801, 0x0104f4ff, 0x0202f703,
	0x09fefe02, 0x0103fdef, 0xfa0107fc, 0x03ff0405,
	0xfc0002ff, 0xff0c08f9, 0xfb0405fe, 0xfef8fb03,
	0xfefb0000, 0x1b03f8fe, 0x06fdfcff, 0xf9fe01fd,
	0xff010404, 0xfdf9f6f9, 0x03050a0a, 0xfdfcfefe,
	0x07050100, 0xecf0fe04, 0x08070400, 0xfffe0002,
	0x110301fe, 0xfffe01fd, 0xfefffeff, 0x00fffbff,
	0x0001fd05, 0x0000fe06, 0xfd00feff, 0xff0801f5,
	0x00000003, 0x01040200, 0x01060002, 0x02fdeefe,
	0x010600f2, 0x01fffefb, 0x010001ff, 0x00040701,
	0xfc0100ff, 0xfc030801, 0x030104fd, 0x01fc01fa,
	0x0303f401, 0xff00f6ff, 0x01020002, 0x04020203,
	0x03000003, 0x01fe0002, 0x00fb0205, 0xfff2ff06,
	0xfdfdfafe, 0x0504ff02, 0x00feff06, 0xfbff0404,
	0x00f501fc, 0x01fc02ff, 0xff03fd02, 0x000ffe01,
	0xfe00ff01, 0x01f9fc01, 0x15fffafe, 0x01ff02fe,
	0x00feff15, 0xfe01fdff, 0xff02fef7, 0xfffc0102,
	0xfa020801, 0x0004fff6, 0x0303fdfc, 0xffff0005,
	0xfe010203, 0x0304fefe, 0xeffc0205, 0x0304fe00,
	0x0300fcf9, 0xff020909, 0xff00faf5, 0x01000105,
	0xf5051100, 0x00fafe03, 0x01fcfe02, 0xff0201fc,
	0xfdfbfffb, 0xfefd05fd, 0xfb021004, 0xffff05fe,
	0x01fc0000, 0x0b0502ff, 0x01feffff, 0xfffdfefc,
	0x000afffb, 0xfd000106, 0x0001fc00, 0xff03fcfe,
	0x00030906, 0x00fe01fe, 0xfefefdfe, 0xfa010001,
	0x01020001, 0x01fe03ff, 0x00f1ff00, 0x060205ff,
	0x02020002, 0x06fcf400, 0xff040100, 0xfc010201,
	0x00f9fe01, 0xff000000, 0x030b02fb, 0xfa000301,
	0xfcf7fd00, 0x00ff0301, 0x00fe0104, 0x06fffd07,
	0x0206fe01, 0xfe03ff00, 0x020004fe, 0x02f202ff,
	0xff000202, 0xf2fd03fe, 0xfd030200, 0x02030105,
	0xf204fd01, 0xff0bfe01, 0x0003ff00, 0x020001ff,
	0x02fd03fe, 0x03fcfffc, 0x030102ff, 0x0702fefa,
	0x000201fe, 0xff0000fe, 0x02ff050c, 0xf901fff8,
	0x02fcfe02, 0xfef5000b, 0xfffd0103, 0xff010300,
	0xfe000300, 0xfdfffa00, 0x00fef90c, 0x0101fe07,
	0x02020201, 0x020002ff, 0x000400e9, 0x03010203,
	0x05fffbfc, 0xff0a05fd, 0xfc030000, 0xfb02ff01
};

static const uint32 s_svq1InterCodebook8x4[768] = {
	0x00040809, 0xfdfcfcfd, 0xff040809, 0xfdfbfbfc,
	0xfe030708, 0xfcfbfbfb, 0xfe010406, 0xfdfcfbfc,
	0xfcf5f2f4, 0x06060501, 0xfbf9f6f8, 0x010101fe,
	0x01030405, 0xffff0000, 0x06090d0d, 0xfeff0003,
	0xfffdfcfc, 0x0b080401, 0xfefcfafb, 0x0c080300,
	0xfcfaf9f9, 0x0a0702fe, 0xfcfbf9f9, 0x080501fe,
	0x01fffefd, 0x06070603, 0x07050302, 0x04060808,
	0x03040504, 0xf9fafe01, 0xf9fe0001, 0xf0eff2f6,
	0x0801fcfb, 0xf9fd0309, 0x0b01faf9, 0xf8fd050c,
	0x0900f9f8, 0xf9fd050b, 0x05fffaf8, 0xfafe0408,
	0xf8f9fbfc, 0xfaf8f7f7, 0xf9fafbfc, 0xfefcfaf9,
	0x03020100, 0x090a0805, 0x06030201, 0x0d0e0c09,
	0x05060605, 0x01020304, 0x07070605, 0x04060606,
	0x010100ff, 0x05050503, 0xefeff0f3, 0xfcfaf6f2,
	0x100d0b09, 0x0a0c0d0f, 0xf9fafbfc, 0xfbfaf9f9,
	0xf9f9fafa, 0xfbfaf9f9, 0x0000fffe, 0xff000000,
	0xf0f1f3f5, 0xf6f4f2f0, 0x05040302, 0x03030304,
	0x08080706, 0x05060708, 0x03030403, 0x03030303,
	0x01040403, 0xeff3f9fe, 0x05070705, 0xedf3fb01,
	0x08090806, 0xf0f7ff05, 0x0a0a0806, 0xf5fc0207,
	0xf6ff0912, 0x00fcf7f3, 0xf4ff0c16, 0x02fcf6f1,
	0xf6000d17, 0x02fdf7f3, 0xfa020c14, 0x02fefaf7,
	0xf9fafafa, 0xfaf9f9f9, 0xf8f8f9fa, 0xf8f7f7f7,
	0xfdfdfdfd, 0xfdfdfdfd, 0x15120f0c, 0x0e111315,
	0x1212100e, 0x0d0f1012, 0x05060605, 0x03040405,
	0xf6f7f9fa, 0xf9f7f6f6, 0xf2f3f5f6, 0xf6f4f3f2,
	0x05fcefe5, 0x070a0a09, 0x07fdede0, 0x080b0c0b,
	0x08fef0e2, 0x070a0c0c, 0x0700f4e9, 0x06090b0a,
	0x0c101110, 0xf4f8ff06, 0x0a0f1211, 0xeef1f801,
	0x040a0e0f, 0xe9ecf2fb, 0xff04080a, 0xeaebf0f7,
	0xf5f4f4f6, 0x140e04fb, 0xf4f1f3f5, 0x1b1307fc,
	0xf5f2f3f5, 0x1c1508fd, 0xf7f4f5f6, 0x191208fe,
	0x01ffffff, 0x05060604, 0x02000000, 0xfe010304,
	0x04020000, 0xf6f9ff04, 0x05030000, 0xf1f5fd03,
	0xfff8f3f2, 0xfcff0303, 0x04fffcfb, 0x00030808,
	0x03020203, 0x01030504, 0xfe000305, 0xfffffffe,
	0xfafa0109, 0xfffefdfb, 0xfafa010c, 0x00fffefc,
	0xfcfc040e, 0xfffffefe, 0xffff060e, 0xffffffff,
	0x0a080604, 0x0507090b, 0x00ffffff, 0xfeffff00,
	0xfbfcfcfe, 0xfcfbfbfb, 0xfcfdfdfe, 0xfffefdfc,
	0x04040302, 0x00000103, 0x050401ff, 0x03040506,
	0x02fefaf8, 0x03040403, 0xfbf7f3f2, 0x0000fffe,
	0xfcfbfcfd, 0x0d0c0700, 0xfbfbfcfd, 0x0a0904fe,
	0xfbfcfdfe, 0x0403fffc, 0xfdfeffff, 0x0100fefd,
	0xf8fe0509, 0xfcf9f6f5, 0x02060a0c, 0x0000ff00,
	0x04030202, 0x01010103, 0x00fcf8f7, 0x00010201,
	0x05080806, 0xf3f5fb01, 0x02020100, 0xf5f8fcff,
	0x0301fefd, 0xfcff0103, 0x0502fffe, 0x01040606,
	0x05050403, 0xfafd0104, 0x02040605, 0xfd000202,
	0xfb000506, 0xfefffefb, 0xf5fd0407, 0xfefdf9f4,
	0xffff0001, 0x000000ff, 0x04040302, 0x03040505,
	0xf6f7f7f9, 0xfaf9f7f6, 0x06050403, 0x05050505,
	0xf9f9f9f9, 0xfcfbfafa, 0xfffdfcfb, 0x0000ffff,
	0x0401fefd, 0x05050505, 0x0603fffe, 0x090a0a09,
	0x030a01f2, 0x010100fe, 0x030d02f0, 0x0001fffd,
	0x030c02f1, 0x0101fefc, 0x020a03f6, 0x0101fffd,
	0x02040100, 0x0bfdf6fb, 0x020401ff, 0x0ffef3fa,
	0x010300ff, 0x0ffff4fa, 0x010201ff, 0x0b00f8fc,
	0xfefe050a, 0xfc010502, 0xfaf80007, 0xfc020501,
	0xf9f4fb02, 0xff040702, 0xfcf6f9ff, 0x02070904,
	0xfafcfbfb, 0xfdfbfbfa, 0xfcfefeff, 0xfcfbfafb,
	0x04070706, 0xfdfdfe00, 0x0a0d0e0d, 0xfeff0105,
	0x02020101, 0x02020202, 0xf7f8fafb, 0xfaf9f8f7,
	0x0b0a0907, 0x0507090b, 0xfdfdfeff, 0xfdfcfcfc,
	0x0000ffff, 0xffff0000, 0xfbfcfdfd, 0xfefdfdfc,
	0xfdff0102, 0x00fffefd, 0x03080c0c, 0x01000001,
	0xfaf8f8fa, 0x080602fe, 0xfeff0101, 0x07050300,
	0xff010303, 0x020000ff, 0xff000100, 0xfffeffff,
	0x00000001, 0x04020000, 0x04030102, 0x02000103,
	0x00000102, 0x0300ffff, 0xf4fa0105, 0x04fff8f3,
	0xfeff00fe, 0x030200ff, 0x00fefdfa, 0x01010101,
	0x0400fbf7, 0x00010305, 0x0703fdf8, 0x00010408,
	0x03020201, 0xfdff0103, 0x06050504, 0x00020506,
	0x00000000, 0xfcfe0001, 0xfdfcfdfd, 0xf8f9fcfd,
	0xff060c0e, 0x0000fdfd, 0xfd010507, 0xfffefcfb,
	0xfefefefe, 0xfffffefe, 0x01fffcfa, 0xff000101,
	0xfd010202, 0xfdfaf9fa, 0xfdff0001, 0x060401fe,
	0x02010000, 0x07080704, 0x00000000, 0xf8f9fcff,
	0xfe010200, 0xfffefdfd, 0xfd0001ff, 0x0200fefb,
	0xfbfefffe, 0x090601fc, 0xfcfdfefd, 0x0d0b05fe,
	0x0602fefc, 0xf2f6fd04, 0x0401fffe, 0xfeff0104,
	0xfeff0000, 0x060400fe, 0xfd000202, 0x090500fd,
	0x01fefcfc, 0xf9030906, 0xfffefefe, 0xf5000804,
	0x00000101, 0xf6ff0602, 0x00010202, 0xf9000402,
	0xfafdfeff, 0xf8f8f8f9, 0x01030302, 0xfcfdfeff,
	0x04050505, 0xff000203, 0x03030303, 0x01010202,
	0xfe020303, 0x0a0700fd, 0xfe020201, 0x0300fcfb,
	0x02040300, 0xfcfafbfd, 0x04040200, 0xf9f9fc01,
	0x05050402, 0x06060505, 0xfbfdfcfc, 0xfefdfdfb,
	0xfbfcfcfd, 0xfefefefc, 0x00000101, 0x04050402,
	0x040300fe, 0x02020304, 0x00fcf9f7, 0x06060603,
	0xfefdfbfb, 0x04030100, 0xfe020505, 0xfdfbfafc,
	0x07fcfa01, 0x01fefe05, 0x06fcfb05, 0x01fcfb04,
	0x06fcfb05, 0x01fdfb04, 0x08fdf901, 0x01fdff07,
	0x00fcf9f8, 0x05050402, 0x02050605, 0xf9f9fbff,
	0x01040605, 0xfbfafbfd, 0xfefbf9f9, 0x0a090601,
	0x01000306, 0xf2f80003, 0x01ff0003, 0xfc000304,
	0x01000001, 0x01010102, 0x0201ffff, 0x00ffff01,
	0x01010101, 0x00fdfe00, 0x00010201, 0xfcf7f8fe,
	0x02030301, 0x01fdfd01, 0x01010100, 0x08040101,
	0x07090502, 0x01ffff02, 0x0001fffc, 0x02fffcfd,
	0x030300fd, 0x0200ff00, 0x0101fffc, 0xfcfbfcfe,
	0xfefeff01, 0x050402ff, 0x00010102, 0x0000ffff,
	0x05040302, 0x00010204, 0xfdfaf7f7, 0xffffffff,
	0x0704fafa, 0xfefffe00, 0x0605feff, 0xff00feff,
	0x0001ff04, 0xfe00fefc, 0xfeff0107, 0x010301fd,
	0x03010204, 0x02010103, 0x00fcfe02, 0x00000103,
	0xfcf8fc01, 0x00010201, 0xfaf7fd02, 0x02030300,
	0xff00ffff, 0x020100ff, 0xf8fc0103, 0x0201fdf9,
	0xfefdff02, 0x000100ff, 0x0b0500ff, 0xfdff0309,
	0xfffefeff, 0x01010101, 0x0300ff00, 0x05050606,
	0xffff0102, 0xfcfafbfe, 0x01020202, 0xfbfbfcff,
	0xf9fafdff, 0x01fffcfa, 0x04030505, 0x05040304,
	0xfefdfeff, 0x0100fefe, 0x00000000, 0x03020100,
	0xfffcfafa, 0x02020202, 0xfefbf9fa, 0x00ffff00,
	0x04020202, 0x04030304, 0xff000102, 0x04020000,
	0xf8fb050c, 0x020200fb, 0xfdfafd02, 0xfeff0000,
	0x03fffdfe, 0xfdfe0104, 0x04030202, 0xffff0103,
	0x00010203, 0x00030401, 0xfb000304, 0x030300fa,
	0xf9010302, 0x0301faf4, 0xff040301, 0x0100fcfa,
	0x0602fcf7, 0x00010407, 0x0604fff9, 0xfdfd0004,
	0x040400fa, 0xfefdfe01, 0x020301fc, 0x00fffe00,
	0xfb020500, 0xfc0103fd, 0xfa0204fe, 0xfd0406fd,
	0xfb0305ff, 0xfc0307ff, 0xfa0002ff, 0xfd0305fd,
	0x0503fdf8, 0xfefe0103, 0xfe040402, 0x0301fdfc,
	0xfbfd0102, 0x030403fd, 0x03fbfafb, 0xfbff080a,
	0xfc020300, 0x0600f9f7, 0x0705fffb, 0xfdfdff04,
	0x03fefbfb, 0xfcff0506, 0xfc000609, 0xff0101fe,
	0x01ffffff, 0x00ff0001, 0x000000ff, 0x00ffff00,
	0xfffe0102, 0x00000101, 0xff02080c, 0xf9f9fcff,
	0x06030102, 0x00020407, 0x00ff0001, 0xf8f9fcff,
	0x00ff0000, 0xfdff0000, 0x00000000, 0xfe000101,
	0x010100ff, 0xfeff0000, 0xfdff0000, 0x01fffdfc,
	0x000000ff, 0x0c0a0401, 0xfefe00ff, 0x01fffdfd,
	0xfcfefffd, 0x07090902, 0xfdff00fd, 0x01ff0200,
	0xfdfe01ff, 0x00fdff00, 0xfefd0000, 0x01ffff00,
	0xfffffeff, 0xfefffffe, 0xfffeff02, 0xfe000100,
	0x02feff03, 0xfdff0305, 0x01fbfb01, 0x00020606,
	0xff000201, 0xfe000100, 0x00fffdfb, 0xfe010201,
	0xfefefbf9, 0x0100fefe, 0x010100ff, 0x0c090300,
	0x01050600, 0x0300fdfe, 0x01050600, 0x03020101,
	0xfdfefefb, 0x00000000, 0xfefdfdfa, 0xfeff0000,
	0x01020404, 0x00ffff00, 0x0100fefe, 0x00010102,
	0xff010202, 0xf6f7fbfd, 0xffff0102, 0x01040401,
	0xfefe0004, 0x00fffefe, 0xfdfc0107, 0x010100fe,
	0xfeff050a, 0x00010100, 0xfcfd0105, 0xfefffffd,
	0xfdff0102, 0xff0101fd, 0x0003fffe, 0x000101ff,
	0x020701fd, 0x00fffefd, 0xff0804fe, 0x0200fbf8,
	0x0201fffc, 0xfefcfd01, 0x01fefdfb, 0x06040404,
	0xfdfcfefd, 0x02010100, 0x01020202, 0xffff0102,
	0xff00fffc, 0xfffffdfd, 0x02040401, 0xfdfeff00,
	0x03050604, 0xfcfe0102, 0x01010100, 0xfafcff01,
	0xff020201, 0x02fffbfa, 0x0101fefd, 0x0502fdfc,
	0x0202fffe, 0x0300fcfd, 0x0602fefe, 0x02010205,
	0x00fdfd02, 0x01030200, 0x0301ff03, 0xfbff0201,
	0xfefcf9fb, 0x01080801, 0x000200ff, 0xfd0100fd,
	0xfefbfbfe, 0xfe00fffd, 0x0400fcff, 0x00040200,
	0x0a080000, 0xff030102, 0x0302fdfc, 0xff01fdfd,
	0x02fcfe01, 0xfffe0307, 0xfffe0406, 0x03ff0002,
	0xfefe0101, 0x04fdfbfe, 0x0101fefa, 0x04fefcff,
	0xfefefffe, 0xfe000100, 0xff0001ff, 0xfdff0000,
	0xfcfe0100, 0x0000fffd, 0x00050806, 0x03020100,
	0x0502fefe, 0x01000002, 0xfffefe02, 0x040201ff,
	0x0100ff02, 0x01000000, 0x01fff9f8, 0x0301ffff,
	0x02060300, 0x000201fe, 0x00fff9f6, 0x0102fffd,
	0x02020000, 0xff010101, 0xfefe0003, 0x00010200,
	0x00000108, 0x00fffdfe, 0x0502fe02, 0x01fffe01,
	0xfffdfafd, 0x02fffdfd, 0x02010002, 0x00000102,
	0xfeffff01, 0x000100ff, 0xff02090f, 0xfdfdfdfe,
	0x00fefd00, 0xffffff00, 0x00010001, 0xffffff00,
	0xfe020200, 0xf8f9fdfd, 0x00020200, 0x01010201,
	0x02020201, 0x03000103, 0xfeff0001, 0x0500feff,
	0x01fffaf5, 0xfd010302, 0xff030401, 0xff0201fe,
	0xff010202, 0xff0100fe, 0xffff0000, 0x02030200,
	0x01020101, 0xfc0001ff, 0xfe000000, 0xfe0402fe,
	0x0000fdfe, 0xfa0102ff, 0x05050200, 0xf9ff0203,
	0x00000204, 0xff010303, 0x03ffff00, 0xff010406,
	0x0200fefe, 0xfefe0002, 0xfbff00ff, 0x01fffbf9,
	0x00feff05, 0xfb020402, 0x02fefb00, 0xfa000201,
	0x01000106, 0x0204fffe, 0x00fdfd02, 0x0000feff,
	0x0200ff01, 0x0b060000, 0x00ffff02, 0x0503fefd,
	0x00fffe00, 0xfd0000ff, 0xffffff01, 0xf9fdfffe,
	0xfefe0101, 0xfe010301, 0xff0002ff, 0x000001ff,
	0xff0302fc, 0x0100fefe, 0x0504fef5, 0xfeff0206,
	0xff01fefa, 0x0901fcfd, 0x030300fd, 0x03fdfd02,
	0x00000101, 0x03feff01, 0xfdfd0002, 0x03ffff00,
	0x01fdff01, 0x06fcfa02, 0xfefbfe00, 0x03fefd00,
	0xfe010202, 0xff0201fe, 0xfe0101ff, 0xff0706ff,
	0xfefc0001, 0x01fdfe01, 0xfefd00fc, 0x00fd0002,
	0x010304fd, 0xff000708, 0xfc0104fd, 0xfdfe0302,
	0xfc0106fd, 0xffff0101, 0xfdfd04fe, 0xffff0003,
	0x02fc0201, 0x02fffd04, 0x05fcff03, 0x02fdfa04
};

static const uint32 s_svq1InterCodebook8x8[1536] = {
	0x0504fdfc, 0x00010102, 0x0505fdfb, 0x00000102,
	0x0505fcfa, 0x00000102, 0x0504fcf9, 0x00000102,
	0x0403fbf8, 0x00000102, 0x0403faf8, 0x00010101,
	0x0402faf8, 0x00010102, 0x0402faf8, 0x01010101,
	0xffffffff, 0xffffffff, 0xfefefeff, 0xfefefefe,
	0xfdfdfdfe, 0xfdfdfdfd, 0xfdfdfdfe, 0xfdfcfdfd,
	0xfefefefe, 0xfefdfdfe, 0x01010101, 0xffff0001,
	0x05050504, 0x02030304, 0x08080707, 0x05060708,
	0x04020102, 0xfafc0004, 0x05020101, 0xf9fb0105,
	0x04010201, 0xf8fb0105, 0x05010101, 0xf8fa0005,
	0x05010100, 0xf7fa0106, 0x04010000, 0xf8fb0005,
	0x04010000, 0xf9fb0005, 0x04010000, 0xf9fc0104,
	0x00030201, 0xfffdfcfd, 0x00040301, 0xfffdfcfd,
	0x01050402, 0xfefdfcfd, 0x01060502, 0xfefcfbfd,
	0x01060603, 0xfefcfbfd, 0x01060603, 0xfefcfbfd,
	0x01060603, 0xfefcfbfd, 0x01050503, 0xfefcfcfd,
	0x02020202, 0xff000001, 0x03040404, 0x00010102,
	0x04040504, 0x01020303, 0x04040404, 0x02020304,
	0x03030302, 0x01020303, 0xffffffff, 0x00000000,
	0xfbfafafb, 0xfdfdfcfb, 0xf8f7f7f9, 0xfbfafaf9,
	0x06060606, 0x04050506, 0x03040404, 0x02030303,
	0xffffff00, 0xfffffefe, 0xfafafbfd, 0xfcfbfafa,
	0xf9fafbfd, 0xfcfbfafa, 0xfefefeff, 0xfffffefe,
	0x01010100, 0x01010101, 0x03030303, 0x03030303,
	0xfbfe0102, 0x050200fc, 0xfafe0102, 0x050300fb,
	0xfafe0102, 0x0603fffa, 0xf9fe0203, 0x070400fa,
	0xf9fe0102, 0x070500fb, 0xfafe0102, 0x070400fb,
	0xfafe0102, 0x060400fc, 0xfbfe0101, 0x060300fc,
	0xfcfaf7f6, 0x020302ff, 0xfdfbf7f6, 0x03040400,
	0xfffdf9f7, 0x03050502, 0x00fefbf9, 0x03050503,
	0x0100fdfa, 0x03050604, 0x0201fefc, 0x02040503,
	0x020100fe, 0x01030402, 0x020201ff, 0x01030302,
	0xfafbfbfc, 0xfbfafafa, 0xfcfcfdfd, 0xfcfcfcfc,
	0x00000000, 0xffffffff, 0x05060505, 0x02030405,
	0x07070605, 0x04050607, 0x04040303, 0x02030404,
	0x0000ff00, 0xff00ffff, 0xfcfcfdfd, 0xfdfdfcfc,
	0x01fbfe01, 0x00020405, 0x01fafd01, 0x00020506,
	0x00f9fc00, 0x01020606, 0xfff7fbff, 0x01030606,
	0xfef6faff, 0x01030606, 0xfef7faff, 0x01030605,
	0xfef7fafe, 0x01030505, 0xfef9fafe, 0x01020404,
	0xf7f8f9fb, 0xfaf9f8f7, 0xf9fafafb, 0xfbfafaf9,
	0xfcfdfdfd, 0xfcfcfbfb, 0xff0000ff, 0xffffffff,
	0x02020100, 0x01020202, 0x05040302, 0x04050505,
	0x06050403, 0x07080808, 0x06050403, 0x06070707,
	0x08070605, 0x090a0a09, 0x07060403, 0x08090908,
	0x03020100, 0x05050504, 0xfffffeff, 0x02020100,
	0xfdfdfdfe, 0x00fffefd, 0xfbfbfcfd, 0xfcfbfbfb,
	0xfafbfbfc, 0xfbfaf9f9, 0xfafbfcfd, 0xfafaf9f9,
	0xfd00070d, 0xfbfcfcfd, 0xfd00070e, 0xfcfcfcfd,
	0xfcff080f, 0xfcfbfcfc, 0xfcff080f, 0xfdfcfbfc,
	0xfcff070f, 0xfcfbfbfb, 0xfcff070e, 0xfdfcfcfc,
	0xfcff060c, 0xfdfcfcfc, 0xfcff050b, 0xfdfcfcfc,
	0x0405fcef, 0x03030404, 0x0405fbee, 0x03030404,
	0x0406fbed, 0x02030404, 0x0406fbec, 0x03030404,
	0x0406fcec, 0x03030504, 0x0406fbed, 0x03030504,
	0x0405fcee, 0x02030404, 0x0304fbef, 0x03030404,
	0xfcfafafa, 0x0b0601fe, 0xfcf9f9fa, 0x0d0802fe,
	0xfcf9f8f8, 0x0e0903fe, 0xfbf9f8f8, 0x100a04ff,
	0xfbf9f8f8, 0x110a04ff, 0xfcf9f8f8, 0x100a0500,
	0xfdfaf8f8, 0x0f090400, 0xfdfbf9f9, 0x0c080400,
	0x05070708, 0xf2f8fe02, 0x05070808, 0xf1f8fe02,
	0x05070808, 0xf0f7fd01, 0x05070808, 0xeff6fd01,
	0x05080908, 0xeff6fd01, 0x04070808, 0xf0f6fc01,
	0x04070707, 0xf2f7fd01, 0x03060706, 0xf3f7fd00,
	0xfcfc0105, 0x0000fffd, 0xfdfd0207, 0x0001fffe,
	0xfdfd0107, 0x010100ff, 0xfefd0106, 0x000101ff,
	0xfefc0006, 0x000100ff, 0xfdfc0005, 0xff0000ff,
	0xfffd0005, 0xfe000000, 0xfffe0104, 0xff000100,
	0x01010202, 0xf8f8fafe, 0x01010101, 0xf8f8fbfe,
	0x00010101, 0xfbfbfdff, 0x00000000, 0xfeffffff,
	0x0000ff00, 0x00010000, 0x00000001, 0x02030201,
	0x01010102, 0x03040302, 0x03030303, 0x04050404,
	0xfefdfcfc, 0x01010000, 0xfefdfcfc, 0x010000ff,
	0xfffefefe, 0x0000ffff, 0x00000100, 0xff000000,
	0x02020202, 0x01010202, 0x04040403, 0x03040404,
	0x03010101, 0x03030403, 0xfcfbfafb, 0xfefefdfd,
	0xfffffefc, 0x0100ffff, 0xfffffefc, 0x0100ffff,
	0xfffffefd, 0x020100ff, 0xfffefdfc, 0x030301ff,
	0xfffdfdfc, 0x050401ff, 0xfefefdfc, 0x070401ff,
	0xfffffefe, 0x08060200, 0x010000ff, 0x08070401,
	0xfefdfdfd, 0x00fffffe, 0x0100ffff, 0x03030202,
	0x04020100, 0x05060605, 0x030200ff, 0x03050605,
	0x0200ffff, 0x01020303, 0x00fffefe, 0xfcfcfdff,
	0xffff0000, 0xf9f8fcfe, 0x00010201, 0xf9fafcff,
	0xfa0104fe, 0x00000300, 0xf90105fe, 0x00000300,
	0xf80105fd, 0xffff0300, 0xf70106fe, 0xff000300,
	0xf80206fe, 0xff000400, 0xf90105fd, 0x00000401,
	0xf90104fe, 0x00010400, 0xfa0104ff, 0x00010300,
	0x03000000, 0x01040504, 0x02010101, 0x00020303,
	0x02010202, 0xfeff0102, 0x01010304, 0xfbfdff00,
	0xff010305, 0xfafcfdfe, 0xfe000305, 0xf9fafbfd,
	0xfe000304, 0xfbfbfcfd, 0xff000304, 0xfdfdfefe,
	0x00000000, 0x06fefbff, 0x01000000, 0x08fefaff,
	0x02000000, 0x09fdfa00, 0x0200ff00, 0x0afef900,
	0x0200ff00, 0x0afdf8ff, 0x02ffff00, 0x09fdf9ff,
	0x0100ff00, 0x08fdfaff, 0x01000000, 0x07fefb00,
	0x02030302, 0xffff0001, 0x02030403, 0xfeff0001,
	0x02040403, 0xfdfeff01, 0x02030302, 0xfdfeff00,
	0x010100ff, 0xfefeff00, 0xfffdfcfb, 0x01010100,
	0xfffbf8f8, 0x03040301, 0x00fbf7f6, 0x05060503,
	0x0504fffb, 0x00000103, 0x0504fffa, 0xfeff0002,
	0x0405fffa, 0xfefeff02, 0x0404fff9, 0xfdfdfe01,
	0x0405fffa, 0xfdfdfe01, 0x040400fb, 0xfefeff01,
	0x040500fc, 0xfeffff01, 0x030401fd, 0xfeffff01,
	0x01fefdfe, 0x03050604, 0x00fcfcfd, 0x02040503,
	0xfffbfbfd, 0x01030402, 0xfffcfafc, 0xff020402,
	0x01fdfcfe, 0xff020402, 0x01fefcfe, 0xfe010303,
	0x01fefdfe, 0xfe010303, 0x01fffefe, 0xfe000303,
	0xfefdfcfc, 0x070502ff, 0xfdfdfcfc, 0x070501fe,
	0xfdfefdfe, 0x0503fffd, 0xfe00ffff, 0x0402fefd,
	0xff010101, 0x0301fdfc, 0xff020304, 0x01fffdfc,
	0x00030406, 0x00fefdfd, 0x01030506, 0xfffefdfe,
	0x04080b0c, 0xfffefe00, 0x0206090a, 0x00fffeff,
	0x00020304, 0x0100ffff, 0xffffffff, 0x020100fe,
	0xfefcfbfd, 0x030200fe, 0xfefcfbfb, 0x020100ff,
	0xfefcfbfb, 0x010100ff, 0xfefdfcfc, 0x0000fffe,
	0xff020303, 0xfefdfcfd, 0xfe000203, 0xfefdfcfc,
	0xff010202, 0xfdfcfbfd, 0x01030303, 0xfdfdfdfe,
	0x03040404, 0xfefefe00, 0x03050505, 0xfefeff00,
	0x02040505, 0xfefdfeff, 0x00030303, 0xfcfcfcfe,
	0xfe04ffff, 0xfb0206fe, 0xfe0400ff, 0xfa0206fd,
	0xfe0400ff, 0xf90307fd, 0xfd04ffff, 0xf90308fc,
	0xfd04ff00, 0xfa0307fc, 0xfd04ffff, 0xfa0307fc,
	0xfd03ffff, 0xfa0306fc, 0xfe0300ff, 0xfb0306fd,
	0x02f9fe01, 0x01fffe05, 0x03f8fe01, 0x02fffd06,
	0x04f7fe02, 0x02fefc07, 0x05f7ff03, 0x03fffc07,
	0x04f7ff03, 0x02fefc07, 0x04f9ff03, 0x01fefc06,
	0x04fa0002, 0x01fffc06, 0x03fb0002, 0x01fffd04,
	0x000202fe, 0xfcfdff00, 0x010202fe, 0xfcfe0001,
	0x020202fe, 0xfeff0102, 0x030302fd, 0xfe000204,
	0x020302fd, 0xfd000204, 0x010201fc, 0xfdff0102,
	0x000100fb, 0xfdfe0101, 0x000000fc, 0xfdfe0001,
	0xfeff0000, 0x080702fe, 0xfdff0000, 0x070601fe,
	0xfdff0100, 0x050400fd, 0xff000100, 0x030100ff,
	0x01010200, 0x0100ff00, 0x020100fe, 0xffff0001,
	0x0100fefb, 0xfdfd0001, 0x01fffcfa, 0xfcfdff01,
	0x0502fefc, 0x02030406, 0x0401fdfb, 0x00000204,
	0x0200fefc, 0xfefeff01, 0x0100fffe, 0xfefdfe00,
	0x000000fe, 0xfffeffff, 0x00fffffe, 0x02010000,
	0xfffffefe, 0x04030100, 0xfffefdfe, 0x05040200,
	0xfefe0102, 0x000100ff, 0xfdfd0001, 0x000100ff,
	0xfdfdff00, 0x010101ff, 0xfffd0000, 0x03030201,
	0xfffdff00, 0x03030301, 0xfefcfefe, 0x04040301,
	0xfefcfdfd, 0x04030301, 0xfefbfdfe, 0x03030201,
	0x04030504, 0x05040404, 0x00010303, 0x01000000,
	0xfeff0101, 0xfefdfcfd, 0xfe000202, 0xfefdfcfe,
	0xff010302, 0xfefdfdff, 0x00000201, 0xfffefeff,
	0xff000100, 0xfffefdff, 0xff000101, 0xfefefeff,
	0x00fffffe, 0x00010201, 0x05030201, 0x03050506,
	0x04030201, 0x03040505, 0xfdfdfefe, 0x0000fffe,
	0xfbfcfdfd, 0xfffefdfc, 0xfefeffff, 0x0000fffe,
	0xff000100, 0x010000ff, 0xfeff00ff, 0xfffefefd,
	0x05060707, 0xfeff0204, 0x02020303, 0xfdfe0001,
	0xffffff00, 0xfefeff00, 0xfffefdff, 0x01000000,
	0xfffefe00, 0x020201ff, 0xffff0103, 0x020201ff,
	0xfdfe0103, 0x0201fffe, 0xfafbfe01, 0x00fefdfb,
	0xfdfeff00, 0xfefe00ff, 0xffff0000, 0xfeff0100,
	0xfffe0000, 0xfe000000, 0xfdfdfeff, 0xfdfdfffe,
	0xfdfdfeff, 0xfcfdfefe, 0x00000202, 0xfeff0000,
	0x02030505, 0xff000202, 0x05060808, 0x01020404,
	0xfdfaf8f9, 0xfffeffff, 0x00fdfbfb, 0x00000102,
	0x0300ffff, 0x01010304, 0x03010102, 0x02020304,
	0x02000203, 0x02010203, 0xffff0204, 0x01010100,
	0xfdfe0203, 0x0100fffe, 0xfcfd0103, 0x0100fefd,
	0x02fffefc, 0x00010303, 0xfefcfbf9, 0xfeff0000,
	0xfcfbfbfa, 0xfdfefefe, 0xffff00ff, 0xff000000,
	0x02020302, 0x00010202, 0x03040503, 0x00010001,
	0x02030403, 0xffffff00, 0x01030505, 0xffffff00,
	0xff000101, 0xfcfafbfd, 0x00000101, 0xfffdfd00,
	0x00ffff00, 0x03010001, 0xfffdfefe, 0x07040202,
	0x00fefefe, 0x06030202, 0x010000ff, 0x03000001,
	0x03030300, 0xfffdfe01, 0x03040301, 0xfcfbfd00,
	0xff000200, 0xfefefcfd, 0x00020401, 0xfffefdfe,
	0x01030603, 0xff00fefe, 0x01040704, 0x00fffdfe,
	0x00030603, 0x00fffdfd, 0xff000301, 0x0101fefd,
	0xfeff0100, 0x0202fffd, 0xfdfdfffe, 0x0201fffd,
	0x00ff0103, 0x00000001, 0xfffeff02, 0xffff0001,
	0x00feff01, 0xfdfe0001, 0x01fffe00, 0xfbfd0103,
	0x02fffe00, 0xfbfd0205, 0x04fffe00, 0xfbfe0306,
	0x0400fe00, 0xfcfe0407, 0x0400fe00, 0xfcfe0406,
	0xfcfdfefe, 0x00fffefd, 0xff000101, 0x0100ffff,
	0x02020303, 0x01010101, 0x02020202, 0x01000001,
	0x00000000, 0xffffffff, 0xfcfcfcfc, 0xfdfcfcfc,
	0xfdfefdfd, 0x00fffffe, 0x05040403, 0x07060605,
	0xfe07feff, 0x00fffffc, 0xff09feff, 0x00fffffc,
	0xff0afdff, 0x01fffffc, 0xfe0afdff, 0x02fffffd,
	0xfe0afeff, 0x02fffffc, 0xfe09feff, 0x02fffffc,
	0xfe08feff, 0x01ff00fc, 0xfe07fe00, 0x0200fffd,
	0x0301fc03, 0x0001fefd, 0x0401fb03, 0x0001fefd,
	0x0502fa03, 0x0003fffd, 0x0502fa03, 0x0002fffd,
	0x0501fa03, 0x0003fefc, 0x0501fa03, 0x0002fefd,
	0x0401fa02, 0x0001fffd, 0x0401fa02, 0x0001fffe,
	0x01010000, 0x02000001, 0x0101ff00, 0x02000001,
	0x0000ff00, 0x02000000, 0x0000ff00, 0x00ff0000,
	0x00010001, 0xfffeff00, 0x00010103, 0xfdfcfe00,
	0x01020305, 0xfcfbfd00, 0x00020405, 0xfbfbfcff,
	0xfeff0001, 0xf7fafdfe, 0xffff0002, 0xfafd0000,
	0xff000001, 0xfbfe0000, 0x01010102, 0xfdff0201,
	0x01020101, 0xff010202, 0x01020101, 0x01010101,
	0x01020000, 0x02020000, 0x02020100, 0x02020000,
	0x0100fdfc, 0x03040604, 0x0000fefd, 0x00010402,
	0x0000ffff, 0xfdfe0101, 0x00010101, 0xfbfd0101,
	0x00010101, 0xfbfd0101, 0xff0000ff, 0xfcfe0101,
	0xff0000ff, 0xfe000201, 0x000000ff, 0x00010302,
	0x040300ff, 0xfbfbfc00, 0x05040000, 0xfefdfe02,
	0x0402ff00, 0x00ffff02, 0x01fffe00, 0x0100fe00,
	0x00fefe01, 0x01ffff00, 0x00fdfe01, 0x00ff0001,
	0x01fefe01, 0x00000103, 0x02fffe01, 0x00000204,
	0x02030201, 0x01020200, 0x000100ff, 0x010101fd,
	0xfe0000ff, 0x010200fc, 0xff0202ff, 0x010200fb,
	0xff0403ff, 0x010200fb, 0x000402fe, 0x0000fffc,
	0x000200fc, 0x0000fefc, 0x0102fffb, 0x020301fe,
	0x00010001, 0xfeff0201, 0xfeff0002, 0xff000301,
	0xfcfe0003, 0x00010300, 0xfbfd0105, 0x010102fe,
	0xfbfe0106, 0x010001fe, 0xfbff0105, 0x00ff00fe,
	0xfcfe0003, 0x00ff00fe, 0xfe00ff01, 0x01000100,
	0x03020101, 0x02010102, 0x0100ffff, 0x01010001,
	0x0000fdfc, 0x02010101, 0x0200fdfc, 0x02030202,
	0x0100fcfb, 0x02010101, 0xfffffcfb, 0x00fffefe,
	0x0000fefd, 0xfffefdfe, 0x04040302, 0x00000002,
	0x0100fefc, 0x00000000, 0x0101fffd, 0x00000000,
	0x020200fe, 0x02000000, 0x010201ff, 0x050300ff,
	0xff010200, 0x060500fe, 0xfd000100, 0x060400fd,
	0xfcfe0101, 0x0201fdfc, 0xfcfe0001, 0x00fefcfb,
	0xfdfdfdff, 0xfffffefd, 0x00010203, 0x01010100,
	0x02030405, 0x02020101, 0xfe000102, 0xfffffefe,
	0xff000000, 0xfefefefe, 0x03030100, 0xffff0102,
	0x04030100, 0xff010203, 0x01fffdfc, 0xfdfdfe00,
	0xf8f9fcfd, 0x02fffcf9, 0xfcfdff00, 0x0200fefc,
	0xff000001, 0x0200fefd, 0x00010102, 0x0200ffff,
	0x01010101, 0x01000000, 0x01010101, 0x01010101,
	0x02010000, 0x02020303, 0x03010000, 0x02030404,
	0x00030303, 0x020100ff, 0xff010101, 0x01fffffe,
	0xfdfffefe, 0x00fefefd, 0xfefefcfc, 0x00fdfefe,
	0x01fffcfc, 0x02ff0001, 0x0302fffd, 0x05030304,
	0x030200fe, 0x03030303, 0x0000fefe, 0x01000000,
	0xff010200, 0xfe03fffd, 0xffff00ff, 0xfe0400fd,
	0xfefefefe, 0xfe0501fe, 0xfffdfefd, 0xfd0401fe,
	0x00ff00fe, 0xfb0300ff, 0x02010201, 0xfb020000,
	0x03020402, 0xfd030101, 0x01010201, 0xfe040100,
	0xfffcfd04, 0x03010303, 0xfffcfc04, 0x02000203,
	0x00fcfd04, 0x01ff0202, 0x01fefd04, 0x00fe0102,
	0x01fefc02, 0x00fd0002, 0x00fefd02, 0x02fe0001,
	0x00ffff03, 0x03000000, 0xfefefe02, 0x02ffffff,
	0x04030202, 0xff000103, 0x02010001, 0xfefeff01,
	0x01020102, 0xffff0001, 0x03040304, 0x01010102,
	0x02020203, 0x01010101, 0x00fffeff, 0xffffffff,
	0xfffefdfd, 0xfefefefe, 0xfdfdfcfc, 0xfdfdfcfc,
	0xfdff0102, 0x0403fefc, 0xff010202, 0x0201fefd,
	0x03030201, 0xfefffe00, 0x040200ff, 0xfdff0002,
	0x0300fefe, 0xfd000203, 0xfffdfe00, 0xff020201,
	0xfbfcff03, 0x000200fd, 0xfafe0306, 0x010300fb,
	0x00fe03fe, 0x01fefe03, 0x00fd04fd, 0x02fffe03,
	0x00fd05fd, 0x02fffe04, 0xfffc04fe, 0x02fefd03,
	0x00fd04fd, 0x02fffd03, 0x00fe05fe, 0x02fffd03,
	0x01fd04fe, 0x02fffe03, 0x01fe03fe, 0x0200fe03,
	0xff000001, 0xfffc0201, 0xff000002, 0xfefc0201,
	0xff010101, 0x00fe0402, 0xff01ff00, 0x01ff0502,
	0xfe00ff00, 0x01ff0501, 0xfeffff00, 0xfffd0300,
	0xfe000101, 0xfffd0300, 0xfd000101, 0x00fe0300,
	0x01ff0001, 0x05040201, 0x01ff0001, 0x07050101,
	0xfffe0000, 0x050300ff, 0xfffeff00, 0x0302ffff,
	0xfffdff00, 0x0201ffff, 0xfefcfeff, 0x0000fefe,
	0xfffefeff, 0x0000fefe, 0x00ffff00, 0x0000ffff,
	0xff000303, 0x040401ff, 0xfe000302, 0x010100fe,
	0xff010302, 0x000100ff, 0xff000201, 0xfe00ffff,
	0xff000100, 0xfe00fffe, 0xff000100, 0x0001fffe,
	0xfdff0101, 0x0301fdfc, 0xfdff0201, 0x0301fcfb,
	0x0100fefd, 0xfe000101, 0x01010100, 0xfdff0000,
	0x01010201, 0xfeffff00, 0xfffdff00, 0xff00ffff,
	0xfdfafd00, 0x0101fffe, 0xfdfcff02, 0x020200fe,
	0x01010405, 0x02030100, 0x01020405, 0x0100ff00,
	0xfbfe00fe, 0x0000fdfa, 0x000100fe, 0x020201ff,
	0x030100fe, 0x01020202, 0x040200fe, 0x01010203,
	0x030200fe, 0x00ff0002, 0x0101fffd, 0x01ffff00,
	0x0001fffc, 0x0200feff, 0xff00fffc, 0x0401feff,
	0xff0000fd, 0x00010101, 0xff0001fd, 0xffff0000,
	0x000303ff, 0x01000001, 0xfe0202fd, 0x010000ff,
	0xfe0000fb, 0x020001ff, 0x0001fef9, 0x02020201,
	0x020300fb, 0x02020303, 0x010402fd, 0xfdfe0000,
	0xfefe0205, 0xffffff00, 0xfdfcff02, 0xfffffeff,
	0x01fefe00, 0x0100ff02, 0x03fffeff, 0x0200ff03,
	0x03000001, 0x02fffe03, 0x03010102, 0x00fefe02,
	0x01ff0001, 0xfefdfd01, 0x02010001, 0x00000003,
	0x00fdfbfc, 0xfffeff01, 0x01fffdfe, 0x00000002,
	0x01020101, 0x01010102, 0x01030403, 0xfffffe00,
	0x00020303, 0xfefdfdfe, 0xff000101, 0xfefefcfe,
	0x00000102, 0x0100ff00, 0x01010102, 0x03010101,
	0xff000000, 0x0001fffe, 0xfefffffe, 0x0000fefd,
	0xff0000ff, 0x010100fe, 0xff000101, 0x010301ff,
	0xfe000202, 0x000302ff, 0xffff0103, 0xfe020401,
	0xfffd0002, 0xfb000503, 0x00feff01, 0xfaff0303,
	0x040300ff, 0x02010002, 0x0100fffe, 0x0100feff,
	0xfdfefdfe, 0xfdfaf9fa, 0x01030202, 0xfefdfeff,
	0x01030202, 0x00000000, 0x00010102, 0x01000101,
	0x00000001, 0x02010100, 0x00ff0001, 0x01020200,
	0x01030101, 0x01ffffff, 0x0000fffe, 0x02fffefe,
	0x0101fefe, 0x03010001, 0xff00fefe, 0x0200ff00,
	0x00010000, 0x01feffff, 0x01020203, 0x01fefe00,
	0x02030305, 0x04010101, 0xfbfcfd00, 0x01fffdfc,
	0x02fffcfa, 0xff000002, 0x0301fefc, 0x00020203,
	0x02fffefd, 0x00020303, 0x01fefefd, 0xff010102,
	0x00fefefe, 0xff010202, 0x01ffffff, 0x00020302,
	0x01feffff, 0xff020202, 0x00feff00, 0xff000102,
	0x01020406, 0x01000000, 0xfeff0204, 0xfffffefe,
	0xfeff0102, 0xfffefefe, 0xfe000202, 0x00fffefe,
	0xfeff0000, 0x0100fffe, 0xfffefdfd, 0x00fffeff,
	0x0302fefd, 0xfeff0002, 0x050400fe, 0xff000205,
	0x00020405, 0xfffffeff, 0x01020304, 0xff00ff00,
	0x01000101, 0xff010001, 0x00fffffe, 0xfdfefe00,
	0x000000ff, 0xfbfdfdff, 0xff010100, 0xfdfefeff,
	0xfeffffff, 0x010301ff, 0xfffefeff, 0x05060502,
	0x01fefdfd, 0xfffffe01, 0x04030201, 0xfdfffd01,
	0x01000203, 0xfdfffdff, 0x00ff0001, 0x0001ffff,
	0x01000101, 0x03050202, 0x02010101, 0x00030202,
	0xfefefffd, 0xfdfffdfd, 0x000101ff, 0xfe00ffff,
	0xfefe0002, 0xfe010402, 0xfffe0001, 0xff020503,
	0xfefdfeff, 0xfe010301, 0xfffffeff, 0xff010200,
	0x01010000, 0x00020201, 0x04040100, 0x01030202,
	0x0102fffe, 0xfdfefdfe, 0x000100ff, 0xfbfcfcfd,
	0xfcfd0004, 0xfffefcfc, 0x00ff0005, 0xfffefdff,
	0x01000004, 0x00000001, 0xfffefd00, 0x00010000,
	0x0000fe00, 0x01020101, 0x00000002, 0x00010101,
	0xffff0002, 0x00010101, 0xfefeff01, 0x02020200,
	0x00fefbfd, 0x00fdfdff, 0x0200fe00, 0x03000002,
	0x00feff02, 0x02ffff00, 0xffff0205, 0x02ffffff,
	0xff000205, 0x02ff00ff, 0x00000102, 0x02000100,
	0x0101ffff, 0x02010202, 0x0000fefd, 0xfffe0000,
	0x00020300, 0xfdfdfdfe, 0x01030300, 0x02010000,
	0xfeff00ff, 0x0301ffff, 0xfeff00ff, 0x0200ffff,
	0xfeff00ff, 0x02ff0000, 0xfeff00ff, 0x01feffff,
	0xfd000100, 0x02ffffff, 0xff020505, 0x0301ffff,
	0xff010000, 0x0200fefd, 0x00030101, 0x0100fefe,
	0x01030101, 0xffff0000, 0x0102ff00, 0xfdff0001,
	0x0101feff, 0xfcfe0001, 0x010200ff, 0xfdff0001,
	0x02030101, 0xfdfe0001, 0x02040202, 0xfcfeff01,
	0x02020201, 0x0200fe00, 0xfdfeffff, 0x01fdfbfc,
	0x00010100, 0x01ffffff, 0x01010100, 0x02000000,
	0x02010100, 0x02010101, 0x0200ffff, 0x03020202,
	0xfffcfcfe, 0x00fefefe, 0x01000001, 0x01000000,
	0xfefdff00, 0x01020200, 0xfdfeff00, 0x02010100,
	0xfdfe0001, 0x010000ff, 0xfeff00ff, 0x00ff0000,
	0x000101ff, 0x00000202, 0x01030200, 0x02030503,
	0xfe0101ff, 0x01010300, 0xfc0000ff, 0xfffffffc,
	0x000101ff, 0x02010201, 0x000100fd, 0x02000101,
	0xfffffdfb, 0x01000100, 0xfdfefdfc, 0x00fffffe,
	0xfeff0000, 0x00fefefe, 0x00020403, 0x01000000,
	0x00000102, 0x00ff0000, 0x03020100, 0x02030404,
	0x040704ff, 0x00000000, 0x030604ff, 0x01010100,
	0x00040300, 0x010000ff, 0xfe010100, 0xffff00ff,
	0xffff00ff, 0x000000ff, 0x00ffffff, 0x00000000,
	0x00fdfdff, 0xfffeff01, 0xfefcfcfd, 0xfffefeff,
	0x00010202, 0xfd000101, 0x0000fffe, 0xfd000101,
	0x0100fffe, 0xfe010102, 0x02020201, 0xff020303,
	0x00010201, 0xff020101, 0xfcfe0100, 0xff0100fe,
	0xfdff0101, 0xfdff00fe, 0xff000201, 0xfcff0100,
	0xfefeffff, 0x03040300, 0xfdff0101, 0x000000fe,
	0x02020202, 0xffff0102, 0x03010101, 0xfefe0003,
	0xffffff00, 0xfffffe00, 0xfdfcfdff, 0x0200fefe,
	0x0100ffff, 0x05030202, 0x00fffffe, 0x01000000,
	0x0002fdfe, 0xff010100, 0xfe01fcff, 0x000202ff,
	0xfe00fc01, 0xff0101fe, 0xff01fd02, 0xff0101ff,
	0x0103fe03, 0xff010100, 0x0102fd01, 0xff000100,
	0x0001fbff, 0x010100ff, 0x0303fd00, 0x03030201,
	0x01feff00, 0xff020505, 0xfffeff01, 0xfbfe0101,
	0xfeff0101, 0xfdffffff, 0xffff0101, 0x030402ff,
	0xffffffff, 0x030400ff, 0x0100ffff, 0xfffffdff,
	0x02000000, 0xff000002, 0x00fdfe00, 0x02030101,
	0x01020302, 0xfefe0000, 0x01000302, 0x02030301,
	0xfffd0000, 0x030202ff, 0x01fdfefe, 0x01010201,
	0x02fefffe, 0xfeff0101, 0x02000100, 0xfefe0000,
	0x02000100, 0xfefe0000, 0x00fefefd, 0xfdfefeff,
	0x03ff0100, 0xff0301ff, 0x03ff0100, 0xfd02ffff,
	0x03fe0101, 0xfd00fdff, 0x03fe0202, 0xfe01fe00,
	0x03fd0101, 0xfd01feff, 0x03fd0101, 0xfe01ff00,
	0x04ff0201, 0xfe01ff00, 0x03ff0100, 0xfd00fdff,
	0x01fffdfd, 0xfeff0102, 0x0200fefe, 0xfefe0001,
	0x0201fefd, 0xfffeff01, 0x0402fefd, 0x01fefe00,
	0x0402fffd, 0x02fefe00, 0x030401ff, 0x02fefdff,
	0x02040200, 0x02fffeff, 0x00020100, 0x030100ff,
	0x01fb0003, 0x01000004, 0x02fbfe01, 0x01feff05,
	0x030000ff, 0xff000103, 0xfe0403fe, 0xfe00fffd,
	0xfd0503fd, 0xfe0000fd, 0xfe0203ff, 0xff0202fe,
	0xff000002, 0x00000000, 0x01fefd00, 0xfefe0003
};

const uint32 *const s_svq1InterCodebooks[6] = {
    s_svq1InterCodebook4x2, s_svq1InterCodebook4x4,
    s_svq1InterCodebook8x4, s_svq1InterCodebook8x8,
    0, 0
};

static const uint32 s_svq1IntraCodebook4x2[192] = {
	0x0b0d0d0c, 0xeff1f6f9, 0xf6f4f1f0, 0x0c0f0f0b,
	0x0f141102, 0x0d02e8d3, 0xdcfa1415, 0xe5ff100c,
	0x2d0aebee, 0x15f9ecf5, 0x00e4f82b, 0x03e4f021,
	0xfaeeeef4, 0x371cf6ec, 0xeeebeefb, 0xecfa1e38,
	0xea1d1bde, 0xe71a1de2, 0x1a21221e, 0xdfdde1e7,
	0xe0dcdde1, 0x1f25241d, 0x2226f4b9, 0x212affc1,
	0xc4e1253a, 0xc3df2237, 0x5d16c7c7, 0x5d15c6c7,
	0x3e46453b, 0xc4bcbcc1, 0xc0b9b9c0, 0x3e48493f,
	0x0f0700fe, 0x05fdf6f5, 0xf6f6f8fb, 0x090e0901,
	0xf5fc080f, 0xf4f5020c, 0x1c1300f8, 0xe6f1ff04,
	0xf2021bf1, 0xf70116f2, 0xf6f3fafc, 0x2f06f2fa,
	0x1706ecdd, 0x04060906, 0xea1702fa, 0xeb1c04f9,
	0x06feea14, 0x08fbe416, 0xf4f0eef6, 0xff021324,
	0x080400fd, 0x1717f6d3, 0xddec0f28, 0x0104fffc,
	0x18dffb09, 0x13e60308, 0xfd0604ff, 0xcff31920,
	0x070f1818, 0xf9ede5ef, 0x182700d1, 0x0407faeb,
	0xf3f600ff, 0x10050101, 0xf7fd0514, 0xfafefcff,
	0x0401f9ef, 0x0000070c, 0x0b0c0003, 0xe90001fd,
	0x00fa1104, 0x00e70306, 0x05080aef, 0x040104f2,
	0x02040a0d, 0x0201f7e9, 0x0701fd03, 0x14f9e901,
	0x0c02eef9, 0x090afcfb, 0xe8070a04, 0xf6040306,
	0x06eaf216, 0xff050500, 0xfcf503fc, 0xf2071ff9,
	0x2afff0fb, 0xfbf7fefc, 0xfdfaf805, 0xfbebfc2a,
	0xf4140cee, 0x07f6f30d, 0xeefef7f8, 0x082806f0,
	0x0400ff0a, 0xf3ff04fd, 0xf10106fe, 0x02010305,
	0x0301fefc, 0xfcf7000f, 0xfcfcfcfd, 0xfd1005fd,
	0x04030d02, 0x00f6f8fd, 0xfffcfefa, 0x17fafdfe,
	0x0107fa06, 0x0105ee04, 0x0ef101ff, 0x04fc06fb,
	0x06020202, 0x030702e8, 0x030300e6, 0x06010705,
	0xfdeefe0e, 0x02fc0507, 0x012003fa, 0xf4fafafa,
	0x0607dc05, 0x000bff09, 0x03050404, 0xda030f04,
	0xd6fb170a, 0x04040400, 0xf3fa1117, 0x1d01dbf3,
	0x01fff205, 0x01030005, 0x02fb0400, 0xf6000008,
	0xfdfe0704, 0x010103f6, 0x030dfff4, 0xfd01ff00,
	0x0103ff00, 0x0903f7fa, 0xfafc01fa, 0x0800ff08,
	0x1200fdfd, 0xfffcfffb, 0xfc03fef8, 0xfbff1100,
	0xf609fe05, 0xfb06fb01, 0x03020204, 0x01f8f20a,
	0xffeefeff, 0x020114ff, 0x01f701ff, 0xfc16f7ff,
	0xfd08fc06, 0x05ed07ff, 0xfcfc1ff9, 0xfbfb00fa,
	0xfcedf8f9, 0x20040101, 0x04f8ff26, 0xf4faf8f9,
	0x01f900ff, 0x00ff09ff, 0x00ffff09, 0xfd01fa02,
	0x010200f4, 0x00080101, 0x02000109, 0x00f501fe,
	0xf6020800, 0x00ff02ff, 0xfb00fcfe, 0x0efffffe,
	0x05ff07fd, 0x0101f600, 0xff0efbff, 0xfefd01fe,
	0x060000fa, 0x04f70302, 0xfffffb04, 0xff0803f9,
	0xf5fffc02, 0x0001020b, 0x090302ff, 0xf1000200,
	0x03ec0503, 0x0303ff03, 0x0110ff01, 0x0209e302,
	0xfdfffaf3, 0xfdf8ff24, 0x02040502, 0x030b09db
};

static const uint32 s_svq1IntraCodebook4x4[384] = {
	0x0603fdf5, 0x0705fff6, 0x0706fff7, 0x0604fff7,
	0xf2000705, 0xf1020906, 0xf1020906, 0xf2000604,
	0xfafb0310, 0xf8f80110, 0xf7f7ff0e, 0xf8f8000c,
	0x11100c08, 0x090602fe, 0x00fcf8f6, 0xf9f5f2f1,
	0x10fef6f9, 0x12fdf5f9, 0x14fff5f9, 0x1301f8fa,
	0xeff0f3f7, 0xf7f9fe02, 0xff04080b, 0x070b0f10,
	0x0f0dfeea, 0x100efee8, 0x0f0dfce7, 0x0d0afae7,
	0x10161a1a, 0x03090f11, 0xf2f5fafe, 0xe4e4e7ec,
	0xebe7e5e5, 0xf9f5f1f0, 0x0d0c0803, 0x1e1f1c17,
	0xdff91014, 0xddfa1316, 0xdefa1316, 0xe0fa1114,
	0x2602ecec, 0x2802eaeb, 0x2802eaeb, 0x2603ecec,
	0x1a18fcd1, 0x1b1afdce, 0x1b1afdce, 0x1a18fcd1,
	0xe5e9062d, 0xe4e70530, 0xe4e60530, 0xe5e8062c,
	0x4cf6dce2, 0x4ef5dbe1, 0x4ef5dbe1, 0x4df6dce1,
	0x3423e0cb, 0x3424deca, 0x3424deca, 0x3322dfcb,
	0x413edea3, 0x423edea3, 0x413edea3, 0x403cdea3,
	0x020200f9, 0x0303fff8, 0x050400f8, 0x050501fa,
	0x0b0b0703, 0x03030202, 0xf9fafe01, 0xf3f5fb01,
	0xfdfcfe03, 0xfbfb0007, 0xf9fb040c, 0xf9fc060e,
	0xfe030e12, 0xfd000406, 0x00fefbf8, 0x02fef5f0,
	0x1207faf8, 0x0d02f8f9, 0x06fefafc, 0x01fdfc00,
	0xeef3fd01, 0xf9fbff00, 0x070601ff, 0x110f04fe,
	0xfef9f2f1, 0x00fffbfa, 0x01030606, 0x01060d0f,
	0x0af5fe02, 0x0bf4ff02, 0x0bf4ff03, 0x0bf5fe02,
	0xfbff0ef7, 0xfbfe0ff7, 0xfbfe10f8, 0xfcff0ff9,
	0x08080602, 0x0c0903fe, 0x0a04fbf5, 0x00f8f0ed,
	0xf1f9080e, 0xf2f9070c, 0xf7fc0508, 0xfcff0305,
	0x02fef20c, 0x03fff10d, 0x03fff10e, 0x03fff20d,
	0xf30a0600, 0xf10a0600, 0xef090700, 0xf0080601,
	0xfe0ffbf8, 0xfe11faf8, 0xfd10faf8, 0xfe0ffbf8,
	0xf6f5f5f7, 0x08090a09, 0x090a0a08, 0xf9f8f7f8,
	0x07090a09, 0xf6f6f6f8, 0xf5f5f6f9, 0x080b0c0b,
	0x00070a00, 0xfa000700, 0xfafb0200, 0xfffcfffe,
	0xf7fa0005, 0x01020202, 0x070500fe, 0x0401fbfa,
	0xff02f803, 0x0003f704, 0x0003f905, 0x0003fb07,
	0x0902fdfb, 0x0801fdfa, 0x0701fdfa, 0x0400fefb,
	0x0103080d, 0xfffcfbfd, 0x00fdf9f8, 0x020301ff,
	0xf4fb0203, 0xf7fe0304, 0xfc010403, 0xff040503,
	0x00fcf8f7, 0x00020608, 0x0003080a, 0xfffdfbfa,
	0xfbf4f7fd, 0x00fbfd00, 0x04020302, 0x06070805,
	0x0c05feff, 0x0905ffff, 0xfeff0102, 0xeff5ff02,
	0xff0303f9, 0xff0403f7, 0xff0604f6, 0x000705f7,
	0x0202f9ee, 0x030501f8, 0x00010403, 0xfdfe0509,
	0x080600fe, 0xfdfbfbfc, 0xf8fafe01, 0xff03090a,
	0x00fefe00, 0x00fbfc00, 0x08fcf8fe, 0x1806f9fb,
	0x01f90109, 0x01f80109, 0x01f60008, 0xfff5ff08,
	0x03060808, 0x02030405, 0x00fffdfe, 0xfcf8f3f6,
	0xfd020400, 0xfb030600, 0xf4020a03, 0xeafc0a05,
	0x03fffc00, 0x05fffc01, 0x0800fb01, 0x07fefaff,
	0xfcfeffff, 0xfafcfeff, 0xfeffffff, 0x090a0501,
	0xfe00030a, 0xfbfeff06, 0xfafeff03, 0xfb000002,
	0x00000306, 0x01010306, 0x01fefe04, 0xfef7f700,
	0x0201fdf5, 0x050402fa, 0x040302fd, 0x020101fe,
	0xfefffcfa, 0xfeffff02, 0xfefe020a, 0xfffc020b,
	0x02fe0006, 0x00000303, 0x000303fa, 0x0005ffef,
	0x0b0a04ff, 0x0100fefd, 0xfdfbfcfd, 0xfffefeff,
	0xf4f7fd02, 0x02030303, 0x04040202, 0xfeff0102,
	0xf60509fe, 0xfb0505fd, 0x000201fe, 0x01fefeff,
	0xfe07fdfe, 0xfd07fdff, 0xfc08feff, 0xfd07fefe,
	0x0cfdf801, 0x04fefe02, 0xfb000301, 0xf90205ff,
	0xfb0103ff, 0x0103fef9, 0x02fef9fe, 0xfffb0314,
	0xfefd0005, 0x0600f9f9, 0x060700fa, 0xf9000602,
	0x01f906fe, 0x03f807fe, 0x03f907fe, 0x02fa07ff,
	0x0705fefb, 0xf8fc0104, 0xfbfe0306, 0x0703fbf9,
	0x0506ffff, 0xfc01ff00, 0xf9000102, 0xfc000001,
	0x010300f8, 0xffff01fe, 0x01fdff01, 0x0901fe01,
	0xfcfd0205, 0xfdff00ff, 0x010301fd, 0x020400fc,
	0x0cfefe02, 0x03fbfe00, 0x01fd00ff, 0x01fefffd,
	0x00030501, 0x01fefcfa, 0x02fefe00, 0xfffc0106,
	0xfffbfbfd, 0x04050503, 0xff010300, 0xfdfe01fe,
	0xfdfbfc02, 0xfefdfe04, 0xffff0006, 0x00000107,
	0x00fefefd, 0xfffbfdfe, 0xff0002fe, 0xff090bff,
	0xf6ff0100, 0xfa0001ff, 0x04010001, 0x0dfffb02,
	0x000504fe, 0x030601fb, 0x0203fefa, 0xfe00fefb,
	0xfe0101ff, 0x0200feff, 0x07050505, 0xf9f8fc00,
	0xfbff0200, 0xfd0202ff, 0xfb030500, 0xf4020803,
	0xfe000408, 0xfffcff0a, 0x00fdfa03, 0x0000fbfc,
	0x02fcf600, 0x0503faff, 0x0406fdff, 0x0204fe00,
	0xff010800, 0xfd010b00, 0xfcfe06ff, 0xfcf9fefd,
	0xffffff00, 0x05060504, 0xfbf8f7fb, 0x02030202,
	0x01060200, 0x00030002, 0xfefffe01, 0xfafdff00,
	0x00020000, 0x01020004, 0x0000fe05, 0x02fff7fe,
	0xf6000100, 0x000801ff, 0x0004feff, 0xff02ff01,
	0xff02fefd, 0xfd02fffd, 0x0001ff00, 0x03ff0108,
	0x02010100, 0x00fefc00, 0xff01fbff, 0x020bfffe,
	0xfefe0501, 0x00fc0200, 0x01fb01fe, 0x01000500,
	0x0600fdfb, 0x000002fc, 0x000105fd, 0x000003fd,
	0x01fdfe03, 0x0800fc01, 0x03fefdfe, 0xffff0201,
	0x02000101, 0x06010002, 0x0102ff01, 0xed000300,
	0x02fefd01, 0xf9fe0506, 0x010301fd, 0x0200ffff,
	0xfcfffff8, 0x02ff0101, 0x03020304, 0x000301fb,
	0x01ff0200, 0x050000fd, 0x0800fefb, 0x06fcfcfc,
	0x02010201, 0x02fd0202, 0x00f70004, 0x01f50007,
	0xfe000000, 0xfaff0303, 0xf6fd0304, 0x020602ff,
	0x05fdfe07, 0xff0300fc, 0xf90102fc, 0x03ffff02,
	0x02020203, 0xfbf9f9fb, 0x02040605, 0x0100fffe
};

static const uint32 s_svq1IntraCodebook8x4[768] = {
	0x06060605, 0x08080707, 0x00000000, 0x03020100,
	0xfbfcfcfd, 0xfefdfcfb, 0xfbfcfcfc, 0xfdfdfcfc,
	0x02020201, 0x03030302, 0x04030302, 0x05050504,
	0x010100ff, 0x04040302, 0xf7f7f6f7, 0xfbfaf9f8,
	0xfafbfcfc, 0xf9f9f9fa, 0xfefeff00, 0xfcfcfdfd,
	0x03030404, 0x00010102, 0x06070707, 0x04040506,
	0x06050402, 0xfafd0104, 0x05050403, 0xf8fb0004,
	0x04040302, 0xf6f9fe02, 0x01020202, 0xf4f7fc00,
	0x01fdf9f7, 0x03030404, 0x03fef9f6, 0x03030505,
	0x03fefaf7, 0x03040506, 0x03fffaf8, 0x02030404,
	0xfbfbfbfb, 0x070401fd, 0xfcfbfbfb, 0x080601fe,
	0xfdfcfbfc, 0x0a0803ff, 0xfefdfcfd, 0x0b090501,
	0xfefefefe, 0xfefefefe, 0xfbfbfbfc, 0xfcfbfbfb,
	0xfcfcfcfd, 0xfdfcfcfc, 0x0b0a0a09, 0x0a0a0b0b,
	0xfe010407, 0xf6f7fafc, 0x00030709, 0xf7f8fcfe,
	0x0204080b, 0xf8fafd00, 0x0305090b, 0xf9fbfe01,
	0xf4f3f3f3, 0xf8f8f6f5, 0x03020100, 0x03040404,
	0x06050403, 0x04050606, 0x04040403, 0x02030303,
	0x0a0b0a0a, 0x07080909, 0x06060606, 0x02030405,
	0xff000000, 0xfcfcfdfe, 0xf4f5f6f6, 0xf2f2f2f3,
	0x10111010, 0x0b0c0d0f, 0xfcfdfeff, 0xfdfcfcfc,
	0xfafafbfc, 0xfafafafa, 0xfafafafb, 0xfbfbfafa,
	0xf4f3f3f3, 0xfaf8f6f5, 0xfaf9f8f7, 0x0100fefc,
	0x0301fffe, 0x09080705, 0x0b090705, 0x0f0f0e0d,
	0x070b0e10, 0xf7f9fd02, 0x03080c0e, 0xf5f7faff,
	0x0004090b, 0xf3f5f8fc, 0xfd010508, 0xf2f4f6fa,
	0xfdf7f1ee, 0x0b090601, 0xfff9f3ef, 0x0c0b0703,
	0x01fbf5f1, 0x0d0c0905, 0x02fdf7f3, 0x0d0b0905,
	0x0f131516, 0xf7fc030a, 0x090f1214, 0xeff4fb02,
	0x01080d10, 0xe8ecf2f9, 0xf8ff060a, 0xe5e7ebf1,
	0xf2ece9e7, 0x0e0901f9, 0xf7f0ebe9, 0x15100900,
	0xfff6f0ec, 0x19161008, 0x06fdf5f1, 0x1b19140e,
	0x0100fefc, 0x02020202, 0x0200fefb, 0x03030303,
	0x01fffcfa, 0x03030302, 0x00fefbf9, 0x02020101,
	0x01010102, 0xfdfe0001, 0x01020303, 0xfcfdff00,
	0x01020304, 0xfafcfe00, 0x01030405, 0xfafbfdff,
	0x04060605, 0xfdfe0002, 0x04040403, 0xff000103,
	0xfffffefe, 0xfefeffff, 0xfefdfcfb, 0xfdfdfefe,
	0xffffffff, 0xffffffff, 0xfcfcfcfd, 0xfdfdfdfd,
	0xffffffff, 0xfeffffff, 0x06060605, 0x03040506,
	0x04040404, 0x07060504, 0xffffff00, 0x020100ff,
	0xfdfdfdfe, 0x00fffefd, 0xfcfcfdfd, 0xfffefdfc,
	0xfcfcfe00, 0x030200fe, 0xfdfdfe00, 0x050402ff,
	0xfdfcfeff, 0x06050300, 0xfdfcfdfe, 0x050402ff,
	0xfd000409, 0x0100fffd, 0xfcff0408, 0x0201fffd,
	0xfbfd0206, 0x0100fefc, 0xfcfd0105, 0x0100fefc,
	0xff010305, 0xf6f6f8fc, 0x01020303, 0xfcfdfe00,
	0x02010101, 0x00010203, 0x020100ff, 0x02030403,
	0x02020100, 0xfdfdff01, 0x01010100, 0xfdfcfeff,
	0xfdfdfdfd, 0x02fffdfd, 0x00fdfcfc, 0x0e0c0703,
	0xfafafbfb, 0xfbfafafa, 0x01020202, 0x00000000,
	0x02030404, 0x00000001, 0x04050606, 0x01010202,
	0xfdfaf9f9, 0x08070400, 0xfdfdfeff, 0x0201fffe,
	0xff010303, 0xfffefefe, 0x02040606, 0xfefefe00,
	0x02fefbfa, 0x0c0b0905, 0x00fefcfc, 0x06050402,
	0xfefefefd, 0x0100fffe, 0xfdfefefe, 0xfefdfdfd,
	0x0301fdf9, 0xfbfd0003, 0x0503fefa, 0xfbfd0104,
	0x0604fffb, 0xfcfd0205, 0x070500fc, 0xfdff0306,
	0x00000000, 0x00000000, 0xfdfdfefe, 0xfffefdfd,
	0x09080706, 0x06070809, 0xfbfbfcfc, 0xfcfbfafa,
	0xfcfaf8f7, 0x06060300, 0x03fffcfb, 0x03050605,
	0x06060301, 0xfbfe0104, 0x01050706, 0xf5f6f9fd,
	0x0105090a, 0xfcfafafd, 0xfbff0305, 0x02fefbfa,
	0xfafafcfe, 0x0a0601fc, 0xfcf9f9fa, 0x0c0b0701,
	0x02030506, 0x00000000, 0xfeff0102, 0xfffffefd,
	0xfcfeff00, 0x01fffefc, 0xfeff0000, 0x030200ff,
	0xfefeff00, 0xfffffefe, 0x01020405, 0x00000000,
	0x01030506, 0x00000000, 0xfcfe0002, 0xfefefdfc,
	0x0200fcf9, 0x01020202, 0x0000fdf9, 0x00000000,
	0x0101fffc, 0x01000000, 0x020201ff, 0x03030202,
	0x020200fe, 0x01010101, 0x020201ff, 0xff000001,
	0x02040200, 0xfdfeff00, 0x01030201, 0xfafafcfe,
	0x04020201, 0x01040605, 0xffffff00, 0xfcfe0000,
	0xfeff0000, 0xfafcfefe, 0x00000102, 0xfdff0101,
	0x01010101, 0x03030201, 0x00010000, 0x04040201,
	0xffffffff, 0x03020100, 0xfbfbfcfc, 0x00fffdfb,
	0xfcfbfbfa, 0xfffffefd, 0x010000ff, 0x03030201,
	0x01010100, 0x04030202, 0xffff0000, 0x03020100,
	0x01010100, 0xffff0000, 0x02030301, 0xfefeff01,
	0x020200fe, 0x01010202, 0xfefcf8f7, 0x03030301,
	0xfeffffff, 0xfcfdfdfd, 0xff000000, 0xfdfdfefe,
	0x00020202, 0xffffffff, 0x03040505, 0x02020202,
	0xfcff0306, 0x0101fffd, 0xfcfdff02, 0x000202ff,
	0x01fefeff, 0xfd010404, 0x0401fffe, 0xf8fd0306,
	0x01020303, 0xfefefeff, 0xfffefcfc, 0x04040301,
	0xfcfbfbfc, 0x020200fe, 0x01040707, 0xfefdfeff,
	0x000301ff, 0x0600fafc, 0x010401fe, 0x07fffafc,
	0x020401fd, 0x06fffafd, 0x020300fe, 0x04fffbfe,
	0x01feff01, 0xf9ff0404, 0xfffcff01, 0xfa000605,
	0xfdfc0003, 0xfc020603, 0xfcfb0003, 0xfd010401,
	0x03030202, 0x02020303, 0xf9fafbfc, 0xfaf9f9f9,
	0x03030201, 0x02020303, 0x01010000, 0x01020201,
	0x03fdfd03, 0x02fefe04, 0x04fcfc03, 0x02fcfc04,
	0x04fcfc04, 0x03fdfc04, 0x03fcfd03, 0x03fdfd03,
	0xfefefefe, 0xfffffefe, 0x08080706, 0x05060708,
	0xf9f9fafb, 0xfbfaf9f8, 0x02020101, 0x01010202,
	0x00000000, 0x0000ff00, 0x000000ff, 0x0000ff00,
	0xfefefdfe, 0xfdfdfdfe, 0x06050302, 0x00010204,
	0x00020608, 0x00000000, 0x00000104, 0xffffff00,
	0x0000ff01, 0xfdfeff00, 0x00fffefe, 0xfbfcfe00,
	0xfeff0103, 0xfbfbfcfd, 0x00000102, 0x00000101,
	0x00ffff00, 0x02020202, 0x01fffeff, 0x02020202,
	0xfffeff00, 0x00ffffff, 0xfffefeff, 0x010000ff,
	0x02010102, 0x00010102, 0x01030506, 0xfcfcfe00,
	0x00fffefd, 0xff000101, 0x04030100, 0x01030505,
	0x00ffffff, 0xfeff0001, 0xfffefefe, 0xfdfeff00,
	0xfefeff00, 0x0200ffff, 0xfffeff01, 0x0200ffff,
	0xfefe0001, 0x0501fefe, 0xfefeff01, 0x0a0500fe,
	0x00000000, 0xffffff00, 0x00ffffff, 0x02010000,
	0x03020201, 0x05060404, 0xfefdfdfd, 0xfdfdfdfe,
	0xfefeff01, 0x07050300, 0xfdfe0002, 0x030200fe,
	0xfdfe0103, 0xfffffefd, 0xff000103, 0xffffffff,
	0x04050301, 0xfcfdff02, 0x0201fefd, 0xfeff0001,
	0x0200fdfb, 0x00000102, 0x0201fffd, 0x00000102,
	0xffffff00, 0x04030201, 0xfdfcfcfd, 0x010000ff,
	0xfffefdfe, 0x01010101, 0x0300fefe, 0x02030404,
	0xfefdfcfc, 0x030201ff, 0x01010100, 0xfdfdfeff,
	0x04050403, 0xfdfdff02, 0x0200fefe, 0x00010202,
	0x070500fc, 0xfcfcff04, 0x030402ff, 0xfefdfd00,
	0xff000102, 0x0100fefe, 0xfeff0000, 0x0201fffe,
	0xfffefdfc, 0x02020100, 0x0005090a, 0xfefdfcfd,
	0xfefeff01, 0x000000ff, 0x01fffefe, 0xff000101,
	0x0300fdfb, 0xfe000204, 0x0100fffe, 0xffff0001,
	0xfeff0203, 0x0101fffe, 0xfbff0507, 0x0402fefa,
	0xfd0303fe, 0xfe0201fc, 0xfd0403fd, 0xfe0302fc,
	0xfd0403fd, 0xfe0302fc, 0xfe0402fc, 0xff0201fd,
	0xfdff0304, 0x0201fffd, 0x00fcfafc, 0x01040504,
	0x06050200, 0xfcfbfd02, 0xfdff0101, 0x0402fefb,
	0x020100ff, 0x04030302, 0x010100ff, 0xffffff00,
	0x020100ff, 0xfeff0102, 0x00fffefd, 0xfdfeff00,
	0x01010101, 0x02010000, 0x00ff0001, 0x00010100,
	0xfffcfe01, 0x00010201, 0xfdf9fc01, 0x01020301,
	0x01010101, 0xff000101, 0x00010101, 0x00020201,
	0x00000101, 0xfd000200, 0xff000203, 0xf7fafeff,
	0x01000000, 0x02010000, 0x00000001, 0x0200ffff,
	0x01010100, 0x00fefdff, 0x0601fbf9, 0xffff0206,
	0xfdff0103, 0x0401fefc, 0xfdfe0002, 0x02fffdfc,
	0x01010202, 0x01000001, 0x00000101, 0x01000000,
	0xfe0101ff, 0xfffcfafb, 0x030401ff, 0x02010002,
	0x030200ff, 0x01000001, 0x000100ff, 0x00ffff00,
	0x02020100, 0x01fffe00, 0xfefffffe, 0x080602ff,
	0xfdfeffff, 0x020100fe, 0xff0000ff, 0xffff00ff,
	0x01010102, 0x00000001, 0x01010000, 0x01ffff01,
	0x020200ff, 0x03fefdff, 0x00030200, 0x04fef9fb,
	0x000000ff, 0xfdfdfeff, 0xfeff00ff, 0xfefefefe,
	0x00000101, 0xff000201, 0x02010201, 0x00020605,
	0x00fdfcfe, 0xfd000202, 0x01000103, 0xfdfe0102,
	0x00000103, 0xff000000, 0xfefeff01, 0x030301ff,
	0x02010203, 0xfe010304, 0xfdfcfcfe, 0xfdfe00ff,
	0xffff0001, 0xff000100, 0x00000203, 0x00010100,
	0x00000101, 0x00000000, 0x02030302, 0x01010202,
	0xfdfeff00, 0xfcfbfbfb, 0xff000101, 0x03030100,
	0x00fefaf7, 0x02020101, 0x0201fefa, 0x01000101,
	0x020201fe, 0x01010101, 0x01020200, 0x01010100,
	0x00000001, 0x00ff0000, 0x00000000, 0x00fefdff,
	0xfefdfdfd, 0x090703ff, 0x02020201, 0xfdfcfe00,
	0xfffe0002, 0xfaff0403, 0xfdfe0001, 0x000303ff,
	0x00030300, 0x0101fffe, 0x0203fffa, 0x0100feff,
	0xfe000305, 0x010200fd, 0x02020101, 0xf9fcfe00,
	0x0201fefd, 0xfcff0102, 0xfe000202, 0x020200fe,
	0xfdfe0000, 0x0000fffe, 0x00000000, 0x02020000,
	0x0100fffe, 0x03020100, 0x0000fefc, 0x030200ff,
	0xfffefefe, 0x040200ff, 0x00000000, 0x0100ffff,
	0xffffff00, 0x0000ffff, 0x00020406, 0xfffffeff,
	0x01010100, 0xf6fbff01, 0x01010101, 0xfc000101,
	0x01010001, 0xff010101, 0x01010102, 0x00000000,
	0x030401fd, 0x00ff0103, 0x000100fc, 0x000000ff,
	0x010200fb, 0xff000101, 0xfe0102ff, 0xff00fffe,
	0x03050402, 0x0201ff00, 0x00010000, 0xfffffefe,
	0xfefefefe, 0x00fffefd, 0x00010000, 0x02010000,
	0xfdfefe00, 0xff0202ff, 0x00000001, 0xfe030501,
	0xff00ffff, 0xfb000200, 0x000100ff, 0xfe020200,
	0xffff0103, 0x02010100, 0x01000001, 0x01010101,
	0x01fef8f6, 0x01010102, 0x010201ff, 0x00000000,
	0x0100ffff, 0x01020202, 0x00ffffff, 0xfcfbfdff,
	0x01020101, 0x02000001, 0xfffffeff, 0x040200ff,
	0x00fbf9fd, 0x00000002, 0x01feff03, 0x02010102,
	0x01fffe01, 0x01000102, 0x0300ff00, 0xffffff02,
	0x00010102, 0x00000000, 0x03fef9f7, 0x01010203,
	0xfe000203, 0x0101fffe, 0x0000ff00, 0x00000101,
	0x0101fffe, 0x00000001, 0xfe010201, 0x0201fdfc,
	0xfe010201, 0x010300fd, 0x0000ffff, 0xfc000301,
	0x01ff0002, 0x03fefe02, 0x02ff0002, 0x01fcfe03,
	0x01010100, 0xfefafe02, 0x000000ff, 0xfffe0002,
	0x0201ffff, 0xfefdfe01, 0xfffeff03, 0x020100ff,
	0x0000040a, 0xfffefeff, 0xfffeff03, 0x00ffff00,
	0x010702fb, 0x0001fefc, 0xff0302fe, 0x000200fd,
	0x00000102, 0xfeff0101, 0xfffefe01, 0x0000feff,
	0xf9fe0300, 0x000003ff, 0xfbfd0301, 0x00ff0302,
	0xfefe0200, 0x00fe0204, 0x00ff01ff, 0x01feff02,
	0xfcfd0004, 0x010201fe, 0x05030000, 0xfeff0103,
	0xff010101, 0x0101fffd, 0xfefeff01, 0xfeff0000
};

static const uint32 s_svq1IntraCodebook8x8[1536] = {
	0x02030404, 0xff000102, 0x02030304, 0xffff0001,
	0x02020303, 0xfeff0001, 0x01020203, 0xfdfeff00,
	0x00010202, 0xfdfeffff, 0x00000102, 0xfcfdfeff,
	0xff000001, 0xfcfcfdfe, 0xfeff0000, 0xfcfcfdfe,
	0x03030302, 0x03030303, 0x02020202, 0x03030202,
	0x02020201, 0x02020202, 0x01010100, 0x01010101,
	0x000000ff, 0x01010000, 0xfffffefe, 0xffffffff,
	0xfdfdfdfd, 0xfefefdfd, 0xfcfcfcfb, 0xfdfcfcfc,
	0x00fffefc, 0x03020201, 0x00fffefc, 0x03030201,
	0x00fffdfc, 0x03030201, 0x00fffdfc, 0x03030201,
	0x00fffdfb, 0x03030201, 0x00fffdfb, 0x03030201,
	0x00fffdfb, 0x03020101, 0xfffefdfb, 0x03020100,
	0x05050404, 0x07070606, 0x03020202, 0x04040403,
	0x00000000, 0x02010101, 0xfefefefe, 0x00ffffff,
	0xfefefefe, 0xfefefefe, 0xfefefefe, 0xfefefefe,
	0xfefefeff, 0xfefefefe, 0xffffffff, 0xfefefefe,
	0xff010305, 0xfdfdfdfe, 0xff010305, 0xfdfdfdfe,
	0xff010305, 0xfdfdfdfe, 0xff010305, 0xfdfdfdfe,
	0x00010405, 0xfdfdfdfe, 0x00020406, 0xfdfdfefe,
	0x00020406, 0xfdfefeff, 0x01020406, 0xfefefeff,
	0x030301ff, 0xfafd0002, 0x040301ff, 0xfafd0003,
	0x040401ff, 0xfafd0103, 0x040301ff, 0xfafd0103,
	0x040301fe, 0xfafd0103, 0x040301fe, 0xf9fd0103,
	0x030301fe, 0xf9fd0002, 0x030200fe, 0xfafd0002,
	0x0608090a, 0x04040506, 0x03040506, 0x01020202,
	0xff000102, 0xfffefefe, 0xfcfdfeff, 0xfdfcfcfc,
	0xfcfcfdfe, 0xfdfcfcfb, 0xfdfdfefe, 0xfefefdfd,
	0xffffffff, 0x00ffffff, 0x01010101, 0x02010101,
	0x0201fffe, 0x08070504, 0x0100fefd, 0x08070503,
	0x00fffdfc, 0x07060402, 0xfffefcfb, 0x07050301,
	0xfefdfbfa, 0x06040200, 0xfefcfbfa, 0x050301ff,
	0xfdfbfaf9, 0x030200fe, 0xfcfbf9f8, 0x0201fffd,
	0x0507090b, 0xffff0103, 0x0306080a, 0xfefe0001,
	0x02050709, 0xfcfdfe00, 0x01030608, 0xfcfcfdff,
	0xff020406, 0xfbfbfcfd, 0xfe000305, 0xfafafbfc,
	0xfdff0103, 0xf9f9fafb, 0xfcfe0002, 0xf9f9fafa,
	0x07070605, 0x08080807, 0x05050403, 0x06060606,
	0x03020200, 0x05040404, 0x0100fffe, 0x03030202,
	0xfffefdfc, 0x02010100, 0xfdfcfbfa, 0x00fffefe,
	0xfafaf9f8, 0xfdfdfcfb, 0xf8f8f7f6, 0xfbfafaf9,
	0x01030506, 0xf8fafdff, 0x02040506, 0xf8fafdff,
	0x02040506, 0xf8fafd00, 0x02040506, 0xf8fafd00,
	0x02040606, 0xf8fafd00, 0x02040506, 0xf8fafd00,
	0x02040506, 0xf8fafd00, 0x02040506, 0xf8fbfdff,
	0x08090a0b, 0x04050607, 0x06070808, 0x02030405,
	0x04040506, 0x00010202, 0x01020303, 0xfeff0000,
	0xff000101, 0xfdfdfefe, 0xfdfeffff, 0xfbfbfcfc,
	0xfbfcfcfd, 0xf9f9fafa, 0xf9fafbfb, 0xf8f8f8f8,
	0xf5f4f3f2, 0xfcfaf9f7, 0xf7f6f5f4, 0xfffdfbf9,
	0xfaf9f7f6, 0x0200fefd, 0xfefcfaf8, 0x05040200,
	0x0200fdfb, 0x08070504, 0x040200fe, 0x0a090806,
	0x07050300, 0x0c0b0a08, 0x08070503, 0x0c0c0b0a,
	0xeeeeeded, 0xf2f1f0ef, 0xf3f2f1f1, 0xf7f6f5f4,
	0xf8f7f6f5, 0xfdfcfbfa, 0xfefdfbfa, 0x020100ff,
	0x030200ff, 0x06060504, 0x08070604, 0x0a0a0a09,
	0x0c0b0a09, 0x0e0e0e0d, 0x0f0e0e0c, 0x10101010,
	0x11131516, 0x05090b0e, 0x0e111314, 0x0104080b,
	0x0a0d0f11, 0xfc000306, 0x05080b0d, 0xf7fbfe01,
	0xff030609, 0xf3f5f9fc, 0xfafd0004, 0xeff1f4f7,
	0xf5f8fbfe, 0xeceef0f2, 0xf0f3f6f8, 0xeaebedef,
	0x12121211, 0x0e101011, 0x0f0f1010, 0x0b0c0d0e,
	0x0a0b0c0c, 0x05070809, 0x04060607, 0xff010203,
	0xfeff0001, 0xfafbfcfd, 0xf8f9fafb, 0xf4f5f6f7,
	0xf2f3f4f5, 0xeff0f0f1, 0xeeefeff0, 0xecececed,
	0x00000000, 0xfdfeffff, 0x00000001, 0xfdfeff00,
	0x00000101, 0xfefeffff, 0x00010101, 0xfeffff00,
	0x01010102, 0xfeffff00, 0x01010202, 0xfeff0000,
	0x01010202, 0xffff0001, 0x01010202, 0xfe000001,
	0x00ffff00, 0x03020100, 0x00ffff00, 0x02020101,
	0xffffffff, 0x02020100, 0xfffeffff, 0x02010100,
	0xfffefeff, 0x02010000, 0xfefefeff, 0x020100ff,
	0xfffeffff, 0x02010000, 0xffffffff, 0x02010100,
	0x02020203, 0x00000101, 0x02020203, 0x00000102,
	0x01020202, 0x00000101, 0x01010202, 0xff000001,
	0x00010101, 0xffff0000, 0xffff0000, 0xffffffff,
	0xfefefefe, 0xfefefefe, 0xfdfdfdfe, 0xfefefefe,
	0x00000205, 0x000000ff, 0xff000204, 0xff00ffff,
	0xffff0104, 0xfffffffe, 0xffff0104, 0xfffffffe,
	0xfeff0104, 0xfffffffe, 0xfeff0104, 0xfffffffe,
	0xffff0104, 0xffffffff, 0xff000204, 0xff000000,
	0x0100fffe, 0x01010101, 0x0100fffd, 0x01010101,
	0x0100fffd, 0x01010101, 0x0100fffd, 0x01010101,
	0x0100fefd, 0x01010202, 0x0100fefc, 0x02020202,
	0x01fffdfb, 0x02010201, 0x00fefdfb, 0x01010101,
	0x00010303, 0xfbfcfcfe, 0x00020303, 0xfcfdfeff,
	0x01010202, 0xfefeff00, 0x01010101, 0x00000001,
	0x01000000, 0x01010101, 0x00fffffe, 0x02020100,
	0xfffefefd, 0x03020100, 0xfffefdfd, 0x03020100,
	0xfdfdfdfd, 0xfefefefd, 0xfefefdfd, 0xfffffffe,
	0xfffefefe, 0xffffffff, 0x00ffffff, 0x00000000,
	0x00000000, 0x01010101, 0x01010000, 0x02020202,
	0x02010101, 0x03030302, 0x02020202, 0x03030303,
	0xfdfbf9f8, 0xff00fffe, 0x00fffdfc, 0x01010201,
	0x030201ff, 0x01020203, 0x03030201, 0x00010202,
	0x02030302, 0xff000001, 0x00010201, 0xffffffff,
	0xff000101, 0xfffefeff, 0x00000101, 0xff00ffff,
	0x00fefdfc, 0x03030201, 0x00fefdfc, 0x02020201,
	0xfffefdfd, 0x01010100, 0xfffefefe, 0x000000ff,
	0xffffff00, 0xffffffff, 0x00010102, 0xfeffff00,
	0x01030303, 0xfefeff00, 0x02040405, 0xfeff0001,
	0x00000000, 0x03030201, 0x0000ff00, 0x03030201,
	0x0000ff00, 0x02030201, 0x01000000, 0x02020201,
	0x01010102, 0x00010101, 0x01020202, 0xfeff0000,
	0x00000102, 0xfafbfdfe, 0xfdffff00, 0xf7f8fafb,
	0x020100fe, 0xfcff0102, 0x020200fe, 0xfcff0102,
	0x020200fe, 0xfdff0102, 0x020200fe, 0xfdff0102,
	0x0202fffe, 0xfdff0102, 0x0201fffe, 0xfdff0102,
	0x0201fffd, 0xfdff0102, 0x0201fffe, 0xfdff0102,
	0xff0101ff, 0x0400fdfd, 0xff0101ff, 0x0400fdfd,
	0x000101ff, 0x0400fdfd, 0x000201ff, 0x0500fdfd,
	0x00020100, 0x0400fcfd, 0x00020100, 0x0500fcfd,
	0x00020100, 0x0400fdfd, 0xff020100, 0x0400fefe,
	0x06050606, 0x05050505, 0x02020202, 0x02020202,
	0x00000000, 0x00000000, 0xffffffff, 0xfefefefe,
	0xfefefefe, 0xfefefefe, 0xfefefefe, 0xfefefefe,
	0xfffeffff, 0xffffffff, 0xffffffff, 0xffffffff,
	0x02020202, 0x02020202, 0x00010100, 0x00000000,
	0xfefefeff, 0xfffefefe, 0xfdfdfdfd, 0xfefdfdfd,
	0xfdfcfcfd, 0xfefefdfd, 0xfefefefe, 0x0000ffff,
	0x01010100, 0x03030202, 0x05040403, 0x06060605,
	0xfdfe0104, 0x0301fffd, 0xfcfe0104, 0x0301fffd,
	0xfcfe0105, 0x0401fffd, 0xfdfe0105, 0x0402fffd,
	0xfdfe0105, 0x0402fffd, 0xfcfd0004, 0x0402fffd,
	0xfdfd0004, 0x0301fffd, 0xfdfe0003, 0x0301fffe,
	0xfcfcfcfd, 0xfcfcfcfc, 0xffffffff, 0xfefeffff,
	0x02010102, 0x01010102, 0x04030303, 0x03030304,
	0x04030303, 0x03030404, 0x02010201, 0x02020202,
	0xfffefefe, 0x0000ffff, 0xfcfcfcfc, 0xfdfdfdfd,
	0xfdfdfeff, 0x00fffefe, 0xfefeff00, 0x0100fffe,
	0xffff0102, 0x0100ffff, 0xff000103, 0x010000ff,
	0xff000203, 0x01000000, 0xff000103, 0x01000000,
	0xff000103, 0x01000000, 0x00000102, 0x01000000,
	0x01000000, 0x04030201, 0x00ff0000, 0x03020000,
	0xffffff00, 0x0100ffff, 0xffffff00, 0x00ffffff,
	0xffff0000, 0xfffefeff, 0xff000001, 0xfffefeff,
	0x00010202, 0xffffffff, 0x01020303, 0x00ffff00,
	0x00010001, 0xfffeff00, 0x00000000, 0xfffeffff,
	0x0000ff00, 0xffffffff, 0x00ffffff, 0x00000000,
	0x00ffffff, 0x01010000, 0x00ffffff, 0x03020101,
	0x00fffefe, 0x04030201, 0x00fffefe, 0x05040201,
	0x0001fffd, 0x0100ff00, 0x000100fd, 0x0200ffff,
	0x000100fd, 0x0200ffff, 0x000201fe, 0x0200ffff,
	0x000201fe, 0x0200ffff, 0x000201fe, 0x0200ffff,
	0x000202ff, 0x0200ffff, 0x000101ff, 0x01ffffff,
	0x01fffefe, 0x01030403, 0x00fffefe, 0x00020302,
	0x00fffefe, 0xff010201, 0x00ffffff, 0xff010201,
	0x00ffffff, 0xfe000101, 0x00ffff00, 0xff000101,
	0x00ffff00, 0xff010101, 0x00ffff00, 0xff000100,
	0x0100fffe, 0x01010101, 0x0000fffe, 0x00000000,
	0x00fffffe, 0xfefeffff, 0xfffffffe, 0xfdfefeff,
	0x010100ff, 0xfefeff00, 0x03030201, 0x00000102,
	0x03030201, 0x00010203, 0x01010000, 0x00000101,
	0xffffff00, 0x00000000, 0x00000001, 0x01010000,
	0x01010101, 0x01010101, 0x02020101, 0x01010101,
	0x01010101, 0x01010101, 0x010000ff, 0x00000001,
	0xfffffefd, 0xff00ffff, 0xfdfcfbfb, 0xfffefefe,
	0x01010101, 0xff000102, 0x02010101, 0xff000101,
	0x01010101, 0xfe000101, 0x01010102, 0xfe000101,
	0x00000101, 0xfdff0000, 0x00000101, 0xfdfeff00,
	0x00000101, 0xfcfeffff, 0xff000001, 0xfcfdfefe,
	0x03050708, 0x01010102, 0x00000102, 0xfffeffff,
	0xfeffffff, 0xfffefefe, 0xffffffff, 0x00ffff00,
	0x01000000, 0x00000001, 0x01010000, 0x00000001,
	0x000000ff, 0xffff0000, 0xfffffefe, 0xfffefeff,
	0xfe000409, 0xfffffefe, 0xfeff0207, 0x0000fffe,
	0xfefe0004, 0x010100ff, 0xfefefe01, 0x010100ff,
	0xfffefeff, 0x01010100, 0x00fffeff, 0x00010101,
	0x0100ffff, 0xff000101, 0x0100ff00, 0xffff0000,
	0x01010100, 0x00000101, 0x02020201, 0x00000001,
	0x02020202, 0xffff0001, 0x00010101, 0xfefefeff,
	0xff000000, 0xfefefdfe, 0xfeffffff, 0x00fffefe,
	0xffffffff, 0x02010000, 0x00ffffff, 0x04030201,
	0x0000ffff, 0xfdfdfeff, 0x0000ffff, 0xffffff00,
	0x00fffefe, 0x01010101, 0x00fefefe, 0x03030201,
	0x00ffffff, 0x03030301, 0x00000001, 0x02020101,
	0x00010202, 0xffffff00, 0x00010203, 0xfdfdfeff,
	0xfeffffff, 0xfbfcfdfe, 0xff000000, 0xfcfdfdff,
	0x00010101, 0xfdfeff00, 0x01020202, 0xffff0001,
	0x02020202, 0xff000101, 0x02020202, 0x00000102,
	0x01020101, 0x00000101, 0x01010000, 0xff000000,
	0x010302fe, 0xff0101ff, 0x000302fd, 0xff0101ff,
	0x000302fd, 0xff0101ff, 0x000302fc, 0xfe0101ff,
	0x000301fc, 0xfe0101ff, 0xff0301fc, 0xfe0101fe,
	0x000201fd, 0xfe0101ff, 0x000201fd, 0xff0101ff,
	0xfeffffff, 0xfefefefe, 0x01010101, 0x00000000,
	0x02020201, 0x02020202, 0x01010000, 0x02020201,
	0xfffffefe, 0x000000ff, 0xfdfdfdfd, 0xfefdfdfd,
	0xffffffff, 0xfefefefe, 0x04040404, 0x02030304,
	0xfffefdfd, 0x05020100, 0xfefdfdfd, 0x060301ff,
	0xfefefdfd, 0x05030200, 0xfefefefd, 0x05030100,
	0xfffefefe, 0x050301ff, 0xfffffefe, 0x04020100,
	0xffffffff, 0x04010100, 0xffffffff, 0x03020100,
	0x0100ff00, 0xffff0001, 0x01000000, 0xffff0002,
	0x00ff0001, 0x00000001, 0xfffeff01, 0x00000000,
	0xfffdfe01, 0x01000000, 0xfefdff01, 0x02010100,
	0xfffeff01, 0x02010100, 0x00ff0002, 0x02020101,
	0x01010101, 0x02010000, 0xff0000ff, 0x01000000,
	0xfffffefd, 0x010100ff, 0x00fffefc, 0x01010100,
	0x0000fefd, 0x01010101, 0x0100fffd, 0x00000101,
	0x010100ff, 0xff000001, 0x02020100, 0xff000001,
	0xfdfcfcfc, 0xfffffffe, 0xfffefefe, 0x00000000,
	0x000000ff, 0x01010101, 0x01010000, 0x01010101,
	0x01010000, 0x00010202, 0x01010000, 0x00010101,
	0x01000000, 0x00010101, 0x010000ff, 0x00000101,
	0x02020201, 0xfcfeff01, 0x02020101, 0xfcfe0001,
	0x01010100, 0xfdff0001, 0x010100ff, 0xfeff0000,
	0x010100ff, 0xff000001, 0x0000fffe, 0xff000000,
	0x0100ffff, 0x00000001, 0x010100ff, 0x00010101,
	0xff000202, 0xfefffffe, 0xfeff0101, 0xfefffffe,
	0xfeff0101, 0xff0000fe, 0xfe000101, 0x000101ff,
	0xff000101, 0x010201ff, 0xff000101, 0x010201ff,
	0xff000101, 0x010101ff, 0xff000101, 0x01010100,
	0xfeff0000, 0xfcfcfcfc, 0x02030303, 0x00000001,
	0x03030303, 0x02020202, 0x00000000, 0x01000000,
	0xffffffff, 0x00ffffff, 0x0000ff00, 0x000000ff,
	0x00000000, 0x00000000, 0x00000000, 0x00ffff00,
	0xff00ffff, 0xff0201ff, 0x00000101, 0xff030200,
	0xff000101, 0xff0301ff, 0xfe000101, 0xfe0100fe,
	0xfe000001, 0xfd0100fe, 0x00000000, 0xfd0101ff,
	0x00010100, 0xfd010201, 0x010100ff, 0xfc010201,
	0x0100fdfc, 0x00000101, 0x0100fefc, 0xff000101,
	0x0101fffd, 0xffff0001, 0x010101ff, 0x00ff0001,
	0x01020201, 0x0000ff00, 0x00010202, 0x0100ffff,
	0xff000102, 0x0100fffe, 0xff000202, 0x0101fffe,
	0x00000101, 0xffffffff, 0xffffff00, 0xffffffff,
	0xffffffff, 0xffffffff, 0xffff0000, 0xffffffff,
	0xff000001, 0xffffffff, 0x00000102, 0xffffffff,
	0x01020305, 0x00000000, 0x02030506, 0x00000001,
	0x01030404, 0x01000000, 0x01020303, 0x01000000,
	0x00010202, 0x0100ffff, 0xff000000, 0x0100ffff,
	0xffff0000, 0x0200fffe, 0xfeffff00, 0x0100fefe,
	0xfeffff00, 0x00fffefe, 0xfeff0000, 0x00fffefe,
	0xffff0000, 0x030200ff, 0xfefeff00, 0x0201ffff,
	0xffff0001, 0x000000ff, 0x00010101, 0xffff0000,
	0x00010201, 0xffffff00, 0x000000ff, 0xffffffff,
	0xfffffefd, 0x02010100, 0x01fffdfc, 0x05050302,
	0x00000000, 0x00000000, 0x01010101, 0x01010101,
	0xff000000, 0x01000000, 0xfefeffff, 0x00fffffe,
	0x00000000, 0x01010100, 0x02020202, 0x03030302,
	0x01010101, 0x01010202, 0xfcfcfdfc, 0xfdfdfcfc,
	0x020100ff, 0x03030302, 0xffffffff, 0x00000000,
	0xfeff0000, 0xfefdfdfe, 0x00010203, 0xfefefeff,
	0x01020304, 0x00000001, 0x01010202, 0x01010100,
	0xffffff00, 0x010000ff, 0xfefefefe, 0x0000fffe,
	0x0200ff01, 0x01fffe01, 0x0200ff01, 0x01fefe01,
	0x0300ff01, 0x01fffe02, 0x0300fe00, 0x01fefe02,
	0x0300fe00, 0x01fefe02, 0x0300fe00, 0x01fffe01,
	0x0200fe00, 0x01fefe01, 0x0200ff00, 0x01fffe01,
	0x02020100, 0x02020303, 0x02010100, 0x01020303,
	0x02010000, 0x01020202, 0x010000ff, 0x01010101,
	0x0000ffff, 0x00000000, 0xffffffff, 0xffffffff,
	0xfefefefe, 0xfffefefe, 0xfefefefe, 0xfffefefe,
	0xfeff0000, 0x050300ff, 0xffff0000, 0x040200ff,
	0x00000101, 0x0201ffff, 0x01010201, 0x00ffff00,
	0x01020100, 0xfefeff00, 0x020100ff, 0xfdfd0001,
	0x0201fffe, 0xfcfe0002, 0x0200fffe, 0xfdff0102,
	0x00000000, 0xffff0000, 0x00ff0000, 0x00000000,
	0xffffffff, 0x00000000, 0xffffffff, 0x00ffffff,
	0xffffffff, 0x00ffffff, 0x000000ff, 0x00ffff00,
	0x01010000, 0x01000000, 0x04030303, 0x03030303,
	0xfefe0105, 0xff000000, 0xfffdff04, 0xff000000,
	0x00ffff03, 0xff000101, 0x01000002, 0xfe000101,
	0x01000001, 0xfe000101, 0xffffff00, 0xff000000,
	0xffffff00, 0xff0000ff, 0x00000102, 0x00000100,
	0x01010001, 0x00000101, 0x01000001, 0x00000001,
	0x00ffff01, 0x00000000, 0xffff0002, 0x00ffffff,
	0xffff0103, 0xfffefefe, 0x00010204, 0xfffefeff,
	0x00000102, 0x0000ffff, 0xffffff00, 0x010100ff,
	0x02020100, 0xfdff0102, 0x01010000, 0xfeff0001,
	0x00000000, 0xffff0000, 0x00ff0000, 0x00010100,
	0x00ff0000, 0x01010101, 0x00000000, 0x00010101,
	0x01010000, 0xfdff0102, 0x01000000, 0xfbfcff01,
	0xfffefefe, 0x02020200, 0x00000000, 0x00010101,
	0x01010101, 0xfdfe0001, 0x01010000, 0xfcfdff00,
	0x0100ffff, 0xfdfe0000, 0x0100ffff, 0xff000101,
	0x01000000, 0x00010101, 0x01010101, 0x00000000,
	0x00000100, 0x02010101, 0x00000201, 0x01ff0000,
	0xff000200, 0x00ff0001, 0x00000100, 0x01000102,
	0x00ff0100, 0x01000202, 0xffff00ff, 0x02010102,
	0xfefdfefe, 0x01010100, 0xfdfdfefe, 0x00ffffff,
	0x0100fffd, 0x00010102, 0x0100fffd, 0x01010102,
	0x010000fe, 0x01010101, 0x000000ff, 0x00000000,
	0x000000fe, 0x00ffff00, 0x000000fe, 0xffff0000,
	0x010100fd, 0x01000101, 0x0100fefb, 0x02010202,
	0x00fffffe, 0x03020100, 0x01010000, 0x02010000,
	0x00010000, 0x0100ffff, 0xffffffff, 0x00fffefe,
	0xfefefefe, 0x0100fffe, 0xff000000, 0x02020100,
	0x00000102, 0x02020100, 0xff000102, 0x0000ffff,
	0x01010100, 0xfcff0101, 0x0100ffff, 0xfd000101,
	0x0000fffe, 0xfe020201, 0x000000ff, 0xff030200,
	0x000000ff, 0x00020100, 0xfeff0000, 0x000101ff,
	0xfeff0000, 0x010200fe, 0xfeff0000, 0x020201ff,
	0x00000001, 0xfdfefdfe, 0x00010000, 0xfffffefe,
	0x0101ff00, 0x0000ffff, 0x0101ff00, 0x0000ffff,
	0x01020100, 0x0100ffff, 0x02030201, 0x02010000,
	0x010200ff, 0x03020000, 0x0000fffe, 0x020100ff,
	0xff000101, 0x01fffefe, 0xff010101, 0x0200fefe,
	0xff010101, 0x0200ffff, 0x00000000, 0x02010000,
	0x00ffffff, 0x02010000, 0x01fffeff, 0x00000101,
	0x01fffeff, 0xff000202, 0x02fffeff, 0xff000202,
	0xfeffffff, 0x0100ffff, 0xffff0000, 0x020100ff,
	0x00000001, 0x02010100, 0x00000101, 0x01010101,
	0x01000101, 0x01010001, 0x01010101, 0xffffff00,
	0x00010201, 0xfdfefeff, 0x00010202, 0xfcfcfdfe,
	0x0101fefc, 0x00000101, 0x000100fe, 0x00000000,
	0xfe010100, 0x0100fffe, 0xfe010202, 0x0201fffe,
	0xfe010201, 0x0201fffe, 0xff0101ff, 0x0100ffff,
	0x010100fe, 0x00ffff00, 0x020200fe, 0x00ffff01,
	0x00000101, 0x00000100, 0xfefdfdfe, 0x0000fffe,
	0xfefdfcfd, 0x000000ff, 0x0100ffff, 0x01020302,
	0x03020100, 0x01020303, 0x02010101, 0xff000001,
	0x00000000, 0xffffffff, 0x00ffff00, 0x00000000,
	0x00000101, 0x0200ffff, 0x00010000, 0x0101ffff,
	0x0100fffe, 0x01010101, 0x0200fdfd, 0x00010102,
	0x0100fefe, 0x00000101, 0x00000001, 0xffff0000,
	0xfdff0103, 0x0100fffe, 0xfdff0204, 0x0201fffd,
	0xff000000, 0xffffffff, 0x00010201, 0xffff0000,
	0x02030302, 0xffff0001, 0x02040403, 0xfeff0001,
	0x01020303, 0xfefeff00, 0xff000101, 0xfdfefeff,
	0xff000000, 0xfefefeff, 0xffffffff, 0xfffefeff,
	0x02020201, 0x02020102, 0x01010100, 0x00000001,
	0x01000000, 0xfeff0001, 0x00000000, 0xfcff0001,
	0x00000001, 0xfbfe0000, 0x00000001, 0xfcff0000,
	0x00ff0001, 0xfdff0000, 0x00ffff00, 0xff010101,
	0x0000fffe, 0xfeffffff, 0x000000ff, 0xfefeffff,
	0x00010100, 0xfeffffff, 0x00010100, 0xffff0000,
	0x00000100, 0x00010101, 0x00000101, 0x01020201,
	0x00000101, 0x01020201, 0xff000101, 0x00010100,
	0x00010204, 0x01010100, 0x00010204, 0x01000000,
	0x00000103, 0x00ffffff, 0xff000001, 0x00fffeff,
	0x00000000, 0x00ffffff, 0x0000ffff, 0x0100ffff,
	0xff00fffe, 0x010000ff, 0xfefffefe, 0x010000ff,
	0x01010100, 0xff000102, 0x00ffffff, 0xfefeff00,
	0x00ff00ff, 0xfffeff00, 0x00000000, 0x02010000,
	0x00000000, 0x03020000, 0xffff00ff, 0x0300ffff,
	0xff0000ff, 0x0300feff, 0x00000000, 0x0401ffff,
	0x00000202, 0x01000000, 0xfeff0101, 0x01fffeff,
	0xfefeffff, 0x00fefdfe, 0xffff00ff, 0x01fffeff,
	0x00000101, 0x01000001, 0x00000202, 0x01000001,
	0x00000202, 0xffff0000, 0x00000202, 0xffff0001,
	0x010100ff, 0xffffff00, 0x02030201, 0x00000001,
	0x01010100, 0x0000ff00, 0x00fffefe, 0x00000001,
	0x02fffefe, 0x00010202, 0x0100fffe, 0xff000001,
	0x0000ffff, 0xfefffeff, 0x01010100, 0x01010000,
	0xfefdfdfd, 0xfefeffff, 0x0100ffff, 0x00000102,
	0x02010101, 0x00000102, 0x01010101, 0x01ff0001,
	0xffff0001, 0x01ff0000, 0xffffff00, 0x01ffff00,
	0x00ff0001, 0x0200ff00, 0x00ff0002, 0x02000000,
	0xfffe0001, 0x00010100, 0xffff0002, 0x00010100,
	0xfffe0001, 0xff000100, 0xffff0001, 0xff000100,
	0x00010100, 0x00000101, 0x010201fe, 0x01000000,
	0x010200fb, 0x0100ff00, 0x0102fffa, 0x0000ff00,
	0xff000305, 0xfffffffe, 0xff000101, 0xffff00ff,
	0x010100ff, 0x00010202, 0x0100fffe, 0x01010102,
	0xfffffffe, 0x0100ff00, 0x00000100, 0x0000ffff,
	0x01010100, 0x00000001, 0x0100fefd, 0xff000001,
	0x000100ff, 0x030200ff, 0xfe0000ff, 0x00fffefc,
	0x00010100, 0xff00fffe, 0x01030201, 0x00010100,
	0x010100ff, 0x00010101, 0x00fefdfe, 0x00010000,
	0x00fefeff, 0xff000001, 0x00000103, 0xffff0001,
	0x0000fffe, 0x0000ffff, 0x000000ff, 0x01010100,
	0x00ffffff, 0x01010101, 0xfffdfe00, 0x00000001,
	0xfffdff01, 0xff000101, 0x01ff0103, 0xff000202,
	0x01000103, 0x00010102, 0xfffefe00, 0x000000ff,
	0xffff0001, 0x00010201, 0xfffeff00, 0x01020201,
	0x00ffffff, 0x00020100, 0x000000fe, 0xff010000,
	0x000100ff, 0xffffffff, 0x02010100, 0x00fffe00,
	0x02020201, 0x00ffff01, 0x01010100, 0xfffefe00,
	0xffff0000, 0xfefeffff, 0x00ff0000, 0x01020201,
	0xffff0000, 0x02020100, 0xfeff0101, 0xffffffff,
	0x00010202, 0xfefeff00, 0x01020201, 0xfefe0000,
	0x00000000, 0xff000101, 0xffffff00, 0x01020302,
	0x0201fe00, 0x010000ff, 0x0302feff, 0x000000ff,
	0x0302fe00, 0x0000ffff, 0x0203ff00, 0x000100fe,
	0x0103ff00, 0x000100fe, 0x0102ff00, 0xff0001ff,
	0xff010000, 0xff0000fe, 0xfe000001, 0xfffffffe,
	0x01010101, 0xfeffff01, 0x01000000, 0x01010101,
	0x01000000, 0x03020101, 0xff000001, 0x02010000,
	0xfeffff00, 0x020100ff, 0xfefefefe, 0x010100ff,
	0xffffffff, 0xff000000, 0x00020202, 0xfcfeffff,
	0xfffffeff, 0x03020100, 0xffffffff, 0x03020100,
	0x00ff0001, 0x020100ff, 0x00000001, 0x020200ff,
	0xffff0001, 0x020100fe, 0xfefefe00, 0x0100fffd,
	0xfefefe00, 0x0101fffe, 0x00000000, 0x02020100
};

const uint32 *const s_svq1IntraCodebooks[6] = {
    s_svq1IntraCodebook4x2, s_svq1IntraCodebook4x4,
    s_svq1IntraCodebook8x4, s_svq1IntraCodebook8x8,
    0, 0
};

} // End of namespace Video

#endif
