/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $URL: https://scummvm.svn.sourceforge.net/svnroot/scummvm/scummvm/tags/release-1-1-0/backends/platform/sdl/events.cpp $
 * $Id: events.cpp 48101 2010-02-21 04:04:13Z bluddy $
 *
 */

#include "backends/platform/sdl/sdl.h"
#include "common/util.h"
#include "common/events.h"
#ifdef MAEMO_SDL
#include "common/config-manager.h"
#endif
// FIXME move joystick defines out and replace with confile file options
// we should really allow users to map any key to a joystick button
#define JOY_DEADZONE 3200

#ifndef __SYMBIAN32__ // Symbian wants dialog joystick i.e cursor for movement/selection
	#define JOY_ANALOG
#endif

// #define JOY_INVERT_Y
#define JOY_XAXIS 0
#define JOY_YAXIS 1
// buttons
#define JOY_BUT_LMOUSE 0
#define JOY_BUT_RMOUSE 2
#define JOY_BUT_ESCAPE 3
#define JOY_BUT_PERIOD 1
#define JOY_BUT_SPACE 4
#define JOY_BUT_F5 5




static int mapKey(SDLKey key, SDLMod mod, Uint16 unicode) {
	if (key >= SDLK_F1 && key <= SDLK_F9) {
		return key - SDLK_F1 + Common::ASCII_F1;
	} else if (key >= SDLK_KP0 && key <= SDLK_KP9) {
		return key - SDLK_KP0 + '0';
	} else if (key >= SDLK_UP && key <= SDLK_PAGEDOWN) {
		return key;
	} else if (unicode) {
		return unicode;
	} else if (key >= 'a' && key <= 'z' && (mod & KMOD_SHIFT)) {
		return key & ~0x20;
	} else if (key >= SDLK_NUMLOCK && key <= SDLK_EURO) {
		return 0;
	}
	return key;
}

void OSystem_SDL::fillMouseEvent(Common::Event &event, int x, int y) {
	event.mouse.x = x;
	event.mouse.y = y;

	// Update the "keyboard mouse" coords
	_km.x = x;
	_km.y = y;

	// Adjust for the screen scaling
	if (!_overlayVisible) {
		event.mouse.x /= _videoMode.scaleFactor;
		event.mouse.y /= _videoMode.scaleFactor;
		if (_videoMode.aspectRatioCorrection)
			event.mouse.y = aspect2Real(event.mouse.y);
	}
}

void OSystem_SDL::handleKbdMouse() {
	uint32 curTime = getMillis();
	if (curTime >= _km.last_time + _km.delay_time) {
		_km.last_time = curTime;
		if (_km.x_down_count == 1) {
			_km.x_down_time = curTime;
			_km.x_down_count = 2;
		}
		if (_km.y_down_count == 1) {
			_km.y_down_time = curTime;
			_km.y_down_count = 2;
		}

		if (_km.x_vel || _km.y_vel) {
			if (_km.x_down_count) {
				if (curTime > _km.x_down_time + _km.delay_time * 12) {
					if (_km.x_vel > 0)
						_km.x_vel++;
					else
						_km.x_vel--;
				} else if (curTime > _km.x_down_time + _km.delay_time * 8) {
					if (_km.x_vel > 0)
						_km.x_vel = 5;
					else
						_km.x_vel = -5;
				}
			}
			if (_km.y_down_count) {
				if (curTime > _km.y_down_time + _km.delay_time * 12) {
					if (_km.y_vel > 0)
						_km.y_vel++;
					else
						_km.y_vel--;
				} else if (curTime > _km.y_down_time + _km.delay_time * 8) {
					if (_km.y_vel > 0)
						_km.y_vel = 5;
					else
						_km.y_vel = -5;
				}
			}

			_km.x += _km.x_vel;
			_km.y += _km.y_vel;

			if (_km.x < 0) {
				_km.x = 0;
				_km.x_vel = -1;
				_km.x_down_count = 1;
			} else if (_km.x > _km.x_max) {
				_km.x = _km.x_max;
				_km.x_vel = 1;
				_km.x_down_count = 1;
			}

			if (_km.y < 0) {
				_km.y = 0;
				_km.y_vel = -1;
				_km.y_down_count = 1;
			} else if (_km.y > _km.y_max) {
				_km.y = _km.y_max;
				_km.y_vel = 1;
				_km.y_down_count = 1;
			}

			SDL_WarpMouse((Uint16)_km.x, (Uint16)_km.y);
		}
	}
}

static void SDLModToOSystemKeyFlags(SDLMod mod, Common::Event &event) {

	event.kbd.flags = 0;

#ifdef LINUPY
	// Yopy has no ALT key, steal the SHIFT key
	// (which isn't used much anyway)
	if (mod & KMOD_SHIFT)
		event.kbd.flags |= Common::KBD_ALT;
#else
	if (mod & KMOD_SHIFT)
		event.kbd.flags |= Common::KBD_SHIFT;
	if (mod & KMOD_ALT)
		event.kbd.flags |= Common::KBD_ALT;
#endif
	if (mod & KMOD_CTRL)
		event.kbd.flags |= Common::KBD_CTRL;

	// Sticky flags
	if (mod & KMOD_NUM)
		event.kbd.flags |= Common::KBD_NUM;
	if (mod & KMOD_CAPS)
		event.kbd.flags |= Common::KBD_CAPS;
}

bool OSystem_SDL::pollEvent(Common::Event &event) {
	SDL_Event ev;

	handleKbdMouse();

	// If the screen mode changed, send an Common::EVENT_SCREEN_CHANGED
	if (_modeChanged) {
		_modeChanged = false;
		event.type = Common::EVENT_SCREEN_CHANGED;
		return true;
	}

	while (SDL_PollEvent(&ev)) {
		preprocessEvents(&ev);
		if (dispatchSDLEvent(ev, event))
			return true;
	}
	return false;
}

bool OSystem_SDL::dispatchSDLEvent(SDL_Event &ev, Common::Event &event) {
	switch (ev.type) {
	case SDL_KEYDOWN:
		return handleKeyDown(ev, event);
	case SDL_KEYUP:
		return handleKeyUp(ev, event);
	case SDL_MOUSEMOTION:
		return handleMouseMotion(ev, event);
	case SDL_MOUSEBUTTONDOWN:
		return handleMouseButtonDown(ev, event);
	case SDL_MOUSEBUTTONUP:
		return handleMouseButtonUp(ev, event);
	case SDL_JOYBUTTONDOWN:
		return handleJoyButtonDown(ev, event);
	case SDL_JOYBUTTONUP:
		return handleJoyButtonUp(ev, event);
	case SDL_JOYAXISMOTION:
		return handleJoyAxisMotion(ev, event);

	case SDL_VIDEOEXPOSE:
		_forceFull = true;
		break;

	case SDL_QUIT:
		event.type = Common::EVENT_QUIT;
		return true;

	}

	return false;
}


bool OSystem_SDL::handleKeyDown(SDL_Event &ev, Common::Event &event) {

#ifdef MAEMO_SDL
// we want to remap first including ctr/shift/alt modifiers
	const bool event_complete = remapKey(ev, event);
	SDLModToOSystemKeyFlags(ev.key.keysym.mod, event);  
#else
	SDLModToOSystemKeyFlags(SDL_GetModState(), event);
#endif
	// Handle scroll lock as a key modifier
	if (ev.key.keysym.sym == SDLK_SCROLLOCK)
		_scrollLock = !_scrollLock;

	if (_scrollLock)
		event.kbd.flags |= Common::KBD_SCRL;

#ifdef MAEMO_SDL
	// fullscreen button or ctrl+space toggle full screen mode
	if (ev.key.keysym.sym == SDLK_F6 || (_have_keyboard && event.kbd.hasFlags(Common::KBD_CTRL) && (ev.key.keysym.sym == SDLK_SPACE) ) ) {
#else
	// Alt-Return and Alt-Enter toggle full screen mode
	if (event.kbd.hasFlags(Common::KBD_ALT) && (ev.key.keysym.sym == SDLK_RETURN || ev.key.keysym.sym == SDLK_KP_ENTER)) {
#endif
		beginGFXTransaction();
			setFullscreenMode(!_videoMode.fullscreen);
		endGFXTransaction();
#ifdef USE_OSD
		if (_videoMode.fullscreen)
			displayMessageOnOSD("Fullscreen mode");
		else
			displayMessageOnOSD("Windowed mode");
#endif

		return false;
	}

	// Alt-S: Create a screenshot
	if (event.kbd.hasFlags(Common::KBD_ALT) && ev.key.keysym.sym == 's') {
		char filename[20];

		for (int n = 0;; n++) {
			SDL_RWops *file;

			sprintf(filename, "scummvm%05d.bmp", n);
			file = SDL_RWFromFile(filename, "r");
			if (!file)
				break;
			SDL_RWclose(file);
		}
		if (saveScreenshot(filename))
			printf("Saved '%s'\n", filename);
		else
			printf("Could not save screenshot!\n");
		return false;
	}

#ifndef MAEMO_SDL
	// Ctrl-m toggles mouse capture
	if (event.kbd.hasFlags(Common::KBD_CTRL) && ev.key.keysym.sym == 'm') {
		toggleMouseGrab();
		return false;
	}
#else
// mouse capture makes no sense for Maemo and ctrl+m is used for global menu
#endif

#if defined(MACOSX)
	// On Macintosh', Cmd-Q quits
	if ((ev.key.keysym.mod & KMOD_META) && ev.key.keysym.sym == 'q') {
		event.type = Common::EVENT_QUIT;
		return true;
	}
#elif defined(UNIX)
	// On other unices, Control-Q quits
	if ((ev.key.keysym.mod & KMOD_CTRL) && ev.key.keysym.sym == 'q') {
		event.type = Common::EVENT_QUIT;
		return true;
	}
#else
	// Ctrl-z and Alt-X quit
	if ((event.kbd.hasFlags(Common::KBD_CTRL) && ev.key.keysym.sym == 'z') || (event.kbd.hasFlags(Common::KBD_ALT) && ev.key.keysym.sym == 'x')) {
		event.type = Common::EVENT_QUIT;
		return true;
	}
#endif

	if ((ev.key.keysym.mod & KMOD_CTRL) && ev.key.keysym.sym == 'u') {
		event.type = Common::EVENT_MUTE;
		return true;
	}

	// Ctrl-Alt-<key> will change the GFX mode
	if ((event.kbd.flags & (Common::KBD_CTRL|Common::KBD_ALT)) == (Common::KBD_CTRL|Common::KBD_ALT)) {
		if (handleScalerHotkeys(ev.key))
			return false;
	}

#ifdef MAEMO_SDL
	if (event_complete)
#else
	if (remapKey(ev, event))
#endif
		return true;

	event.type = Common::EVENT_KEYDOWN;
	event.kbd.keycode = (Common::KeyCode)ev.key.keysym.sym;
	event.kbd.ascii = mapKey(ev.key.keysym.sym, ev.key.keysym.mod, ev.key.keysym.unicode);

	return true;
}

bool OSystem_SDL::handleKeyUp(SDL_Event &ev, Common::Event &event) {
	if (remapKey(ev, event))
		return true;

	event.type = Common::EVENT_KEYUP;
	event.kbd.keycode = (Common::KeyCode)ev.key.keysym.sym;
	event.kbd.ascii = mapKey(ev.key.keysym.sym, ev.key.keysym.mod, ev.key.keysym.unicode);

	// Ctrl-Alt-<key> will change the GFX mode
#ifdef MAEMO_SDL
	// we can't call SDL_GetModState(), modifiers can be remapped too
	SDLModToOSystemKeyFlags(ev.key.keysym.mod, event);  
#else
	SDLModToOSystemKeyFlags(SDL_GetModState(), event);
#endif

	// Set the scroll lock sticky flag
	if (_scrollLock)
		event.kbd.flags |= Common::KBD_SCRL;

	if ((event.kbd.flags & (Common::KBD_CTRL|Common::KBD_ALT)) == (Common::KBD_CTRL|Common::KBD_ALT)) {
		// Swallow these key up events
		return false;
	}

	return true;
}

bool OSystem_SDL::handleMouseMotion(SDL_Event &ev, Common::Event &event) {
	event.type = Common::EVENT_MOUSEMOVE;
	fillMouseEvent(event, ev.motion.x, ev.motion.y);

	setMousePos(event.mouse.x, event.mouse.y);
	return true;
}

bool OSystem_SDL::handleMouseButtonDown(SDL_Event &ev, Common::Event &event) {
#ifdef	MAEMO_SDL
	if (ev.button.button == SDL_BUTTON_LEFT){
	SDLMod mod=SDL_GetModState();
	if (mod & KMOD_SHIFT)
		event.type = Common::EVENT_RBUTTONDOWN;
	else if ( mod & KMOD_CTRL)
		event.type = Common::EVENT_MOUSEMOVE;
	else
		event.type = Common::EVENT_LBUTTONDOWN;
	}
#else
	if (ev.button.button == SDL_BUTTON_LEFT)
		event.type = Common::EVENT_LBUTTONDOWN;
#endif
	else if (ev.button.button == SDL_BUTTON_RIGHT)
		event.type = Common::EVENT_RBUTTONDOWN;
#if defined(SDL_BUTTON_WHEELUP) && defined(SDL_BUTTON_WHEELDOWN)
	else if (ev.button.button == SDL_BUTTON_WHEELUP)
		event.type = Common::EVENT_WHEELUP;
	else if (ev.button.button == SDL_BUTTON_WHEELDOWN)
		event.type = Common::EVENT_WHEELDOWN;
#endif
#if defined(SDL_BUTTON_MIDDLE)
	else if (ev.button.button == SDL_BUTTON_MIDDLE)
		event.type = Common::EVENT_MBUTTONDOWN;
#endif
	else
		return false;

#ifdef MAEMO_SDL
	// we have touchscreen so we may have no mousemotion events between taps
	setMousePos(event.mouse.x, event.mouse.y);
	// this is trying to fix wrong action done by mouse click in some engines
	// it looks like clicking affects objects in previous mouse position
	// if this does not help we should perhaps generate some fake mouse motion event(s) 
#endif
	fillMouseEvent(event, ev.button.x, ev.button.y);

	return true;
}

bool OSystem_SDL::handleMouseButtonUp(SDL_Event &ev, Common::Event &event) {
#ifdef	MAEMO_SDL
	if (ev.button.button == SDL_BUTTON_LEFT){
		SDLMod mod=SDL_GetModState();
		if (mod & KMOD_SHIFT)
			event.type = Common::EVENT_RBUTTONUP;
		else if ( mod & KMOD_CTRL)
			event.type = Common::EVENT_MOUSEMOVE;
		else
			event.type = Common::EVENT_LBUTTONUP;
	}
#else
	if (ev.button.button == SDL_BUTTON_LEFT)
		event.type = Common::EVENT_LBUTTONUP;
#endif
	else if (ev.button.button == SDL_BUTTON_RIGHT)
		event.type = Common::EVENT_RBUTTONUP;
#if defined(SDL_BUTTON_MIDDLE)
	else if (ev.button.button == SDL_BUTTON_MIDDLE)
		event.type = Common::EVENT_MBUTTONUP;
#endif
	else
		return false;
	fillMouseEvent(event, ev.button.x, ev.button.y);

	return true;
}

bool OSystem_SDL::handleJoyButtonDown(SDL_Event &ev, Common::Event &event) {
	if (ev.jbutton.button == JOY_BUT_LMOUSE) {
		event.type = Common::EVENT_LBUTTONDOWN;
		fillMouseEvent(event, _km.x, _km.y);
	} else if (ev.jbutton.button == JOY_BUT_RMOUSE) {
		event.type = Common::EVENT_RBUTTONDOWN;
		fillMouseEvent(event, _km.x, _km.y);
	} else {
		event.type = Common::EVENT_KEYDOWN;
		switch (ev.jbutton.button) {
		case JOY_BUT_ESCAPE:
			event.kbd.keycode = Common::KEYCODE_ESCAPE;
			event.kbd.ascii = mapKey(SDLK_ESCAPE, ev.key.keysym.mod, 0);
			break;
		case JOY_BUT_PERIOD:
			event.kbd.keycode = Common::KEYCODE_PERIOD;
			event.kbd.ascii = mapKey(SDLK_PERIOD, ev.key.keysym.mod, 0);
			break;
		case JOY_BUT_SPACE:
			event.kbd.keycode = Common::KEYCODE_SPACE;
			event.kbd.ascii = mapKey(SDLK_SPACE, ev.key.keysym.mod, 0);
			break;
		case JOY_BUT_F5:
			event.kbd.keycode = Common::KEYCODE_F5;
			event.kbd.ascii = mapKey(SDLK_F5, ev.key.keysym.mod, 0);
			break;
		}
	}
	return true;
}

bool OSystem_SDL::handleJoyButtonUp(SDL_Event &ev, Common::Event &event) {
	if (ev.jbutton.button == JOY_BUT_LMOUSE) {
		event.type = Common::EVENT_LBUTTONUP;
		fillMouseEvent(event, _km.x, _km.y);
	} else if (ev.jbutton.button == JOY_BUT_RMOUSE) {
		event.type = Common::EVENT_RBUTTONUP;
		fillMouseEvent(event, _km.x, _km.y);
	} else {
		event.type = Common::EVENT_KEYUP;
		switch (ev.jbutton.button) {
		case JOY_BUT_ESCAPE:
			event.kbd.keycode = Common::KEYCODE_ESCAPE;
			event.kbd.ascii = mapKey(SDLK_ESCAPE, ev.key.keysym.mod, 0);
			break;
		case JOY_BUT_PERIOD:
			event.kbd.keycode = Common::KEYCODE_PERIOD;
			event.kbd.ascii = mapKey(SDLK_PERIOD, ev.key.keysym.mod, 0);
			break;
		case JOY_BUT_SPACE:
			event.kbd.keycode = Common::KEYCODE_SPACE;
			event.kbd.ascii = mapKey(SDLK_SPACE, ev.key.keysym.mod, 0);
			break;
		case JOY_BUT_F5:
			event.kbd.keycode = Common::KEYCODE_F5;
			event.kbd.ascii = mapKey(SDLK_F5, ev.key.keysym.mod, 0);
			break;
		}
	}
	return true;
}

bool OSystem_SDL::handleJoyAxisMotion(SDL_Event &ev, Common::Event &event) {
	int axis = ev.jaxis.value;
	if ( axis > JOY_DEADZONE) {
		axis -= JOY_DEADZONE;
		event.type = Common::EVENT_MOUSEMOVE;
	} else if ( axis < -JOY_DEADZONE ) {
		axis += JOY_DEADZONE;
		event.type = Common::EVENT_MOUSEMOVE;
	} else
		axis = 0;

	if ( ev.jaxis.axis == JOY_XAXIS) {
#ifdef JOY_ANALOG
		_km.x_vel = axis/2000;
		_km.x_down_count = 0;
#else
		if (axis != 0) {
			_km.x_vel = (axis > 0) ? 1:-1;
			_km.x_down_count = 1;
		} else {
			_km.x_vel = 0;
			_km.x_down_count = 0;
		}
#endif

	} else if (ev.jaxis.axis == JOY_YAXIS) {
#ifndef JOY_INVERT_Y
		axis = -axis;
#endif
#ifdef JOY_ANALOG
		_km.y_vel = -axis / 2000;
		_km.y_down_count = 0;
#else
		if (axis != 0) {
			_km.y_vel = (-axis > 0) ? 1: -1;
			_km.y_down_count = 1;
		} else {
			_km.y_vel = 0;
			_km.y_down_count = 0;
		}
#endif
	}

	fillMouseEvent(event, _km.x, _km.y);

	return true;
}

// called on SDL KEYUP and KEYDOWN events
bool OSystem_SDL::remapKey(SDL_Event &ev, Common::Event &event) {
#ifdef MAEMO_SDL
	static int engine=0;
#define	ENG_OTHER	-1
//#define	ENG_SCUMM	1
	static int game=0;
#define GAME_OTHER	-1	
#define GAME_LURE	1
#define GAME_SWORD1	2
#define GAME_SWORD2	3
#define GAME_SAGA	4
#define GAME_FW		5
//#define GAME_SIMON1	6
//#define GAME_SIMON2	7
#define GAME_FEEBLE	8
//#define GAME_TOUCHE	9
#define GAME_DISCWORLD	10
#define GAME_CRUISE	11


	if (engine == 0){
		// one time initialization
		Common::String gameid(ConfMan.get("gameid"));
		if (gameid.hasPrefix("lure")) {
			game=GAME_LURE;
			engine=ENG_OTHER;
		} else if (gameid.hasPrefix("sword2")) {
			game=GAME_SWORD2;
			engine=ENG_OTHER;
		} else if (gameid.hasPrefix("cine")) {
			game=GAME_FW;
			engine=ENG_OTHER;
/*		} else if (gameid == "touche") {
			game=GAME_TOUCHE;
			engine=ENG_OTHER;
		} else if (gameid == "simon1") {
			game=GAME_SIMON1;
			engine=ENG_OTHER;
		} else if (gameid == "simon2") {
			game=GAME_SIMON2;
			engine=ENG_OTHER;
*/
		} else if (gameid.hasPrefix("feeble")) {
			game=GAME_FEEBLE;
			engine=ENG_OTHER;
		} else if (gameid.hasPrefix("sword1")) {
			game=GAME_SWORD1;
			engine=ENG_OTHER;
		} else if (gameid.hasPrefix("saga")) {
			game=GAME_SAGA;
			engine=ENG_OTHER;
		} else if (gameid.hasPrefix("tinsel")) {
			game=GAME_DISCWORLD;
			engine=ENG_OTHER;
		} else if (gameid.hasPrefix("cruise")) {
			game=GAME_CRUISE;
			engine=ENG_OTHER;
		} else {
			game=GAME_OTHER;
			engine=ENG_OTHER;
		}
	}
	// global mapping - N810, N900
	if (_have_keyboard && (ev.key.keysym.mod & KMOD_CTRL)){
		// map ctrl-m to ctrl F5 = global scummvm menu 
		if (ev.key.keysym.sym==SDLK_m) ev.key.keysym.sym=SDLK_F5 ;
	}
        if (_have_keyboard && (ev.key.keysym.mod & KMOD_SHIFT)){
                // map shift backspace to escape, shift enter to menu key
                if (ev.key.keysym.sym==SDLK_BACKSPACE) { ev.key.keysym.sym=SDLK_ESCAPE ; ev.key.keysym.mod = (SDLMod) (ev.key.keysym.mod & ~KMOD_SHIFT); }
                if (ev.key.keysym.sym==SDLK_KP_ENTER) { ev.key.keysym.sym=SDLK_F4; ev.key.keysym.mod = (SDLMod) (ev.key.keysym.mod & ~KMOD_SHIFT); }
        }
														 
	// engine specific mappings
	switch (engine){
		// nothing now
	}
	// game specific mapping
	switch (game) {
		case GAME_LURE:
			if ((ev.key.keysym.sym==SDLK_F8 && _have_keyboard ) || (ev.key.keysym.sym==SDLK_F4 && !_have_keyboard)){
				// map zoom - to right click if we have keyboard (N810), otherwise map menu key (770,N800)
				event.type = ((ev.type==SDL_KEYUP) ? Common::EVENT_RBUTTONUP : Common::EVENT_RBUTTONDOWN );
				event.mouse.x = _mouseCurState.x;
				event.mouse.y = _mouseCurState.y;
				return true;			 
			
			}
			switch(ev.key.keysym.sym){
				case SDLK_F5: // map F5 (home key) to f9 = restart game
					ev.key.keysym.sym=SDLK_F9;
					break;
				case SDLK_F8: // map F8 (zoom - key) to F5 (save dialog) in game
					ev.key.keysym.sym=SDLK_F5;
					break;
				case SDLK_F4: // same as above, only one mapping happens due to right click maping above 
					ev.key.keysym.sym=SDLK_F5;
				default:
					;
			}
			break;
		case GAME_FW:
			// Future Wars - no mapping here, done in game engine
			break;
		case GAME_FEEBLE:
			if ((ev.key.keysym.sym==SDLK_F8 && _have_keyboard ) || (ev.key.keysym.sym==SDLK_F4 && !_have_keyboard)){
				// map zoom - to right click if we have keyboard (N810), otherwise map menu key (770,N800)
				event.type = ((ev.type==SDL_KEYUP) ? Common::EVENT_RBUTTONUP : Common::EVENT_RBUTTONDOWN );
				event.mouse.x = _mouseCurState.x;
				event.mouse.y = _mouseCurState.y;
				return true;			 
			
			}
			if (!_have_keyboard) switch(ev.key.keysym.sym){
				case SDLK_F7: // map F7 (zoom + key) to letter y
					ev.key.keysym.sym=SDLK_y;
					break;
				case SDLK_F8: // map F8 (zoom - key) to letter 1
					ev.key.keysym.sym=SDLK_1;
					break;
				default:
					;
			} 
			break;
		case GAME_DISCWORLD:
			switch(ev.key.keysym.sym) {
				case SDLK_F8: // map F8 (zoom - key) to right click
					event.type = ((ev.type==SDL_KEYUP) ? Common::EVENT_RBUTTONUP : Common::EVENT_RBUTTONDOWN );
					event.mouse.x = _mouseCurState.x;
					event.mouse.y = _mouseCurState.y;
					return true;
				// now map F7 (=zoom+) to Enter for N810 (useful when closed)				 
				case SDLK_F7:
					if (_have_keyboard) ev.key.keysym.sym=SDLK_RETURN; else ev.key.keysym.sym=SDLK_y;
					break;
				case SDLK_F4: // map menu key to game menu
				case SDLK_F5: // swap/home key too
					ev.key.keysym.sym=SDLK_F1;
					break; 
				default:
					;			
			}
			break;
		case GAME_CRUISE:
			switch(ev.key.keysym.sym) {
				case SDLK_F8: // map F8 (zoom - key) to right click
					event.type = ((ev.type==SDL_KEYUP) ? Common::EVENT_RBUTTONUP : Common::EVENT_RBUTTONDOWN );
					event.mouse.x = _mouseCurState.x;
					event.mouse.y = _mouseCurState.y;
					return true;
				// now map F7 (=zoom+) to menu for N810 (useful when closed)				 
				case SDLK_F7:
					if (_have_keyboard) ev.key.keysym.sym=SDLK_F10; else ev.key.keysym.sym=SDLK_p;
					break;
				case SDLK_F4: // map menu key to game menu
					ev.key.keysym.sym=SDLK_F10;
					break; 
				default:
					;			
			}
			break;
		default:
		//case GAME_SWORD2:
		//case GAME_SWORD1:
		//case GAME_SAGA: //I Have No Mouth
			if (!_have_keyboard) switch(ev.key.keysym.sym){
				case SDLK_F7: // map F7 (zoom + key) to letter y for save game entry and 'yes' replies (simon, touche)
					ev.key.keysym.sym=SDLK_y;
					break;
				case SDLK_F8: // map F8 (zoom - key) to letter 1 for save game entry and copyprotection in monkey2
					ev.key.keysym.sym=SDLK_1;
					break;
				default:
					;
			} else switch(ev.key.keysym.sym) {
				case SDLK_F8: // map F8 (zoom - key) to right click
					event.type = ((ev.type==SDL_KEYUP) ? Common::EVENT_RBUTTONUP : Common::EVENT_RBUTTONDOWN );
					event.mouse.x = _mouseCurState.x;
					event.mouse.y = _mouseCurState.y;
					return true;
				// now map F7 (=zoom+) to menu (=F4) so we can have same mapping for N810 and 770/800 for menu key
				// N800's real menu key is hidden on retractable keyboard so we use zoom+ for it instead too				 
				case SDLK_F7:
					ev.key.keysym.sym=SDLK_F4;
					break;
				/* with real keyboard we can afford to lose F7, do not remap F4 back
				case SDLK_F4:
					ev.key.keysym.sym=SDLK_F7;
					break; */
				default:
					;			
			}
			break;
	}
#endif //SDL_MAEMO

#ifdef LINUPY
	// On Yopy map the End button to quit
	if ((ev.key.keysym.sym == 293)) {
		event.type = Common::EVENT_QUIT;
		return true;
	}
	// Map menu key to f5 (scumm menu)
	if (ev.key.keysym.sym == 306) {
		event.type = Common::EVENT_KEYDOWN;
		event.kbd.keycode = Common::KEYCODE_F5;
		event.kbd.ascii = mapKey(SDLK_F5, ev.key.keysym.mod, 0);
		return true;
	}
	// Map action key to action
	if (ev.key.keysym.sym == 291) {
		event.type = Common::EVENT_KEYDOWN;
		event.kbd.keycode = Common::KEYCODE_TAB;
		event.kbd.ascii = mapKey(SDLK_TAB, ev.key.keysym.mod, 0);
		return true;
	}
	// Map OK key to skip cinematic
	if (ev.key.keysym.sym == 292) {
		event.type = Common::EVENT_KEYDOWN;
		event.kbd.keycode = Common::KEYCODE_ESCAPE;
		event.kbd.ascii = mapKey(SDLK_ESCAPE, ev.key.keysym.mod, 0);
		return true;
	}
#endif

#ifdef QTOPIA
	// Quit on fn+backspace on zaurus
	if (ev.key.keysym.sym == 127) {
		event.type = Common::EVENT_QUIT;
		return true;
	}

	// Map menu key (f11) to f5 (scumm menu)
	if (ev.key.keysym.sym == SDLK_F11) {
		event.type = Common::EVENT_KEYDOWN;
		event.kbd.keycode = Common::KEYCODE_F5;
		event.kbd.ascii = mapKey(SDLK_F5, ev.key.keysym.mod, 0);
	}
	// Nap center (space) to tab (default action )
	// I wanted to map the calendar button but the calendar comes up
	//
	else if (ev.key.keysym.sym == SDLK_SPACE) {
		event.type = Common::EVENT_KEYDOWN;
		event.kbd.keycode = Common::KEYCODE_TAB;
		event.kbd.ascii = mapKey(SDLK_TAB, ev.key.keysym.mod, 0);
	}
	// Since we stole space (pause) above we'll rebind it to the tab key on the keyboard
	else if (ev.key.keysym.sym == SDLK_TAB) {
		event.type = Common::EVENT_KEYDOWN;
		event.kbd.keycode = Common::KEYCODE_SPACE;
		event.kbd.ascii = mapKey(SDLK_SPACE, ev.key.keysym.mod, 0);
	} else {
	// Let the events fall through if we didn't change them, this may not be the best way to
	// set it up, but i'm not sure how sdl would like it if we let if fall through then redid it though.
	// and yes i have an huge terminal size so i dont wrap soon enough.
		event.type = Common::EVENT_KEYDOWN;
		event.kbd.keycode = ev.key.keysym.sym;
		event.kbd.ascii = mapKey(ev.key.keysym.sym, ev.key.keysym.mod, ev.key.keysym.unicode);
	}
#endif
	return false;
}
