#!/usr/bin/env python

"""
===============================================================
    contains functions to access config file
===============================================================
"""

# importing 'with' statment - necessary for python2.5
from __future__ import with_statement

__author__ = "Mezhenin Artoym <mezhenin@cs.karelia.ru>"
__version__ = "$Revision: 2 $"
__date__ = "$Date: 2010/02/24 $"
__copyright__ = ""
__license__ = "GPLv2"


import os
import ConfigParser
from Misc import get_home



class ConfigDriver(object):
    """
    Config driver for parsing options from file     
    """

    def __init__(self, config_file=None):
        """
        Read existing config file or initialize new with default values.
        
        @param config_file (string) path to file with config, None by default
        @return None
        """

        self._parser = ConfigParser.RawConfigParser()
        self._config_file = config_file

        if not self._config_file:
            self._config_file = get_home() + "/config"


        """
        Initialize config file
        """
        if not os.path.exists(self._config_file):
            from About import __version__ as version
            self._parser.add_section('Main')
            self.set('Main', 'version', str(version))
            self.set('Main', 'proxy', None)

            self._parser.add_section('UI')
            self.set('UI', 'warning_press_refresh', False)
            self.set('UI', 'autorefresh', False)
            self.set('UI', 'font', None)

        else:
            self._parser.read(self._config_file)


    def get(self, section, option):
        """
        Read value from config file.
        
        @param section (string) name of section, where we should search for
                        option(for example 'Main', 'UI', 'Network')
        @param option (scting) option name(for example 'version', 'proxy')
        @return (string, None) value of option from selected section 
        """

        res = self._parser.get(section, option)
        if res and res.lower() == 'none':
            return None

        return res

    def get_bool(self, section, option):
        """
        Read bool value from config file.
        
        @param section (string) name of section, where we should search for
                        option(for example 'Main', 'UI', 'Network')
        @param option (scting) option name(for example 'version', 'proxy')
        @return (bool) value of option from selected section
        @exception ValueError if option can't be converted to bool 
        """
        res = self._parser.getboolean(section, option)
        return res


    def set(self, section, option, value):
        """
        Write value to config file. 
        
        @param section (string) name of section, where we should search for
                        option(for example 'Main', 'UI', 'Network')
        @param option (scting) option name(for example 'version', 'proxy')
        @param value (string, None) value of option
        return None 
        """

        # we have to conver bools to strings in lower case, because 
        # RawConfigParser have problems with reading them
        if type(value) == bool:
            if value:
                value = 'true'
            else:
                value = 'false'

        self._parser.set(section, option, value)
        # Writing our configuration file
        with open(self._config_file, 'wb') as file_obj:
            self._parser.write(file_obj)



config = ConfigDriver()
