#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
===============================================================
    This file contains functions to work with LJ
===============================================================
"""

__author__ = "Mezhenin Artoym <mezhenin@cs.karelia.ru>"
__version__ = "$Revision: 9 $"
__date__ = "$Date: 2010/04/04 $"
__copyright__ = ""
__license__ = "GPLv2"


import xmlrpclib
import urllib2, urllib
from urllib import urlencode
import os
import time
import hashlib
import socket
import re
import string

from ldm.Message import Message, POST, COMMENT
from ldm.Account import Account
from ldm.SqlDriver import TIME_FORMAT, sql_driver
from ProxyedTransp import ProxyedTransp
from Service import Service
from ScriboExc import XmlrpcExc, SocketExc, WarningExc
from LjCommentUtils import *


MAX_EVENTS = 50
MAX_SKIP = 100
MAX_ITEMS = 50
ONE_PAGE = 20

LJ_SERVER = "http://www.livejournal.com/interface/xmlrpc:80"
LJ_USERPICS = 'http://l-userpic.livejournal.com/'
LJ_SHORT = "http://www.livejournal.com/"

class LivejAccount(Account, Service):
    """
    This class provide funct to access LivrJournal API
    """

    def __init__(self, id, server_url=LJ_SERVER, userpics_url=LJ_USERPICS):

        super(LivejAccount, self).__init__(id)

        self._server_url = server_url
        self._userpics_url = userpics_url

        ## flag, become True when XML-RPC structure is ready to use
        self.initialized = False
        ## instance of class, to send XML-RPC requests
        self._server = None
        ## instance of class, to work with http requests
        self._opener = None

        self._base_param = None

    def reset(self):
        """
        Reset date in account
        
        @return None
        """
        super(LivejAccount, self).reset()

        self.set_option('friends_last', '0000-00-00 00:00:00')
        self.set_option('friends', [])


    @staticmethod
    def get_name():
        """
        Get name of this service. It will be used in UI in list of 
        avalible services.
        
        @return (string) name of this service
        """

        return 'LiveJournal'


    @staticmethod
    def get_icon():
        """
        Get path to icon of this service. It will be used in UI in list of 
        avalible services.
        
        @return (string) path to icon of this service
        """

        return '../data/faviconLJ.ico'


    def get_username(self):
        """
        We have to override parent getter method for username, because of 
        setter method. (see LivejAccount.set_username)
        
        @return (string) username 
        """

        return super(LivejAccount, self).get_username()


    def set_username(self, value):
        """
        We have to override parent setter for username, because in LJ 
        'vasja-pupkin' and 'vasja_pupkin' are same user
        
        @return None 
        """

        value = str(value).replace('-', '_').lower()
        super(LivejAccount, self).set_username(value)


    def get_hpassword(self):
        """
        Return encripted password
        """

        return hashlib.md5(self.passwd).hexdigest()


    def get_userpics(self):
        """
        Return dictionary with user avatars: 'key' is tag of avatar in LJ, and
        'value' is path to local copy of this file 
        
        @return (dict) dictinary with avatars
        """
        return self.get_option('userpisc')


    ## property
    hpassword = property(get_hpassword)
    ## property
    username = property(get_username, set_username)


    def connect(self):
        """
        Create classes to connect to server
        """

        if self.initialized:
            return

        # get info about proxy 
        proxy_arg = self.get_proxy()

        if not proxy_arg:
            self._server = xmlrpclib.Server(self._server_url)
            self._opener = urllib2.build_opener(urllib2.HTTPHandler)
        else:

            #print "Proxy: " + proxy_arg
            proxyed = ProxyedTransp()
            proxyed.set_proxy(proxy_arg)
            self._server = xmlrpclib.ServerProxy(self._server_url,
                                                transport=proxyed)

            """
            proxy_info = {
            'user' : 'username',
            'pass' : 'password',
            'host' : "proxy.name.com",
            'port' : 80 # or 8080 or whatever
            }
            
            # build a new opener that uses a proxy requiring authorization
            proxy_support = urllib2.ProxyHandler({"http" : \
            "http://%(user)s:%(pass)s@%(host)s:%(port)d" % proxy_info})
            opener = urllib2.build_opener(proxy_support, urllib2.HTTPHandler)
            """

            proxy_support = urllib2.ProxyHandler(
                                        {'http': 'http://' + proxy_arg})
            self._opener = urllib2.build_opener(proxy_support,
                                               urllib2.HTTPHandler)

        urllib2.install_opener(self._opener)

        self.initialized = True


    def disconnect(self):

        self.initialized = False


    def exec_xmlrpc(self, funct, param):
        """
        execute XML-RPc query, return result and convert for errors
        """
        self.connect()

        param.update({
                      'username': self.username,
                      'hpassword': self.hpassword,
                      'ver': '1'
                     })

        req = 'self._server.LJ.XMLRPC.' + funct + '(' + repr(param) + ')'
        try:
            res = eval(req)
        except xmlrpclib.Fault, exc:
            raise XmlrpcExc(self, exc)
        except socket.error, exc:
            raise SocketExc(self, exc)

        return res


    def check_login(self):
        """
        Check correctness of login/passwod.
        
        @return (None) None if login/password is correct
        @exception (LoginExc) if password is not correct
        """

        param = {}
        param.update({
                      'getpickws': '1',
                      'getpickwurls': '1'
                     })

        res = self.exec_xmlrpc('login', param)

        ## we use this propetry from base class Account 
        self.nick = self._parse_string(res['fullname'])

        if 'defaultpicurl' in res and len(res['defaultpicurl']) > 0:
            self.path_to_ava = self._get_avatar(res['defaultpicurl'])
        else:
            self.path_to_ava = None

        journal_names = res['usejournals']
        journal_list = []
        for i in journal_names:
            journal = Account.register(type(self), i)
            journal.group = True
            journal_list.append(journal.id)

        self.journals_list = journal_list

        userpics = {}
        pict_num = len(res['pickwurls'])
        for i in xrange(pict_num):
            key = self._parse_string(res['pickws'][i])
            url = res['pickwurls'][i]
            userpics[key] = self._get_avatar(url)

        self.set_option('userpics', userpics)



    def send_post(self, msg):
        """
        Send post to LiveJournal
        
        @return None
        """

        moment = time.localtime()
        if not msg.tags:
            msg.tags = ''

        param = {}
        param.update({
                      'event': msg.text.replace('\n', '<br />'),
                      'subject': msg.title,
                      'props': {
                                'taglist': msg.tags,
                                'opt_preformatted': True,
                                'opt_nocomments': 0
                               },
                      'security': 'public',
                      'allowmask': '1',
                      'year': moment[0],
                      'mon': moment[1],
                      'day': moment[2],
                      'hour': moment[3],
                      'min': moment[4],
                      'lineendings': 'unix'
                     })

        if msg.journal:
            param['usejournal'] = msg.journal.username

        return self.exec_xmlrpc('postevent', param)


    def edit_post(self, msg):
        """
        Edit existing post on LiveJournal
        
        @return None
        """

        moment = time.strptime(msg.date, TIME_FORMAT)

        if not msg.tags:
            msg.tags = ''

        param = {}
        param.update({
                      'itemid': msg.custom_id,
                      'event': msg.text,
                      'subject': msg.title,
                      'props': {
                                'taglist': msg.tags,
                                'opt_preformatted': True,
                                'opt_nocomments': 0
                               },
                      'security': 'public',
                      'allowmask': '1',
                      'year': moment[0],
                      'mon': moment[1],
                      'day': moment[2],
                      'hour': moment[3],
                      'min': moment[4],
                      'lineendings': 'unix'
                     })
        self.exec_xmlrpc('editevent', param)
        # TODO: return result
        return None


    def refresh_post(self, msg):

        result = self._get_post(msg.custom_id)
        if result:
            self._parse_post(result, msg)
        else:
            raise WarningExc("LivejAccount","Post was deleted! Try update list of post")



    def refresh_posts(self, callback):
        """
        Download post and search for updates in journal and sync. with local 
        cache
        
        @param callback (function) run this funct. to ask user for patient and 
                                   redraw UI
        @return None
        """
        # if we need to init new account
        if not self.date:
            self._get_all_posts(callback)
        else:
            self._get_new_posts(callback)


    def _parse_string(self, lj_field):
        """
        Convert response from LJ to utf-8. sqlite3 need utf-8 strings
        """

        return str(lj_field).decode('utf-8')


    def _parse_post(self, lj_post, new_post=None):
        """
        Converts answer with post from LJ into Message instance
        """
        if not new_post:
            new_post = Message()

        if 'subject' in lj_post:
            new_post.title = self._parse_string(lj_post['subject'])

        #convert text in our (HTML) format (\n to <br> )
        event = self._parse_string(lj_post['event'])
        event = event.replace('\n', '<br>')
        new_post.text = event

        new_post.date = self._parse_string(lj_post['eventtime'])

        if 'url' in lj_post:
            new_post.url = self._parse_string(lj_post['url'])

        new_post.custom_id = str(lj_post['itemid']) #TODO rewrite

        if 'taglist' in lj_post['props']:
            new_post.tags = self._parse_string(lj_post['props']['taglist'])
        else:
            new_post.tags = ''

        # TODO: what to do with 'reply_count'?
        #new_post. = lj_post['reply_count']

        return new_post

    def _get_new_posts(self, callback):
        """
        For updates from LJ we can use getevent or syncitems functions
        """


        """
        #we can search for updates using getevent, but it doesn't works well
        param = {
         'username': self.username,
         'hpassword': self.hpassword,
         'ver': '1',
         'lineendings': 'unix',
         "selecttype": "syncitems",
         #'beforedate': self.date,
         "lastsync": self.date
        }
        result = self._server.LJ.XMLRPC.getevents(param)
        """

        # syncitems works better than getevents, but getting posts one by one 
        # may take much longer
        param = {}
        param['lastsync'] = self.date

        result = self.exec_xmlrpc('syncitems', param)['syncitems']
        callback()

        for i in result:
            #TODO: i['item'] can be 'L-'(posts) or 'C-'(new comments)
            if 'L' == i['item'][0]:
                item_id = i['item'][2:]
                lj_post = self._get_post(item_id)

                callback()

                if 'update' == i['action']:
                    # action is 'update'
                    old_msg = Message.find(custom_id=item_id)
                    self._parse_post(lj_post, old_msg)
                else:
                    new_post = self._parse_post(lj_post)
                    self.add_post(new_post)

        # save new date for syncitems
        ## we use this propetry from base class Account
        if result:
            self.date = result[-1]['time']

        #search for deleted posts
        param = {}
        result = self.exec_xmlrpc('syncitems', param)['syncitems']
        callback()

        cur_posts = self.posts
        for i in result:
            item_id = i['item'][2:]
            cur_msg = Message.find(custom_id=item_id)
            try:
                cur_posts.remove(cur_msg)
            except ValueError:
                pass

        for i in cur_posts:
            i.del_msg_tree()


    def _get_all_posts(self, callback):
        """
        Get all posts for new account and add it
        
        @return None
        """

        param = {}
        param.update({
                      'lineendings': 'unix',
                      'selecttype': 'lastn',
                      'howmany': str(MAX_EVENTS),
                      'itemid': '-1',
                     })

        ret = self.exec_xmlrpc('getevents', param)
        callback()

        lj_posts = ret['events']

        if not lj_posts: # TODO: test this funct on account without posts
            return

        while MAX_EVENTS == len(ret['events']):

            param = {}
            param.update({
                          'lineendings': 'unix',
                          'selecttype': 'lastn',
                          'howmany': str(MAX_EVENTS),
                          'itemid': '-1',
                          'beforedate': lj_posts[-1]['eventtime']
                         })

            ret = self.exec_xmlrpc('getevents', param)
            callback()

            lj_posts = lj_posts + ret['events']

        """
        TODO: check posts order 
        """
        lj_posts.reverse()
        # WARNING: don't use "for i in lj_posts:" - it breaks unicode data
        #          in lj_posts.
        for i in xrange(len(lj_posts)):
            #FIXME: user syncitems's seconds
            new_post = self._parse_post(lj_posts[i])
            self.add_post(new_post)

        """
        ============== BUG FIX TO UPDATE TIME ==================
        Note: We cant take lj_posts[-1]['eventtime'] as 'lastsync' in 
              syncitems. So we have to run syncitems and take date of last
              result + 1 second as last update time. 
        """

        param = {}
        param['lastsync'] = ''

        result = self.exec_xmlrpc('syncitems', param)['syncitems']
        callback()

        # add 1 second to date
        last_sync = result[-1]['time']
        last_sync = time.strptime(last_sync, TIME_FORMAT)
        last_sync = time.mktime(last_sync)
        last_sync = last_sync + 1
        last_sync = time.localtime(last_sync)
        last_sync = time.strftime(TIME_FORMAT, last_sync)

        #self.set_option('posts_last', last_sync)
        self.date = last_sync


    def del_post(self, msg):
        """
        Delete post with this itemid from LJ
        """

        param = {}
        param['itemid'] = msg.custom_id

        result = self.exec_xmlrpc('editevent', param)
        msg.del_msg_tree()
        return result


    def _parse_user(self, lj_user):
        """
        parse information about LJ-user. upadate existing users and create new. 
        """
        username = self._parse_string(lj_user['username'])

        user = Account.register(type(self), username)

        if 'birthday' in lj_user:
            user.date_of_birth = self._parse_string(lj_user['birthday'])

        '''
        http://www.livejournal.com/doc/server/ljp.csp.xml-rpc.getfriends.html
        
        type(optional):
        * [scalar](required) The type of journal the friend item is. This value 
        can be one of 'community', 'syndicated', 'news', 'shared', or identity 
        (I), depending on the journaltype of the account in question. The 
        account is a normal personal account when this value is not sent. 
        '''
        user.group = 'type' in lj_user


        try:
            user.date_of_birth = self._parse_string(lj_user['birthday'])
        except KeyError:
            pass

        user.nick = self._parse_string(lj_user['fullname'])
        return user


    def refresh_friends_list(self, callback):
        """
        Refresh list of current frinds and groups.
        
        @return None
        """

        param = {}
        param.update({
                      #'includefriendof': '1',
                      #'includegroups': '1',
                      'includebdays': '1'
                     })
        callback()
        result = self.exec_xmlrpc('getfriends', param)
        callback()   
        #parse result from server
        self.friends_list = []
        old_friends = self.get_option('friends')
        new_friends = []
        has_new = False
        for i in result['friends']:
            super(LivejAccount, self).add_friend(self._parse_user(i))
            username = self._parse_string(i['username'])
            new_friends.append(username)
            try:
                old_friends.remove(username)
            except ValueError:
                has_new = True
            callback()

        '''
        deleting posts of old friends from DB, we souldnt see their posts 
        in our friendlist
        '''
        for i in old_friends:
            bore = self.find(LivejAccount, i)
            bore.del_posts()

        self.set_option('friends', new_friends)
        '''
        friendofs = result['friendofs']
        for i in friendofs:
            self.add_reader(self._parse_user(i))
        '''
        callback()
        return has_new


    def _parse_friend_post(self, lj_post, new_post=None):
        """
        Converts answer with post from LJ into Message instance
        """
        if not new_post:
            new_post = Message()

        # parse result
        #convert text in our (HTML) format (\n to <br> )
        event = self._parse_string(lj_post['event_raw'])
        event = event.replace('\n', '<br>')
        new_post.text = event

        new_post.title = self._parse_string(lj_post['subject_raw'])

        url = lj_post['poster_userpic_url']
        if len(url) > 0:
            new_post.path_to_ava = self._get_avatar(url)
        else:
            new_post.path_to_ava = None

        date = lj_post['logtime']
        new_post.date = time.strftime("%Y-%m-%d %H:%M:%S",
                                 time.localtime(date))

        poster = self.find(LivejAccount,
                           lj_post['postername'])
        if not poster:
            # Note: LivejAccount is base class so we have to take type(self) 
            # as class 
            poster = Account.register(type(self), lj_post['postername'])
            poster.group = ('P' != lj_post['postertype'])
        new_post.poster = poster

        new_post.custom_id = lj_post['ditemid']

        journal = self.find(LivejAccount,
                            lj_post['journalname'])
        journal.add_post(new_post)

        return new_post


    def refresh_friends_page(self, callback):
        """
        Check for new posts in friend-page, then updating frind list and 
        downloading posts if there are something new
        
        @param callback (function) this function will be called after 
               downloading some part of information
        @return None
        """

        param = {}
        param.update({
                      'lastupdate': self.get_option('friends_last'),
                      #'includegroups': '1',
                      #'includebdays': '1' #TODO: parse bdays
                     })

        result = self.exec_xmlrpc('checkfriends', param)
        callback()

        if 1 == result['new']:
            new_friends = self.refresh_friends_list(callback)
            callback()
            self._get_friends_posts(new_friends, callback)

        self.set_option('friends_last', result['lastupdate'])



    def _get_friends_posts(self, new_friends, callback):
        """
        Download posts from friendpage, parse and save it on DB.
        
        @param new_friendes (bool) if this flag is False, then we dont need to
               to download all friend-page, we get posts to first message that
               we already have and stop. if flag is True we have to download all
               posts, but we will skip 'old' posts
        @param callback (function) this function will be called after 
               downloading some part of information
        @return None 
        """

        skip = 0

        if new_friends:
            page = MAX_ITEMS
        else:
            page = ONE_PAGE

        if self in self.get_friends():
            self.refresh_posts(callback)

        while True:
            param = {}
            param['itemshow'] = str(page)
            param['skip'] = str(skip)

            result = self.exec_xmlrpc('getfriendspage', param)['entries']
            callback()

            getted = len(result)
            for i in xrange(getted):
                if self.username == result[i]['postername']:
                    url_id = 'url LIKE "%%%s.html"' % result[i]['ditemid']
                    msg = Message.find(where=url_id)
                else:
                    msg = Message.find(custom_id=result[i]['ditemid'])

                if not msg:
                    self._parse_friend_post(result[i])

                else:
                    # end loop if we start to parse old messages and we have no
                    # new friends
                    if not new_friends:
                        return

            skip += MAX_ITEMS
            if getted < MAX_ITEMS or skip > MAX_SKIP:
                return

        return


    def add_friend(self, user):
        """
        override function of adding friend. if adding was success, we make 
        changes in local cache through base function 
        
        @param user (LivejAccount) new friend
        @return (bool) True if account was successfully added, False otherwise 
        """

        param = {}
        param['add'] = ({'username': user.username},)

        try:
            # Note: editfriends return result
            self.exec_xmlrpc('editfriends', param)
        except xmlrpclib.Fault:
            return False

        return super(LivejAccount, self).add_friend(user)


    def remove_friend(self, user):
        """
        override function of removing friend. if removing was success, we make 
        changes in local cache through base function 
        
        @param user (LivejAccount) our friend
        @return (bool) True if account was successfully removed, False otherwise 
        """

        param = {}
        param['delete'] = (user.username,)

        self.exec_xmlrpc('editfriends', param)

        return super(LivejAccount, self).remove_friend(user)


    def _get_post(self, postid):
        """
        Get post from LJ by itemid
        """

        param = {
                 'lineendings': 'unix',
                 'selecttype': 'one',
                 'itemid': postid,
                 'howmany': '1'
                }
        result = self.exec_xmlrpc('getevents', param)
        if result['events'] == []:
            return  None
        return result['events'][0]


    def _get_avatar(self, url):
        """
        Download avatar if from LJ and save it on local file system'. If such 
        file already exist, funct. will skip downloading 
        
        @param url (string) URL of avatar with 'http://' prefix
        @return path to avatar on local file system
        """

        self.connect()

        from Misc import get_userpic_path

        pict_name = url[len(self._userpics_url):]
        pict_name = pict_name.replace('/', '_')

        dir = get_userpic_path() + '/' + type(self).__name__
        filename = dir + '/' + pict_name

        # if we have this file of disk we shouldn't download it again
        if os.path.isfile(filename):
            return filename

        if not os.path.exists(dir):
            os.makedirs(dir)

        try:
            socket = urllib2.urlopen(url)
            file = open(filename, 'w')
            file.write(socket.read())
            file.close()
        except IOError, exc:
            # sometimes may happen error with writing local files and user will
            # see "Networking error... disk is full", but most of time it will
            # work good
            raise SocketExc(self, exc)

        return filename

    """
    ===============================================================
    Support of comment
    ===============================================================
    """

    def get_session (self):
        """
        Generate session for using cookie autentification
        
        @return Cookie string
        """
        self.connect()
        params = {
            'username': self.username,
            'ver': 1,
            'password': self.passwd,
            }
        lj_sess = self._server.LJ.XMLRPC.sessiongenerate(params)
        
        return lj_sess['ljsession']

    def send_comment (self, msg, parent, addit = None):
        """
        send comment to livejournal
        
        @param msg (Message) message for sending
        @param parent (Message) parent message
        @param addit (Dict) unused
        @return None
        """
        """
        def lookup_for_root(mesg):
            if mesg.type == POST:
                return mesg
            else:
                lookup_for_root(mesg.parent)
          """
          
        req = None
        print "sending reach LiveJAccount"
        try:
            misc = parent.misc
            if parent.type == POST:
                ptid = 0
                itemid = re.findall("\d+\.html$", parent.url)[0].split('.')[0]
            else:
                ptid = parent.custom_id
                mid = sql_driver.select_cmd("SELECT id FROM Messages WHERE custom_id=? AND journal=? AND type=?",\
                  (misc['parenttalkid'], self.id, POST))
                print mid
                itemid = re.findall("\d+\.html$", Message(mid).url)[0].split('.')[0]
                #itemid = re.findall("\d+\.html$", lookup_for_root(Message(misc['parenttalkid'])).url)[0].split('.')[0]
          
            print "1"
            req = {
                'parenttalkid': ptid,
                'itemid': itemid,
                'journal': parent.journal.username,
                'usertype': 'user',
                'userpost': self.username,
                'password': self.passwd,
                'do_login': '1',
                'subject': msg.title.encode( "utf-8" ),
                'subjection':'none',
                'body':msg.text.encode( "utf-8" ),
            }
            print "2"
            req = urlencode(req)
        except Exception, x:
            print  x
        print req
        try:
            resp = urllib2.urlopen(urllib2.Request(LJ_SHORT+"/talkpost_do.bml",req))
            tmp = resp.readlines()
            resp.close()
            #tfl = open("/home/user/log.html","w")
            #for i in tmp:
            #    tfl.write(i)
            #tfl.close()
        except IOError, exc:
            print exc
            raise SocketExc(self, exc)
        


    def del_comment (self, msg, addit = None):
        """
        delete comment from livejournal
        
        @param msg (Message) message for deleting
        @param addit (Dict) must have 'delauthor' and 'delthread' keys only
          'delthread' - delete thread flag 1-yes, 0-no (default 0)
          'delauthor' - delete all by author flag 1-yes, 0-no (default 0) now unused
        @return None
        """
        ##FIXME: deleting by author is not supported yet
        session = self.get_session()
        print msg.id
        print msg.title
        print msg.custom_id
        #FIXME dirty hack!!!
        req = urllib2.Request(LJ_SHORT+"/delcomment.bml?journal=%s&id=%s" % (msg.journal.username,\
             int(str((int(msg.custom_id)) * 256 +5))),headers={'Cookie': "ljsession=%s" %(session)})
        #FIXME end of dirty hack!!!
        print req.get_full_url()
        r = urllib2.urlopen(req)
        page = r.readlines()
        try:
            dthr = addit['delthread']
        except:
            dthr = 0
        try:
            daut = addit['delauthor']
        except:
            daut = 0
        self.__confirm(page,session,{'delauthor':daut,'delthread':dthr})
        msg.del_comment_local()
        ##FIXME: add removing from DB!!!
        
    def freeze_comment (self, msg, addit = None):
        """
        freeze/unfreeze comment from livejournal
        
        @param msg (Message) message for freezing
        @param addit (Dict) unused
        @return None
        """
        session = self.get_session()
        misc = msg.misc
        if misc['frozen']:
            mode = 'freeze'
        else:
            mode = 'unfreeze'
        #FIXME dirty hack!!!
        req=urllib2.Request(LJ_SHORT+"/talkscreen.bml?mode=%s&journal=%s&talkid=%i" % (mode, msg.journal,\
                    int(str((int(msg.custom_id)) * 256 +5))),headers = {'Cookie': "ljsession=%s" %(session)})
        #FIXME end of dirty hack!!!
        r = urllib2.urlopen(req)
        page = r.readlines()
        self.__confirm(page,session,None)
        
        if misc['frozen']:
            msg.unfroze_childs()
        else:
            msg.froze_childs()
        ##FIXME: add marking of frozen to misc
        ##NOTE: Fix frozen flag when get comments
        
    def screen_comment (self, msg, addit = None):
        """
        screen/unscreen comment from livejournal
        
        @param msg (Message) message for screening
        @param addit (Dict) unused
        @return None
        """
        session = self.get_session()
        misc = msg.misc
        if misc['screened']:
            mode = 'screen'
        else:
            mode = 'unscreen'
        #FIXME dirty hack!!!
        req=urllib2.Request(LJ_SHORT+"/talkscreen.bml?mode=%s&journal=%s&talkid=%i" % (mode, msg.journal,\
                    int(str((int(msg.custom_id)) * 256 +5))),headers = {'Cookie': "ljsession=%s" %(session)})
        #FIXME end of dirty hack!!!
        r = urllib2.urlopen(req)
        page = r.readlines()
        self.__confirm(page,session,None)
        
        if misc['screened']:
            msg.unscreen_childs()
        else:
            msg.screen_childs()


    def refresh_comments (self, callback, addit = None):
        """
        update comments from livejournal
        
        @param addit (Dict) unused
        @return None
        """
        cp = CommentProcessor()
        acc_misc = self.misc
        if acc_misc <> None and 'max_getted_comment' in acc_misc:
            cp.maxid = acc_misc['acc_misc']
            cp.current = acc_misc['acc_misc']-1
        session = self.get_session()
        #Gets metadata for all comments, get data only for new
        cp.current = -1
        
        once = True
        while once or cp.maxid>cp.current:
            once = False
            req = urllib2.Request(LJ_SHORT+"/export_comments.bml?get=comment_meta&startid=%i"\
                      %(cp.current+1), headers = {'Cookie': "ljsession=%s"%(session)})
            rxml = urllib2.urlopen(req).read()
            cp.feedmeta(rxml)
            callback()
        
        try:
            cp.current = acc_misc['acc_misc']
        except:
            cp.current = -1
         
        while cp.maxid>cp.current: 
            req = urllib2.Request(LJ_SHORT+"/export_comments.bml?get=comment_body&startid=%i"\
                      % (cp.current+1), headers = {'Cookie': "ljsession=%s"%(session)})
            rxml = urllib2.urlopen(req).read()
            cp.feeddata(rxml)
            callback()
            
        try:
            _i = acc_misc['acc_misc']
        except:
            acc_misc['acc_misc'] = 0
        finally:
            self.misc = acc_misc
            
        self.__savecomments(cp.changeset, cp.umap)
        acc_misc['acc_misc'] = cp.maxid
        self.misc = acc_misc
        
        
        


        
    def __savecomments (self, changeset, umap):
        """
        update comments in DB by changeset
        
        @param changeset (List of Dict) unused
        @param umap (List of Dict) unused
        @return None
        """
        #kl = changeset.keys()
        #print changeset
        #kl.sort()
        
        #list of changes data
        changelist = {}
        #list of deleted, frozen and screened comments
        deleted = []
        frozen = []
        screened = []
        
        #function  for lookup parent of message in DB           
        def __is_parent_added(comment):
            try:
                par_id = comment['parentid']
                if par_id == 0:
                    par_id = comment['jitemid']
                mid = sql_driver.select_cmd("SELECT id FROM Messages WHERE custom_id=? AND journal=?",\
                        (par_id, self.id))
                return True
            except:
                return False
        
        #fill list of id of deleted, frozen, screened post
        def ___split_changeset():     
            for i in changeset:
                if changeset[i]['state']=='D':
                    deleted.append(i)
                elif changeset[i]['state']=='S':
                    screened.append(i)
                elif changeset[i]['state']=='F':
                    frozen.append(i)
                    
        #generate message by custom_id
        def __msg_by_cid(cid, mtype=COMMENT):

            try:
                mid = sql_driver.select_cmd("SELECT id FROM Messages WHERE custom_id=? AND journal=? AND type=?",\
                                (str(cid), str(self.id), str(mtype)))
                msg = Message(mid)
                return (msg, False)
            except:
                msg = Message()
                msg.type = COMMENT
                msg.custom_id = cid
                msg.journal = self              
                return (msg, True)
            
            
        
        #generate lists
        ___split_changeset()
       
        #add all comments from changeset
        while len(changeset)>0:
            #resolve deps
            
            added = []
            for i in changeset:
            
                #get or create message
                comment, isnew = __msg_by_cid(i)
                #if message is new add data
                if isnew:
                    
                    comment.title = changeset[i]['subject']
                    comment.date = changeset[i]['date']
                    comment.text = string.replace(changeset[i]['body'], "\n", "<br/>")
                    comment.misc = {'parenttalkid':changeset[i]['jitemid'],'frozen':False,'screened':False,
                                      'deleted':False}
                    comment.poster = Account.register(type(self), umap[changeset[i]['posterid']])
                
                    #add message to tree of comments
                    par_id = changeset[i]['parentid']
                    mtp = COMMENT
                    if par_id == 0:
                        print 'jitemid'
                        par_id = changeset[i]['jitemid']
                        mtp = POST
                    par_msg, isnew = __msg_by_cid(par_id, mtp)
                    
                    if isnew:
                        par_msg.del_msg_tree()
                        comment.del_msg_tree()
                        continue
                    if (comment.date == None) or (comment.date == ""):
                        comment.date = par_msg.date
                        #raise LookupError("Message not founded in DB", "Trying to comment not existing message")
                      
                    jitem_msg, isnew = __msg_by_cid(changeset[i]['jitemid'],POST)
                    print jitem_msg.id
                    print jitem_msg.custom_id
                    print jitem_msg.journal.id
                    print jitem_msg.type
                    if isnew:
                        jitem_msg.del_msg_tree()
                        raise LookupError("Message not founded in DB", "Trying to comment not existing message")
                    
                    par_msg.add_comment(comment, jitem_msg)
                    sql_driver._db.commit()
                added.append(i)
            print added
            for i in added:
                #delet alredy exists comments
                del changeset[i]
            if len(added)==0:
                break
            #print '==========================================='
            #print len(changeset)
            #print '===========================================' 
            #if st_len < 0:
            #   raise LookupError("Comment tree corrupted", "Some comments haven't parents")
        
        self.clean_tree(frozen,screened)
        
        #froze messages
        for i in frozen:
            comment, isnew = __msg_by_cid(i)
            if isnew:
                comment.del_msg_tree()
                continue
                #raise LookupError("Message not founded in DB", "Trying to froze not existing message")
            
            misc = comment.misc
            misc['frozen'] = True
            comment.misc = misc
            #TODO: add to message class
            comment.froze_childs()
            self.add_marked(comment,'frozen')
            
        #screen messages
        for i in screened:
            comment, isnew = __msg_by_cid(i)
            if isnew:
                comment.del_msg_tree()
                continue
                #raise LookupError("Message not founded in DB", "Trying to screen not existing message")
            
            misc = comment.misc
            misc['screened'] = True
            comment.misc = misc
            #TODO: add to message class
            comment.screen_childs()
            self.add_marked(comment,'screened')
            
        #delete messages
        for i in deleted:
            comment, isnew = __msg_by_cid(i)
            if isnew:
                comment.del_msg_tree()
                continue
                #raise LookupError("Message not founded in DB", "Trying to delete not existing message")
            
            misc = comment.misc
            misc['deleted'] = True
            comment.misc = misc
            comment.del_comment_local()

        
    def __confirm (self, page, session=None, uparams=None):
        """
        Confirm operation. (get form from HTML-page, fill it and send)
        
        @param page (string list) HTML-page source
        @param session (string) Cookie string for auhtentification (Default None)
        @param uparams (dict) additional params for form (Default None)
        @return None
        """
        def strsinstr(srcstr, pats):
            for i in pats:
                if string.find(srcstr, i)>=0:
                    return True
            return False  
            
            
        if session:
            headers = {'Cookie':"ljsession=%s" %(session)}
        else:
            headers = None
        # Finds useful content
        pat = ("<!-- your content here -->","<!-- Content -->")
        while not strsinstr(page.pop(0), pat):
            pass

        pat = ("<!-- /content -->","<!-- End \#content -->")
        while not strsinstr(page.pop(), pat):
            pass
          
        #Look for form in HTML-page
        html_data = string.join(page)
        parcer = CHTMLParser()
        parcer.feed(html_data)
        #Update params by user's params
        params = parcer.params
        if uparams:
            params.update(uparams)
        #send request
        print "============================="
        print parcer.request
        print urllib.urlencode(params)
        print headers
        print "============================="
        req=urllib2.Request(parcer.request, urllib.urlencode(params), headers=headers)
        print req
        r = urllib2.urlopen(req)

    """
    ===============================================================
    deprecated and testing finct.
    ===============================================================
    """



    def _refresh_inbox(self):

        param = {
                 'username': self.username,
                 'hpassword': self.hpassword,
                 'ver': '1',
                 'itemshow': 100,
                 'skip': 20
                }
        result = self.exec_xmlrpc('getinbox', param)

        return super(LivejAccount, self).get_inbox()


    def refresh_user_info(self):
        """
        Get informarion about user. Use rdf parser and don't new password.
        
        @return None
        """

        '''
        from rdflib.Graph import Graph
        from rdflib import Namespace
        FOAF = Namespace("http://xmlns.com/foaf/0.1/")

        g = Graph()
        g.parse("http://scribo-rpc.livejournal.com/data/foaf.xml/")

        r = g.query('SELECT ?aname ?bname WHERE { ?aname foaf:name ?bname }',
                    initNs=dict(foaf=FOAF)).serialize('python')
        self.nick = unicode(r[0][1])

        r = g.query('SELECT ?aname ?bname WHERE { ?aname foaf:dateOfBirth ?bname }',
                    initNs=dict(foaf=FOAF)).serialize('python')
        self.date_of_birth = unicode(r[0][1])

        r = g.query('SELECT ?aname ?bname WHERE { ?aname foaf:img ?bname }',
                    initNs=dict(foaf=FOAF)).serialize('python')
        self.path_to_ava = unicode(r[0][1])

        pass 
        '''

    '''
    def _get_posts_nopass(self):
        import feedparser
        atom_lenta = feedparser.parse(
                        'http://scribo-rpc.livejournal.com/data/atom.xml/')
        print atom_lenta.feed.title
        print atom_lenta.feed.description

        for entry in atom_lenta.entries:
            print entry.title


    def add_comment(self):
        
        param = {
                 'username': self.username,
                 'hpassword': self.hpassword,
                 'ver': '1',
                 'body': 'comment text',
                 'subject': ' repl subj))',
                 'ditemid': 13757
                }
        result = self._server.LJ.XMLRPC.addcomment(param)

        return result


    def get_last(self):
        """
        Get last event from LJ
        """
        
        result = self._server.LJ.XMLRPC.getevents({
                                                  'username': self.username,
                                                  'hpassword': self.hpassword,
                                                  'ver': '1',
                                                  'lineendings': 'unix',
                                                  "selecttype": "lastn",
                                                  "itemid":'-1',
                                                  "howmany": '1'
                                                 })

        return result['events'][0]
    '''



