#!/usr/bin/env python

"""
===============================================================
    Contains definition of class Profile
===============================================================
"""

__author__ = "Mezhenin Artoym <mezhenin@cs.karelia.ru>"
__version__ = "$Revision: 18 $"
__date__ = "$Date: 2010/02/06 $"
__copyright__ = ""
__license__ = "GPLv2"

import time

from ScriboExc import WarningExc
from Structure import Structure
from Message import Message, POST
from Account import Account
from SqlDriver import sql_driver, SQL_PROFILE_FIELDS, TIME_FORMAT


class Profile (Structure):

    """
        Class contains user profile and profiles of friends (in future). 
    
    @exception WarningExc All methods can raise WarningExc.
    @author Mezhenin Artoym <mezhenin@cs.karelia.ru> 
    @version 0.3
    @date 2010/01/12
    """

    #def __new__(cls, id=None):
    #    return super(Profile, cls).__new__(cls, id)


    def __init__(self, id=None):
        """
        Constuctor for class.
        
        @param id (string) app_id of profile in table. If there is no record
                           with such app_id it will be created.
        @return None
        @todo test
        """
        super(Profile, self).__init__("Profiles", id)


    """
    ===============================================================
        Getter methods
    ===============================================================
    """

    def get_accounts(self):
        """
        Get list of accounts in this profile.

        @return (list) a list of Account instance
        @todo test
        """

        query = 'SELECT id FROM Accounts WHERE prof=?'

        id_list = sql_driver.select_mcmd(query, (self._id,))

        accounts = []
        for i in id_list:
            accounts.append(Account(int(i[0])))

        return accounts

    def get_def_account(self):
        """
        Get field from Profiles table.

        @return (Account) account by default
        """
        res = self._get_field("def_account")
        if res:
            return Account(int(res))

        return None


    def get_posts(self):
        """
        Get list of posts from all account in this profile
        @return (list) list of Message instances
        """
        acc_list = self.accounts

        if not acc_list:
            return []

        """
        compose a complex query to the database that would select 
        all the posts belong to one of the profile accounts
        """

        #SELECT id from messages  order by  date desc, id

        cmd = "SELECT id FROM Messages WHERE type=? AND ( journal=?"
        for i in xrange(1, len(acc_list)):
            cmd += " OR journal=?"

        cmd += ") ORDER BY date DESC, id DESC"

        arg_list = [POST, ]
        for i in acc_list:
            arg_list.append(i.id)

        # get a list of id of posts
        id_list = sql_driver.select_mcmd(cmd, arg_list)

        # create list of Messages from id's 
        post_list = []
        for i in id_list:
            post_list.append(Message(i[0]))

        return post_list

    """
    ===============================================================
        Setter methods
    ===============================================================
    """

    def set_def_account(self, value):
        """
        Change field value in Profiles table.

        @param value (Account) account to be set as default for this profile
        @exception WarningExc if value is not Account, or value wasn't added to
                   this profile
        @return None
        """
        if value == None:
            self._set_field("def_account", None)
        elif value.prof == self:
            self._set_field("def_account", value._id)
        else:
            raise WarningExc("Profile setter",
                             "value is not a memder of this profile")

    """
    ===============================================================
        Init getter/setter  methods
    ===============================================================
    """
    ## property
    accounts = property(get_accounts)
    ## property
    def_account = property(get_def_account, set_def_account)

    """
    ===============================================================
        Other methods
    ===============================================================
    """
    def add_account(self, acc):
        """
        Add account to this profile.
        
        @param acc (Account) account we want to add 
        @return None
        """

        acc.prof = self


    def remove_account(self, acc):
        if acc in self.accounts:
            acc.prof = None


    '''
    def del_profile(self):
        """
        Delete this profile with all accounts from DB
        
        @return None
        """
        acc_list = self.accounts
        for i in acc_list:
            i.prof = None

        self._del_struct()
    '''

    def copy(self):
        """
        Make full copy of this profile.
        
        @return (Profile) new profile with exact values
        """

        return Profile(self._copy_fields(SQL_PROFILE_FIELDS))

    def get_friends_posts(self, last_ndays=None):
        """
        Get list with friends posts for last N days
        @param last_ndays (int) number of last days
        @return (list) list with posts, if last_ndays is not setm, all avalible
                friends posts will be returned 
        """

        friends = []
        for i in self.accounts:
            friends += i.friends_list

        if not friends:
            return []

        """
        compose a complex query to the database that would select 
        all the posts belong to one of our friends
        """

        #SELECT id from messages  order by  date desc, id

        cmd = "SELECT id FROM Messages WHERE type=? AND ( journal=?"
        for i in xrange(1, len(friends)):
            cmd += " OR journal=?"

        if last_ndays:
            date = time.localtime()
            date = time.mktime(date)
            date = date - 60 * 60 * 24 * last_ndays
            date = time.localtime(date)
            date = time.strftime(TIME_FORMAT, date)
            cmd += " ) AND ( date>='%s'" % date

        cmd += ") ORDER BY date DESC, id DESC"

        arg_list = [POST, ] + friends

        # get a list of id of posts
        id_list = sql_driver.select_mcmd(cmd, arg_list)

        # create list of Messages from id's 
        post_list = []
        for i in id_list:
            post_list.append(Message(i[0]))

        return post_list



"""
Init profile of user
"""
main_prof = Profile(1)
