/*
 *  scout Maemo 5 calendar, contact and conversations search tool
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <clockd/libtime.h>
#include <CMulticalendar.h>
#include <CCalendar.h>
#include <CJournal.h>
#include <libintl.h>
#include <locale.h>

#include "calendar-event-preview.h"
#include "calendar-journal-editor.h"

typedef struct _journal_ui_data_t {
  GtkWidget* window;
  GtkWidget *text_view;
  GtkWidget* last_modified_label;
  CCalendar* calendar;
  CJournal* journal;

} journal_ui_data_t;

static void 
_set_journal_modified_label_text(CJournal* journal, GtkLabel* label)
{
  gchar* time_formatted = _format_date_time_stamp(journal->getLastModified());
  gchar* last_modified_text = g_strdup_printf(dgettext("calendar", "cal_va_note_saved"), time_formatted);
  gtk_label_set_text(label, last_modified_text);
  g_free(last_modified_text);
  g_free(time_formatted);
}

static void
_save_journal_entry(GtkButton* button, gpointer user_data)
{
  journal_ui_data_t* journal_ui_data = (journal_ui_data_t*)user_data;
  GtkTextBuffer* text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(journal_ui_data->text_view));
  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(text_buffer, &start);
  gtk_text_buffer_get_end_iter(text_buffer, &end);
  gchar* text = gtk_text_buffer_get_text(text_buffer,
					 &start, 
					 &end,
					 FALSE);
  string update_text = string(text);
  journal_ui_data->journal->setSummary(update_text);
  int error = 0;
  journal_ui_data->calendar->modifyJournal(journal_ui_data->journal, error);
  g_free(text);
  gtk_widget_hide(GTK_WIDGET(journal_ui_data->window));
}

static void
_delete_journal_ui_data(GtkWidget* widget,
		      gpointer user_data)
{
  journal_ui_data_t* journal_ui_data = (journal_ui_data_t*)user_data;
  delete journal_ui_data->calendar;
  delete journal_ui_data->journal;
  g_free(journal_ui_data);
}

void
open_calendar_journal_window(osso_context_t* osso, int calendar_id, const gchar* event_id)
{
  int error = 0;  
  CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
  if(calendar)
  {
    CJournal* journal = calendar->getJournal(event_id, error);
    if(journal)
    {
      GtkWidget* window;
      GtkWidget* text_view;
      GtkWidget* last_modified_label;
      GtkWidget* save_entry_button;
      window = hildon_stackable_window_new();
      text_view = hildon_text_view_new();
      last_modified_label = gtk_label_new("");
      _set_journal_modified_label_text(journal, GTK_LABEL(last_modified_label));
      
      journal_ui_data_t* journal_ui_data = g_new0(journal_ui_data_t, 1);
      journal_ui_data->window = window;
      journal_ui_data->text_view = text_view;
      journal_ui_data->last_modified_label = last_modified_label;
      journal_ui_data->calendar = calendar;
      journal_ui_data->journal = journal;
      save_entry_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      dgettext("hildon-libs", "wdgt_bd_save"),
						      NULL);
      g_signal_connect(save_entry_button, "clicked", G_CALLBACK(_save_journal_entry), journal_ui_data);
      calendar_add_open_calendar_button_menu(window, osso, time_get_time());

      g_signal_connect(window, "hide", G_CALLBACK(_delete_journal_ui_data), journal_ui_data);
      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_CHAR);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(text_view), TRUE);
      GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
      GtkWidget* vbox = gtk_vbox_new(FALSE, 3);
      
      gtk_text_buffer_insert_at_cursor(buffer,
				       journal->getSummary().c_str(),
				       journal->getSummary().size());
      gtk_box_pack_start(GTK_BOX(vbox), last_modified_label, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), text_view, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), save_entry_button, FALSE, FALSE, 0);
      gtk_container_add(GTK_CONTAINER(window), vbox);
      gtk_window_set_title(GTK_WINDOW(window), dgettext("calendar", "cal_ti_edit_note"));
      gtk_widget_show_all(window);
    }
  }
}
