/*
 *  scout Maemo 5 calendar, contact and conversations search tool
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "calendar-event-editor.h"
#include <gtk/gtk.h>
#include <libintl.h>
#include <locale.h>
#include <hildon/hildon.h>
#include <CMulticalendar.h>
#include <CCalendar.h>
#include <CEvent.h>
#include <Common.h>
#include "calendar-event-preview.h"
#include "calendar-recurrence-editor.h"

typedef struct _event_ui_data_t {
  GtkWidget* window;
  GtkWidget* summary_text_view;
  GtkWidget* location_text_view;
  GtkWidget* description_text_view;
  GtkWidget* all_day_check_button;
  GtkWidget* start_date_button;
  GtkWidget* start_time_button;
  GtkWidget* end_date_button;
  GtkWidget* end_time_button;
  GtkWidget* alarm_picker_selector;
  CCalendar* calendar;
  CEvent* event;
} event_ui_data_t;

static void
_set_alarm_type_from_selector(CEvent* event, HildonTouchSelector* alarm_picker_selector)
{
  int alarm_type = hildon_touch_selector_get_active(alarm_picker_selector, 0);
  if(alarm_type == 0)
  {
    event->removeAlarm();
  }
  else
  {
    switch(alarm_type)
    {
    case 1:
      event->setAlarmBefore(0);
      break;
    case 2:
      event->setAlarmBefore(5 * 60);
      break;
    case 3:
      event->setAlarmBefore(15 * 60);
      break;
    case 4:
      event->setAlarmBefore(30 * 60);
      break;
    case 5:
      event->setAlarmBefore(60 * 60);
      break;
    case 6:
      event->setAlarmBefore(3 * 60 * 60);
      break;
    case 7:
      event->setAlarmBefore(12 * 60 * 60);
      break;
    }
  }
}

static void
_delete_event_ui_data(GtkWidget* widget,
		      gpointer user_data)
{
  event_ui_data_t* event_ui_data = (event_ui_data_t*)user_data;
  delete event_ui_data->calendar;
  delete event_ui_data->event;
  g_free(event_ui_data);
}

static gchar*
_create_text_for_recurrence(CEvent* event)
{
  CRecurrence* recurrence = event->getRecurrence();
  if(recurrence)
  {
    const gchar* recurrence_text = NULL;
    switch(event->getRtype())
    {
    case E_DAILY:
      recurrence_text = dgettext("calendar", "cal_va_every_day");
      break;
    case E_WEEKDAY:
      recurrence_text = dgettext("calendar", "cal_va_every_workday");
      break;
    case E_WEEKLY:
      recurrence_text = dgettext("calendar", "cal_va_every_week");
      break;
    case E_MONTHLY:
      recurrence_text = dgettext("calendar", "cal_va_every_month");
      break;
    case E_YEARLY:
      recurrence_text = dgettext("calendar", "cal_va_every_year");
      break;
    default:
      return NULL;
    }
    vector<CRecurrenceRule*> rules = recurrence->getRecurrenceRule();
    if(rules.size()==1)
    {
      if(rules[0]->checkRuleIsComplex())
      {
	return NULL;
      }
      CRecurrenceRule cr;
      cr.rruleParser(rules[0]->toString());
      int interval = cr.getInterval();
      if(interval>1)
	return g_strdup_printf("%s (%s %d)", recurrence_text, gettext("Interval"), interval);
      else
	return g_strdup_printf("%s", recurrence_text);
    }
  }
  return g_strdup(dgettext("calendar", "cal_va_never"));  
}

static
GtkWidget*
_create_alarm_touch_selector_for_event(CEvent* event)
{
  GtkWidget* alarm_picker_selector = hildon_touch_selector_new_text();
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				      dgettext("calendar", "cal_va_none"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				    dgettext("calendar", "cal_va_0_min"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				    dgettext("calendar", "cal_va_5_min"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				    dgettext("calendar", "cal_va_15_min"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				    dgettext("calendar", "cal_va_30_min"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				    dgettext("calendar", "cal_va_1_hour"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				    dgettext("calendar", "cal_va_3_hours"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				    dgettext("calendar", "cal_va_day_before"));
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				   0, 0);
  int alarm_seconds = event->getAlarmBefore();

  if(alarm_seconds == 0)
  {
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				     0, 1);
  }
  else if(alarm_seconds == (5 * 60))
  {
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				     0, 2);
  }
  else if(alarm_seconds == (15 * 60))
  {
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				     0, 3);
  }
  else if(alarm_seconds == (30 * 60))
  {
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				     0, 4);
  }
  else if(alarm_seconds == (60 * 60))
  {
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				     0, 5);
  }
  else if(alarm_seconds == (3 * 60 * 60))
  {
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				     0, 6);
  }
  else if(alarm_seconds == (12 * 60 * 60))
  {
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(alarm_picker_selector),
				     0, 7);
  }
  return alarm_picker_selector;
}

static void
_set_date_time(GtkWidget* date_button, GtkWidget* time_button, time_t time_stamp)
{
  struct tm* time_stamp_tm = localtime(&time_stamp);
  hildon_date_button_set_date(HILDON_DATE_BUTTON(date_button),
			      time_stamp_tm->tm_year + 1900,
			      time_stamp_tm->tm_mon,
			      time_stamp_tm->tm_mday);
  hildon_time_button_set_time(HILDON_TIME_BUTTON(time_button),
			      time_stamp_tm->tm_hour,
			      time_stamp_tm->tm_min);
  
}
static void
_set_date_time_for_event(event_ui_data_t* event_ui_data)
{
  struct tm time_stamp;
  time_t time_stamp_time;
  guint year, mon, day, hour, min;
  hildon_date_button_get_date(HILDON_DATE_BUTTON(event_ui_data->start_date_button),
			 &year, &mon, &day);
  hildon_time_button_get_time(HILDON_TIME_BUTTON(event_ui_data->start_time_button),
			 &hour, &min);
  year-=1900;
  time_stamp.tm_year = year;
  time_stamp.tm_mon = mon;
  time_stamp.tm_mday = day;
  time_stamp.tm_hour = hour;
  time_stamp.tm_min = min;

  time_stamp_time = mktime(&time_stamp);
  event_ui_data->event->setDateStart(time_stamp_time);
  hildon_date_button_get_date(HILDON_DATE_BUTTON(event_ui_data->end_date_button),
			      &year, &mon, &day);
  hildon_time_button_get_time(HILDON_TIME_BUTTON(event_ui_data->end_time_button),
			      &hour, &min);
  year-=1900;
  time_stamp.tm_year = year;
  time_stamp.tm_mon = mon;
  time_stamp.tm_mday = day;
  time_stamp.tm_hour = hour;
  time_stamp.tm_min = min;
  time_stamp_time = mktime(&time_stamp);
  event_ui_data->event->setDateEnd(time_stamp_time);
}
static void
_show_hide_time_selector(HildonCheckButton* check_button, gpointer user_data)
{
  event_ui_data_t* event_ui_data = (event_ui_data_t*)user_data;
  if(hildon_check_button_get_active(check_button))
  {
    gtk_widget_hide(event_ui_data->start_time_button);
    gtk_widget_hide(event_ui_data->end_time_button);
  }
  else
  {
    gtk_widget_show(event_ui_data->start_time_button);
    gtk_widget_show(event_ui_data->end_time_button);
  }
}

static void
_save_event_entry(GtkButton* button, gpointer user_data)
{

  event_ui_data_t* event_ui_data = (event_ui_data_t*)user_data;
  const gchar* title_c = hildon_entry_get_text(HILDON_ENTRY(event_ui_data->summary_text_view));
  if(!title_c || strlen(title_c)==0)
  {
    hildon_banner_show_information(NULL, NULL, dgettext("calendar", "cal_ib_add_title"));
    return;
  }
  string update_text;

  update_text = string(title_c);
  event_ui_data->event->setSummary(update_text);

  const gchar* location_c = hildon_entry_get_text(HILDON_ENTRY(event_ui_data->location_text_view));
  update_text = string(location_c);
  event_ui_data->event->setLocation(update_text);

  GtkTextBuffer* description_text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(event_ui_data->description_text_view));
  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(description_text_buffer, &start);
  gtk_text_buffer_get_end_iter(description_text_buffer, &end);
  gchar* text = gtk_text_buffer_get_text(description_text_buffer,
					 &start, 
					 &end,
					 FALSE);
  update_text = string(text);

  event_ui_data->event->setDescription(update_text);
  if(hildon_check_button_get_active(HILDON_CHECK_BUTTON(event_ui_data->all_day_check_button)))
  {
    event_ui_data->event->setAllDay(1);
  }
  else
  {
    event_ui_data->event->setAllDay(0);
  }
  _set_date_time_for_event(event_ui_data);
  _set_alarm_type_from_selector(event_ui_data->event, HILDON_TOUCH_SELECTOR(event_ui_data->alarm_picker_selector));
  
  int error = 0;
  event_ui_data->calendar->modifyEvent(event_ui_data->event, error);
  g_free(text);
  gtk_widget_hide(GTK_WIDGET(event_ui_data->window));
}

void
open_calendar_event_window(osso_context_t* osso, int calendar_id, const gchar* event_id)
{
  int error = 0;  
  CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
  if(calendar)
  {
    CEvent* event = calendar->getEvent(event_id, error);
    if(event)
    {
      GtkWidget* window;
      GtkWidget* summary_text_view;
      GtkWidget* location_text_view;
      GtkWidget* description_text_view;
      GtkWidget* open_calendar_button;
      GtkWidget* save_entry_button;
      GtkWidget* all_day_check_button;
      GtkWidget* start_date_button;
      GtkWidget* start_time_button;
      GtkWidget* end_date_button;
      GtkWidget* end_time_button;
      GtkWidget* alarm_button;
      GtkWidget* recurrence_button;
      GtkWidget* interval_button;

      window = hildon_stackable_window_new();

      calendar_add_open_calendar_button_menu(window, osso, event->getDateStart());

      summary_text_view = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
      location_text_view = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
      description_text_view = hildon_text_view_new();
      start_date_button = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
      end_date_button = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
      start_time_button = hildon_time_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
      end_time_button = hildon_time_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
      all_day_check_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
      
      _set_date_time(start_date_button, start_time_button, event->getDateStart());
      _set_date_time(end_date_button, end_time_button, event->getDateEnd());

      hildon_button_set_title(HILDON_BUTTON(start_date_button),
			      dgettext("calendar", "cal_fi_startdate"));
      hildon_button_set_title(HILDON_BUTTON(end_date_button),
			      dgettext("calendar", "cal_fi_enddate"));

      hildon_button_set_title(HILDON_BUTTON(start_time_button),
			      NULL);
      hildon_button_set_title(HILDON_BUTTON(end_time_button),
			      NULL);
      gtk_button_set_label(GTK_BUTTON(all_day_check_button),
			   dgettext("calendar", "cal_va_allday_long"));
			      
      event_ui_data_t* event_ui_data = g_new0(event_ui_data_t, 1);
      event_ui_data->window = window;
      event_ui_data->summary_text_view = summary_text_view;
      event_ui_data->location_text_view = location_text_view;
      event_ui_data->description_text_view = description_text_view;
      event_ui_data->all_day_check_button = all_day_check_button;
      event_ui_data->calendar = calendar;
      event_ui_data->event = event;
      event_ui_data->start_date_button = start_date_button;
      event_ui_data->end_date_button = end_date_button;
      event_ui_data->start_time_button = start_time_button;
      event_ui_data->end_time_button = end_time_button;

      open_calendar_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
							 HILDON_BUTTON_ARRANGEMENT_VERTICAL,
							 dgettext("calendar", "cal_ap_name"), 
							 NULL);

      hildon_check_button_set_active(HILDON_CHECK_BUTTON(all_day_check_button),
				     event_ui_data->event->getAllDay()==1);

      save_entry_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      dgettext("hildon-libs", "wdgt_bd_save"), 
						      NULL);

      alarm_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL);

      recurrence_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
						   HILDON_BUTTON_ARRANGEMENT_VERTICAL);

      interval_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
      
      hildon_button_set_title(HILDON_BUTTON(alarm_button), 
			      dgettext("calendar", "cal_fi_alarm"));
      hildon_button_set_title(HILDON_BUTTON(recurrence_button), 
			      dgettext("calendar", "cal_fi_repeat"));
      hildon_button_set_title(HILDON_BUTTON(interval_button), 
			      gettext("Interval"));

      GtkWidget* alarm_picker_selector = _create_alarm_touch_selector_for_event(event);
      event_ui_data->alarm_picker_selector = alarm_picker_selector;
      hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(alarm_button),
					HILDON_TOUCH_SELECTOR(alarm_picker_selector));
      
      g_signal_connect(recurrence_button, "clicked", G_CALLBACK(open_recurrence_selection_dialog), event_ui_data->event);
      gchar* recurrence_value = _create_text_for_recurrence(event_ui_data->event);
      if(recurrence_value == NULL)
      {
	gtk_widget_set_sensitive(GTK_WIDGET(recurrence_button), FALSE);
	hildon_button_set_value(HILDON_BUTTON(recurrence_button),
				dgettext("calendar", "cal_fi_repeat_complex"));
	
      }
      else
      {
	gtk_widget_set_sensitive(GTK_WIDGET(recurrence_button), TRUE);
	hildon_button_set_value(HILDON_BUTTON(recurrence_button),
				recurrence_value);
	g_free(recurrence_value);	
      }

      
      g_signal_connect(save_entry_button, "clicked", G_CALLBACK(_save_event_entry), event_ui_data);
      g_signal_connect(window, "hide", G_CALLBACK(_delete_event_ui_data), event_ui_data);
      g_signal_connect(all_day_check_button, "toggled", G_CALLBACK(_show_hide_time_selector), event_ui_data);
      hildon_entry_set_placeholder(HILDON_ENTRY(summary_text_view), "summary");
      hildon_entry_set_text(HILDON_ENTRY(summary_text_view),
			    event->getSummary().c_str());

      hildon_entry_set_placeholder(HILDON_ENTRY(location_text_view), dgettext("calendar", "cal_fi_location"));
      hildon_entry_set_text(HILDON_ENTRY(location_text_view),
			    event->getLocation().c_str());

      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(description_text_view), GTK_WRAP_WORD);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(description_text_view), TRUE);

      hildon_gtk_text_view_set_placeholder_text(GTK_TEXT_VIEW(description_text_view), dgettext("calendar", "cal_va_placeholder_description"));

      GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(description_text_view));
      gtk_text_buffer_insert_at_cursor(buffer,
				       event->getDescription().c_str(),
				       event->getDescription().size());

      GtkWidget* vbox = gtk_vbox_new(FALSE, 3);

      gtk_box_pack_start(GTK_BOX(vbox), summary_text_view, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), description_text_view, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), location_text_view, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), all_day_check_button, FALSE, FALSE, 0);
      GtkWidget* date_box = gtk_vbox_new(TRUE, 0);
      GtkWidget* time_box = gtk_vbox_new(TRUE, 0);
      GtkWidget* date_time_box = gtk_hbox_new(FALSE, 0);

      gtk_box_pack_start(GTK_BOX(date_box), start_date_button, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(date_box), end_date_button, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(time_box), start_time_button, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(time_box), end_time_button, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(date_time_box), date_box, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(date_time_box), time_box, FALSE, FALSE, 0);

      gtk_box_pack_start(GTK_BOX(vbox), date_time_box, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), alarm_button, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), recurrence_button, FALSE, FALSE, 0);

      GtkWidget* vbox2 = gtk_vbox_new(FALSE,3);
      GtkWidget* pan = hildon_pannable_area_new();
      hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pan), vbox);
      gtk_box_pack_start(GTK_BOX(vbox2), pan, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox2), save_entry_button, FALSE, FALSE, 0);

      gtk_container_add(GTK_CONTAINER(window), vbox2);
      gtk_window_set_title(GTK_WINDOW(window), dgettext("calendar", "cal_ti_edit_event"));
      gtk_widget_show_all(window);
      if(event_ui_data->event->getAllDay() == 1)
      {
	gtk_widget_hide(start_time_button);
	gtk_widget_hide(end_time_button);
      }
    }
  }

}

