#include "config.h"

#include <hildon/hildon.h>
#include <libosso.h>
#include "scout-window.h"
#include "calendar-event-preview.hpp"
#include <rtcom-eventlogger/event.h>
#include <string.h>
#include <libintl.h>
#include <locale.h>
//#include <CalInterface.h>
#include <CMulticalendar.h>
#include <CComponent.h>
#include <Common.h>
#include <vector>
#include <ctime>
#include <clockd/libtime.h>
#include <rtcom-eventlogger-plugins/chat.h>
#include <rtcom-eventlogger-ui/rtcom-log-model.h>
#include <rtcom-eventlogger-ui/rtcom-log-view.h>
#include <rtcom-eventlogger-ui/rtcom-log-columns.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/dbus.h>
#include <rtcom-telepathy-glib/extensions.h>
#include <gconf/gconf-client.h>	


#define GC_ROOT "/apps/maemo/" PACKAGE 
#define GCONF_SEARCH_CONTACTS GC_ROOT "/search_contacts"
#define GCONF_SEARCH_CONVERSATIONS GC_ROOT "/search_conversations"
#define GCONF_SEARCH_CALENDARS GC_ROOT "/search_calendars"
#define GCONF_SEARCH_CASE_SENSITIVE GC_ROOT "/search_case_sensitive"
struct _ScoutWindowPrivate
{
  osso_context_t* osso;
  OssoABookAggregator* abook;
  OssoABookAccountManager* acc_manager;
  GtkListStore* result_list;
  RTComEl* rt_el;
  RTComElQuery* rt_com_chat_query;
  RTComElQuery* rt_com_sms_query;

  GtkWidget* search_entry_field;
  GtkWidget* search_touch_list;
  GtkWidget* search_button;
  GHashTable* cached_contact_avatar;
  GHashTable* cached_contact_attribute_icons;
  GHashTable* cached_account_icon;

  GHashTable* cached_calendar_id_icon;
  GHashTable* cached_calendar_type_icon;

  GdkPixbuf* search_category_contacts_icon;
  GdkPixbuf* search_category_calendar_icon;
  GdkPixbuf* search_category_conversation_sms_icon;
  GdkPixbuf* search_category_conversation_outgoing_sms_icon;
  GdkPixbuf* search_category_conversation_im_icon;
  GdkPixbuf* search_category_conversation_outgoing_im_icon;
  gboolean search_in_contacts;
  gboolean search_in_conversations;
  gboolean search_in_calendars;
  gboolean search_case_sensitive;
  gchar* current_search_text;
  GThread* search_thread;
  gboolean search_active;
};

#define SCOUT_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), SCOUT_TYPE_WINDOW, ScoutWindowPrivate))

G_DEFINE_TYPE(ScoutWindow, scout_window, HILDON_TYPE_STACKABLE_WINDOW)

enum
{
  SEARCH_CATEGORY_CONTACT = 0,
  SEARCH_CATEGORY_CONVERSATION_SMS = 1,
  SEARCH_CATEGORY_CONVERSATION_IM = 2,
  SEARCH_CATEGORY_CALENDAR_EVENT = 3,
  SEARCH_CATEGORY_CALENDAR_TODO = 4,
  SEARCH_CATEGORY_CALENDAR_JOURNAL = 5,

};

enum
{
  SEARCH_CATEGORY_ICON_COL = 0,
  SEARCH_RESULT_TEXT_COL = 1,
  SEARCH_RESULT_ITEM_TYPE_ICON_COL = 2,
  SEARCH_RESULT_ITEM_ID_ICON_COL = 3,
  SEARCH_RESULT_ITEM_TYPE_COL = 4,
  SEARCH_RESULT_ITEM_STRING_ID_COL = 5,
  SEARCH_RESULT_ITEM_INT_ID_COL = 6
};

void 
scout_window_set_abook(ScoutWindow* window, 
		       OssoABookAggregator* abook)
{
  window->priv->abook = abook;
  g_object_ref(window->priv->abook);
}

void 
scout_window_set_account_manager(ScoutWindow* window, 
				 OssoABookAccountManager* acc_manager)
{
  window->priv->acc_manager = acc_manager;
}

ScoutWindow*
scout_window_new(osso_context_t* osso)
{
  ScoutWindow* self = SCOUT_WINDOW(g_object_new(SCOUT_TYPE_WINDOW, NULL));
  SCOUT_WINDOW_GET_PRIVATE(self)->osso = osso;
  return self;
} 

static gchar*
_get_remote_name(ScoutWindow* self, const gchar* remote_uid)
{
  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL, NULL);
  RTComElIter* iter = NULL;
  iter = rtcom_el_get_events(self->priv->rt_el, query);
  g_object_unref(query);
  gboolean it = rtcom_el_iter_first(iter);
  if(it)
  {
    gchar* remote_name = NULL;
    rtcom_el_iter_get_values(iter, "remote-name", &remote_name, NULL);
    if(iter)
      g_object_unref(iter);
    return remote_name;
  }
  else
  {
    if(iter)
      g_object_unref(iter);
    return NULL;
  }
}

static gchar*
_get_abook_uid_name(ScoutWindow* self, const gchar* remote_uid)
{
  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL, NULL);
  RTComElIter* iter = NULL;
  iter = rtcom_el_get_events(self->priv->rt_el, query);
  g_object_unref(query);
  gboolean it = rtcom_el_iter_first(iter);
  if(it)
  {
    //    gchar* remote_name = NULL;
    gchar* abook_uid = NULL;
    rtcom_el_iter_get_values(iter, "remote-ebook-uid", &abook_uid, NULL);
    if(iter)
      g_object_unref(iter);
    return abook_uid;
  }
  else
  {
    if(iter)
      g_object_unref(iter);
    return NULL;
  }
}

const gchar*
_get_date_time_format()
{
  static gchar* format = NULL;
  if(format == NULL)
  {
    const gchar* time_format = NULL;
    const gchar* date_format = NULL;

    date_format = dgettext("hildon-libs", "wdgt_va_date");
    time_format = dgettext("hildon-libs", "wdgt_va_24h_time");
    format = g_strdup_printf("%s | %s", date_format, time_format);
  }
  return format;
}
const gchar*
_get_secondary_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "SecondaryTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

const gchar*
_get_active_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "ActiveTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

const gchar*
_get_default_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "DefaultTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

static const gchar*
_get_event_markup_template()
{
  static gchar* markup_template = NULL;
  if(markup_template == NULL)
  {
    markup_template = g_strdup_printf("<span color=\"%s\">%%s</span>  <span size=\"x-small\"><sup>%%s</sup></span>\n"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span size=\"x-small\" color=\"%s\">%%s</span>"
				      "<span size=\"x-small\" color=\"%s\">%%s</span>",
				      _get_default_text_color(),
				      _get_secondary_text_color(),
				      _get_active_text_color(),
				      _get_secondary_text_color());
  }
  return markup_template;
}

static const gchar*
_get_calendar_markup_template()
{
  static gchar* markup_template = NULL;
  if(markup_template == NULL)
  {
    markup_template = g_strdup_printf("<span color=\"%s\">%%s</span>  <span size=\"x-small\"><sup>%%s</sup></span>\n"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>",
				      _get_default_text_color(),
				      _get_secondary_text_color(),
				      _get_active_text_color(),
				      _get_secondary_text_color());
  }
  return markup_template;
}

static const gchar*
_get_contact_markup_template()
{
  static gchar* markup_template = NULL;
  if(markup_template == NULL)
  {
    markup_template = g_strdup_printf("<span color=\"%s\">%%s</span>\n"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>",
				      _get_default_text_color(),
				      _get_secondary_text_color(),
				      _get_active_text_color(),
				      _get_secondary_text_color());
  }
  return markup_template;
}

static GdkPixbuf*
_get_event_type_icon(ScoutWindow* self, const gint event_type)
{
  GdkPixbuf* event_type_icon = NULL;
  event_type_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_calendar_type_icon,
						   &event_type));
  if(!event_type_icon)
  {
    switch(event_type)
    {
    case /*E_TASK*/ 2:
      event_type_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						 "calendar_todo",
						 HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      break;
    case /*E_JOURNAL*/ 3:
      event_type_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						 "general_notes",
						 HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      break;
    case /*E_BIRTHDAY*/ 4:
      event_type_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						 "calendar_birthday",
						 HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      break;
    case 1:

      event_type_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						 "general_calendar",
						 HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    gint* key;
    key = (gint*)g_malloc(sizeof key);
    *key = event_type;
    g_hash_table_insert(self->priv->cached_calendar_type_icon,
			key,
			event_type_icon);
  }
  return event_type_icon;
}

static GdkPixbuf*
_get_calendar_id_icon(ScoutWindow* self, const gint calendar_id)
{
  GdkPixbuf* calendar_id_icon = NULL;
  calendar_id_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_calendar_id_icon,
						    &calendar_id));
  if(!calendar_id_icon)
  {
    int error = 0;
    CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
    int color = calendar->getCalendarColor();
    delete calendar;
    switch(color)
    {
      case COLOUR_DARKBLUE:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_darkblue",
					  HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_DARKGREEN:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_darkgreen", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_DARKRED:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_darkred", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_ORANGE:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_orange", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_VIOLET:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_violet", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_YELLOW:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_yellow", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_BLUE:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_blue", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_RED:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_red", 
					  HILDON_ICON_PIXEL_SIZE_FINGER ,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_GREEN:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_green",
					  HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      default:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_white", 
					  HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    gint* key;
    key = (gint*) g_malloc(sizeof key);
    *key = calendar_id;
    g_hash_table_insert(self->priv->cached_calendar_id_icon,
			key,
			calendar_id_icon);
  }
  return calendar_id_icon;
}

static GdkPixbuf*
_get_contact_attribute_icon(ScoutWindow* self, const gchar* attribute)
{
  GdkPixbuf* attribute_icon = NULL;
  attribute_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_contact_attribute_icons,
					       attribute));
  if(!attribute_icon)
  {
    if(g_strcmp0(attribute, EVC_TEL)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					     "general_call",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      
    }
    if(g_strcmp0(attribute, EVC_FN)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					     "general_business_card",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      
    }
    if(g_strcmp0(attribute, EVC_N)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_business_card",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      
    }
    else if(g_strcmp0(attribute, EVC_EMAIL)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_email",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else if(g_strcmp0(attribute, EVC_LABEL)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_map",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else if(g_strcmp0(attribute, EVC_BDAY)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_calendar",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else if(g_strcmp0(attribute, EVC_NICKNAME)==0 ||
	    g_strcmp0(attribute, EVC_ORG)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_business_card",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else if(g_strcmp0(attribute, EVC_NOTE) == 0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_notes",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else
    {
      McProfile* mc_profile = mc_profile_lookup_default_for_vcard_field(attribute);
      if(mc_profile)
      {
	const gchar* icon_name = mc_profile_get_branding_icon_name (mc_profile);
	
	if (!icon_name)
	  icon_name = mc_profile_get_icon_name (mc_profile);
	g_object_unref (mc_profile);	
	if(!icon_name)
	{
	  return NULL;
	}
        attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						  icon_name,
						  HILDON_ICON_PIXEL_SIZE_SMALL,
						  GTK_ICON_LOOKUP_NO_SVG, NULL);
      }
      else
      {
	g_warning("no profile for %s\n", attribute);
      }
    }
    if(attribute_icon)
      g_hash_table_insert(self->priv->cached_contact_attribute_icons,
			  g_strdup(attribute),
			  attribute_icon);
      
  }
  return attribute_icon;
}

static GdkPixbuf*
_get_account_icon(ScoutWindow* self, const gchar* account_uid)
{
  GdkPixbuf* account_icon = NULL;
  account_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_account_icon,
					       account_uid));
  if(!account_icon)
  {

    McAccount* account = osso_abook_account_manager_lookup_by_name(self->priv->acc_manager,
									   account_uid);
    if(account)
    {
      const gchar* id = mc_account_compat_get_profile(account);
      McProfile* profile = mc_profile_lookup(id);
      if(profile)
      {
	const gchar* icon_name = mc_profile_get_branding_icon_name (profile);
	
	if (!icon_name)
	  icon_name = mc_profile_get_icon_name (profile);
	
	g_object_unref (profile);
	
	if(icon_name)
	{
	  account_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						  icon_name,
						  HILDON_ICON_PIXEL_SIZE_SMALL,
						  GTK_ICON_LOOKUP_NO_SVG, NULL);
	}
      }
    }
    if(account_icon)
      g_hash_table_insert(self->priv->cached_account_icon,
			  g_strdup(account_uid),
			  account_icon);
      
    
  }
  return account_icon;
}

static GdkPixbuf*
_get_contact_avatar_icon(ScoutWindow* self, const gchar* abook_uid)
{
  GdkPixbuf* avatar_icon = NULL;
  avatar_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_contact_avatar,
					       abook_uid));
  if(!avatar_icon)
  {
    GList* entries = osso_abook_aggregator_lookup(self->priv->abook, abook_uid);
    if(entries)
    {
      OssoABookContact *original_contact = (OssoABookContact*)entries->data;
      GdkPixbuf* pixbuf = osso_abook_contact_get_avatar_pixbuf(original_contact,
							       NULL, NULL);
      if(pixbuf)
      {
	avatar_icon = gdk_pixbuf_scale_simple(pixbuf, 
					      48, 48,
					      GDK_INTERP_BILINEAR);
	g_object_unref(pixbuf);
      }
      g_list_free(entries);
    }

    if(avatar_icon == NULL)
    {
      avatar_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					     "general_default_avatar",
					     HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    if(avatar_icon == NULL)
      {
	g_warning("pushd NULL icon \n");
      }
    g_hash_table_insert(self->priv->cached_contact_avatar,
			g_strdup(abook_uid),
			avatar_icon);
    g_object_ref(avatar_icon);
  }
  return avatar_icon;
}


gboolean
_find_in_text(ScoutWindow* self, const gchar* text0, gchar** prefix, gchar** suffix)
{
  gchar* text = NULL;
  const gchar* faulty_char = NULL;
  if(!g_utf8_validate(text0, -1, &faulty_char))
  {
    /*
    g_warning("text invalid %s %c\n", text0, faulty_char);
    gchar* new_try = g_strndup(text0, faulty_char - text0);
    if(_find_in_text(self, new_try, prefix, suffix))
    {
      g_free(new_try);
      return TRUE;
    }
    g_free(new_try);
    new_try = g_strndup(faulty_char+1, strlen(faulty_char));
    if(_find_in_text(self, new_try, prefix, suffix))
    {
      g_free(new_try);
      return TRUE;
    }
    g_free(new_try);
    */
    return FALSE;
  }
  if(!self->priv->search_case_sensitive)
  {
    text = g_utf8_strdown(text0, -1);
  }
  else
  {
    text = g_strdup(text0);
  }
  gchar* pos = strstr(text, self->priv->current_search_text);
  int i = 0;
  int max_length = 80;
  if(pos!=NULL)
  {
    glong query_offset = g_utf8_pointer_to_offset(text, pos);
    glong query_char_len = g_utf8_strlen(self->priv->current_search_text, -1);
    glong text_char_len = g_utf8_strlen(text, -1);
    glong prefix_char_len = g_utf8_pointer_to_offset(text, pos);
    glong prefix_size = (pos - text);
    glong suffix_char_len = text_char_len - (query_char_len + query_offset);
    glong suffix_size = strlen(text) - (strlen(self->priv->current_search_text)+(pos - text));
    
    if(text_char_len <= max_length)
    {
      *prefix = g_strndup(text0, pos - text);
      *suffix = g_strndup(pos + strlen(self->priv->current_search_text), suffix_size);
    }
    else
    {
      prefix_char_len = (max_length/2 - (query_char_len/2 + query_char_len%2));
      suffix_char_len = (max_length/2 - query_char_len/2);
      if((query_offset - prefix_char_len < 0))
      {
	suffix_char_len += prefix_char_len - query_offset;
	prefix_char_len = query_offset;
      }
      else if((suffix_char_len + query_offset + query_char_len)>text_char_len)
      {
	prefix_char_len += (suffix_char_len + query_offset + query_char_len) - text_char_len;
	suffix_char_len = text_char_len - (query_offset + query_char_len);
      }
      *prefix = g_strndup(g_utf8_offset_to_pointer(text0, query_offset - prefix_char_len),
			  pos - g_utf8_offset_to_pointer(text, query_offset - prefix_char_len));
      *suffix = g_strndup(g_utf8_offset_to_pointer(text0, query_offset + query_char_len),
			  g_utf8_offset_to_pointer(text, 
						   query_offset + query_char_len + suffix_char_len) - 
			  g_utf8_offset_to_pointer(text,query_offset + query_char_len));
    }
    g_free(text);
    return TRUE;
  }
  g_free(text);
  return FALSE;
}

struct _result_data_t
{
  ScoutWindow* self;
  GdkPixbuf* category_icon;
  gchar* result_item_text;
  GdkPixbuf* item_type_icon;
  GdkPixbuf* item_id_icon;
  int search_category;
  gchar* string_id;
  int int_id;
};
typedef _result_data_t result_data_t;

static gboolean
_append_result_entry(gpointer data)
{
  result_data_t* result_data = (result_data_t*)data;
  GtkTreeIter iter;
  gtk_list_store_append(result_data->self->priv->result_list, &iter);
  gtk_list_store_set(result_data->self->priv->result_list, &iter,
		     SEARCH_CATEGORY_ICON_COL, result_data->category_icon,
		     SEARCH_RESULT_TEXT_COL, result_data->result_item_text,
		     SEARCH_RESULT_ITEM_TYPE_ICON_COL, result_data->item_type_icon,
		     SEARCH_RESULT_ITEM_ID_ICON_COL, result_data->item_id_icon,
		     SEARCH_RESULT_ITEM_TYPE_COL, result_data->search_category,
		     SEARCH_RESULT_ITEM_STRING_ID_COL, result_data->string_id,
		     SEARCH_RESULT_ITEM_INT_ID_COL, result_data->int_id,
		     -1);
  g_free(result_data->result_item_text);
  g_free(result_data->string_id);
  g_free(result_data);
  return FALSE;
}

static void
_add_result_entry(ScoutWindow* self, 
		  GdkPixbuf* category_icon,
		  const gchar* result_item_text,
		  GdkPixbuf* item_type_icon,
		  GdkPixbuf* item_id_icon,
		  int search_category,
		  const gchar* string_id,
		  int int_id)
{
  result_data_t* result_data = g_new0(result_data_t, 1);
  result_data->self = self;
  result_data->category_icon = category_icon;
  result_data->result_item_text = g_strdup(result_item_text);
  result_data->item_type_icon = item_type_icon;
  result_data->item_id_icon = item_id_icon;
  result_data->search_category = search_category;
  result_data->string_id = g_strdup(string_id);
  result_data->int_id = int_id;
  g_idle_add((GSourceFunc)_append_result_entry, result_data);
}

static void
_search_in_events(ScoutWindow* self, 
		  RTComElQuery* query, 
		  GdkPixbuf* icon,
		  GdkPixbuf* outgoing_icon,
		  int search_type)
{
  
  RTComElIter* rtcom_iter = rtcom_el_get_events(self->priv->rt_el, query);
  gboolean it = rtcom_el_iter_first(rtcom_iter);
  while(self->priv->search_active && it)
  {
    gchar* free_text = NULL;
    rtcom_el_iter_get_values(rtcom_iter, "free-text", &free_text, NULL);
    glong pos = -1;
    gchar* prefix = NULL;
    gchar* suffix = NULL;
    if(_find_in_text(self, free_text, &prefix, &suffix))
    {
      gchar* remote_uid = NULL;
      gchar* local_uid = NULL;
      gint event_id;
      time_t start_time;
      gint outgoing;
      rtcom_el_iter_get_values(rtcom_iter, "id", &event_id, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "remote-uid", &remote_uid, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "start-time", &start_time, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "local-uid", &local_uid, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "outgoing", &outgoing, NULL);
      if(remote_uid!=NULL)
      {
	gchar time_string[256];
	struct tm* time_stamp;
	time_stamp = localtime(&start_time);
	strftime(time_string, 256, _get_date_time_format(), time_stamp);
	gchar* remote_name = _get_remote_name(self, remote_uid);
	gchar* abook_uid = _get_abook_uid_name(self, remote_uid);

	GdkPixbuf* scaled_pixbuf = NULL;

	GdkPixbuf* account_icon = _get_account_icon(self, local_uid);

	if(abook_uid!=NULL)
	{
	  scaled_pixbuf = _get_contact_avatar_icon(self, abook_uid);
	}


	gchar* label_text = NULL;
	if(remote_name)
	  label_text = g_strdup(remote_name);
	else
	  label_text = g_strdup(remote_uid);
	gchar* text_name = 
	  g_markup_printf_escaped(_get_event_markup_template(),
				  label_text, 
				  time_string,
				  prefix,
				  self->priv->current_search_text,
				  suffix);
	g_free(label_text);


	if(outgoing == 1)
	{
	  _add_result_entry(self,
			    outgoing_icon,
			    text_name,
			    account_icon,
			    scaled_pixbuf,
			    search_type,
			    remote_uid,
			    event_id);
	}
	else
	{
	  _add_result_entry(self,
			    icon,
			    text_name,
			    account_icon,
			    scaled_pixbuf,
			    search_type,
			    remote_uid,
			    event_id);
	}
	g_free(remote_name);
	g_free(abook_uid);

	g_free(text_name);
      }
      g_free(remote_uid);
      g_free(local_uid);
    }
    g_free(prefix);
    g_free(suffix);
    g_free(free_text);
    it = rtcom_el_iter_next(rtcom_iter);
  }
  if(rtcom_iter)
    g_object_unref(rtcom_iter);
}

static void
search_in_conversations_chat(ScoutWindow* self)
{
  _search_in_events(self, 
		    self->priv->rt_com_chat_query, 
		    self->priv->search_category_conversation_im_icon,
		    self->priv->search_category_conversation_outgoing_im_icon,
		    SEARCH_CATEGORY_CONVERSATION_IM);
}

static void
search_in_conversations_sms(ScoutWindow* self)
{
  _search_in_events(self, 
		    self->priv->rt_com_sms_query, 
		    self->priv->search_category_conversation_sms_icon,
		    self->priv->search_category_conversation_outgoing_sms_icon,
		    SEARCH_CATEGORY_CONVERSATION_SMS);
}

static gchar*
_format_time(time_t start_time)
{
  gchar* time_string = (gchar*)g_malloc0(256);
  struct tm* time_stamp;
  time_stamp = localtime(&start_time);
  strftime(time_string, 256, _get_date_time_format(), time_stamp);
  return time_string;
}

static void
_find_and_insert_calendar_event(ScoutWindow* self, CComponent* component, int cal_id)
{
  gchar* prefix = NULL;
  gchar* suffix = NULL;

  int event_type = component->getType();
  if(_find_in_text(self, component->getSummary().c_str(), &prefix, &suffix) ||
     (event_type != 3 && 
      _find_in_text(self, component->getDescription().c_str(), &prefix, &suffix)))
  {
    gchar* title_text = NULL;
    const gchar* faulty_char = NULL;
    if(!g_utf8_validate(component->getSummary().c_str(), -1, &faulty_char))
    {
      g_warning("not valid %s\n", component->getSummary().c_str());
      title_text = g_strndup(component->getSummary().c_str(),
			     faulty_char - component->getSummary().c_str());
    }
    else
    {
      title_text = g_strdup(component->getSummary().c_str());
    }
    gchar* time_string = _format_time(component->getDateStart());
    gchar* result_text =
      g_markup_printf_escaped(_get_calendar_markup_template(),
			      title_text,
			      time_string,
			      prefix,
			      self->priv->current_search_text,
			      suffix);
    g_free(title_text);
    g_free(time_string);
    int search_category = 0;
    if(component->getType() == E_EVENT ||
       component->getType() == E_BDAY)
    {
      search_category = SEARCH_CATEGORY_CALENDAR_EVENT;
    }else if(component->getType() == E_TODO)
    {
      search_category = SEARCH_CATEGORY_CALENDAR_TODO;
    }else if(component->getType() == E_JOURNAL)
    {
      search_category = SEARCH_CATEGORY_CALENDAR_JOURNAL;
    }


    _add_result_entry(self,
		      self->priv->search_category_calendar_icon,
		      result_text,
		      _get_event_type_icon(self, event_type),
		      _get_calendar_id_icon(self, cal_id),
		      search_category,
		      component->getId().c_str(),
		      cal_id);
		      
    g_free(result_text);
  }
  g_free(prefix);
  g_free(suffix);
}

static void
search_in_calendar(ScoutWindow* self)
{
  int error = 0;
  time_t start_time = -1;
  time_t end_time = 2145909599;
  vector<CCalendar*> calendar_list = CMulticalendar::MCInstance()->getListCalFromMc();
  for(int j=0;self->priv->search_active && j<calendar_list.size();++j)
  {
    int offset = 0;
    int limit = 100;
    vector<CComponent*> component_list = calendar_list[j]->getAllEventsTodos(start_time,
									     end_time,
									     limit,
									     offset,
									     error);
    while(component_list.size()>0)
    {
      offset += component_list.size();
      for(int i = 0;i<component_list.size();++i)
      {
	_find_and_insert_calendar_event(self, component_list[i], calendar_list[j]->getCalendarId());
	delete component_list[i];
      }
      component_list = calendar_list[j]->getAllEventsTodos(start_time,
							   end_time,
							   limit,
							   offset,
							   error);
    }

    vector<CJournal*> journal_list = calendar_list[j]->getJournalListDetailed(error);
    for(int i = 0;i<journal_list.size();++i)
    {
      _find_and_insert_calendar_event(self, journal_list[i], calendar_list[j]->getCalendarId());
      delete journal_list[i];
    }
  }
  CMulticalendar::MCInstance()->releaseListCalendars(calendar_list);
}

static gboolean
_search_and_insert_contact_attribute(ScoutWindow* self, OssoABookContact* contact, const gchar* e_contact_field)
{
  GList* attributes = osso_abook_contact_get_attributes(E_CONTACT(contact), e_contact_field);
  for(;attributes;attributes=attributes->next)
  {
    GList* values = e_vcard_attribute_get_values((EVCardAttribute*)attributes->data);
    const gchar* name = e_vcard_attribute_get_name((EVCardAttribute*)attributes->data);
    const gchar* value = NULL;
    gchar* prefix = NULL;
    gchar* suffix = NULL;
    if(values)
    {
      value = (const gchar*) values->data;
    }
    if(value && 
       _find_in_text(self, value, &prefix, &suffix))
    {
      const gchar* abook_uid = osso_abook_contact_get_persistent_uid(contact);
      GdkPixbuf* avatar_icon = _get_contact_avatar_icon(self, abook_uid);
      GdkPixbuf* attribute_icon = _get_contact_attribute_icon(self, e_contact_field);

      gchar* text_name = 
	g_markup_printf_escaped(_get_contact_markup_template(), 
				osso_abook_contact_get_display_name(contact),
				prefix,
				self->priv->current_search_text,
				suffix);

      _add_result_entry(self,
			self->priv->search_category_contacts_icon,
			text_name,
			attribute_icon,
			avatar_icon,
			SEARCH_CATEGORY_CONTACT,
			osso_abook_contact_get_persistent_uid(contact),
			0);

      g_free(prefix);
      g_free(suffix);
      return true;
    }
  }
  return false;
}

static void
search_in_contacts(ScoutWindow* self)
{
  GList *contacts_list;
  GList *contacts;
  contacts = osso_abook_aggregator_list_master_contacts(self->priv->abook);
  contacts_list = contacts;
  for (;self->priv->search_active && contacts; contacts = contacts->next) 
  {
    OssoABookContact *original_contact = (OssoABookContact*)contacts->data;
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_FN))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_TEL))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_N))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_EMAIL))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_LABEL))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_BDAY))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_NICKNAME))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_NOTE))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, original_contact, EVC_ORG))
    {
      continue;
    }
    const GList *fields = (GList*)osso_abook_account_manager_get_primary_vcard_fields(self->priv->acc_manager);
    for (;fields; fields = fields->next) 
    {
      if(_search_and_insert_contact_attribute(self, original_contact, (const gchar*)fields->data))
	continue;
    }
  }
  g_list_free(contacts_list);
}

static gboolean
_threaded_search_done(gpointer data)
{
  g_return_val_if_fail(IS_SCOUT_WINDOW(data), FALSE);

  ScoutWindow* self = SCOUT_WINDOW(data);
  self->priv->search_active=FALSE;
  hildon_gtk_window_set_progress_indicator(GTK_WINDOW(self), 0);
  hildon_button_set_text(HILDON_BUTTON(self->priv->search_button),
			 dgettext("hildon-libs", "wdgt_bd_search"),NULL);
  gtk_widget_set_sensitive(GTK_WIDGET(self->priv->search_entry_field),
			   TRUE);
  gtk_widget_grab_focus(self->priv->search_entry_field);

  return FALSE;
}

static gpointer
_threaded_search_func(gpointer data)
{
  g_return_val_if_fail(IS_SCOUT_WINDOW(data), NULL);
  ScoutWindow* self = SCOUT_WINDOW(data);
  if(self->priv->search_active == FALSE)
    return NULL;
  if(self->priv->search_in_contacts)
    search_in_contacts(self);
  if(self->priv->search_in_conversations)
  {
    search_in_conversations_sms(self);
    search_in_conversations_chat(self);
  }
  if(self->priv->search_in_calendars)
    search_in_calendar(self);
  g_idle_add((GSourceFunc)_threaded_search_done, self);
  return NULL;
}

static void
_do_search(ScoutWindow* self)
{
  GtkEntry* entry = GTK_ENTRY(self->priv->search_entry_field);
  if(gtk_entry_get_text_length(entry)>0)
  {
    if(self->priv->current_search_text)
      g_free(self->priv->current_search_text);

    if(!self->priv->search_case_sensitive)
      self->priv->current_search_text = g_utf8_strdown(gtk_entry_get_text(entry), -1);
    else
      self->priv->current_search_text = g_strdup(gtk_entry_get_text(entry));
    gtk_list_store_clear(self->priv->result_list);
    hildon_button_set_text(HILDON_BUTTON(self->priv->search_button),
			   dgettext("hildon-libs", "wdgt_bd_stop"),NULL);

    hildon_gtk_window_set_progress_indicator(GTK_WINDOW(self), 1);
    gtk_widget_set_sensitive(GTK_WIDGET(self->priv->search_entry_field),
			     FALSE);

    GError* error = NULL;
    self->priv->search_active = TRUE;
    self->priv->search_thread = 
      g_thread_create(_threaded_search_func,
		      self,
		      TRUE,
		      &error);
    if(error)
    {
      hildon_gtk_window_set_progress_indicator(GTK_WINDOW(self), 0);

      hildon_button_set_text(HILDON_BUTTON(self->priv->search_button),
			     dgettext("hildon-libs", "wdgt_bd_search"), NULL);
      gtk_widget_set_sensitive(GTK_WIDGET(self->priv->search_entry_field),
			       TRUE);
      g_critical("can not create search thread.\n%s\n", error->message);
      g_error_free(error);
      error = NULL;
    }
  }
}

static void
on_search_enter(GtkEntry* entry, gpointer user_data)
{
  _do_search(SCOUT_WINDOW(user_data));
}

static void
on_search_clicked(GtkButton* button, gpointer user_data)
{
  if(!SCOUT_WINDOW(user_data)->priv->search_active)
  {
    _do_search(SCOUT_WINDOW(user_data));
  }
  else
  {
    SCOUT_WINDOW(user_data)->priv->search_active=FALSE;
    if(SCOUT_WINDOW(user_data)->priv->search_thread)
      g_thread_join(SCOUT_WINDOW(user_data)->priv->search_thread);
    SCOUT_WINDOW(user_data)->priv->search_thread = NULL;
  }
}

typedef struct _event_logger_window_data_t
{
  GtkWidget* view;
  RTComLogModel* model;
  GtkWidget* text_view;
  GtkWidget* window;
  int event_id;
  ScoutWindow* scout;
} event_logger_window_data_t;

static void
_scroll_to_event(GtkTreeModel *tree_model,
		 GtkTreePath  *path,
		 GtkTreeIter  *iter,
		 gpointer      user_data)
{
  event_logger_window_data_t* event_logger_window_data = (event_logger_window_data_t*)user_data;
  int event_id = 0;
  gtk_tree_model_get(tree_model,
		     iter,
		     RTCOM_LOG_VIEW_COL_EVENT_ID, &event_id,
		     -1);
  if(event_logger_window_data->event_id == event_id)
  {
    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(event_logger_window_data->view),
				 path, NULL,
				 TRUE,
				 0, 0);
    gchar* text = NULL;
    gtk_tree_model_get(tree_model, iter,
		       RTCOM_LOG_VIEW_COL_TEXT, &text,
		       -1);
    
    GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(event_logger_window_data->text_view));
    gtk_text_buffer_insert_at_cursor(buffer,
				     text,
				     strlen(text));
    g_free(text);
    hildon_gtk_window_set_progress_indicator(GTK_WINDOW(event_logger_window_data->window), 0);
  }
}

static void
_delete_event_scroller(GtkWidget* widget,
		       GdkEvent* event,
		       gpointer user_data)
{
  event_logger_window_data_t* event_logger_window_data  = (event_logger_window_data_t*)user_data;
  g_object_unref(event_logger_window_data->model);
  g_free(event_logger_window_data);
}

static void
_show_current_text(GtkTreeView* tree_view, 
		   GtkTreePath* path, 
		   GtkTreeViewColumn* column,
		   gpointer data)
{
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
  GtkTreeIter iter;
  GtkTextView* text_view = GTK_TEXT_VIEW(data);
  if(gtk_tree_model_get_iter(GTK_TREE_MODEL(gtk_tree_view_get_model(tree_view)),
			     &iter,
			     path))
  {
    gchar* text = NULL;
    gtk_tree_model_get(GTK_TREE_MODEL(gtk_tree_view_get_model(tree_view)), &iter,
		       RTCOM_LOG_VIEW_COL_TEXT, &text,
		       -1);
    GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
    GtkTextIter start;
    GtkTextIter end;
    gtk_text_buffer_get_start_iter(buffer, &start);
    gtk_text_buffer_get_end_iter(buffer, &end);
    gtk_text_buffer_delete(buffer, &start, &end);
    gtk_text_buffer_insert_at_cursor(buffer,
				     text,
				     strlen(text));
    g_free(text);
  }
}

static void
scout_open_conversation_app(ScoutWindow* self,
			    int event_id)
{
  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "id", event_id, RTCOM_EL_OP_EQUAL, NULL);
  RTComElIter* iter = rtcom_el_get_events(self->priv->rt_el, query);
  g_object_unref(query);
  if(rtcom_el_iter_first(iter))
  {
    gchar* remote_uid = NULL;
    gchar* channel = NULL;
    gint flags = 0;
    gchar* local_uid = NULL;
    rtcom_el_iter_get_values(iter, 
			     "remote-uid", &remote_uid, 
			     "channel", &channel, 
			     "flags", &flags, 
			     "local-uid", &local_uid, 
			     NULL);
    if(iter)
      g_object_unref(iter);
    if (!((remote_uid || channel) && local_uid))
      return;

    McAccount *account = osso_abook_account_manager_lookup_by_name (self->priv->acc_manager, local_uid);
    if (!account)
      goto finally;
    
    if ((flags & RTCOM_EL_FLAG_CHAT_GROUP) && (flags & RTCOM_EL_FLAG_CHAT_OPAQUE))
    {
      if (!channel)
	goto finally;
      
      GHashTable *properties = tp_asv_new(TP_IFACE_CHANNEL ".ChannelType", G_TYPE_STRING,
					  TP_IFACE_CHANNEL_TYPE_TEXT,
					  TP_IFACE_CHANNEL ".TargetHandleType", G_TYPE_UINT,
					  TP_HANDLE_TYPE_NONE,
					  RTCOM_TP_IFACE_CHANNEL_INTERFACE_PERSISTENT ".PersistentID",
					  G_TYPE_STRING, channel,
					  NULL);
      
      mc_account_channelrequest_ht (account,
				    properties,
				    time (NULL),
				    NULL,
				    MC_ACCOUNT_CR_FLAG_USE_EXISTING,
				    NULL, NULL, NULL, NULL);
      
      g_hash_table_unref (properties);
    }
    else
    {
      McAccountChannelrequestData request;
    
      MC_ACCOUNT_CRD_INIT (&request);
      MC_ACCOUNT_CRD_SET (&request, channel_type, TP_IFACE_QUARK_CHANNEL_TYPE_TEXT);

      if ((flags & RTCOM_EL_FLAG_CHAT_ROOM) && channel)
      {
	MC_ACCOUNT_CRD_SET (&request, target_handle_type, TP_HANDLE_TYPE_ROOM);
	MC_ACCOUNT_CRD_SET (&request, target_id, channel);
      }
      else if (remote_uid)
      {
	MC_ACCOUNT_CRD_SET (&request, target_handle_type, TP_HANDLE_TYPE_CONTACT);
	MC_ACCOUNT_CRD_SET (&request, target_id, remote_uid);
      }
      else
	goto finally;

      mc_account_channelrequest (account,
				 &request,
				 time (NULL),
				 NULL,
				 MC_ACCOUNT_CR_FLAG_USE_EXISTING,
				 NULL, NULL, NULL, NULL);
    }


  finally:
    g_free (remote_uid);
    g_free (channel);
    g_free (local_uid);
  }
}

static void
open_conversation_button_cb(GtkButton* button, gpointer user_data)
{  
  event_logger_window_data_t* event_logger_window_data  = (event_logger_window_data_t*)user_data;
  scout_open_conversation_app(event_logger_window_data->scout,
			      event_logger_window_data->event_id);
}

static void
scout_open_event_log_window(ScoutWindow* self,
			    const gchar* user_id,
			    int event_id)
{
  GtkWidget* window;
  window = hildon_stackable_window_new();
  const gchar* services[] = {"RTCOM_EL_SERVICE_CHAT", "RTCOM_EL_SERVICE_SMS", NULL};

  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "remote-uid", user_id, RTCOM_EL_OP_EQUAL, 
			  "service", services, RTCOM_EL_OP_IN_STRV, 
			  "id", event_id, RTCOM_EL_OP_EQUAL,
			  NULL);
  RTComElIter* rtcom_iter = rtcom_el_get_events(self->priv->rt_el, query);
  if(rtcom_el_iter_first(rtcom_iter))
  {
    int start_time = 0;
    int seven_days = 7 * 24 * 60 * 60;
    rtcom_el_iter_get_values(rtcom_iter, "start-time", &start_time, NULL);
    g_object_unref(query);
    g_object_unref(rtcom_iter);
    query = rtcom_el_query_new (self->priv->rt_el);    
    rtcom_el_query_prepare (query, 
			    "remote-uid", user_id, RTCOM_EL_OP_EQUAL, 
			    "service", services, RTCOM_EL_OP_IN_STRV, 
			    "start-time", start_time - seven_days, RTCOM_EL_OP_GREATER_EQUAL,
			    "start-time", start_time + seven_days, RTCOM_EL_OP_LESS_EQUAL,
			    NULL);
  }
  else
  {
    g_object_unref(query);
    query = rtcom_el_query_new (self->priv->rt_el);    
    rtcom_el_query_prepare (query, 
			    "remote-uid", user_id, RTCOM_EL_OP_EQUAL, 
			    "service", services, RTCOM_EL_OP_IN_STRV, 
			    NULL);

  }
  event_logger_window_data_t* event_logger_window_data = g_new0(event_logger_window_data_t,1);
  RTComLogModel* rtcom_model = rtcom_log_model_new();
  GtkWidget* rtcom_log_view = rtcom_log_view_new();
  GtkWidget* text_view = hildon_text_view_new();
  GtkWidget* open_conversation_button = 
    hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
				HILDON_BUTTON_ARRANGEMENT_VERTICAL,
				dgettext("rtcom-messaging-ui", "messaging_ap_conversations"), 
				NULL);
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_CHAR);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(text_view), FALSE);

  event_logger_window_data->view = rtcom_log_view;
  event_logger_window_data->model = rtcom_model;
  event_logger_window_data->window = window;
  event_logger_window_data->text_view = text_view;
  event_logger_window_data->event_id = event_id;
  event_logger_window_data->scout = self;
  rtcom_log_model_set_abook_aggregator(rtcom_model, OSSO_ABOOK_AGGREGATOR(self->priv->abook));
  rtcom_log_view_set_model(RTCOM_LOG_VIEW(rtcom_log_view),
			   GTK_TREE_MODEL(rtcom_model));
  g_signal_connect(rtcom_model, "row-inserted", G_CALLBACK(_scroll_to_event), event_logger_window_data);
  g_signal_connect(open_conversation_button, "clicked", G_CALLBACK(open_conversation_button_cb), event_logger_window_data);
  GtkWidget* pan = hildon_pannable_area_new();
  GtkWidget* content = gtk_vbox_new(FALSE, 0);
  GtkWidget* content2 = gtk_vbox_new(TRUE, 0);
  gtk_container_add(GTK_CONTAINER(pan), rtcom_log_view);
  gtk_box_pack_start(GTK_BOX(content), text_view, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(content), open_conversation_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(content2), pan, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(content2), content, TRUE, TRUE, 0);
  gtk_container_add(GTK_CONTAINER(window), content2);
  gtk_window_set_title(GTK_WINDOW(window), user_id);
  hildon_gtk_window_set_progress_indicator(GTK_WINDOW(window), 1);
  rtcom_log_model_populate_query(rtcom_model, query);
  g_object_unref(query);
  g_signal_connect(rtcom_log_view, "row-activated", G_CALLBACK(_show_current_text), text_view);
  g_signal_connect(window, "destroy-event", G_CALLBACK(_delete_event_scroller), event_logger_window_data);
  gtk_widget_show_all(window);
}

static void
scout_open_text_message_window(ScoutWindow* self,
			       const gchar* user_id,
			       int event_id)
{
  GtkWidget* window;
  window = hildon_stackable_window_new();
  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "id", event_id, RTCOM_EL_OP_EQUAL, NULL);
  RTComElIter* iter = rtcom_el_get_events(self->priv->rt_el, query);
  g_object_unref(query);
  GtkWidget* text_view = hildon_text_view_new();
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_CHAR);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(text_view), FALSE);
  gboolean it = rtcom_el_iter_first(iter);
  if(it)
  {
    gchar* text = NULL;
    rtcom_el_iter_get_values(iter, "free-text", &text, NULL);
    if(iter)
      g_object_unref(iter);
    hildon_text_view_set_placeholder (HILDON_TEXT_VIEW (text_view),
				      text);
    GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
    gtk_text_buffer_insert_at_cursor(buffer,
				     text,
				     strlen(text));
  }
  else
  {
    hildon_text_view_set_placeholder (HILDON_TEXT_VIEW (text_view),
				      "no text");
  }
  gtk_container_add(GTK_CONTAINER(window), text_view);
  gtk_window_set_title(GTK_WINDOW(window), user_id);
  gtk_widget_show_all(window);
}



static void 
open_result_from_list(GtkTreeView* tree_view, 
		      GtkTreePath* path, 
		      GtkTreeViewColumn* column,
		      gpointer data)
{
  ScoutWindow* self = SCOUT_WINDOW(data);
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
  GtkTreeIter iter;
  if(gtk_tree_model_get_iter(GTK_TREE_MODEL(self->priv->result_list),
			     &iter,
			     path))
  {
    gchar* name = NULL;
    gchar* id = NULL;
    gchar* string_id = NULL;
    int type;
    int int_id;
    gtk_tree_model_get(GTK_TREE_MODEL(self->priv->result_list), &iter,
		       SEARCH_RESULT_TEXT_COL, &name,
		       SEARCH_RESULT_ITEM_TYPE_COL, &type,
		       SEARCH_RESULT_ITEM_STRING_ID_COL, &id,
		       SEARCH_RESULT_ITEM_INT_ID_COL, &int_id,
		       SEARCH_RESULT_ITEM_STRING_ID_COL, &string_id,
		       -1);

    if(type == SEARCH_CATEGORY_CONTACT)
    {
      GList* entries = osso_abook_aggregator_lookup(self->priv->abook, id);
      if(entries)
      {
	OssoABookContact *original_contact = (OssoABookContact*)entries->data;
	const char* n = osso_abook_contact_get_name(original_contact);
	GtkWidget* dialog = osso_abook_touch_contact_starter_dialog_new
	  (GTK_WINDOW(self),
	   ( OssoABookTouchContactStarter*)osso_abook_touch_contact_starter_new_with_contact(GTK_WINDOW(self), original_contact));
	gtk_widget_show_all(dialog);
      }
      g_list_free(entries);
    }
    else if(type == SEARCH_CATEGORY_CALENDAR_EVENT)
    {
      open_calendar_event_window(self->priv->osso, int_id, string_id);
    }
    else if(type == SEARCH_CATEGORY_CALENDAR_TODO)
    {
      open_calendar_todo_window(self->priv->osso, int_id, string_id);
    }
    else if(type == SEARCH_CATEGORY_CALENDAR_JOURNAL)
    {
      open_calendar_journal_window(self->priv->osso, int_id, string_id);
    }
    else if(type == SEARCH_CATEGORY_CONVERSATION_SMS ||
	    type == SEARCH_CATEGORY_CONVERSATION_IM)
    {
      //      scout_open_conversation_app(self, id, event_id);
      //      scout_open_text_message_window(self, id, event_id);
      scout_open_event_log_window(self, id, int_id);
    }
    g_free(name);
    g_free(string_id);
    g_free(id);
    gtk_tree_selection_unselect_all(selection);
  }
}


static void
_init_rtcom(ScoutWindow* self)
{
  self->priv->rt_el = rtcom_el_new();
  self->priv->rt_com_chat_query = rtcom_el_query_new(self->priv->rt_el);
  rtcom_el_query_prepare(self->priv->rt_com_chat_query, "service", "RTCOM_EL_SERVICE_CHAT", RTCOM_EL_OP_EQUAL, NULL);
  self->priv->rt_com_sms_query = rtcom_el_query_new(self->priv->rt_el);
  rtcom_el_query_prepare(self->priv->rt_com_sms_query, "service", "RTCOM_EL_SERVICE_SMS", RTCOM_EL_OP_EQUAL, NULL);
}



static void 
_read_settings(ScoutWindow* self)
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  g_return_if_fail(GCONF_IS_CLIENT(client));
  if(!gconf_client_dir_exists(client, GC_ROOT, NULL))
  {
    g_object_unref(client);
    return;
  }
  self->priv->search_in_contacts = gconf_client_get_bool(client, GCONF_SEARCH_CONTACTS, NULL);
  self->priv->search_in_conversations = gconf_client_get_bool(client, GCONF_SEARCH_CONVERSATIONS , NULL);
  self->priv->search_in_calendars = gconf_client_get_bool(client, GCONF_SEARCH_CALENDARS , NULL);
  self->priv->search_case_sensitive = gconf_client_get_bool(client, GCONF_SEARCH_CASE_SENSITIVE, NULL);
  g_object_unref(client);
}

static void 
_store_settings(ScoutWindow* self)
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  g_return_if_fail(GCONF_IS_CLIENT(client));
  gconf_client_set_bool(client, GCONF_SEARCH_CONTACTS, self->priv->search_in_contacts, NULL);
  gconf_client_set_bool(client, GCONF_SEARCH_CONVERSATIONS, self->priv->search_in_conversations, NULL);
  gconf_client_set_bool(client, GCONF_SEARCH_CALENDARS, self->priv->search_in_calendars, NULL);
  gconf_client_set_bool(client, GCONF_SEARCH_CASE_SENSITIVE, self->priv->search_case_sensitive, NULL);
  g_object_unref(client);
}

static void
_show_settings_dialog_cb(GtkButton* button, gpointer data)
{
  g_return_if_fail(IS_SCOUT_WINDOW(data));
  ScoutWindow* self = SCOUT_WINDOW(data);

  GtkWidget* dialog;
  dialog = gtk_dialog_new_with_buttons("Settings",
				       GTK_WINDOW(self),
				       GTK_DIALOG_NO_SEPARATOR,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_ACCEPT,
				       NULL);

  GtkWidget* button1 = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* button2 = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* button3 = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* button4 = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);

  gtk_button_set_label(GTK_BUTTON(button1), dgettext("osso-addressbook", "addr_ap_address_book"));
  gtk_button_set_label(GTK_BUTTON(button2), dgettext("rtcom-messaging-ui", "messaging_ap_conversations"));
  gtk_button_set_label(GTK_BUTTON(button3), dgettext("calendar", "cal_ap_name"));
  gtk_button_set_label(GTK_BUTTON(button4), "case sensitive");
  
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button1), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button2), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button3), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button4), FALSE);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button1), self->priv->search_in_contacts);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button2), self->priv->search_in_conversations);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button3), self->priv->search_in_calendars);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button4), self->priv->search_case_sensitive);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), button1, TRUE, TRUE, 3);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), button2, TRUE, TRUE, 3);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), button3, TRUE, TRUE, 3);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), button4, TRUE, TRUE, 3);

  gtk_widget_show_all(dialog);
  int result = gtk_dialog_run(GTK_DIALOG(dialog));
  
  if(result = GTK_RESPONSE_ACCEPT)
  {
    self->priv->search_in_contacts = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button1));
    self->priv->search_in_conversations = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button2));
    self->priv->search_in_calendars = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button3));
    self->priv->search_case_sensitive = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button4));
  }
  gtk_widget_destroy(dialog);
}

static void
_init_menu(ScoutWindow* self)
{
  HildonAppMenu* menu;
  menu = HILDON_APP_MENU(hildon_app_menu_new());
  GtkWidget* settings_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(settings_button), "settings");
  g_signal_connect(settings_button, "clicked", G_CALLBACK(_show_settings_dialog_cb), self);
  hildon_app_menu_append(menu, GTK_BUTTON(settings_button));
  gtk_widget_show_all(GTK_WIDGET(menu));
  hildon_window_set_app_menu(HILDON_WINDOW(self), menu);
}

static void
scout_window_init(ScoutWindow* self)
{
  self->priv = SCOUT_WINDOW_GET_PRIVATE(self);
  _init_rtcom(self);
  self->priv->cached_calendar_id_icon = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->cached_calendar_type_icon = 
    g_hash_table_new_full(g_int_hash, g_int_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);
  self->priv->cached_contact_attribute_icons = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->cached_account_icon = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->cached_contact_avatar = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->search_category_contacts_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_contacts",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_calendar_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_calendar",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_sms_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_sms",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_outgoing_sms_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "chat_replied_sms",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_im_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_chat",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_outgoing_im_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "chat_replied_chat",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_thread = NULL;
  self->priv->search_active = FALSE;
  self->priv->current_search_text = NULL;
  self->priv->search_in_contacts = TRUE;
  self->priv->search_in_conversations = TRUE;
  self->priv->search_in_calendars = TRUE;
  self->priv->search_case_sensitive = FALSE;

  GtkWidget* box = gtk_vbox_new(FALSE, 0);
  self->priv->search_entry_field = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  g_object_set(self->priv->search_entry_field,"hildon-input-mode",
               HILDON_GTK_INPUT_MODE_FULL, NULL);
  self->priv->result_list = gtk_list_store_new(7, 
					       GDK_TYPE_PIXBUF, 
					       G_TYPE_STRING, 
					       GDK_TYPE_PIXBUF, 
					       GDK_TYPE_PIXBUF, 
					       G_TYPE_INT, 
					       G_TYPE_STRING, 
					       G_TYPE_INT);
  GtkWidget* pan = hildon_pannable_area_new();
  GtkWidget* entry_alignment = gtk_alignment_new(0, 1, 1, 0);
  GtkCellRenderer* renderer = NULL;
  GtkCellRenderer* renderer2 = NULL;
  GtkCellRenderer* renderer3 = NULL;
  GtkCellRenderer* renderer4 = NULL;
  self->priv->search_touch_list = hildon_gtk_tree_view_new(HILDON_UI_MODE_NORMAL);


  self->priv->search_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
							  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
							  dgettext("hildon-libs", "wdgt_bd_search"), NULL);
  renderer = gtk_cell_renderer_pixbuf_new();
  renderer2 = gtk_cell_renderer_text_new();
  renderer3 = gtk_cell_renderer_pixbuf_new();
  renderer4 = gtk_cell_renderer_pixbuf_new();
  
  GtkTreeViewColumn* column = gtk_tree_view_column_new();
  GtkTreeViewColumn* column2 = gtk_tree_view_column_new();
  gtk_tree_view_column_set_sizing (column2,
				   GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width (column2,
					785 - (48+48+48));

  GtkTreeViewColumn* column3 = gtk_tree_view_column_new();
  GtkTreeViewColumn* column4 = gtk_tree_view_column_new();

  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column);
  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column2);
  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column3);
  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column4);

  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_pack_start(column2, renderer2, TRUE);  
  gtk_tree_view_column_pack_start(column3, renderer3, FALSE);  
  gtk_tree_view_column_pack_start(column4, renderer4, FALSE);  

  gtk_tree_view_column_set_attributes(column,
				      renderer,
				      "pixbuf",SEARCH_CATEGORY_ICON_COL,
				      NULL);
  gtk_tree_view_column_set_attributes(column2,
				      renderer2,
				      "markup", SEARCH_RESULT_TEXT_COL,
				      NULL);
  gtk_tree_view_column_set_attributes(column3,
				      renderer3,
				      "pixbuf",SEARCH_RESULT_ITEM_TYPE_ICON_COL,
				      NULL);
  gtk_tree_view_column_set_attributes(column4,
				      renderer4,
				      "pixbuf",SEARCH_RESULT_ITEM_ID_ICON_COL,
				      NULL);

  gtk_tree_view_set_model(GTK_TREE_VIEW(self->priv->search_touch_list),
			 GTK_TREE_MODEL(self->priv->result_list));

  GtkWidget* search_touch_alignment = gtk_alignment_new(0, 0, 1.0f, 1.0f);

  GtkWidget* search_box = gtk_hbox_new(FALSE, 3);

  gtk_container_add(GTK_CONTAINER(search_touch_alignment), pan);


  gtk_container_add(GTK_CONTAINER(pan), self->priv->search_touch_list);

  g_signal_connect(self->priv->search_entry_field, "activate", G_CALLBACK(on_search_enter), self);
  g_signal_connect(self->priv->search_button, "clicked", G_CALLBACK(on_search_clicked), self);
  g_signal_connect(self->priv->search_touch_list, "row-activated", G_CALLBACK(open_result_from_list), self);

  gtk_box_pack_start(GTK_BOX(search_box), self->priv->search_entry_field, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(search_box), self->priv->search_button, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(entry_alignment), search_box);
  gtk_box_pack_start(GTK_BOX(box), search_touch_alignment, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(box), entry_alignment, FALSE, FALSE, 0);
  gtk_widget_show_all(box);
  gtk_container_add(GTK_CONTAINER(self), box);
  gtk_widget_grab_focus(self->priv->search_entry_field);
  _read_settings(self);
  _init_menu(self);
}

static void
scout_window_finalize(GObject* object)
{
  ScoutWindow* self = SCOUT_WINDOW(object);
  g_object_unref(self->priv->result_list);
  if(self->priv->rt_el)
    g_object_unref(self->priv->rt_el);
  if(self->priv->rt_com_chat_query)
    g_object_unref(self->priv->rt_com_chat_query);
  if(self->priv->rt_com_sms_query)
    g_object_unref(self->priv->rt_com_sms_query);
  if(self->priv->abook)
    g_object_unref(self->priv->abook);
  if(self->priv->search_category_contacts_icon)
    g_object_unref(self->priv->search_category_contacts_icon);
  if(self->priv->search_category_calendar_icon)
    g_object_unref(self->priv->search_category_calendar_icon);
  if(self->priv->search_category_conversation_sms_icon)
    g_object_unref(self->priv->search_category_conversation_sms_icon);
  if(self->priv->search_category_conversation_im_icon)
    g_object_unref(self->priv->search_category_conversation_im_icon);


  g_hash_table_destroy(self->priv->cached_contact_attribute_icons);

  g_hash_table_destroy(self->priv->cached_contact_avatar);
  g_hash_table_destroy(self->priv->cached_calendar_id_icon);
  g_hash_table_destroy(self->priv->cached_calendar_type_icon);
  g_hash_table_destroy(self->priv->cached_account_icon);
  _store_settings(SCOUT_WINDOW(object));
  G_OBJECT_CLASS(scout_window_parent_class)->finalize(object);
}

static void
scout_window_dispose(GObject* object)
{
  G_OBJECT_CLASS(scout_window_parent_class)->dispose(object);
}

static void
scout_window_class_init(ScoutWindowClass* klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS(klass);
  g_type_class_add_private(klass, sizeof(ScoutWindowPrivate));
  object_class->finalize = scout_window_finalize;
  object_class->dispose = scout_window_dispose;
}

