#include "config.h"

#include <hildon/hildon.h>
#include <libosso.h>
#include "scout-window.h"
#include <rtcom-eventlogger/event.h>
#include <string.h>
#include <libintl.h>
#include <locale.h>
#include <CalInterface.h>
#include <CMulticalendar.h>
#include <vector>
#include <CComponent.h>
#include <CCalendar.h>
#include <ctime>
#include <clockd/libtime.h>

struct _ScoutWindowPrivate
{
  osso_context_t* osso;
  OssoABookAggregator* abook;
  OssoABookAccountManager* acc_manager;
  GtkListStore* result_list;
  RTComEl* rt_el;
  RTComElQuery* rt_com_chat_query;
  RTComElQuery* rt_com_sms_query;

  GtkWidget* search_entry_field;
  GtkWidget* search_touch_list;
  GHashTable* cached_contact_avatar;
  GHashTable* cached_contact_attribute_icons;
  GHashTable* cached_account_icon;

  GHashTable* cached_calendar_id_icon;
  GHashTable* cached_calendar_type_icon;

  GdkPixbuf* search_category_contacts_icon;
  GdkPixbuf* search_category_calendar_icon;
  GdkPixbuf* search_category_conversation_sms_icon;
  GdkPixbuf* search_category_conversation_outgoing_sms_icon;
  GdkPixbuf* search_category_conversation_im_icon;
  GdkPixbuf* search_category_conversation_outgoing_im_icon;
};

#define SCOUT_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), SCOUT_TYPE_WINDOW, ScoutWindowPrivate))

G_DEFINE_TYPE(ScoutWindow, scout_window, HILDON_TYPE_STACKABLE_WINDOW)

enum
{
  SEARCH_CATEGORY_CONTACT = 0,
  SEARCH_CATEGORY_CONVERSATION_SMS = 1,
  SEARCH_CATEGORY_CALENDAR = 2,
  SEARCH_CATEGORY_CONVERSATION_IM = 3
};

enum
{
  SEARCH_CATEGORY_ICON_COL = 0,
  SEARCH_RESULT_TEXT_COL = 1,
  SEARCH_RESULT_ITEM_TYPE_ICON_COL = 2,
  SEARCH_RESULT_ITEM_ID_ICON_COL = 3,
  SEARCH_RESULT_ITEM_TYPE_COL = 4,
  SEARCH_RESULT_ITEM_STRING_ID_COL = 5,
  SEARCH_RESULT_ITEM_INT_ID_COL = 6
};

void 
scout_window_set_abook(ScoutWindow* window, 
		       OssoABookAggregator* abook)
{
  window->priv->abook = abook;
  g_object_ref(window->priv->abook);
}

void 
scout_window_set_account_manager(ScoutWindow* window, 
				 OssoABookAccountManager* acc_manager)
{
  window->priv->acc_manager = acc_manager;
}

ScoutWindow*
scout_window_new(osso_context_t* osso)
{
  ScoutWindow* self = SCOUT_WINDOW(g_object_new(SCOUT_TYPE_WINDOW, NULL));
  SCOUT_WINDOW_GET_PRIVATE(self)->osso = osso;
  return self;
} 

static gchar*
_get_remote_name(ScoutWindow* self, const gchar* remote_uid)
{
  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL, NULL);
  RTComElIter* iter = rtcom_el_get_events(self->priv->rt_el, query);
  g_object_unref(query);
  gboolean it = rtcom_el_iter_first(iter);
  if(it)
  {
    gchar* remote_name = NULL;
    rtcom_el_iter_get_values(iter, "remote-name", &remote_name, NULL);
    return remote_name;
  }
  else
  {
    return NULL;
  }
}

static gchar*
_get_abook_uid_name(ScoutWindow* self, const gchar* remote_uid)
{
  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL, NULL);
  RTComElIter* iter = rtcom_el_get_events(self->priv->rt_el, query);
  g_object_unref(query);
  gboolean it = rtcom_el_iter_first(iter);
  if(it)
  {
    //    gchar* remote_name = NULL;
    gchar* abook_uid = NULL;
    rtcom_el_iter_get_values(iter, "remote-ebook-uid", &abook_uid, NULL);
    return abook_uid;
  }
  else
  {
    return NULL;
  }
}

const gchar*
_get_date_time_format()
{
  static gchar* format = NULL;
  if(format == NULL)
  {
    const gchar* time_format = NULL;
    const gchar* date_format = NULL;
    date_format = dgettext("hildon-libs", "wdgt_va_date");
    time_format = dgettext("hildon-libs", "wdgt_va_24h_time");
    format = g_strdup_printf("%s | %s", date_format, time_format);
  }
  return format;
}
const gchar*
_get_secondary_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "SecondaryTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

const gchar*
_get_active_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "ActiveTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

const gchar*
_get_default_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "DefaultTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

static const gchar*
_get_event_markup_template()
{
  static gchar* markup_template = NULL;
  if(markup_template == NULL)
  {
    markup_template = g_strdup_printf("<span color=\"%s\">%%s</span>  <span size=\"x-small\"><sup>%%s</sup></span>\n"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span size=\"x-small\" color=\"%s\">%%s</span>"
				      "<span size=\"x-small\" color=\"%s\">%%s</span>",
				      _get_default_text_color(),
				      _get_secondary_text_color(),
				      _get_active_text_color(),
				      _get_secondary_text_color());
  }
  return markup_template;
}

static const gchar*
_get_calendar_markup_template()
{
  static gchar* markup_template = NULL;
  if(markup_template == NULL)
  {
    markup_template = g_strdup_printf("<span color=\"%s\">%%s</span>  <span size=\"x-small\"><sup>%%s</sup></span>\n"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>",
				      _get_default_text_color(),
				      _get_secondary_text_color(),
				      _get_active_text_color(),
				      _get_secondary_text_color());
  }
  return markup_template;
}

static const gchar*
_get_contact_markup_template()
{
  static gchar* markup_template = NULL;
  if(markup_template == NULL)
  {
    markup_template = g_strdup_printf("<span color=\"%s\">%%s</span>\n"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>",
				      _get_default_text_color(),
				      _get_secondary_text_color(),
				      _get_active_text_color(),
				      _get_secondary_text_color());
  }
  return markup_template;
}

static GdkPixbuf*
_get_event_type_icon(ScoutWindow* self, const gint event_type)
{
  GdkPixbuf* event_type_icon = NULL;
  event_type_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_calendar_type_icon,
						   &event_type));

  if(!event_type_icon)
  {
    switch(event_type)
    {
    case /*E_TASK*/ 2:
      event_type_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						 "calendar_todo",
						 HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      break;
    case /*E_BIRTHDAY*/ 4:
      event_type_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						 "calendar_birthday",
						 HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      break;
    case 1:
    case E_NORMAL_EVENT:
      event_type_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						 "general_calendar",
						 HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    gint* key;
    key = (gint*)g_malloc(sizeof key);
    *key = event_type;
    g_hash_table_insert(self->priv->cached_calendar_type_icon,
			key,
			event_type_icon);
  }
  return event_type_icon;
}

static GdkPixbuf*
_get_calendar_id_icon(ScoutWindow* self, const gint calendar_id)
{
  GdkPixbuf* calendar_id_icon = NULL;
  calendar_id_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_calendar_id_icon,
						    &calendar_id));
  if(!calendar_id_icon)
  {
    int error = 0;
    CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
    int color = calendar->getCalendarColor();
    delete calendar;
    switch(color)
    {
      case COLOUR_DARKBLUE:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_darkblue",
					  HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_DARKGREEN:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_darkgreen", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_DARKRED:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_darkred", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_ORANGE:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_orange", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_VIOLET:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_violet", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_YELLOW:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_yellow", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_BLUE:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_blue", 
					  HILDON_ICON_PIXEL_SIZE_FINGER,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_RED:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_red", 
					  HILDON_ICON_PIXEL_SIZE_FINGER ,GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      case COLOUR_GREEN:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_green",
					  HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
	break;
      default:
	calendar_id_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					  "calendar_colors_white", 
					  HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    gint* key;
    key = (gint*) g_malloc(sizeof key);
    *key = calendar_id;
    g_hash_table_insert(self->priv->cached_calendar_id_icon,
			key,
			calendar_id_icon);
  }
  return calendar_id_icon;
}

static GdkPixbuf*
_get_contact_attribute_icon(ScoutWindow* self, const gchar* attribute)
{
  GdkPixbuf* attribute_icon = NULL;
  attribute_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_contact_attribute_icons,
					       attribute));
  if(!attribute_icon)
  {
    if(g_strcmp0(attribute, EVC_TEL)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					     "general_call",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      
    }
    if(g_strcmp0(attribute, EVC_FN)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					     "general_business_card",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      
    }
    if(g_strcmp0(attribute, EVC_N)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_business_card",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
      
    }
    else if(g_strcmp0(attribute, EVC_EMAIL)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_email",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else if(g_strcmp0(attribute, EVC_LABEL)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_map",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else if(g_strcmp0(attribute, EVC_BDAY)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_calendar",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else if(g_strcmp0(attribute, EVC_NICKNAME)==0)
    {
      attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"general_business_card",
						HILDON_ICON_PIXEL_SIZE_SMALL, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else
    {
      McProfile* mc_profile = mc_profile_lookup_default_for_vcard_field(attribute);
      if(mc_profile)
      {
	const gchar* icon_name = mc_profile_get_branding_icon_name (mc_profile);
	
	if (!icon_name)
	  icon_name = mc_profile_get_icon_name (mc_profile);
	g_object_unref (mc_profile);	
	if(!icon_name)
	{
	  return NULL;
	}
        attribute_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						  icon_name,
						  HILDON_ICON_PIXEL_SIZE_SMALL,
						  GTK_ICON_LOOKUP_NO_SVG, NULL);
      }
      else
      {
	g_warning("no profile for %s\n", attribute);
      }
    }
    if(attribute_icon)
      g_hash_table_insert(self->priv->cached_contact_attribute_icons,
			  g_strdup(attribute),
			  attribute_icon);
      
  }
  return attribute_icon;
}

static GdkPixbuf*
_get_account_icon(ScoutWindow* self, const gchar* account_uid)
{
  GdkPixbuf* account_icon = NULL;
  account_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_account_icon,
					       account_uid));
  if(!account_icon)
  {

    McAccount* account = osso_abook_account_manager_lookup_by_name(self->priv->acc_manager,
									   account_uid);
    if(account)
    {
      const gchar* id = mc_account_compat_get_profile(account);
      McProfile* profile = mc_profile_lookup(id);
      if(profile)
      {
	const gchar* icon_name = mc_profile_get_branding_icon_name (profile);
	
	if (!icon_name)
	  icon_name = mc_profile_get_icon_name (profile);
	
	g_object_unref (profile);
	
	if(icon_name)
	{
	  account_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						  icon_name,
						  HILDON_ICON_PIXEL_SIZE_SMALL,
						  GTK_ICON_LOOKUP_NO_SVG, NULL);
	}
      }
    }
    if(account_icon)
      g_hash_table_insert(self->priv->cached_account_icon,
			  g_strdup(account_uid),
			  account_icon);
      
    
  }
  return account_icon;
}

static GdkPixbuf*
_get_contact_avatar_icon(ScoutWindow* self, const gchar* abook_uid)
{
  GdkPixbuf* avatar_icon = NULL;
  avatar_icon = GDK_PIXBUF(g_hash_table_lookup(self->priv->cached_contact_avatar,
					       abook_uid));
  if(!avatar_icon)
  {
    GList* entries = osso_abook_aggregator_lookup(self->priv->abook, abook_uid);
    if(entries)
    {
      OssoABookContact *original_contact = (OssoABookContact*)entries->data;
      GdkPixbuf* pixbuf = osso_abook_contact_get_avatar_pixbuf(original_contact,
							       NULL, NULL);
      if(pixbuf)
      {
	avatar_icon = gdk_pixbuf_scale_simple(pixbuf, 
					      48, 48,
					      GDK_INTERP_BILINEAR);
	g_object_unref(pixbuf);
      }
      g_list_free(entries);
    }

    if(avatar_icon == NULL)
    {
      avatar_icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					     "general_default_avatar",
					     HILDON_ICON_PIXEL_SIZE_FINGER, GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    if(avatar_icon == NULL)
      {
	g_warning("pushd NULL icon \n");
      }
    g_hash_table_insert(self->priv->cached_contact_avatar,
			g_strdup(abook_uid),
			avatar_icon);
    g_object_ref(avatar_icon);
  }
  return avatar_icon;
}

static void
_split_text_for_search_result(const gchar* text, 
			      const gchar* search_text,
			      const gchar* search_in_text_pos,
			      const gint max_text_len,
			      gchar** pprefix,
			      gchar** psuffix)
{
  int textlen = strlen(text);
  if(textlen<=max_text_len)
  {
    int prefix_len = search_in_text_pos - text;
    int suffix_len = textlen - (search_in_text_pos-text) - strlen(search_text);
    (*pprefix) = g_strndup(text, prefix_len);
    (*psuffix) = g_strndup(search_in_text_pos+strlen(search_text),suffix_len);
    return;
  }
  int search_text_len = strlen(search_text);
  int prefix_len = max_text_len/2 - (search_text_len/2) + (search_text_len % 2);
  int suffix_len = max_text_len/2 - (search_text_len/2);
  if(prefix_len > (search_in_text_pos - text))
  {
    suffix_len += prefix_len - (search_in_text_pos - text);
    prefix_len = search_in_text_pos - text;
  }
  else if(textlen < (suffix_len + (search_in_text_pos - text) + search_text_len))
  {
    prefix_len += (suffix_len + (search_in_text_pos - text) + search_text_len) - textlen;
    suffix_len = textlen - ((search_in_text_pos - text ) + search_text_len);
  }
  (*pprefix) = g_strndup(search_in_text_pos - prefix_len, prefix_len);
  (*psuffix) = g_strndup(search_in_text_pos + search_text_len, suffix_len);
}

static void
_search_in_events(ScoutWindow* self, 
		  const gchar* search, 
		  RTComElQuery* query, 
		  GdkPixbuf* icon,
		  GdkPixbuf* outgoing_icon,
		  int search_type)
{
  RTComElIter* rtcom_iter = rtcom_el_get_events(self->priv->rt_el, query);
  gboolean it = rtcom_el_iter_first(rtcom_iter);
  GtkTreeIter iter;
  while(it)
  {
    gchar* free_text = NULL;
    rtcom_el_iter_get_values(rtcom_iter, "free-text", &free_text, NULL);
    gchar* pos = NULL;
    if(free_text && (pos = strstr(free_text, search))!=NULL)
    {
      gchar* remote_uid = NULL;
      gchar* local_uid = NULL;
      gint event_id;
      time_t start_time;
      gint outgoing;
      rtcom_el_iter_get_values(rtcom_iter, "id", &event_id, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "remote-uid", &remote_uid, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "start-time", &start_time, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "local-uid", &local_uid, NULL);
      rtcom_el_iter_get_values(rtcom_iter, "outgoing", &outgoing, NULL);
      if(remote_uid!=NULL)
      {
	gchar time_string[256];
	struct tm* time_stamp;
	time_stamp = localtime(&start_time);
	strftime(time_string, 256, _get_date_time_format(), time_stamp);
	gchar* remote_name = _get_remote_name(self, remote_uid);
	gchar* abook_uid = _get_abook_uid_name(self, remote_uid);
	GdkPixbuf* scaled_pixbuf = NULL;
	GdkPixbuf* account_icon = _get_account_icon(self, local_uid);

	if(abook_uid!=NULL)
	{
	  scaled_pixbuf = _get_contact_avatar_icon(self, abook_uid);
	}

	gchar* prefix = NULL;
	gchar* suffix = NULL;
	_split_text_for_search_result(free_text, search, pos, 80, &prefix, &suffix);


	gchar* label_text = NULL;
	if(remote_name)
	  label_text = g_strdup(remote_name);
	else
	  label_text = g_strdup(remote_uid);
	gchar* text_name = 
	  g_markup_printf_escaped(_get_event_markup_template(),
				  label_text, 
				  time_string,
				  prefix,
				  search,
				  suffix);
	g_free(label_text);
	gtk_list_store_append(self->priv->result_list, &iter);
	if(outgoing == 1)
	{
	  gtk_list_store_set(self->priv->result_list, &iter,
			     SEARCH_CATEGORY_ICON_COL, outgoing_icon,
			     SEARCH_RESULT_TEXT_COL, text_name,
			     SEARCH_RESULT_ITEM_TYPE_ICON_COL, account_icon,
			     SEARCH_RESULT_ITEM_ID_ICON_COL, scaled_pixbuf,
			     SEARCH_RESULT_ITEM_TYPE_COL, search_type,
			     SEARCH_RESULT_ITEM_STRING_ID_COL, remote_uid,
			     SEARCH_RESULT_ITEM_INT_ID_COL, event_id,
			     -1);
	}
	else
	{

	  gtk_list_store_set(self->priv->result_list, &iter,
			     SEARCH_CATEGORY_ICON_COL, icon,
			     SEARCH_RESULT_TEXT_COL, text_name,
			     SEARCH_RESULT_ITEM_TYPE_ICON_COL, account_icon,
			     SEARCH_RESULT_ITEM_ID_ICON_COL, scaled_pixbuf,
			     SEARCH_RESULT_ITEM_TYPE_COL, search_type,
			     SEARCH_RESULT_ITEM_STRING_ID_COL, remote_uid,
			     SEARCH_RESULT_ITEM_INT_ID_COL, event_id,
			     -1);
	}

	g_free(remote_name);
	g_free(abook_uid);

	g_free(text_name);
	g_free(prefix);
	g_free(suffix);
      }
      g_free(remote_uid);
      g_free(local_uid);
    }
    g_free(free_text);
    it = rtcom_el_iter_next(rtcom_iter);
  }
  if(rtcom_iter)
    g_object_unref(rtcom_iter);
}

static void
search_in_conversations_chat(ScoutWindow* self)
{
  GtkEntry* entry = GTK_ENTRY(self->priv->search_entry_field);
  const gchar* search = gtk_entry_get_text(entry);
  _search_in_events(self, 
		    search,
		    self->priv->rt_com_chat_query, 
		    self->priv->search_category_conversation_im_icon,
		    self->priv->search_category_conversation_outgoing_im_icon,
		    SEARCH_CATEGORY_CONVERSATION_IM);
}

static void
search_in_conversations_sms(ScoutWindow* self)
{
  GtkEntry* entry = GTK_ENTRY(self->priv->search_entry_field);
  const gchar* search = gtk_entry_get_text(entry);
  _search_in_events(self, 
		    search,
		    self->priv->rt_com_sms_query, 
		    self->priv->search_category_conversation_sms_icon,
		    self->priv->search_category_conversation_outgoing_sms_icon,
		    SEARCH_CATEGORY_CONVERSATION_SMS);
}

static void
search_in_calendar(ScoutWindow* self)
{
  GtkEntry* entry = GTK_ENTRY(self->priv->search_entry_field);
  const gchar* search = gtk_entry_get_text(entry);
  int error = 0;
  int offset = 0;
  int size = 0;
  int limit = 100;
  time_t start_time = -1;
  time_t end_time = 2145909599;
  vector<CCalendar*> calendar_list = CMulticalendar::MCInstance()->getListCalFromMc();
  GtkTreeIter iter;
  const gchar* time_format = NULL;
  const gchar* date_format = NULL;
  date_format = dgettext("hildon-libs", "wdgt_va_date");
  time_format = dgettext("hildon-libs", "wdgt_va_24h_time");
  gchar* full_format = g_strdup_printf("%s | %s", date_format, time_format);
  for(int j=0;j<calendar_list.size();++j)
  {
    vector<CComponent*> component_list = calendar_list[j]->getAllEventsTodos(start_time,
									     end_time,
									     limit,
									     offset,
									     error);
    for(int i = 0;i<component_list.size();++i)
    {
      const gchar* pos = NULL;
      const gchar* text = NULL;
      if((pos = strstr(component_list[i]->getDescription().c_str(),search))!=NULL)
      {
	text = component_list[i]->getDescription().c_str();
      }
      else if((pos = strstr(component_list[i]->getSummary().c_str(),search))!=NULL)
      {
	text = component_list[i]->getSummary().c_str();
      }
      if(text != NULL)
      {
	int event_type = component_list[i]->getType();
	GdkPixbuf* event_type_icon = _get_event_type_icon(self, event_type);
	GdkPixbuf* calendar_id_icon = _get_calendar_id_icon(self, component_list[i]->getCalendarId());

	gchar* prefix = NULL;
	gchar* suffix = NULL;
	gchar time_string[256];
	struct tm* time_stamp;
	time_t start_time = component_list[i]->getDateStart();
	time_stamp = localtime(&start_time);
	strftime(time_string, 256, full_format, time_stamp);

	_split_text_for_search_result(text, search, pos, 78, &prefix, &suffix);
	gchar* text_name = g_markup_printf_escaped(_get_calendar_markup_template(),
						   component_list[i]->getSummary().c_str(), 
						   time_string,
						   prefix,
						   search,
						   suffix);
      
	gtk_list_store_append(self->priv->result_list, &iter);
	gtk_list_store_set(self->priv->result_list, &iter,
			   SEARCH_CATEGORY_ICON_COL, self->priv->search_category_calendar_icon,
			   SEARCH_RESULT_TEXT_COL, text_name,
			   SEARCH_RESULT_ITEM_TYPE_ICON_COL, event_type_icon,
			   SEARCH_RESULT_ITEM_ID_ICON_COL, calendar_id_icon,
			   SEARCH_RESULT_ITEM_TYPE_COL, SEARCH_CATEGORY_CALENDAR,
			   SEARCH_RESULT_ITEM_INT_ID_COL, component_list[i]->getDateStart(),
			   -1);
	g_free(text_name);
	g_free(prefix);
	g_free(suffix);
      }
      delete component_list[i];
    }
  }
  CMulticalendar::MCInstance()->releaseListCalendars(calendar_list);
  g_free(full_format);
}

static gboolean
_search_and_insert_contact_attribute(ScoutWindow* self, const gchar* search, OssoABookContact* contact, const gchar* e_contact_field)
{
  GList* attributes = osso_abook_contact_get_attributes(E_CONTACT(contact), e_contact_field);
  for(;attributes;attributes=attributes->next)
  {
    GList* values = e_vcard_attribute_get_values((EVCardAttribute*)attributes->data);
    const gchar* name = e_vcard_attribute_get_name((EVCardAttribute*)attributes->data);
    const gchar* pos = NULL;
    const gchar* value = NULL;
    if(values)
    {
      value = (const gchar*) values->data;
    }
    if(value && (pos = strstr(value, search)) != NULL)
    {
      GtkTreeIter iter;
      const gchar* abook_uid = osso_abook_contact_get_persistent_uid(contact);
      GdkPixbuf* avatar_icon = _get_contact_avatar_icon(self, abook_uid);
      GdkPixbuf* attribute_icon = _get_contact_attribute_icon(self, e_contact_field);
      gchar* prefix = NULL;
      gchar* suffix = NULL;
      _split_text_for_search_result(value, search, pos, 80, &prefix, &suffix);
      gchar* text_name = 
	g_markup_printf_escaped(_get_contact_markup_template(), 
				osso_abook_contact_get_display_name(contact),
				prefix,
				search,
				suffix);
      gtk_list_store_append(self->priv->result_list, &iter);
      gtk_list_store_set(self->priv->result_list, &iter,
			 SEARCH_CATEGORY_ICON_COL, self->priv->search_category_contacts_icon,
			 SEARCH_RESULT_TEXT_COL, text_name,
			 SEARCH_RESULT_ITEM_TYPE_ICON_COL, attribute_icon,
			 SEARCH_RESULT_ITEM_ID_ICON_COL, avatar_icon,
			 SEARCH_RESULT_ITEM_TYPE_COL, SEARCH_CATEGORY_CONTACT,
			 SEARCH_RESULT_ITEM_STRING_ID_COL, osso_abook_contact_get_persistent_uid(contact),
			 SEARCH_RESULT_ITEM_INT_ID_COL, 0,
			 -1);
      return true;
    }
  }
  return false;
}

static void
search_in_contacts(ScoutWindow* self)
{
  GtkEntry* entry = GTK_ENTRY(self->priv->search_entry_field);
  const gchar* search = gtk_entry_get_text(entry);
  GList *contacts_list;
  GList *contacts;
  contacts = osso_abook_aggregator_list_master_contacts(self->priv->abook);
  contacts_list = contacts;
  for (; contacts; contacts = contacts->next) 
  {
    OssoABookContact *original_contact = (OssoABookContact*)contacts->data;
    if(_search_and_insert_contact_attribute(self, search, original_contact, EVC_FN))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, search, original_contact, EVC_TEL))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, search, original_contact, EVC_N))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, search, original_contact, EVC_EMAIL))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, search, original_contact, EVC_LABEL))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, search, original_contact, EVC_BDAY))
    {
      continue;
    }
    if(_search_and_insert_contact_attribute(self, search, original_contact, EVC_NICKNAME))
    {
      continue;
    }
    
    const GList *fields = (GList*)osso_abook_account_manager_get_primary_vcard_fields(self->priv->acc_manager);
    for (;fields; fields = fields->next) 
    {
      if(_search_and_insert_contact_attribute(self, search, original_contact, (const gchar*)fields->data))
	continue;
    }
  }
  g_list_free(contacts_list);
}

static void
_do_search(ScoutWindow* self)
{
  GtkEntry* entry = GTK_ENTRY(self->priv->search_entry_field);
  if(gtk_entry_get_text_length(entry)>0)
  {
    hildon_gtk_window_set_progress_indicator(GTK_WINDOW(self), 1);

    gtk_list_store_clear(self->priv->result_list);
    search_in_contacts(self);
    search_in_conversations_sms(self);
    search_in_conversations_chat(self);
    search_in_calendar(self);
    hildon_gtk_window_set_progress_indicator(GTK_WINDOW(self), 0);
  }
}

static void
on_search_enter(GtkEntry* entry, gpointer user_data)
{
  _do_search(SCOUT_WINDOW(user_data));
}

static void
on_search_clicked(GtkButton* button, gpointer user_data)
{
  _do_search(SCOUT_WINDOW(user_data));
}

static void
scout_open_text_message_window(ScoutWindow* self,
			       const gchar* user_id,
			       int event_id)
{
  GtkWidget* window;
  window = hildon_stackable_window_new();
  RTComElQuery *query = rtcom_el_query_new (self->priv->rt_el);
  rtcom_el_query_prepare (query, "id", event_id, RTCOM_EL_OP_EQUAL, NULL);
  RTComElIter* iter = rtcom_el_get_events(self->priv->rt_el, query);
  
  g_object_unref(query);
  GtkWidget* text_view = hildon_text_view_new();
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_CHAR);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(text_view), FALSE);
  gboolean it = rtcom_el_iter_first(iter);
  if(it)
  {
    gchar* text = NULL;
    rtcom_el_iter_get_values(iter, "free-text", &text, NULL);

    hildon_text_view_set_placeholder (HILDON_TEXT_VIEW (text_view),
				      text);
    GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
    gtk_text_buffer_insert_at_cursor(buffer,
				     text,
				     strlen(text));
  }
  else
  {
    hildon_text_view_set_placeholder (HILDON_TEXT_VIEW (text_view),
				      "no text");
  }
  gtk_container_add(GTK_CONTAINER(window), text_view);
  gtk_window_set_title(GTK_WINDOW(window), user_id);
  gtk_widget_show_all(window);
}

static void
_open_calendar(ScoutWindow* self, int start_time)
{
  int type = 3;
  const char* text = "DUMMY";  

  osso_rpc_run(self->priv->osso,
	       CALENDAR_SERVICE, 
	       CALENDAR_PATH, 
	       CALENDAR_INTERFACE,
	       CALENDAR_LAUNCH_METHOD,
	       NULL,
	       DBUS_TYPE_UINT32, type,
	       DBUS_TYPE_INT32, start_time, 
	       DBUS_TYPE_STRING, text,
	       DBUS_TYPE_INVALID);
}

static void 
open_result_from_list(GtkTreeView* tree_view, 
		      GtkTreePath* path, 
		      GtkTreeViewColumn* column,
		      gpointer data)
{
  ScoutWindow* self = SCOUT_WINDOW(data);
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
  GtkTreeIter iter;
  if(gtk_tree_model_get_iter(GTK_TREE_MODEL(self->priv->result_list),
			  &iter,
			     path))
  {
    gchar* name = NULL;
    gchar* id = NULL;
    int type;
    int event_id;
    gtk_tree_model_get(GTK_TREE_MODEL(self->priv->result_list), &iter,
		       SEARCH_RESULT_TEXT_COL, &name,
		       SEARCH_RESULT_ITEM_TYPE_COL, &type,
		       SEARCH_RESULT_ITEM_STRING_ID_COL, &id,
		       SEARCH_RESULT_ITEM_INT_ID_COL, &event_id,
		       -1);

    if(type == SEARCH_CATEGORY_CONTACT)
    {
      GList* entries = osso_abook_aggregator_lookup(self->priv->abook, id);
      if(entries)
      {
	OssoABookContact *original_contact = (OssoABookContact*)entries->data;
	const char* n = osso_abook_contact_get_name(original_contact);
	GtkWidget* dialog = osso_abook_touch_contact_starter_dialog_new
	  (GTK_WINDOW(self),
	   ( OssoABookTouchContactStarter*)osso_abook_touch_contact_starter_new_with_contact(GTK_WINDOW(self), original_contact));
	gtk_widget_show_all(dialog);
      }
      g_list_free(entries);
    }
    else if(type == SEARCH_CATEGORY_CALENDAR)
    {
      _open_calendar(self, event_id);
    }
    else if(type == SEARCH_CATEGORY_CONVERSATION_SMS ||
	    type == SEARCH_CATEGORY_CONVERSATION_IM)
    {
      scout_open_text_message_window(self, id, event_id);
    }
    g_free(name);
    g_free(id);
    gtk_tree_selection_unselect_all(selection);
  }
}

static void
_init_rtcom(ScoutWindow* self)
{
  self->priv->rt_el = rtcom_el_new();
  self->priv->rt_com_chat_query = rtcom_el_query_new(self->priv->rt_el);
  rtcom_el_query_prepare(self->priv->rt_com_chat_query, "service", "RTCOM_EL_SERVICE_CHAT", RTCOM_EL_OP_EQUAL, NULL);
  self->priv->rt_com_sms_query = rtcom_el_query_new(self->priv->rt_el);
  rtcom_el_query_prepare(self->priv->rt_com_sms_query, "service", "RTCOM_EL_SERVICE_SMS", RTCOM_EL_OP_EQUAL, NULL);
}

static void
scout_window_init(ScoutWindow* self)
{
  self->priv = SCOUT_WINDOW_GET_PRIVATE(self);
  _init_rtcom(self);
  self->priv->cached_calendar_id_icon = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->cached_calendar_type_icon = 
    g_hash_table_new_full(g_int_hash, g_int_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);
  self->priv->cached_contact_attribute_icons = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->cached_account_icon = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->cached_contact_avatar = 
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  (GDestroyNotify) g_free, (GDestroyNotify) g_object_unref);

  self->priv->search_category_contacts_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_contacts",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_calendar_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_calendar",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_sms_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_sms",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_outgoing_sms_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "chat_replied_sms",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_im_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "general_chat",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  self->priv->search_category_conversation_outgoing_im_icon = 
    gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
			     "chat_replied_chat",
			     HILDON_ICON_PIXEL_SIZE_FINGER, 
			     GTK_ICON_LOOKUP_NO_SVG, NULL);

  GtkWidget* box = gtk_vbox_new(FALSE, 0);
  self->priv->search_entry_field = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  self->priv->result_list = gtk_list_store_new(7, 
					       GDK_TYPE_PIXBUF, 
					       G_TYPE_STRING, 
					       GDK_TYPE_PIXBUF, 
					       GDK_TYPE_PIXBUF, 
					       G_TYPE_INT, 
					       G_TYPE_STRING, 
					       G_TYPE_INT);
  GtkWidget* pan = hildon_pannable_area_new();
  GtkWidget* entry_alignment = gtk_alignment_new(0, 1, 1, 0);
  GtkCellRenderer* renderer = NULL;
  GtkCellRenderer* renderer2 = NULL;
  GtkCellRenderer* renderer3 = NULL;
  GtkCellRenderer* renderer4 = NULL;

  self->priv->search_touch_list = hildon_gtk_tree_view_new(HILDON_UI_MODE_NORMAL);

  GtkWidget* search_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
							 HILDON_BUTTON_ARRANGEMENT_VERTICAL,
							 "search", NULL);
  renderer = gtk_cell_renderer_pixbuf_new();
  renderer2 = gtk_cell_renderer_text_new();
  renderer3 = gtk_cell_renderer_pixbuf_new();
  renderer4 = gtk_cell_renderer_pixbuf_new();
  
  GtkTreeViewColumn* column = gtk_tree_view_column_new();
  GtkTreeViewColumn* column2 = gtk_tree_view_column_new();
  gtk_tree_view_column_set_sizing (column2,
				   GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width (column2,
					785 - (48+48+48));

  GtkTreeViewColumn* column3 = gtk_tree_view_column_new();
  GtkTreeViewColumn* column4 = gtk_tree_view_column_new();

  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column);
  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column2);
  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column3);
  gtk_tree_view_append_column(GTK_TREE_VIEW(self->priv->search_touch_list),
			      column4);

  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_pack_start(column2, renderer2, TRUE);  
  gtk_tree_view_column_pack_start(column3, renderer3, FALSE);  
  gtk_tree_view_column_pack_start(column4, renderer4, FALSE);  

  gtk_tree_view_column_set_attributes(column,
				      renderer,
				      "pixbuf",SEARCH_CATEGORY_ICON_COL,
				      NULL);
  gtk_tree_view_column_set_attributes(column2,
				      renderer2,
				      "markup", SEARCH_RESULT_TEXT_COL,
				      NULL);
  gtk_tree_view_column_set_attributes(column3,
				      renderer3,
				      "pixbuf",SEARCH_RESULT_ITEM_TYPE_ICON_COL,
				      NULL);
  gtk_tree_view_column_set_attributes(column4,
				      renderer4,
				      "pixbuf",SEARCH_RESULT_ITEM_ID_ICON_COL,
				      NULL);
  //  gtk_tree_view_column_set_expand(column2, TRUE);
  gtk_tree_view_set_model(GTK_TREE_VIEW(self->priv->search_touch_list),
			 GTK_TREE_MODEL(self->priv->result_list));

  GtkWidget* search_touch_alignment = gtk_alignment_new(0, 0, 1.0f, 1.0f);

  GtkWidget* search_box = gtk_hbox_new(FALSE, 0);

  gtk_container_add(GTK_CONTAINER(search_touch_alignment), pan);


  gtk_container_add(GTK_CONTAINER(pan), self->priv->search_touch_list);
  g_signal_connect(self->priv->search_entry_field, "activate", G_CALLBACK(on_search_enter), self);
  g_signal_connect(search_button, "clicked", G_CALLBACK(on_search_clicked), self);
  GtkTreeSelection* selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(self->priv->search_touch_list));
  g_signal_connect(self->priv->search_touch_list, "row-activated", G_CALLBACK(open_result_from_list), self);
  gtk_box_pack_start(GTK_BOX(search_box), self->priv->search_entry_field, TRUE, TRUE, 3);
  gtk_box_pack_end(GTK_BOX(search_box), search_button, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(entry_alignment), search_box);
  gtk_box_pack_start(GTK_BOX(box), search_touch_alignment, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(box), entry_alignment, FALSE, FALSE, 0);
  gtk_widget_show_all(box);
  gtk_container_add(GTK_CONTAINER(self), box);
  gtk_widget_grab_focus(self->priv->search_entry_field);
}

static void
scout_window_finalize(GObject* object)
{
  ScoutWindow* self = SCOUT_WINDOW(object);
  g_object_unref(self->priv->result_list);
  if(self->priv->rt_el)
    g_object_unref(self->priv->rt_el);
  if(self->priv->rt_com_chat_query)
    g_object_unref(self->priv->rt_com_chat_query);
  if(self->priv->rt_com_sms_query)
    g_object_unref(self->priv->rt_com_sms_query);
  if(self->priv->abook)
    g_object_unref(self->priv->abook);
  if(self->priv->search_category_contacts_icon)
    g_object_unref(self->priv->search_category_contacts_icon);
  if(self->priv->search_category_calendar_icon)
    g_object_unref(self->priv->search_category_calendar_icon);
  if(self->priv->search_category_conversation_sms_icon)
    g_object_unref(self->priv->search_category_conversation_sms_icon);
  if(self->priv->search_category_conversation_im_icon)
    g_object_unref(self->priv->search_category_conversation_im_icon);

  g_warning("destroy contact_attribute_icons\n");
  g_hash_table_destroy(self->priv->cached_contact_attribute_icons);
  g_warning("destroy contact_avatar_icons\n");
  g_hash_table_destroy(self->priv->cached_contact_avatar);
  g_warning("destroy calendar_id_icons\n");
  g_hash_table_destroy(self->priv->cached_calendar_id_icon);
  g_warning("destroy calendar_type_icons\n");
  g_hash_table_destroy(self->priv->cached_calendar_type_icon);
  g_hash_table_destroy(self->priv->cached_account_icon);

  G_OBJECT_CLASS(scout_window_parent_class)->finalize(object);
}

static void
scout_window_dispose(GObject* object)
{

  G_OBJECT_CLASS(scout_window_parent_class)->dispose(object);
}

static void
scout_window_class_init(ScoutWindowClass* klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS(klass);
  g_type_class_add_private(klass, sizeof(ScoutWindowPrivate));
  object_class->finalize = scout_window_finalize;
  object_class->dispose = scout_window_dispose;
}

