/*
    This file is part of Scoresheet, a simple scorekeeping application.
    Copyright (C) 2010 Heikki Holstila

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <QtGui>
#include <QMaemo5InformationBox>
#include <iostream>
#include "scoresheet.h"
#include "csvfile.h"

CScoresheet::CScoresheet(QWidget *win_)
{
	win = win_;
	defaultSizeX = 4;
	defaultSizeY = 30;
	minSizeX = 1;
	maxSizeX = 20;
	minSizeY = 5;
	maxSizeY = 200;
	sizeX = defaultSizeX;
	sizeY = defaultSizeY;
	defaultRowHeight = 55;
	minSectSize = 100;
	typing = false;
	hideKeys = true;
	portrait = false;
	loadingFile = false;
	defaultName = "Player ";
	defaultDirectory = "/home/user/MyDocs";
	lastFileName = "scores.csv";
	
	for(int i=0; i<sizeX; i++)
	{
		QString s;
		s.setNum(i+1);
		names << defaultName+s;
	}

	QHBoxLayout *hbox = new QHBoxLayout(win);
	QVBoxLayout *vbox = new QVBoxLayout();
	hbox->addLayout(vbox);

	table = new QTableWidget(sizeY,sizeX);
	vbox->addWidget(table);
	table->setSelectionMode(QAbstractItemView::NoSelection);
	resizeTable();
	connect( table, SIGNAL(cellChanged(int,int)), this, SLOT(updateHeaders()) );
	connect( table, SIGNAL(cellClicked(int,int)), this, SLOT(cellClicked(int,int)) );
	vbox->setContentsMargins(0,0,0,0);
	hbox->setContentsMargins(0,0,0,0);	
	table->setContentsMargins(0,0,0,0);
	
	keys = new QWidget();
	QGridLayout *keyslayout = new QGridLayout(keys);
	vbox->addWidget(keys);
	keyslayout->setHorizontalSpacing(0);
	keyslayout->setVerticalSpacing(0);
	keyslayout->setContentsMargins(0,0,0,0);
	keys->setContentsMargins(0,0,0,0);

	QPushButton *pb1 = new QPushButton("1");
	keyslayout->addWidget(pb1, 0, 0);
	connect( pb1, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb1->setObjectName("pb1");
	QPushButton *pb2 = new QPushButton("2");
	keyslayout->addWidget(pb2, 0, 1);
	connect( pb2, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb2->setObjectName("pb2");
	QPushButton *pb3 = new QPushButton("3");
	keyslayout->addWidget(pb3, 0, 2);
	connect( pb3, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb3->setObjectName("pb3");
	QPushButton *pb4 = new QPushButton("4");
	keyslayout->addWidget(pb4, 0, 3);
	connect( pb4, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb4->setObjectName("pb4");
	QPushButton *pb5 = new QPushButton("5");
	keyslayout->addWidget(pb5, 0, 4);
	connect( pb5, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb5->setObjectName("pb5");
	QPushButton *pb6 = new QPushButton("6");
	keyslayout->addWidget(pb6, 1, 0);
	connect( pb6, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb6->setObjectName("pb6");
	QPushButton *pb7 = new QPushButton("7");
	keyslayout->addWidget(pb7, 1, 1);
	connect( pb7, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb7->setObjectName("pb7");
	QPushButton *pb8 = new QPushButton("8");
	keyslayout->addWidget(pb8, 1, 2);
	connect( pb8, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb8->setObjectName("pb8");
	QPushButton *pb9 = new QPushButton("9");
	keyslayout->addWidget(pb9, 1, 3);
	connect( pb9, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb9->setObjectName("pb9");
	QPushButton *pb0 = new QPushButton("0");
	keyslayout->addWidget(pb0, 1, 4);
	connect( pb0, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pb0->setObjectName("pb0");
	QPushButton *pbHide = new QPushButton();
	keyslayout->addWidget(pbHide, 2, 0);
	connect( pbHide, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );	
	pbHide->setObjectName("pbHide");
	QPushButton *pbFS = new QPushButton();
	keyslayout->addWidget(pbFS, 2, 1);
	connect( pbFS, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pbFS->setObjectName("pbFS");
	QPushButton *pbPM = new QPushButton("+/-");
	keyslayout->addWidget(pbPM, 2, 3);
	connect( pbPM, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pbPM->setObjectName("pbPM");
	QPushButton *pbBS = new QPushButton();
	keyslayout->addWidget(pbBS, 2, 4);
	connect( pbBS, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	pbBS->setObjectName("pbBS");
	
	if( hideKeys )
		keys->hide();
	
	barV = new QWidget();
	QVBoxLayout *barlayout = new QVBoxLayout(barV);
	QPushButton *barbKB = new QPushButton();
	connect( barbKB, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	barbKB->setObjectName("barbKB");
	QPushButton *barbFS = new QPushButton();
	connect( barbFS, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	barbFS->setObjectName("barbFS");
	barbKB->setMaximumWidth(70);
	barbFS->setMaximumWidth(70);
	barlayout->addStretch(0);
	barlayout->addWidget(barbFS);
	barlayout->addWidget(barbKB);
	barlayout->setContentsMargins(0,0,0,0);
	barV->setContentsMargins(0,0,0,0);
	hbox->addWidget(barV);

	if( !hideKeys ) {
		barV->hide();
	}

	barH = new QWidget();
	QHBoxLayout *barlayout2 = new QHBoxLayout(barH);
	QPushButton *barbKB2 = new QPushButton();
	connect( barbKB2, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	barbKB2->setObjectName("barbKB");
	QPushButton *barbFS2 = new QPushButton();
	connect( barbFS2, SIGNAL(clicked()), this, SLOT(softkeyClicked()) );
	barbFS2->setObjectName("barbFS");
	barbKB2->setMaximumWidth(120);
	barbFS2->setMaximumWidth(120);
	barlayout2->addWidget(barbKB2);
	barlayout2->addWidget(barbFS2);
	barlayout2->addStretch(0);
	barlayout2->setContentsMargins(0,0,0,0);
	barH->setContentsMargins(0,0,0,0);
	vbox->addWidget(barH);

	barH->hide();
		
	QIcon icon_fs("/usr/share/icons/hicolor/48x48/hildon/general_fullsize.png");
	pbFS->setIcon(icon_fs);
	barbFS->setIcon(icon_fs);
	barbFS2->setIcon(icon_fs);
	QIcon icon_hide("/usr/share/icons/hicolor/48x48/hildon/rss_reader_move_down.png");
	pbHide->setIcon(icon_hide);
	QIcon icon_keys("/usr/share/icons/hicolor/48x48/hildon/call_dialpad_button.png");
	barbKB->setIcon(icon_keys);
	barbKB2->setIcon(icon_keys);
	QIcon icon_bs("/usr/share/icons/hicolor/48x48/hildon/general_backspace.png");
	pbBS->setIcon(icon_bs);
	
	
	dlg_confirmation = new QDialog(win,Qt::Dialog);
	dlg_confirmation->setWindowTitle("Clear scores?");
	QHBoxLayout *confirm_hbox = new QHBoxLayout(dlg_confirmation);
	QDialogButtonBox *confirm_bb = new QDialogButtonBox(QDialogButtonBox::Yes|QDialogButtonBox::No|
		QDialogButtonBox::Cancel, Qt::Vertical,dlg_confirmation);
	confirm_hbox->addWidget(confirm_bb);
	connect(confirm_bb, SIGNAL(accepted()), dlg_confirmation, SLOT(accept()));
	connect(confirm_bb, SIGNAL(rejected()), dlg_confirmation, SLOT(reject()));	

	dlg_about = new QDialog(win,Qt::Dialog);
	dlg_about->setWindowTitle("About");
	QHBoxLayout *abt_hbox = new QHBoxLayout(dlg_about);
	QLabel *aboutl = new QLabel("Scoresheet version "+PROGRAM_VERSION+" by Heikki Holstila",dlg_about);
	aboutl->setWordWrap(true);
	abt_hbox->addWidget(aboutl);

	progress = new QProgressBar(win);
	vbox->addWidget(progress);
	progress->hide();

	createSetup();
}

CScoresheet::~CScoresheet()
{
}

void CScoresheet::orientationChanged()
{
	QRect screenGeometry = QApplication::desktop()->screenGeometry();
	if (screenGeometry.width() > screenGeometry.height()) {
		keys->hide();
		barV->show();
		barH->hide();
		portrait = false;
	} else {
		barV->hide();
		if( hideKeys ) {
			keys->hide();
			barV->hide();
			barH->show();
		} else {
			barH->hide();
			barV->hide();
			keys->show();
		}
		portrait = true;
	}
}

void CScoresheet::updateHeaders()
{
	if( loadingFile ) return;

	totalScores.clear();
	for(int i=0; i<sizeX; i++)
	{
		totalScores.append(0);
		for(int j=0; j<sizeY; j++)
		{
			QString data = "0";
			if( table->item(j,i) ) {
				data = table->item(j,i)->text();
			}
			totalScores[i] += data.toInt();
		}
	} 

	QStringList labels;
	for(int i=0; i<sizeX; i++)
	{
		QString s;
		s.setNum(totalScores.at(i));
		labels << names.at(i) + "\n" + s;
	}
	table->setHorizontalHeaderLabels(labels);
}


void CScoresheet::cellClicked(int row_, int col_)
{
	typing = false;
	if( table->item(lastcellRow,lastcellCol) )
	{
		table->item(lastcellRow,lastcellCol)->setSelected(false);
	}

	lastcellCol = col_;
	lastcellRow = row_;

	if( !table->item(lastcellRow,lastcellCol) )
	{
		QTableWidgetItem *item = new QTableWidgetItem();
		table->setItem(lastcellRow,lastcellCol,item);
		table->item(lastcellRow,lastcellCol)->setTextAlignment(Qt::AlignCenter);
	}

	typing = true;
	table->item(lastcellRow,lastcellCol)->setSelected(true);
}

void CScoresheet::softkeyClicked()
{
	if( sender() )
	{
		QString s = sender()->objectName();
		//std::cout << s.toStdString() << std::endl;
		
		if( s=="pbHide" ) {
			hideKeys = true;
			keys->hide();
			if( portrait ) {
				barH->show();
				barV->hide();
			} else {
				barH->hide();
				barV->show();
			}
		}
		if( s=="barbKB" ) {
			hideKeys = false;
			keys->show();
			barV->hide();
			barH->hide();
		}

		if( s=="pbOK" && typing==true ) {
			typing = false;
			if( table->item(lastcellRow,lastcellCol) ) {
				table->item(lastcellRow,lastcellCol)->setSelected(false);			
			}
			updateHeaders();
		}
		
		if( s=="pbFS" || s=="barbFS" ) {
			win->setWindowState(win->windowState() ^ Qt::WindowFullScreen);
		}

		if( typing == false ) return;
		if( !table->item(lastcellRow,lastcellCol) )
		{
			std::cerr << "ERROR: Trying to operate on a non-existent table item" << std::endl;
			return;
		}
		QString data = table->item(lastcellRow,lastcellCol)->text();

		if( s=="pb1" ) data.append("1");
		if( s=="pb2" ) data.append("2");
		if( s=="pb3" ) data.append("3");
		if( s=="pb4" ) data.append("4");
		if( s=="pb5" ) data.append("5");
		if( s=="pb6" ) data.append("6");
		if( s=="pb7" ) data.append("7");
		if( s=="pb8" ) data.append("8");
		if( s=="pb9" ) data.append("9");
		if( s=="pb0" ) data.append("0");
		
		if( s=="pbBS" ) {
			if(data.length()>0)
				data.remove(data.length()-1,1);
		}

		if( s=="pbPM" ) {
			if(data.length()>0)
			{
				if( data.at(0) == '-' )
				{
					data.remove(0,1);
				} else {
					data.prepend("-");
				}
			} else {
				data.append("-");
			}
		}
		
		table->item(lastcellRow,lastcellCol)->setText(data);
	}
}

void CScoresheet::clear()
{
    if( win->rect().width() < win->rect().height() )
        win->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

	if( dlg_confirmation->exec() )
	{
		if( table->item(lastcellRow,lastcellCol) ) {
			table->item(lastcellRow,lastcellCol)->setSelected(false);			
		}
		typing=false;
		
		table->clearContents();
		resizeTable();
	}

	win->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
}

void CScoresheet::resizeTable()
{
	table->setColumnCount(sizeX);
	table->setRowCount(sizeY);
	
	table->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
	table->horizontalHeader()->setMinimumSectionSize(minSectSize);

	typing=false;

	for(int j=0; j<sizeY; j++) {
		table->setRowHeight(j, defaultRowHeight);
	}

	updateHeaders();
}

void CScoresheet::showSetup()
{
	stp_players->setValue(sizeX);
	stp_rows->setValue(sizeY);

	stp_list->clear();
	stp_list->addItems(names);
	for(int i=0; i<sizeX; i++)
		stp_list->item(i)->setFlags(stp_list->item(i)->flags() | Qt::ItemIsEditable);

    if( win->rect().width() < win->rect().height() )
        win->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

	win_setup->show();

    win->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
}

void CScoresheet::saveSetup()
{
	win_setup->hide();
	
	if( table->item(lastcellRow,lastcellCol) ) {
		table->item(lastcellRow,lastcellCol)->setSelected(false);			
	}
	
	sizeX = stp_players->value();
	sizeY = stp_rows->value();
	
	names.clear();
	for(int i=0; i<sizeX; i++)
	{
		names.append( stp_list->item(i)->text() );
	}
	
	resizeTable();
}

void CScoresheet::playersNumChanged(int num)
{
	int count = stp_list->count();
	if( num == count ) return;
	if( num < count )
	{
		for(int i=0; i<count-num; i++)
		{
			delete stp_list->takeItem(num);
		}
	}
	if( num > count ) 
	{
		QString s;
		for(int i=0; i<num-count; i++)
		{
			s.setNum(i+count+1);
			stp_list->addItem(defaultName+s);
			stp_list->item(i+count)->setFlags(stp_list->item(i+count)->flags() | Qt::ItemIsEditable);
		}
	}
}

void CScoresheet::createSetup()
{
	win_setup = new QDialog(win,Qt::Dialog);
	win_setup->setWindowTitle("Scoresheet setup");

	QHBoxLayout *stp_hbox0 = new QHBoxLayout(win_setup);
	QVBoxLayout *stp_vbox1 = new QVBoxLayout();
	stp_hbox0->addLayout(stp_vbox1);

	QLabel *stp_l1 = new QLabel("Players");
	QLabel *stp_l2 = new QLabel("Rows");
	stp_players = new QSpinBox();
	stp_players->setRange(minSizeX,maxSizeX);
	stp_players->setValue(sizeX);
	stp_rows = new QSpinBox();
	stp_rows->setRange(minSizeY,maxSizeY);
	stp_rows->setValue(sizeY);
	stp_vbox1->addWidget(stp_l1);
	stp_vbox1->addWidget(stp_players);
	stp_vbox1->addWidget(stp_l2);
	stp_vbox1->addWidget(stp_rows);
	stp_vbox1->addStretch();
	stp_list = new QListWidget();
	stp_hbox0->addWidget(stp_list);
	stp_list->addItems(names);
	stp_list->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	
	for(int i=0; i<sizeX; i++)
		stp_list->item(i)->setFlags(stp_list->item(i)->flags() | Qt::ItemIsEditable);

	QVBoxLayout *stp_vbox2 = new QVBoxLayout();
	QPushButton *stp_save = new QPushButton("Save");
	connect( stp_save, SIGNAL(clicked()), this, SLOT(saveSetup()) );
	stp_vbox2->addStretch();
	stp_vbox2->addWidget(stp_save);
	stp_hbox0->addLayout(stp_vbox2);
	
	connect(stp_players, SIGNAL(valueChanged(int)), this, SLOT(playersNumChanged(int)));
}

void CScoresheet::save()
{

	if( win->rect().width() < win->rect().height() )
		win->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

	QString fname = QFileDialog::getSaveFileName(win, "Save file",
		defaultDirectory+"/"+lastFileName, "Comma Separated Values (*.csv)");
		
	win->setAttribute(Qt::WA_Maemo5AutoOrientation, true);

	if( fname!=NULL && !fname.endsWith(".csv") )
		fname.append(".csv");
	
	if( fname != NULL  )
	{
		defaultDirectory = fname.section("/",0,-2);
		lastFileName = fname.section("/",-1);

		CCSVFile csv( fname );
		
		csv.appendRow(names);

		QStringList row;		
		for( int i=0; i<table->rowCount(); i++ )
		{
			row.clear();
			for( int j=0; j<table->columnCount(); j++)
			{
				if( table->item(i,j) )
					row.append( table->item(i,j)->text() );
				else
					row.append("");
			}
			csv.appendRow(row);
		}
		if( !csv.save() ) {
			QMaemo5InformationBox::information ( win, "Could not open file for writing!",
				QMaemo5InformationBox::DefaultTimeout );
		} else {
			QMaemo5InformationBox::information ( win, "Saved",
				QMaemo5InformationBox::DefaultTimeout );
		}
	}
}

void CScoresheet::load()
{
    if( win->rect().width() < win->rect().height() )
        win->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

	QString fname = QFileDialog::getOpenFileName(win, "Load file",
		defaultDirectory, "Comma Separated Values (*.csv)");

	win->setAttribute(Qt::WA_Maemo5AutoOrientation, true);

	if( fname != NULL )
	{
		defaultDirectory = fname.section("/",0,-2);
		lastFileName = fname.section("/",-1);
		
		CCSVFile csv( fname );
		
		if( !csv.load() ) {
			QMaemo5InformationBox::information ( win, "Could not load file!",
				QMaemo5InformationBox::DefaultTimeout );
			return;
		}
		sizeX = csv.columns();
		sizeY = csv.rows()-1;  //first line is for headers
		if( sizeX < minSizeX ) sizeX=minSizeX;
		if( sizeX > maxSizeX ) sizeX=maxSizeX;
		if( sizeY < minSizeY ) sizeY=minSizeY;
		if( sizeY > maxSizeY ) sizeY=maxSizeY;
		
		names.clear();
		table->clear();
		for( int i=0; i<sizeX; i++ ) {
			QString n = csv.dataAt(0,i);
			QString s;
			s.setNum(i+1);
			if( n=="" ) n=defaultName+s;
			names.append(n);
		}
		resizeTable();
		loadingFile = true;

		progress->setRange(0,sizeX);
		progress->setValue(0);
		keys->hide();
		barH->hide();
		progress->show();
		table->setEnabled(false);

		for( int i=0; i<sizeX; i++ )
		{
			for(int j=0; j<sizeY; j++ )
			{
				QString t = csv.dataAt(j+1,i);
				if( !table->item(j,i) && t.length()>0 )
				{
					QTableWidgetItem *item = new QTableWidgetItem();
					table->setItem(j,i,item);
					table->item(j,i)->setTextAlignment(Qt::AlignCenter);
				}
				if( t.length()>0 ) table->item(j,i)->setText(t);
			}
			progress->setValue(i);
		}

   		progress->hide();
   		if( !hideKeys ) keys->show();
   		else if( portrait ) barH->show();
		table->setEnabled(true);

		loadingFile = false;		
		updateHeaders();
	}
}

void CScoresheet::showAbout()
{
    if( win->rect().width() < win->rect().height() )
        win->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

	dlg_about->exec();

	win->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
}
