/*
    CSV (Comma Separated Values) file handler for Qt
    Copyright (C) 2010 Heikki Holstila

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <QtCore>
#include "csvfile.h"

#include <iostream> //not needed after testing


CCSVFile::CCSVFile(QString fileName_, bool load_)
{
	fileName = fileName_;
	if( load_ )
		load();
}

CCSVFile::~CCSVFile()
{
	clear();
}

bool CCSVFile::load()
{
	clear();

	QFile file( fileName );	
	if( !file.open(QIODevice::ReadOnly | QIODevice::Text) ) {
		return false;
	}
	QTextStream in(&file);
	
	QString s;
	QStringList row;
	
	s = in.readLine();
	bool quotesMatch = true;
	QString cellData;
	int corr=0;
	while (!s.isNull()) {
		int prevpos = 0;
		for( int i=0; i<s.length(); i++ )
		{
			if( s.at(i)=='"' ) {
				if( quotesMatch )
					quotesMatch=false;
				else
					quotesMatch=true;
			}
			//std::cout << s.at(i).toAscii() << std::endl;
			if( (s.at(i)==',' || s.at(i)=='\n' || i==s.length()-1) && quotesMatch )
			{
				if( (s.at(i)=='\n' || i==s.length()-1) && s.at(i)!=',' ) corr=1;
					else corr=0;
				cellData += s.mid(prevpos,i-prevpos+corr);
				if( cellData.at(0)=='"' && cellData.at(cellData.length()-1)=='"' )
				{
					cellData.remove(0,1);
					cellData.remove(cellData.length()-1,1);
				}
				cellData.replace("\"\"","\"");
				//std::cout << "\"" << cellData.toStdString() << "\"" << std::endl;
				row.append( cellData );
				prevpos=i+1;
				cellData.clear();
			}
			else if( (s.at(i)=='\n' || i==s.length()-1) && !quotesMatch )
			{
				cellData += s.mid(prevpos,i-prevpos+1) + "\n";
			}
		}
		if( quotesMatch )
		{
			appendRow(row);
			row.clear();
		}
	
		s = in.readLine();
	}
	
	file.close();
	return true;
}

bool CCSVFile::save() const
{
	QFile file( fileName );
	if( !file.open(QIODevice::WriteOnly | QIODevice::Text) ) {
		return false;
	}
	QTextStream out(&file);

	int cols = columns();
	int rws = rows();
	
	for(int h=0; h<rws; h++)
	{
		for(int w=0; w<cols; w++)
		{
			QString outstr = dataAt(h,w);
			if( outstr.contains(',') || outstr.contains('\n') )
				out << "\"";
			outstr.replace("\"","\"\"");
			out << outstr;
			if( outstr.contains(',') || outstr.contains('\n') )
				out << "\"";

			if( w < cols-1 ) out << ",";
		}
		out << "\n";
	}
	
	out.flush();
	file.close();
	return true;
}

void CCSVFile::setFileName(QString fileName_)
{
	fileName = fileName_;
}

void CCSVFile::clear()
{
	if(data.count()==0) return;
	for(int i=0; i<data.count(); i++)
	{
		data[i].clear();
	}
	data.clear();
}

void CCSVFile::appendRow(QStringList row_)
{
	data.append(row_);
}

QStringList* CCSVFile::rowAt(int row_)
{
	if( data.count()==0 || data.count() < row_ )
		return 0;
	
	return &data[row_];
}

QString* CCSVFile::cellAt(int row_, int col_)
{
	if( data.count()==0 || data.count() < row_ )
		return 0;
	if( data[row_].count() < col_ )
		return 0;
	return &data[row_][col_];
}

QString CCSVFile::dataAt(int row_, int col_) const
{
	if( data.count()==0 || data.count()-1 < row_ )
		return "";
	if( data[row_].count()-1 < col_ )
		return "";
	return data[row_][col_];
}

void CCSVFile::setDataAt(int row_, int col_, QString cellData_)
{
	while( row_>data.count()-1 ) {
		QStringList empty;
		data.append(empty);
	}
	while( col_>data[row_].count()-1 ) {
		QString empty;
		data[row_].append(empty);
	}
	data[row_][col_] = cellData_;
}

bool CCSVFile::isEmpty() const
{
	if( data.count() == 0 ) {
		return true;
	} else {
		for(int i=0; i<data.count(); i++)
			if(data.at(i).count() > 0) return false;
	}
	return true;
}

int CCSVFile::rows() const
{
	return data.count();
}

int CCSVFile::columns() const
{
	if(data.count()==0) return 0;
	int h=0;
	for(int i=0; i<data.count(); i++)
	{
		if( data.at(i).count() > h )
			h = data.at(i).count();
	}
	return h;
}

