/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QColor>
#include <QBrush>
#include <QFont>
#include "table-model.h"
#include "score-common.h"

QString empty("");

ScoreTableModel::ScoreTableModel(QObject *parent) 
    : QAbstractTableModel(parent)
{
    score = 0;
    course = 0;
    handicap = -1;
}

void ScoreTableModel::set(Score * s, Course * c, int h)
{
    score = s;
    course = c;
    handicap = h;
}

int ScoreTableModel::rowCount(const QModelIndex &) const
{
    return ROWS;
}
 
int ScoreTableModel::columnCount(const QModelIndex &) const
{
    return COLS + 2; // 2 for in/out and tot columns
}

QVariant ScoreTableModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    int row = index.row();
    int col = index.column();

    //
    // ALIGNMENT
    //
    if (role == Qt::TextAlignmentRole ) {
        return Qt::AlignCenter;
    }

    if (index.column() > 10)
        return QVariant();

    //
    // COLORS
    //
    if (role == Qt::ForegroundRole) {
        // Hole numbers 1-18. All hole nums, in, out and tot cell but not
        // the empty cell.
        if ((row == ROW_HOLE && col != 10) || row == ROW_HOLE_2) {
            QBrush brush(ScoreColor::holeBg());
            return brush;
        }
        if (score && course && (row == ROW_SCORE || row == ROW_SCORE_2)) {
            int par;
            int shots;
            if (row == ROW_SCORE) {
                par = course->getPar(col).toInt();
                shots = score->getScore(col).toInt();
            }
            else {
                par = course->getPar(col + 9).toInt();
                shots = score->getScore(col + 9).toInt();
            }

            if (col == (COLS+1) && row == ROW_SCORE_2) {
                // Total score
                QBrush brush(ScoreColor::total());
                return brush;
            }
            if (col == COLS) {
                // In and Out scores
                QBrush brush(ScoreColor::subTotal());
                return brush;
            }
            if (col < COLS) {
                if (shots == par) {
                    // Par
                    QBrush brush(ScoreColor::par());
                    return brush;
                }
                if (shots == (par-1)) {
                    // Birdie
                    QBrush brush(ScoreColor::birdie());
                    return brush;
                }
                if (shots == (par+1)) {
                    // Bogey
                    QBrush brush(ScoreColor::bogey());
                    return brush;
                }
                if (shots == (par+2)) {
                    // Double Bogey
                    QBrush brush(ScoreColor::doubleBogey());
                    return brush;
                }
                if (shots > (par+2)) {
                    // Very bad
                    QBrush brush(ScoreColor::bad());
                    return brush;
                }
            }
        }
        return QVariant();
    }
    //
    // FONT
    //
    if (role == Qt::FontRole) {
        QFont font;
        if (col == (COLS+1) && row == ROW_SCORE_2) {
            font.setBold(true);
            font.setPointSize(fontSize+4);
        }
        else 
            font.setPointSize(fontSize);
        if (row == ROW_HOLE || row == ROW_HOLE_2) {
            font.setBold(true);
        }
        return font;
    }

    //
    // NUMBERS
    //
    if (role == Qt::DisplayRole) {

        if (col == COLS) {
            // In/out label
            if (row == ROW_HOLE)
                return QString("Out");
            if (row == ROW_HOLE_2)
                return QString("In");

            // In/Out for par
            if (score && course && row == ROW_PAR)
                return course->getTotal(TotalOut);
            if (score && course && row == ROW_PAR_2)
                return course->getTotal(TotalIn);

            // In/Out for score
            if (score && row == ROW_SCORE)
                return score->getTotal(TotalOut);
            if (score && row == ROW_SCORE_2)
                return score->getTotal(TotalIn);
      
        }
        else if (col == (COLS+1)) {
            // Total label
            if (row == ROW_HOLE_2)
                return QString("Tot");
            // Total score
            if (score && course && row == ROW_PAR_2)
                return course->getTotal(Total);
            if (score && row == ROW_SCORE_2)
                return score->getTotal(Total);
            // calculate net score
            if (score && course && row == ROW_HCP_2) {
                int scoreTotal = score->getTotal(Total).toInt();
                int courseTotal = course->getTotal(Total).toInt();
                int n = scoreTotal - courseTotal;
                return QString("+%1").arg(n);
            }
        }
        else {
            // data cells
            switch(row) {
            case ROW_HOLE:
                return col + 1;

            case ROW_HOLE_2:
                return col + 10;

            case ROW_PAR:
                if (course)
                    return course->getPar(col);
                else
                    return QString("");

            case ROW_PAR_2:
                if (course)
                    return course->getPar(col + 9);
                else
                    return QString("");

            case ROW_HCP: 
                if (course)
                    return course->getHcp(col); 
                else
                    return QString("");

            case ROW_HCP_2:
                if (course)
                    return course->getHcp(col + 9);
                else
                    return QString("");

            case ROW_SCORE:
                if (score)
                    return score->getScore(col);
                else
                    return QString("");
              
            case ROW_SCORE_2: 
                if (score)
                    return score->getScore(col + 9);
                else
                    return QString("");
            }
        }
    }
    return QVariant();
}

QVariant ScoreTableModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    // Only vertical header -- horizontal is hidden
    if (orientation == Qt::Horizontal)
        return QVariant();

    if (role == Qt::FontRole) {
        QFont font;
        font.setPointSize(fontSize);
        return font;
    }
    if (role == Qt::DisplayRole) {
        switch(section) {
        case ROW_HOLE: 
        case ROW_HOLE_2: 
            return QString("Hole");
        case ROW_PAR: 
        case ROW_PAR_2: 
            return QString("Par");
        case ROW_HCP: 
        case ROW_HCP_2: 
            return QString("HCP");
        case ROW_SCORE: 
        case ROW_SCORE_2: 
            return QString("Score");
        }
        return QVariant();
    }

    return QVariant();
}

//
// CourseTableModel
//
CourseTableModel::CourseTableModel(QObject *parent) 
    : QAbstractTableModel(parent)
{
    course = 0;
}

void CourseTableModel::set(Course *c)
{
    course = c;
}

int CourseTableModel::rowCount(const QModelIndex &) const
{
    return ROWS;
}
 
int CourseTableModel::columnCount(const QModelIndex &) const
{
    return COLS + 2;
}

QVariant CourseTableModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    int row = index.row();
    int col = index.column();

    //
    // ALIGNMENT
    //
    if (role == Qt::TextAlignmentRole ) {
        return Qt::AlignCenter;
    }

    //
    // FONT
    //
    if (role == Qt::FontRole) {
        QFont font;
        font.setPointSize(fontSize);
        if (row == ROW_HOLE || row == ROW_HOLE_2) {
            font.setBold(true);
        }
        return font;
    }
    //
    // COLORS
    //
    if (role == Qt::ForegroundRole) {
        // Hole numbers 1-18. All hole nums, in, out and tot cell but not
        // the empty cell.
        if ((row == ROW_HOLE && col != 10) || row == ROW_HOLE_2) {
            QBrush brush(ScoreColor::holeBg());
            return brush;
        }
        return QVariant();
    }
    //
    // NUMBERS
    //
    if (role == Qt::DisplayRole) {

        if (col == COLS) {
            // In/out label
            if (row == ROW_HOLE)
                return QString("Out");
            if (row == ROW_HOLE_2)
                return QString("In");

            // In/Out for par
            if (course && row == ROW_PAR)
                return course->getTotal(TotalOut);
            if (course && row == ROW_PAR_2)
                return course->getTotal(TotalIn);
        }
        else if (col == (COLS+1)) {
            // Total label
            if (row == ROW_HOLE_2)
                return QString("Tot");
            // Total score
            if (course && row == ROW_PAR_2)
                return course->getTotal(Total);
        }
        else {
            // data cells
            switch(row) {
            case ROW_HOLE:
                return col + 1;
            case ROW_HOLE_2:
                return col + 10;
            case ROW_PAR:
                if (course)
                    return course->getPar(col); 
            case ROW_PAR_2:
                if (course)
                    return course->getPar(col + 9); 
            case ROW_HCP: 
                if (course)
                    return course->getHcp(col); 
            case ROW_HCP_2:
                if (course)
                    return course->getHcp(col + 9);
            }
        }
    }
    return QVariant();
}

QVariant CourseTableModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    // Only vertical header -- horizontal is hidden
    if (orientation == Qt::Horizontal)
        return QVariant();

    if (role == Qt::FontRole) {
        QFont font;
        font.setPointSize(fontSize);
        return font;
    }
    if (role == Qt::DisplayRole) {
        switch(section) {
        case ROW_HOLE: 
        case ROW_HOLE_2: 
            return QString("Hole");
        case ROW_PAR: 
        case ROW_PAR_2: 
            return QString("Par");
        case ROW_HCP: 
        case ROW_HCP_2: 
            return QString("HCP");
        case ROW_LEN: 
        case ROW_LEN_2: 
            return QString("Len");
        }
        return QVariant();
    }

    return QVariant();
}
