/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QtGui>
#include <QInputContext>
#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#endif

#include "score-dialog.h"
#include "score-common.h"
#include "table-model.h"

////////////////////////////////////////////////////////////////////////////////
// ScoreWindow based on QMainWindow
////////////////////////////////////////////////////////////////////////////////
ScoreWindow::ScoreWindow(QWidget *parent) : QMainWindow(parent)
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    QAction *editAction = new QAction(tr("Edit"), this);
    connect(editAction, SIGNAL(triggered()), parent, SLOT(editScore()));
    menuBar()->addAction(editAction);

    QAction *delAction = new QAction(tr("Delete"), this);
    connect(delAction, SIGNAL(triggered()), parent, SLOT(deleteScore()));
    menuBar()->addAction(delAction);

    model = new ScoreTableModel;
    
    QTableView * table = new QTableView;
    table->showGrid();
    table->setSelectionMode(QAbstractItemView::NoSelection);
    table->setStyleSheet(defaultStyleSheet);
    table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
    table->horizontalHeader()->hide();
    table->setModel(model);
    
    QWidget *central = new QWidget(this);
    setCentralWidget(central);
    
    QVBoxLayout *layout = new QVBoxLayout;
    layout->addWidget(table);
    
    central->setLayout(layout);
}

ScoreWindow::~ScoreWindow()
{
    TRACE;
}

void ScoreWindow::setup(Score *score, Course *course)
{
    TRACE;
    QString title = QString("Score: %1, %2 - %3").arg(score->getClubName()).arg(score->getCourseName()).arg(score->getDate());
    setWindowTitle(title);
    model->set(score, course);
}


////////////////////////////////////////////////////////////////////////////////
// SelectDialog based on QDialog
////////////////////////////////////////////////////////////////////////////////
SelectDialog::SelectDialog(QWidget *parent) : QDialog(parent)
{
    resize(800, 350);

    QWidget *centralWidget = new QWidget(this);
    createLayout(centralWidget);

    setWindowTitle(tr("ScoreCard: Select Course and Date"));
}

void SelectDialog::createLayout(QWidget *parent)
{
    listWidgetClub = new QListWidget(parent);
    pushButtonNext = new QPushButton(tr("Next"));

    connect(pushButtonNext, SIGNAL(clicked()), this, SLOT(next()));

    QDialogButtonBox * buttonBox = new QDialogButtonBox(Qt::Vertical);
    buttonBox->addButton(pushButtonNext, QDialogButtonBox::ActionRole);

    leftLayout = new QVBoxLayout;
    leftLayout->addWidget(listWidgetClub);

#ifdef Q_WS_MAEMO_5
    dateButton = new QMaemo5ValueButton();
    dateButton->setValueLayout(QMaemo5ValueButton::ValueUnderText);
    dateButton->setPickSelector(new QMaemo5DatePickSelector());
    dateButton->setText(QString::fromUtf8("Date"));
    leftLayout->addWidget(dateButton);
#else
    QDate today(QDate::currentDate());
    lineEditDate = new QLineEdit;
    lineEditDate->setText(today.toString("yyyy-MM-dd"));
    date = new QDateEdit;
    leftLayout->addWidget(date);
    leftLayout->addWidget(lineEditDate);
#endif

    rightLayout = new QVBoxLayout;
    rightLayout->addStretch();
    rightLayout->addWidget(buttonBox);

    QHBoxLayout *mainLayout = new QHBoxLayout(parent);
    mainLayout->addLayout(leftLayout);
    mainLayout->addLayout(rightLayout);

    setLayout(mainLayout);
}

void SelectDialog::init(QList<Club *> &list)
{
    TRACE;
    clubList = list;

    QListIterator<Club *> i(clubList);
    int index = 0;
    bool markedFlag = false;

    while (i.hasNext()) {
        Club *club = i.next();

        QList<Course *> courseList = club->getCourseList();

        QListIterator<Course *> j(courseList);
        while (j.hasNext()) {
            Course *course = j.next();

            QListWidgetItem *newItem = new QListWidgetItem;
            
            QString entry = club->getName() + ", " + course->getName();

            newItem->setText(entry);
            listWidgetClub->insertItem(index, newItem);

            if (!markedFlag & club->isHomeClub()) {
                listWidgetClub->setCurrentRow(index);
                // Mark the 1st course of the home club the selection
                markedFlag = true;
            }
            index++;
        }
    }
}

void SelectDialog::results(QString &club, 
			   QString &course, 
			   QString &date)
{  
    QListWidgetItem *current = listWidgetClub->currentItem();

    if (current) {
        QString tmp = current->text();

        QStringList stringList = tmp.split(", ");
        club = stringList[0];
        course = stringList[1];
#ifdef Q_WS_MAEMO_5
        QMaemo5DatePickSelector *sel = (QMaemo5DatePickSelector *)dateButton->pickSelector();
        QDate d = sel->currentDate();
        // TODO: change to QDate
        date = d.toString(Qt::ISODate);
#else
        date = lineEditDate->text();
#endif
    }
}

bool SelectDialog::validate(void)
{
    return true;
}

void SelectDialog::next(void)
{
    if (validate())
        done(1);
    else {
        qDebug() << "SelectDialog: invalid data, cancel or correct";
    }
}

void SelectDialog::reject(void)
{
    done(0);
}

////////////////////////////////////////////////////////////////////////////////
// ScoreDialog based on QDialog
////////////////////////////////////////////////////////////////////////////////
ScoreDialog::ScoreDialog(QWidget *parent) : QDialog(parent)
{
    TRACE;
    resize(800, 400);
    QFont font;
    font.setPointSize(fontSize);
    setFont(font);

    QWidget *centralWidget = new QWidget(this);

    createTable();
    createButton();
    
    createLayout(centralWidget);
}

ScoreDialog::~ScoreDialog()
{
    //if (centralWidget)
    //  delete centralWidget;
    if (leftLayout)
        delete leftLayout;
    if (rightLayout)
        delete rightLayout;
    //if (mainLayout)
    //  delete mainLayout;
    if (table)
        delete table;
}

void ScoreDialog::createLayout(QWidget *parent)
{
    TRACE;
    leftLayout = new QVBoxLayout;
    leftLayout->addWidget(table);

    QDialogButtonBox * buttonBoxUp = new QDialogButtonBox(Qt::Vertical);
    buttonBoxUp->addButton(pushButtonUp, QDialogButtonBox::ActionRole);
    buttonBoxUp->addButton(pushButtonDown, QDialogButtonBox::ActionRole);
    buttonBoxUp->addButton(pushButtonNext, QDialogButtonBox::ActionRole);

    QDialogButtonBox * buttonBoxDown = new QDialogButtonBox(Qt::Vertical);
    buttonBoxDown->addButton(pushButtonFinish, QDialogButtonBox::ActionRole);

    rightLayout = new QVBoxLayout;
    rightLayout->addWidget(buttonBoxUp);
    rightLayout->addStretch();
    rightLayout->addWidget(buttonBoxDown);

    QHBoxLayout *mainLayout = new QHBoxLayout(parent);
    mainLayout->addLayout(leftLayout);
    mainLayout->addLayout(rightLayout);
    setLayout(mainLayout);
}

void ScoreDialog::createTable(QWidget *parent)
{
    TRACE;
    table = new QTableWidget(ROWS, COLS, parent);

    table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
    table->horizontalHeader()->hide();

    table->setStyleSheet(defaultStyleSheet);

    QStringList headers;
    headers << "" << "Par" << "HCP" << "Score" << "" << "Par" << "HCP" << "Score";
    table->setVerticalHeaderLabels(headers);
}

void ScoreDialog::createButton(QWidget *parent)
{
    TRACE;
    Q_UNUSED(parent);
    pushButtonUp = new QPushButton(tr("+"));
    connect(pushButtonUp, SIGNAL(clicked()), this, SLOT(up()));
    
    pushButtonDown = new QPushButton(tr("-"));
    connect(pushButtonDown, SIGNAL(clicked()), this, SLOT(down()));
  
    pushButtonNext = new QPushButton(tr("Next"));
    connect(pushButtonNext, SIGNAL(clicked()), this, SLOT(next()));

    pushButtonFinish = new QPushButton(tr("Finish"));
    connect(pushButtonFinish, SIGNAL(clicked()), this, SLOT(finish()));
}

void ScoreDialog::init(Course *course, Score *score)
{
    TRACE;
    QTableWidgetItem *par, *hcp, *scoreItem, *holeNum;

    for (int i = 0; i < 18; i++) {
        par = new QTableWidgetItem(course->getPar(i));
        hcp = new QTableWidgetItem(course->getHcp(i));
        if (score)
            scoreItem = new QTableWidgetItem(score->getScore(i));
        else
            scoreItem = new QTableWidgetItem("");
        holeNum = new QTableWidgetItem(QString::number(i+1));

        holeNum->setTextAlignment(Qt::AlignCenter);
        holeNum->setFlags(Qt::NoItemFlags);
        holeNum->setForeground(ScoreColor::holeBg());
        
        par->setTextAlignment(Qt::AlignCenter);
        par->setFlags(Qt::NoItemFlags);

        hcp->setTextAlignment(Qt::AlignCenter);
        hcp->setFlags(Qt::NoItemFlags);
        
        scoreItem->setTextAlignment(Qt::AlignCenter);

        if (i < 9) {
            table->setItem(ROW_HOLE, i, holeNum);
            table->setItem(ROW_PAR, i, par);
            table->setItem(ROW_HCP, i, hcp);
            table->setItem(ROW_SCORE, i, scoreItem);
        }
        else {
            table->setItem(ROW_HOLE_2, i-9, holeNum);
            table->setItem(ROW_PAR_2, i-9, par);
            table->setItem(ROW_HCP_2, i-9, hcp);
            table->setItem(ROW_SCORE_2, i-9, scoreItem);
        }
    }

    // Set focus to 1st cell
    table->setCurrentCell(ROW_SCORE, 0);
    if (!score)
        setDefaultScore(table);
}

// Set default score to par if not set
void ScoreDialog::setDefaultScore(QTableWidget *table)
{
    int row = table->currentRow();
    int col = table->currentColumn();
  
    if (row == ROW_SCORE)
        row = ROW_PAR;
    else if (row == ROW_SCORE_2)
        row = ROW_PAR_2;
    else {
        qDebug() << "ERROR: unknown row in default score";
        return;
    }
    QTableWidgetItem *par = table->item(row, col);
    QTableWidgetItem *score = table->item(row + 2, col);

    if (par && score && score->text() == "") {
        QVariant value(par->text());
        score->setData(Qt::DisplayRole, value);
    }
}

void ScoreDialog::up(void)
{
    QTableWidgetItem *item = table->currentItem();

    if (!item) {
        qWarning() << "ERROR: no current item";
        return;
    }

    int i = (item->text()).toInt();
    QVariant value(i+1);
    item->setData(Qt::DisplayRole, value);
}

void ScoreDialog::down(void)
{
    QTableWidgetItem *item = table->currentItem();

    if (!item)
        return;

    int i = (item->text()).toInt();
    QVariant value(i-1);
    item->setData(Qt::DisplayRole, value);
}

void ScoreDialog::next(void)
{
    if (table) {
        QTableWidgetItem *item = table->currentItem();
        moveToNextCell(item);
        setDefaultScore(table);
    }
}

void ScoreDialog::moveToNextCell(QTableWidgetItem *item)
{
    if (!item)
        return;

    QTableWidget *table = item->tableWidget();

    if (!table)
        return;

    int row = table->currentRow();
    int col = table->currentColumn();

    if (col < (COLS-1)) {
        col++;
    }
    else if (col == (COLS-1)) {
        col = 0;
        row = (row == ROW_SCORE_2) ? ROW_SCORE : ROW_SCORE_2;
    }
    table->setCurrentCell(row, col);
}

void ScoreDialog::results(QVector<QString> &scores)
{
    TRACE;
    for (int i = 0; i < 9; i++) {
        QTableWidgetItem *frontItem = table->item(ROW_SCORE, i);
        QTableWidgetItem *backItem = table->item(ROW_SCORE_2, i);

        if (frontItem)
            scores[i] = frontItem->text();

        if (backItem)
            scores[i+9] = backItem->text();
    }
}

bool ScoreDialog::validate(void)
{
    for (int i=0; i<9; i++) {
        QTableWidgetItem *frontItem = table->item(ROW_SCORE, i);
        QTableWidgetItem *backItem = table->item(ROW_SCORE_2, i);
    
        if (!frontItem || !backItem)
            return false;
    
        QString str1 = frontItem->text();
        QString str2 = backItem->text();
    
        if (str1.isEmpty() || str2.isEmpty())
            return false;
    }
    return true;
}

void ScoreDialog::finish(void)
{
    if (validate())
        done(1);
    else {
        showNote("Invalid data - cancel or correct");
    }
}

void ScoreDialog::reject(void)
{
    done(0);
}

void ScoreDialog::showNote(QString msg)
{
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, 
                                       msg,
                                       QMaemo5InformationBox::DefaultTimeout);
#endif
}
