/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QtGui>
#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#endif

#include "score-common.h"
#include "main-window.h"
#include "score-dialog.h"
#include "course-dialog.h"
#include "settings-dialog.h"
#include "stat-model.h"
#include "xml-dom-parser.h"

QString appName("scorecard");
QString topDir("/opt");
QString mmcDir("/media/mmc1");
QString dataDirName("data");
QString dataDir;
QString imgDir(topDir + "/pixmaps");
QString scoreFileName("score.xml");
QString scoreFile;
QString clubFileName("club.xml");
QString clubFile;
QString masterFileName("club-master.xml");
QString masterFile;
QString logFile("/tmp/scorecard.log");
QString titleScores("ScoreCard - Scores");
QString titleCourses("ScoreCard - Courses");

bool dateLessThan(const Score *s1, const Score *s2)
{
    return (*s1) < (*s2);
}

bool dateMoreThan(const Score *s1, const Score *s2)
{
    return (*s1) > (*s2);
}

bool scoreMoreThan(const Score *s1, const Score *s2)
{
    return s1->getTotal(Total) > s2->getTotal(Total);
}

// Find score based on club and course name
Score *MainWindow::findScore(QString & clubName, QString & courseName)
{
    TRACE;
    QListIterator<Score *> i(scoreList);
    Score * s;

    while (i.hasNext()) {
        s = i.next();
        if ((s->getClubName() == clubName) &&
            (s->getCourseName() == courseName))
            return s;
    }
    return 0;
}

// Find club based on name
Club *MainWindow::findClub(QString &name)
{
    TRACE;
    QListIterator<Club *> i(clubList);
    Club *c;

    while (i.hasNext()) {
        c = i.next();
        if (c->getName() == name)
            return c;
    }
    return 0;
}

// Find course based on club & course name
Course *MainWindow::findCourse(const QString &clubName, 
                               const QString &courseName)
{
    TRACE;
    QListIterator<Club *> i(clubList);
    Club *c;

    while (i.hasNext()) {
        c = i.next();
        if (c->getName() == clubName) {
            return c->getCourse(courseName);
        }
    }
    return 0;
}

// Find course based on current selection on the list
// TODO: make sure this is only called when course list is the model...
Course *MainWindow::currentCourse()
{
    TRACE;
    QModelIndex index = selectionModel->currentIndex();

    if (!index.isValid())
        return 0;

    const QAbstractItemModel *model = selectionModel->model();
    QString str = model->data(index, Qt::DisplayRole).toString();

    QStringList strList = str.split(", ");
    if (strList.count() != 2) {
        showNote(tr("Invalid course selection"));
        return 0;
    }
    return findCourse(strList[0], strList[1]);
}

// Find score based on current selection on the list
// TODO: make sure this is only called when score list is the model...
Score *MainWindow::currentScore()
{
    TRACE;
    QModelIndex index = selectionModel->currentIndex();

    if (!index.isValid())
        return 0;

    return scoreList.at(index.row());
}

void MainWindow::flushReadOnlyItems()
{
    TRACE;
    QMutableListIterator<Club *> i(clubList);
    Club *c;

    while (i.hasNext()) {
        c = i.next();
        if (c->isReadOnly()) {
            qDebug() << "Del:" << c->getName();
            i.remove();
        }
    }
}

void MainWindow::markHomeClub()
{
    TRACE;
    QListIterator<Club *> i(clubList);
    Club *c;

    while (i.hasNext()) {
        c = i.next();
        if (c->getName() == conf.homeClub)
            c->setHomeClub(true);
        else
            c->setHomeClub(false);
    }
}

void MainWindow::sortScoreList()
{
    if (conf.sortOrder == "Date")
        qSort(scoreList.begin(), scoreList.end(), dateMoreThan); 
    else if (conf.sortOrder == "Score")
        qSort(scoreList.begin(), scoreList.end(), scoreMoreThan); 
}

MainWindow::MainWindow(QMainWindow *parent): QMainWindow(parent)
{
    resize(800, 480);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    loadSettings();

    centralWidget = new QWidget(this);

    setCentralWidget(centralWidget);

    loadScoreFile(scoreFile, scoreList);
    if (conf.defaultCourses == "Yes")
        loadClubFile(masterFile, clubList, true);
    loadClubFile(clubFile, clubList);
    markHomeClub();

    // Sort the scores based on settings
    sortScoreList();

    createActions();
    createMenus();

    createListView(scoreList, clubList);

    createLayoutList(centralWidget);

    scoreWindow = new ScoreWindow(this);
    courseWindow = new CourseWindow(this);
}

void MainWindow::loadSettings(void)
{
    TRACE;
    bool external = false;

    QDir mmc(mmcDir);
    if (mmc.exists())
        external = true;

    // TODO: make via user option, automatic will never work
    external = false;

#ifndef Q_WS_MAEMO_5
    dataDir = "./" + dataDirName;
#else
    if (external) {
        dataDir = mmcDir + "/" + appName + "/" + dataDirName;
    }
    else {
        dataDir = topDir + "/" + appName + "/" + dataDirName;
    }
#endif
    scoreFile = dataDir + "/" + scoreFileName;
    clubFile = dataDir + "/" + clubFileName;
    masterFile = dataDir + "/" + masterFileName;

    QDir dir(dataDir);
    if (!dir.exists())
        if (!dir.mkpath(dataDir)) {
            qWarning() << "Unable to create: " + dataDir;
            return;
        }
    qDebug() << "Data is at:" + dataDir;

    settings.beginGroup(settingsGroup);
    conf.hcp = settings.value(settingsHcp);
    conf.homeClub = settings.value(settingsHomeClub);
    conf.sortOrder = settings.value(settingsSortOrder);
    conf.defaultCourses = settings.value(settingsDefaultCourses);
    settings.endGroup();

    // Use default courses if no settings for that
    if (!conf.defaultCourses.isValid())
        conf.defaultCourses = "Yes";

    // Use date sort order if no settings for that
    if (!conf.sortOrder.isValid())
        conf.sortOrder = "Yes";

    qDebug() << "Settings: " << conf.hcp << conf.homeClub << conf.sortOrder << conf.defaultCourses;
}

void MainWindow::saveSettings(void)
{
    TRACE;
    settings.beginGroup(settingsGroup);
    if (conf.hcp.isValid())
        settings.setValue(settingsHcp, conf.hcp);
    if (conf.homeClub.isValid())
        settings.setValue(settingsHomeClub, conf.homeClub);
    if (conf.sortOrder.isValid())
        settings.setValue(settingsSortOrder, conf.sortOrder);
    if (conf.defaultCourses.isValid())
        settings.setValue(settingsDefaultCourses, conf.defaultCourses);
    settings.endGroup();
}

void MainWindow::createLayoutList(QWidget *parent)
{
    TRACE;
    QVBoxLayout * tableLayout = new QVBoxLayout;
    tableLayout->addWidget(list);

    QHBoxLayout *mainLayout = new QHBoxLayout(parent);
    mainLayout->addLayout(tableLayout);
    parent->setLayout(mainLayout);
}

void MainWindow::createListView(QList<Score *> &scoreList, 
                                QList <Club *> &clubList)
{
    TRACE;
    list = new QListView(this);

    scoreListModel = new ScoreListModel(scoreList, clubList);
    courseListModel = new CourseListModel(clubList);

    list->setStyleSheet(defaultStyleSheet);

    list->setSelectionMode(QAbstractItemView::SingleSelection);
    list->setProperty("FingerScrolling", true);

    // Initial view
    listScores();

    connect(list, SIGNAL(clicked(QModelIndex)),
            this, SLOT(clickedList(QModelIndex)));
}

void MainWindow::listScores()
{
    TRACE;
    list->setModel(scoreListModel);
    selectionModel = list->selectionModel();
    updateTitleBar(titleScores);
}

void MainWindow::listCourses()
{
    TRACE;
    list->setModel(courseListModel);
    selectionModel = list->selectionModel();
    updateTitleBar(titleCourses);
}

void MainWindow::createActions()
{
    TRACE;
    newScoreAction = new QAction(tr("New Score"), this);
    connect(newScoreAction, SIGNAL(triggered()), this, SLOT(newScore()));

    newCourseAction = new QAction(tr("New Course"), this);
    connect(newCourseAction, SIGNAL(triggered()), this, SLOT(newCourse()));

    statAction = new QAction(tr("Statistics"), this);
    connect(statAction, SIGNAL(triggered()), this, SLOT(viewStatistics()));

    settingsAction = new QAction(tr("Settings"), this);
    connect(settingsAction, SIGNAL(triggered()), this, SLOT(viewSettings()));

    // Maemo5 style menu filters
    filterGroup = new QActionGroup(this);
    filterGroup->setExclusive(true);

    listScoreAction = new QAction(tr("Scores"), filterGroup);
    listScoreAction->setCheckable(true);
    listScoreAction->setChecked(true);
    connect(listScoreAction, SIGNAL(triggered()), this, SLOT(listScores()));

    listCourseAction = new QAction(tr("Courses"), filterGroup);
    listCourseAction->setCheckable(true);
    connect(listCourseAction, SIGNAL(triggered()), this, SLOT(listCourses()));
}

void MainWindow::createMenus()
{
    TRACE;
#ifdef Q_WS_MAEMO_5
    menu = menuBar()->addMenu("");
#else
    menu = menuBar()->addMenu("Menu");
#endif

    menu->addAction(newScoreAction);
    menu->addAction(newCourseAction);
    menu->addAction(statAction);
    menu->addAction(settingsAction);
    menu->addActions(filterGroup->actions());
}

void MainWindow::updateTitleBar(QString & msg)
{
    TRACE;
    setWindowTitle(msg);
}

void MainWindow::showNote(QString msg)
{
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, 
                                       msg,
                                       QMaemo5InformationBox::DefaultTimeout);
#endif
}

void MainWindow::clickedList(const QModelIndex &index)
{
    TRACE;
    int row = index.row();

    const QAbstractItemModel *m = index.model();
    if (m == scoreListModel) {
        if (row < scoreList.count()) {
            Score * score = scoreList.at(row);
            Course * course = findCourse(score->getClubName(), score->getCourseName());
            viewScore(score, course);
        }
    }
    else if (m == courseListModel) {
        QString str = courseListModel->data(index, Qt::DisplayRole).toString();
        QStringList strList = str.split(", ");

        if (strList.count() != 2) {
            showNote(QString("Invalid course selection"));
            return;
        }
        Course * course = findCourse(strList.at(0), strList.at(1));
        viewCourse(course);
    }
}

void MainWindow::viewScore(Score * score, Course * course)
{
    TRACE;
    scoreWindow->setup(score, course);
    scoreWindow->show();
}

void MainWindow::newScore()
{
    TRACE;
    SelectDialog *selectDialog = new SelectDialog(this);

    selectDialog->init(clubList);

    int result = selectDialog->exec();
    if (result) {
        QString clubName;
        QString courseName;
        QString date;

        selectDialog->results(clubName, courseName, date);

        ScoreDialog *scoreDialog = new ScoreDialog(this);
        QString title = "New Score: " + courseName + ", " + date;
        scoreDialog->setWindowTitle(title);

        Club *club = findClub(clubName);
        if (!club) {
            showNote(tr("Error: no such club"));
            return;
        }
        Course *course = club->getCourse(courseName);
        if (!course) {
            showNote(tr("Error: no such course:"));
            return;
        }
        scoreDialog->init(course);
        result = scoreDialog->exec();
        if (result) {
            QVector<QString> scores(18);

            scoreDialog->results(scores);
            Score *score = new Score(scores, clubName, courseName, date);
            scoreList << score;

            // Sort the scores based on settings
            sortScoreList();
            // Save it
            saveScoreFile(scoreFile, scoreList);
            scoreListModel->update(scoreList);
            list->update();
        }
    }
}

void MainWindow::editScore()
{
    TRACE;
    Course * course = 0;
    Score *score = currentScore();

    if (score) 
        course = findCourse(score->getClubName(), score->getCourseName());

    if (!course || !score) {
        qDebug() << "No score/course to edit";
        return;
    }

    QString date = score->getDate();

    ScoreDialog *scoreDialog = new ScoreDialog(this);
    scoreDialog->init(course, score);
  
    QString title = "Edit Score: " + course->getName() + ", " + date;
    scoreDialog->setWindowTitle(title);

    int result = scoreDialog->exec();
    if (result) {
        QVector<QString> scores(18);

        scoreDialog->results(scores);
    
        score->update(scores);

        // Sort the scores based on dates
        qSort(scoreList.begin(), scoreList.end(), dateMoreThan); 
        // Save it
        saveScoreFile(scoreFile, scoreList);
    }
    if (scoreDialog)
        delete scoreDialog;
}

void MainWindow::deleteScore()
{
    TRACE;
    if (scoreWindow)
        scoreWindow->close();

    QModelIndex index = selectionModel->currentIndex();
    if (!index.isValid()) {
        qDebug() << "Invalid index";
        return;
    }
    
    scoreList.removeAt(index.row());
    // Save it
    saveScoreFile(scoreFile, scoreList);
    scoreListModel->update(scoreList);
    list->update();
}

void MainWindow::viewCourse(Course * course)
{
    TRACE;
    courseWindow->setup(course);
    courseWindow->show();
}

void MainWindow::newCourse()
{
    TRACE;
    CourseSelectDialog *selectDialog = new CourseSelectDialog(this);

    int result = selectDialog->exec();
    if (result) {
        QString clubName;
        QString courseName;
        QString date;

        selectDialog->results(clubName, courseName);

        CourseDialog *courseDialog = new CourseDialog(this);
        courseDialog->init();
        QString title = "New Course: " + clubName + "," + courseName;
        courseDialog->setWindowTitle(title);

        int result = courseDialog->exec();
        if (result) {
            QVector<QString> par(18);
            QVector<QString> hcp(18);
            QVector<QString> len(18);

            courseDialog->results(par, hcp, len);

            Course *course = 0;
            Club *club = findClub(clubName);
            if (club) {
                course = club->getCourse(courseName);
                if (course) {
                    showNote(tr("Club/course already in the database"));
                    return;
                }
                else {
                    course = new Course(courseName, par, hcp);
                    club->addCourse(course);
                }
            }
            else {
                // New club and course
                club = new Club(clubName);
                course = new Course(courseName, par, hcp);
                club->addCourse(course);
                clubList << club;
            }
            // Save it
            saveClubFile(clubFile, clubList);
            courseListModel->update(clubList);
            list->update();
        }
    }
}

void MainWindow::editCourse()
{
    TRACE;
    Course *course = currentCourse();

    if (!course) {
        showNote(tr("No course on edit"));
        return;
    }

    CourseDialog *courseDialog = new CourseDialog(this);
    courseDialog->init(course);

    QString title = "Edit Course: " + course->getName();
    courseDialog->setWindowTitle(title);
  
    int result = courseDialog->exec();
    if (result) {
        QVector<QString> par(18);
        QVector<QString> hcp(18);
        QVector<QString> len(18);
    
        courseDialog->results(par, hcp, len);
    
        course->update(par, hcp, len);
        saveClubFile(clubFile, clubList);
    }
    if (courseDialog)
        delete courseDialog;
}

void MainWindow::deleteCourse()
{
    TRACE;
    Club *club = 0;
    Course * course = currentCourse();

    if (!course) {
        qDebug() << "Invalid course for deletion";
        return;
    }
    club = course->parent();

    // Can not delete course if it has scores -- check
    if (findScore(club->getName(), course->getName()) != 0) {
        showNote(tr("Can not delete course, delete scores on the course first"));
        return;
    }
    // Close the window
    if (courseWindow)
        courseWindow->close();

    club->delCourse(course);

    if (club->isEmpty()) {
        int index = clubList.indexOf(club);
        if (index != -1)
            clubList.removeAt(index);
    }

    // Save it
    saveClubFile(clubFile, clubList);
    courseListModel->update(clubList);
    list->update();
}

void MainWindow::viewStatistics()
{
    TRACE;
    QMainWindow *win = new QMainWindow(this);
    QString title = "Statistics";
    win->setWindowTitle(title);
#ifdef Q_WS_MAEMO_5
    win->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    StatModel *model = new StatModel(clubList, scoreList);

    QTableView *table = new QTableView;
    table->showGrid();
    table->setSelectionMode(QAbstractItemView::NoSelection);
    table->setStyleSheet(statStyleSheet);
    table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
    table->verticalHeader()->setAutoFillBackground(true);
    table->setModel(model);

    QWidget *central = new QWidget(win);
    win->setCentralWidget(central);

    QTextEdit *textEdit = new QTextEdit;

    textEdit->setReadOnly(true);

    QVBoxLayout *infoLayout = new QVBoxLayout;
    infoLayout->addWidget(table);

    QHBoxLayout *mainLayout = new QHBoxLayout(central);
    mainLayout->addLayout(infoLayout);
    central->setLayout(mainLayout);

    win->show();
}

void MainWindow::viewSettings()
{
    TRACE;
    SettingsDialog *dlg = new SettingsDialog(this);

    dlg->init(conf, clubList);

    int result = dlg->exec();
    if (result) {
        QString oldValue = conf.defaultCourses.toString();
        dlg->results(conf);
        QString newValue = conf.defaultCourses.toString();
        saveSettings();

        // Reload club list, or drop r/o courses from list
        if (oldValue == "Yes" && newValue == "No") {
            flushReadOnlyItems();
            courseListModel->update(clubList);
            list->update();
        }
        else if ((oldValue == "No" || oldValue == "") && newValue == "Yes") {
            loadClubFile(masterFile, clubList, true);
            courseListModel->update(clubList);
            list->update();
        }
        // TODO: do these only if needed
        markHomeClub();
        sortScoreList();
        scoreListModel->update(scoreList);
        list->update();
    }
}

void MainWindow::loadScoreFile(QString &fileName, QList<Score *> &list)
{
    ScoreXmlHandler handler(list);

    if (handler.parse(fileName))
        qDebug() << "File loaded:" << fileName << " entries:" << list.size();
}

void MainWindow::saveScoreFile(QString &fileName, QList<Score *> &list)
{
    ScoreXmlHandler handler(list);

    if (handler.save(fileName))
        // TODO: banner
        qDebug() << "File saved:" << fileName << " entries:" << list.size();
    else
        qWarning() << "Unable to save:" << fileName;
}

void MainWindow::loadClubFile(QString &fileName, QList<Club *> &list, bool readOnly)
{
    ClubXmlHandler handler(list);

    if (handler.parse(fileName, readOnly))
        qDebug() << "File loaded:" << fileName << " entries:" << list.size();
}

void MainWindow::saveClubFile(QString &fileName, QList<Club *> &list)
{
    ClubXmlHandler handler(list);

    if (handler.save(fileName))
        // TODO: banner
        qDebug() << "File saved:" << fileName << " entries:" << list.size();
    else
        qWarning() << "Unable to save:" << fileName;
}
