/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QtGui>
#include <QLayout>
#include <QLineEdit>
#include <QInputContext>
#include <QtGui/QTableWidget>

#include "course-dialog.h"
#include "score-common.h"
#include "table-model.h"

////////////////////////////////////////////////////////////////////////////////
// CourseWindow based on QMainWindow
////////////////////////////////////////////////////////////////////////////////
CourseWindow::CourseWindow(QWidget *parent) : QMainWindow(parent)
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    QAction *editAction = new QAction(tr("Edit"), this);
    connect(editAction, SIGNAL(triggered()), parent, SLOT(editCourse()));
    menuBar()->addAction(editAction);

    QAction *delAction = new QAction(tr("Delete"), this);
    connect(delAction, SIGNAL(triggered()), parent, SLOT(deleteCourse()));
    menuBar()->addAction(delAction);

    model = new CourseTableModel;
    
    QTableView * table = new QTableView;
    table->showGrid();
    table->setSelectionMode(QAbstractItemView::NoSelection);
    table->setStyleSheet(ScoreStyle::style());
    table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
    table->horizontalHeader()->hide();
    table->setModel(model);
    
    QWidget *central = new QWidget(this);
    setCentralWidget(central);
    
    QVBoxLayout *layout = new QVBoxLayout;
    layout->addWidget(table);
    
    central->setLayout(layout);
}

void CourseWindow::setup(Course *course)
{
    QString title = QString("Course: %1, Par - %2").arg(course->getName()).arg(course->getTotal(Total));
    setWindowTitle(title);

    model->set(course);
}

////////////////////////////////////////////////////////////////////////////////
// CourseSelectDialog based on QDialog
////////////////////////////////////////////////////////////////////////////////
CourseSelectDialog::CourseSelectDialog(QWidget *parent) : QDialog(parent)
{
  QWidget *centralWidget = new QWidget(this);
  createLayout(centralWidget);

  setWindowTitle(tr("ScoreCard: New Club and Course"));
}

void CourseSelectDialog::createLayout(QWidget *parent)
{
  labelClub = new QLabel(tr("Club"));
  labelCourse = new QLabel(tr("Course"));
  lineEditClub = new QLineEdit;
  lineEditCourse = new QLineEdit;
  pushButtonNext = new QPushButton(tr("Next"));

  connect(pushButtonNext, SIGNAL(clicked()), this, SLOT(next()));

  QDialogButtonBox * buttonBox = new QDialogButtonBox(Qt::Vertical);
  buttonBox->addButton(pushButtonNext, QDialogButtonBox::ActionRole);

  leftLayout = new QVBoxLayout;
  leftLayout->addWidget(lineEditClub);
  leftLayout->addWidget(lineEditCourse);

  rightLayout = new QVBoxLayout;
  rightLayout->addStretch();
  rightLayout->addWidget(buttonBox);

  QHBoxLayout *mainLayout = new QHBoxLayout(parent);
  mainLayout->addLayout(leftLayout);
  mainLayout->addLayout(rightLayout);

  setLayout(mainLayout);
}

void CourseSelectDialog::results(QString &club, 
				 QString &course)
{  
  club = lineEditClub->text();
  course = lineEditCourse->text();
}

bool CourseSelectDialog::validate(void)
{
  QString str1 = lineEditClub->text();
  QString str2 = lineEditCourse->text();

  if (str1.isEmpty() || str2.isEmpty())
    return false;
  
  return true;
}

void CourseSelectDialog::next(void)
{
  if (validate())
    done(1);
  else {
    qDebug() << "CourseDialog: invalid data, cancel or correct";
  }
}

////////////////////////////////////////////////////////////////////////////////
// CourseDialog based on QDialog
////////////////////////////////////////////////////////////////////////////////
CourseDialog::CourseDialog(QWidget *parent) : QDialog(parent)
{
  resize(800, 400);

  QWidget *centralWidget = new QWidget(this);

  createTable();
  createButton();

  createLayout(centralWidget);
}

void CourseDialog::createLayout(QWidget *parent)
{
  leftLayout = new QVBoxLayout;
  leftLayout->addWidget(table);

  QDialogButtonBox * buttonBoxUp = new QDialogButtonBox(Qt::Vertical);
  buttonBoxUp->addButton(pushButtonUp, QDialogButtonBox::ActionRole);
  buttonBoxUp->addButton(pushButtonDown, QDialogButtonBox::ActionRole);
  buttonBoxUp->addButton(pushButtonNext, QDialogButtonBox::ActionRole);

  QDialogButtonBox * buttonBoxDown = new QDialogButtonBox(Qt::Vertical);
  buttonBoxDown->addButton(pushButtonFinish, QDialogButtonBox::ActionRole);

  rightLayout = new QVBoxLayout;
  rightLayout->addWidget(buttonBoxUp);
  rightLayout->addStretch();
  rightLayout->addWidget(buttonBoxDown);

  QHBoxLayout *mainLayout = new QHBoxLayout(parent);
  mainLayout->addLayout(leftLayout);
  mainLayout->addLayout(rightLayout);
  setLayout(mainLayout);
}

void CourseDialog::createButton(QWidget *parent)
{
  Q_UNUSED(parent);
  pushButtonUp = new QPushButton(tr("+"));
  connect(pushButtonUp, SIGNAL(clicked()), this, SLOT(up()));

  pushButtonDown = new QPushButton(tr("-"));
  connect(pushButtonDown, SIGNAL(clicked()), this, SLOT(down()));

  pushButtonNext = new QPushButton(tr("Next"));
  connect(pushButtonNext, SIGNAL(clicked()), this, SLOT(next()));

  pushButtonFinish = new QPushButton(tr("Finish"));
  connect(pushButtonFinish, SIGNAL(clicked()), this, SLOT(finish()));
}

void CourseDialog::createTable(QWidget *parent)
{
  table = new QTableWidget(ROWS, COLS, parent);

  QStringList headers;
  headers << "" << "Par" << "HCP" << "Len" << "" << "Par" << "HCP" << "Len";
  table->setVerticalHeaderLabels(headers);
  table->horizontalHeader()->hide();
  table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
  table->setStyleSheet(ScoreStyle::style());
}

void CourseDialog::init(Course *course)
{
  QTableWidgetItem *par, *hcp;

  for (int i=0; i<18; i++) {
    if (course) {
      par = new QTableWidgetItem(course->getPar(i));
      hcp = new QTableWidgetItem(course->getHcp(i));
    }
    else {
      par = new QTableWidgetItem("4");
      hcp = new QTableWidgetItem("9");
    }
    QTableWidgetItem *len = new QTableWidgetItem("");

    QTableWidgetItem *holeNum = new QTableWidgetItem(QString::number(i+1));
    holeNum->setForeground(ScoreColor::holeBg());
    holeNum->setFlags(Qt::NoItemFlags);

    holeNum->setTextAlignment(Qt::AlignCenter);
    par->setTextAlignment(Qt::AlignCenter);
    hcp->setTextAlignment(Qt::AlignCenter);
    len->setTextAlignment(Qt::AlignCenter);
    // len is not in use - here just to confuse ;(
    len->setFlags(Qt::NoItemFlags);
    
    if (i < 9) {
      table->setItem(0, i, holeNum);
      table->setItem(1, i, par);
      table->setItem(2, i, hcp);
      table->setItem(3, i, len);
    }
    else {
      table->setItem(4, i-9, holeNum);
      table->setItem(5, i-9, par);
      table->setItem(6, i-9, hcp);
      table->setItem(7, i-9, len);
    }
  }
  table->setCurrentCell(1, 0);
}

void CourseDialog::up(void)
{
  QTableWidgetItem *item = table->currentItem();

  if (!item)
    return;

  int i = (item->text()).toInt();
  QVariant value(i+1);
  item->setData(Qt::DisplayRole, value);
}

void CourseDialog::down(void)
{
  QTableWidgetItem *item = table->currentItem();

  if (!item)
    return;

  int i = (item->text()).toInt();
  QVariant value(i-1);
  item->setData(Qt::DisplayRole, value);
}

void CourseDialog::next(void)
{
  if (table) {
    QTableWidgetItem *item = table->currentItem();
    moveToNextCell(item);
    //setDefaultScore(table);
  }
}

void CourseDialog::moveToNextCell(QTableWidgetItem *item)
{
  if (!item)
    return;

  QTableWidget *table = item->tableWidget();

  if (!table)
    return;

  int row = table->currentRow();
  int col = table->currentColumn();

  if (col < (COLS-1)) {
    col++;
  }
  else if (col == (COLS-1)) {
    col = 0;
    row = (row == ROW_PAR_2) ? ROW_PAR : ROW_PAR_2;
  }
  //qDebug() << "new cell: " << row << "/" << col;
  table->setCurrentCell(row, col);
}

void CourseDialog::results(QVector<QString> &par,
			  QVector<QString> &hcp,
			  QVector<QString> &len)
{
  for (int i = 0; i < 9; i++) {
    QTableWidgetItem *frontPar = table->item(ROW_PAR, i);
    QTableWidgetItem *backPar = table->item(ROW_PAR_2, i);
    QTableWidgetItem *frontHcp = table->item(ROW_HCP, i);
    QTableWidgetItem *backHcp = table->item(ROW_HCP_2, i);
    QTableWidgetItem *frontLen = table->item(ROW_LEN, i);
    QTableWidgetItem *backLen = table->item(ROW_LEN_2, i);

    if (frontPar)
      par[i] = frontPar->text();
    if (backPar)
      par[i+9] = backPar->text();
    if (frontHcp)
      hcp[i] = frontHcp->text();
    if (backHcp)
      hcp[i+9] = backHcp->text();
    if (frontLen)
      len[i] = frontLen->text();
    if (backLen)
      len[i+9] = backLen->text();
  }
}

// Only par is mandatory
bool CourseDialog::validate(void)
{
  for (int i = 0; i < 9; i++) {
    QTableWidgetItem *frontItem = table->item(ROW_PAR, i);
    QTableWidgetItem *backItem = table->item(ROW_PAR_2, i);
    
    if (!frontItem || !backItem)
      return false;
    
    QString str1 = frontItem->text();
    QString str2 = backItem->text();
    
    if (str1.isEmpty() || str2.isEmpty())
      return false;
  }
  return true;
}

void CourseDialog::finish(void)
{
  if (validate())
    done(1);
  else {
    qDebug() << "CourseDialog: invalid data, cancel or correct";
  }
}


