/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QtGui>
#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#endif

#include "score-common.h"
#include "main-window.h"
#include "score-dialog.h"
#include "course-dialog.h"
#include "stat-model.h"
#include "xml-dom-parser.h"

QString appName("scorecard");
QString topDir("/opt");
QString mmcDir("/media/mmc1");
QString dataDirName("data");
QString dataDir;
QString imgDir(topDir + "/pixmaps");
QString scoreFileName("score.xml");
QString scoreFile;
QString clubFileName("club.xml");
QString clubFile;
QString logFile("/tmp/scorecard.log");
QString titleScores("ScoreCard - Scores");
QString titleCourses("ScoreCard - Courses");

bool dateLessThan(const Score *s1, const Score *s2)
{
  return (*s1) < (*s2);
}

bool dateMoreThan(const Score *s1, const Score *s2)
{
  return (*s1) > (*s2);
}
// Find score based on club and course name
Score *MainWindow::findScore(QString & clubName, QString & courseName)
{
    QListIterator<Score *> i(scoreList);
    Score * s;

    while (i.hasNext()) {
        s = i.next();
        if ((s->getClubName() == clubName) &&
            (s->getCourseName() == courseName))
            return s;
    }
    return 0;
}

// Find club based on name
Club *MainWindow::findClub(QString &name)
{
    QListIterator<Club *> i(clubList);
    Club *c;

    while (i.hasNext()) {
        c = i.next();
        if (c->getName() == name)
            return c;
    }
    return 0;
}

// Find course based on club & course name
Course *MainWindow::findCourse(const QString &clubName, 
                               const QString &courseName)
{
    QListIterator<Club *> i(clubList);
    Club *c;

    while (i.hasNext()) {
        c = i.next();
        if (c->getName() == clubName) {
            return c->getCourse(courseName);
        }
    }
    return 0;
}

// Find course based on current selection on the list
// TODO: make sure this is only called when course list is the model...
Course *MainWindow::findCourse()
{
    QModelIndex index = selectionModel->currentIndex();
    const QAbstractItemModel *model = selectionModel->model();
    QString str = model->data(index, Qt::DisplayRole).toString();

    QStringList strList = str.split(",");
    if (strList.count() != 2) {
        showNote(tr("Invalid course selection"));
        return 0;
    }
    return findCourse(strList[0], strList[1]);
}

MainWindow::MainWindow(QMainWindow *parent): QMainWindow(parent)
{
  resize(800, 480);

#ifdef Q_WS_MAEMO_5
  setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

  loadSettings();

  centralWidget = new QWidget(this);

  setCentralWidget(centralWidget);

  loadScoreFile(scoreFile, scoreList);
  loadClubFile(clubFile, clubList);

  // Sort the scores based on dates
  qSort(scoreList.begin(), scoreList.end(), dateMoreThan); 
  createActions();
  createMenus();

  createListView(scoreList, clubList);

  createLayoutList(centralWidget);
}

void MainWindow::loadSettings(void)
{
  bool external = false;

  QDir mmc(mmcDir);
  if (mmc.exists())
    external = true;

  // TODO: make via user option, automatic will never work
  external = false;

#ifndef Q_WS_MAEMO_5
  dataDir = "./" + dataDirName;
#else
  if (external) {
    dataDir = mmcDir + "/" + appName + "/" + dataDirName;
  }
  else {
    dataDir = topDir + "/" + appName + "/" + dataDirName;
  }
#endif
  scoreFile = dataDir + "/" + scoreFileName;
  clubFile = dataDir + "/" + clubFileName;

  QDir dir(dataDir);
  if (!dir.exists())
    if (!dir.mkpath(dataDir)) {
      qWarning() << "Unable to create: " + dataDir;
      return;
    }
  qDebug() << "Data is at:" + dataDir;
}

void MainWindow::createLayoutList(QWidget *parent)
{
    QVBoxLayout * tableLayout = new QVBoxLayout;
    tableLayout->addWidget(list);

    QHBoxLayout *mainLayout = new QHBoxLayout(parent);
    mainLayout->addLayout(tableLayout);
    parent->setLayout(mainLayout);
}

void MainWindow::createListView(QList<Score *> &scoreList, 
                                QList <Club *> &clubList)
{
    list = new QListView(this);

    scoreListModel = new ScoreListModel(scoreList, clubList);
    courseListModel = new CourseListModel(clubList);

    list->setStyleSheet(ScoreStyle::style());

    list->setSelectionMode(QAbstractItemView::SingleSelection);
    list->setProperty("FingerScrolling", true);

    // Initial view
    listScores();

    connect(list, SIGNAL(clicked(QModelIndex)),
            this, SLOT(clickedList(QModelIndex)));
}

void MainWindow::listScores()
{
    list->setModel(scoreListModel);
    selectionModel = list->selectionModel();
    updateTitleBar(titleScores);
}

void MainWindow::listCourses()
{
    list->setModel(courseListModel);
    selectionModel = list->selectionModel();
    updateTitleBar(titleCourses);
}

void MainWindow::createActions()
{
    newScoreAction = new QAction(tr("New Score"), this);
    connect(newScoreAction, SIGNAL(triggered()), this, SLOT(newScore()));

    newCourseAction = new QAction(tr("New Course"), this);
    connect(newCourseAction, SIGNAL(triggered()), this, SLOT(newCourse()));

    statAction = new QAction(tr("Statistics"), this);
    connect(statAction, SIGNAL(triggered()), this, SLOT(viewStatistics()));

    // Maemo5 style menu filters
    filterGroup = new QActionGroup(this);
    filterGroup->setExclusive(true);

    listScoreAction = new QAction(tr("Scores"), filterGroup);
    listScoreAction->setCheckable(true);
    listScoreAction->setChecked(true);
    connect(listScoreAction, SIGNAL(triggered()), this, SLOT(listScores()));

    listCourseAction = new QAction(tr("Courses"), filterGroup);
    listCourseAction->setCheckable(true);
    connect(listCourseAction, SIGNAL(triggered()), this, SLOT(listCourses()));
}

void MainWindow::createMenus()
{
#ifdef Q_WS_MAEMO_5
    menu = menuBar()->addMenu("");
#else
    menu = menuBar()->addMenu("Menu");
#endif

    menu->addAction(newScoreAction);
    menu->addAction(newCourseAction);
    menu->addAction(statAction);
    menu->addActions(filterGroup->actions());
}

void MainWindow::updateTitleBar(QString & msg)
{
    setWindowTitle(msg);
}

void MainWindow::showNote(QString msg)
{
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, 
                                       msg,
                                       QMaemo5InformationBox::DefaultTimeout);
#endif
}

void MainWindow::clickedList(const QModelIndex &index)
{
    int row = index.row();

    const QAbstractItemModel *m = index.model();
    if (m == scoreListModel) {
        if (row < scoreList.count()) {
            Score * score = scoreList.at(row);
            Course * course = findCourse(score->getClubName(), score->getCourseName());
            viewScore(score, course);
        }
    }
    else if (m == courseListModel) {
        QString str = courseListModel->data(index, Qt::DisplayRole).toString();
        QStringList strList = str.split(",");

        if (strList.count() != 2) {
            showNote(QString("Invalid course selection"));
            return;
        }
        Course * course = findCourse(strList.at(0), strList.at(1));
        viewCourse(course);
    }
}


void MainWindow::newCourse()
{
    CourseSelectDialog *selectDialog = new CourseSelectDialog(this);

    int result = selectDialog->exec();
    if (result) {
        QString clubName;
        QString courseName;
        QString date;

        selectDialog->results(clubName, courseName);

        CourseDialog *courseDialog = new CourseDialog(this);
        courseDialog->init();
        QString title = "New Course: " + clubName + "," + courseName;
        courseDialog->setWindowTitle(title);

        int result = courseDialog->exec();
        if (result) {
            QVector<QString> par(18);
            QVector<QString> hcp(18);
            QVector<QString> len(18);

            courseDialog->results(par, hcp, len);

            Course *course = 0;
            Club *club = findClub(clubName);
            if (club) {
                course = club->getCourse(courseName);
                if (course) {
                    qDebug() << "Error: club/course already in the database";
                    return;
                }
                else {
                    course = new Course(courseName, par, hcp);
                    club->addCourse(course);
                }
            }
            else {
                // New club and course
                club = new Club(clubName);
                course = new Course(courseName, par, hcp);
                club->addCourse(course);
                clubList << club;
            }
            // Save it
            saveClubFile(clubFile, clubList);
            courseListModel->update(clubList);
            list->update();
        }
    }
}

void MainWindow::deleteCourse()
{
    Course * course = findCourse();
    Club * club = course->parent();

    // Can not delete course if it has scores -- check
    if (findScore(club->getName(), course->getName()) != 0) {
        showNote(tr("Can not delete course, delete scores on the course first"));
        return;
    }
    // Close the window
    if (courseWin)
        courseWin->close();

    club->delCourse(course);

    if (club->isEmpty()) {
        int index = clubList.indexOf(club);
        if (index != -1)
            clubList.removeAt(index);
    }

    // Save it
    saveClubFile(clubFile, clubList);
    courseListModel->update(clubList);
    list->update();
}

void MainWindow::editCourse()
{
    Course *course = findCourse();

    if (!course) {
        showNote(tr("No course on edit"));
        return;
    }

    CourseDialog *courseDialog = new CourseDialog(this);
    courseDialog->init(course);

    QString title = "Edit Course: " + course->getName();
    courseDialog->setWindowTitle(title);
  
    int result = courseDialog->exec();
    if (result) {
        QVector<QString> par(18);
        QVector<QString> hcp(18);
        QVector<QString> len(18);
    
        courseDialog->results(par, hcp, len);
    
        course->update(par, hcp, len);
        saveClubFile(clubFile, clubList);
    }
}

void MainWindow::newScore()
{
    SelectDialog *selectDialog = new SelectDialog(this);

    selectDialog->init(clubList);

    int result = selectDialog->exec();
    if (result) {
        QString clubName;
        QString courseName;
        QString date;

        selectDialog->results(clubName, courseName, date);

        ScoreDialog *scoreDialog = new ScoreDialog(this);
        QString title = "New Score: " + courseName + ", " + date;
        scoreDialog->setWindowTitle(title);

        Club *club = findClub(clubName);
        if (!club) {
            showNote(tr("Error: no such club"));
            return;
        }
        Course *course = club->getCourse(courseName);
        if (!course) {
            showNote(tr("Error: no such course:"));
            return;
        }
        scoreDialog->init(course);
        result = scoreDialog->exec();
        if (result) {
            QVector<QString> scores(18);

            scoreDialog->results(scores);
            Score *score = new Score(scores, clubName, courseName, date);
            scoreList << score;

            // Sort the scores based on dates
            qSort(scoreList.begin(), scoreList.end(), dateMoreThan); 
            // Save it
            saveScoreFile(scoreFile, scoreList);
            scoreListModel->update(scoreList);
            list->update();
        }
    }
}

void MainWindow::deleteScore()
{
    if (scoreWin)
        scoreWin->close();

    QModelIndex index = selectionModel->currentIndex();
    scoreList.removeAt(index.row());
    // Save it
    saveScoreFile(scoreFile, scoreList);
    scoreListModel->update(scoreList);
    list->update();
}

void MainWindow::editScore()
{
    QModelIndex index = selectionModel->currentIndex();
    Score * score = scoreList.at(index.row());
    Course * course = 0;
    if (score) 
        course = findCourse(score->getClubName(), score->getCourseName());

    if (!course || !score) {
        qDebug() << "No score/course to edit";
        return;
    }

    QString date = score->getDate();

    ScoreDialog *scoreDialog = new ScoreDialog;
  
    QString title = "Edit Score: " + course->getName() + ", " + date;
    scoreDialog->setWindowTitle(title);

    scoreDialog->init(course, score);

    int result = scoreDialog->exec();

    if (result) {
        QVector<QString> scores(18);

        scoreDialog->results(scores);
    
        score->update(scores);

        // Sort the scores based on dates
        qSort(scoreList.begin(), scoreList.end(), dateMoreThan); 
        // Save it
        saveScoreFile(scoreFile, scoreList);
        // Update the model
        scoreListModel->update(scoreList);
    }
}

void MainWindow::viewScore(Score * score, Course * course)
{
    scoreWin = new QMainWindow(this);
    QString title = QString("Score: %1, %2 - %3").arg(score->getClubName()).arg(score->getCourseName()).arg(score->getDate());
    scoreWin->setWindowTitle(title);
#ifdef Q_WS_MAEMO_5
    scoreWin->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    QAction *editAction = new QAction(tr("Edit"), scoreWin);
    connect(editAction, SIGNAL(triggered()), this, SLOT(editScore()));
    scoreWin->menuBar()->addAction(editAction);

    QAction *delAction = new QAction(tr("Delete"), scoreWin);
    connect(delAction, SIGNAL(triggered()), this, SLOT(deleteScore()));
    scoreWin->menuBar()->addAction(delAction);

    ScoreTableModel *model = new ScoreTableModel(score, course);
    
    QTableView * table = new QTableView;
    table->showGrid();
    table->setSelectionMode(QAbstractItemView::NoSelection);
    //table->setStyleSheet(ScoreStyle::headerView());
    table->setStyleSheet(ScoreStyle::style());
    table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
    table->horizontalHeader()->hide();
    table->setModel(model);
    
    QWidget *central = new QWidget(scoreWin);
    scoreWin->setCentralWidget(central);
    
    QVBoxLayout *layout = new QVBoxLayout;
    layout->addWidget(table);
    
    central->setLayout(layout);
    scoreWin->show();
}

void MainWindow::viewCourse(Course * course)
{
    courseWin = new QMainWindow(this);
    QString title = QString("Course: %1, Par - %2").arg(course->getName()).arg(course->getTotal(Total));
    courseWin->setWindowTitle(title);
#ifdef Q_WS_MAEMO_5
    courseWin->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    QAction *editAction = new QAction(tr("Edit"), courseWin);
    connect(editAction, SIGNAL(triggered()), this, SLOT(editCourse()));
    courseWin->menuBar()->addAction(editAction);

    QAction *delAction = new QAction(tr("Delete"), courseWin);
    connect(delAction, SIGNAL(triggered()), this, SLOT(deleteCourse()));
    courseWin->menuBar()->addAction(delAction);

    CourseTableModel *model = new CourseTableModel(course);
    
    QTableView * table = new QTableView;
    table->showGrid();
    table->setSelectionMode(QAbstractItemView::NoSelection);
    //table->setStyleSheet(ScoreStyle::headerView());
    table->setStyleSheet(ScoreStyle::style());
    table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
    table->horizontalHeader()->hide();
    table->setModel(model);
    
    QWidget *central = new QWidget(courseWin);
    courseWin->setCentralWidget(central);
    
    QVBoxLayout *layout = new QVBoxLayout;
    layout->addWidget(table);
    
    central->setLayout(layout);
    courseWin->show();
}

void MainWindow::viewStatistics()
{
  QMainWindow *win = new QMainWindow(this);
  QString title = "Statistics";
  win->setWindowTitle(title);
#ifdef Q_WS_MAEMO_5
  win->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

  StatModel *model = new StatModel(clubList, scoreList);

  QTableView *table = new QTableView;
  table->showGrid();
  table->setSelectionMode(QAbstractItemView::NoSelection);
  table->setStyleSheet(ScoreStyle::headerView());
  table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setAutoFillBackground(true);
  table->setModel(model);

  QWidget *central = new QWidget(win);
  win->setCentralWidget(central);

  QTextEdit *textEdit = new QTextEdit;

  textEdit->setReadOnly(true);

  QVBoxLayout *infoLayout = new QVBoxLayout;
  infoLayout->addWidget(table);

  QHBoxLayout *mainLayout = new QHBoxLayout(central);
  mainLayout->addLayout(infoLayout);
  central->setLayout(mainLayout);

  win->show();
}

void MainWindow::loadScoreFile(QString &fileName, QList<Score *> &list)
{
  ScoreXmlHandler handler(list);

  if (handler.parse(fileName))
    qDebug() << "File loaded:" << fileName << " entries:" << list.size();
}

void MainWindow::saveScoreFile(QString &fileName, QList<Score *> &list)
{
  ScoreXmlHandler handler(list);

  if (handler.save(fileName))
    // TODO: banner
    qDebug() << "File saved:" << fileName << " entries:" << list.size();
  else
    qWarning() << "Unable to save:" << fileName;
}

void MainWindow::loadClubFile(QString &fileName, QList<Club *> &list)
{
  ClubXmlHandler handler(list);

  if (handler.parse(fileName))
    qDebug() << "File loaded:" << fileName << " entries:" << list.size();
}

void MainWindow::saveClubFile(QString &fileName, QList<Club *> &list)
{
  ClubXmlHandler handler(list);

  if (handler.save(fileName))
    // TODO: banner
    qDebug() << "File saved:" << fileName << " entries:" << list.size();
  else
    qWarning() << "Unable to save:" << fileName;

}
