/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include "data.h"

////////////////////////////////////////////////////////////////////////
// Hole
////////////////////////////////////////////////////////////////////////

Hole::Hole(const QXmlAttributes &attrs) {
  if (attrs.index("num") != -1)
    num = attrs.value("num");
  if (attrs.index("shots") != -1)
    shots = attrs.value("shots");
  if (attrs.index("putts") != -1)
    putts = attrs.value("putts");
  if (attrs.index("hcp") != -1)
    hcp = attrs.value("hcp");
  if (attrs.index("length") != -1)
    length = attrs.value("length");
  if (attrs.index("par") != -1)
    par = attrs.value("par");
}    

Hole::Hole(const QDomElement node) {
  num = node.attribute("num", "");
  shots = node.attribute("shots", "");
  putts = node.attribute("putts", "");
  hcp = node.attribute("hcp", "");
  length = node.attribute("length", "");
  par = node.attribute("par", "");
}

Hole::Hole(int num, QString &par, QString &hcp)
{
  this->num = QString::number(num);
  this->par = par;
  this->hcp = hcp;
}

Hole::Hole(int num, QString &shots)
{
  this->num = QString::number(num);
  this->shots = shots;
}

QDomElement Hole::toElement(QDomDocument doc)
{
  QDomElement node = doc.createElement("hole");

  if (!num.isEmpty())
    node.setAttribute("num", num);
  if (!shots.isEmpty())
    node.setAttribute("shots", shots);
  if (!putts.isEmpty())
    node.setAttribute("putts", putts);
  if (!hcp.isEmpty())
    node.setAttribute("hcp", hcp);
  if (!length.isEmpty())
    node.setAttribute("length", length);
  if (!par.isEmpty())
    node.setAttribute("par", par);

  return node;
}

QString Hole::getShots() {
  return shots;
}

void Hole::setShots(QString& s) {
  shots = s;
}

QString Hole::getHcp() {
  return hcp;
}

void Hole::setHcp(QString& s) {
  hcp = s;
}

QString Hole::getPar() {
  return par;
}

void Hole::setPar(QString& s) {
  par = s;
}

void Hole::dump() {
  qDebug() << num << "(" << par << ") : " << shots << "/" << putts ; 
}

////////////////////////////////////////////////////////////////////////
// Score
////////////////////////////////////////////////////////////////////////

Score::Score(const QXmlAttributes &attrs) 
{
  club = attrs.value("club");
  course = attrs.value("course");
  date = attrs.value("date");
}

Score::Score(QString &iClub, QString &iCourse, QString &iDate) 
{
  club = iClub;
  course = iCourse;
  date = iDate;
}

Score::Score(QVector<QString> scores, QString &club, QString &course, QString &date) 
{
  this->club = club;
  this->course = course;
  this->date = date;

  for (int i = 0; i < scores.size(); i++) {
    Hole *hole = new Hole(i+1, scores[i]);
    holeList << hole;
  }
}

Score::Score(const QDomElement node) {
  club = node.attribute("club", "");
  course = node.attribute("course", "");
  date = node.attribute("date", "");
}

QDomElement Score::toElement(QDomDocument doc)
{
  QDomElement node = doc.createElement("score");

  node.setAttribute("club", club);
  node.setAttribute("course", course);
  node.setAttribute("date", date);

  for (int i=0; i < holeList.size(); i++) {
    Hole *hole = holeList.at(i);
    node.appendChild(hole->toElement(doc));
  }
  return node;
}

int Score::update(QVector<QString> &scores)
{
  for (int i = 0; i < scores.size(); i++) {
    Hole *hole = holeList.at(i);
    if (hole->getShots() != scores[i])
      hole->setShots(scores[i]);
  }
  return 0;
}

void Score::addHole(Hole *iHole) {
  holeList << iHole;
}
  
QString Score::getScore(int i) const 
{
  if (i >= 0 && i < holeList.size())
    return holeList.at(i)->getShots();
  else
    return QString("-");
}
  
QString Score::getTotal(int what) const
{
  int tot = 0;

  if (what == Total)
    for (int i=0; i <= 17; i++)
      tot += holeList.at(i)->getShots().toInt();

  if (what == TotalOut)
    for (int i=0; i <= 8; i++)
      tot += holeList.at(i)->getShots().toInt();
 
  if (what == TotalIn)
    for (int i=9; i <= 17; i++)
      tot += holeList.at(i)->getShots().toInt();

  return QString("%1").arg(tot);
}

const QString& Score::getClubName() const
{
  return club;
}

const QString& Score::getCourseName() const
{
  return course;
}

const QString& Score::getDate() const
{
  return date;
}

void Score::dump()
{
  qDebug() << club << course << date ; 
  for (int i=0; i<holeList.size(); i++)
    holeList.at(i)->dump();
}

////////////////////////////////////////////////////////////////////////
// Course
////////////////////////////////////////////////////////////////////////

Course::Course(const QXmlAttributes &attrs) {
  name = attrs.value("name");
}

Course::Course(const QDomElement node, Club * parent)
    : club(parent)
{
    name = node.attribute("name", "");
}

Course::Course(QString &name, 
	       QVector<QString> &par,
	       QVector<QString> &hcp)
{
  this->name = name;
  
  for (int i = 0; i < par.size(); i++) {
    Hole *hole = new Hole(i+1, par[i], hcp[i]);
    holeList << hole;
  }
}

Club * Course::parent()
{
    return club;
}

void Course::setParent(Club *parent)
{
    club = parent;
}

QDomElement Course::toElement(QDomDocument doc)
{
  QDomElement node = doc.createElement("course");

  node.setAttribute("name", name);

  for (int i=0; i < holeList.size(); i++) {
    Hole *hole = holeList.at(i);
    node.appendChild(hole->toElement(doc));
  }
  return node;
}

int Course::update(QVector<QString> &par,
		   QVector<QString> &hcp,
		   QVector<QString> &len)
{
  Q_UNUSED(len);
  for (int i = 0; i < par.size(); i++) {
    Hole *hole = holeList.at(i);
    if (hole->getPar() != par[i])
      hole->setPar(par[i]);
    if (hole->getHcp() != hcp[i])
      hole->setHcp(hcp[i]);
  }
  return 0;
}

void Course::addHole(Hole *iHole) {
  holeList << iHole;
}

QString Course::getPar(int i) {
  if (i >= 0 && i < holeList.size())
    return holeList.at(i)->getPar();
  else
    return QString("-");
}

QString Course::getHcp(int i) {
  if (i >= 0 && i < holeList.size())
    return holeList.at(i)->getHcp();
  else
    return QString("-");
}
  
QString& Course::getName() 
{ 
  return name; 
}

QString Course::getTotal(int what) {
  int tot = 0;

  if (what == Total)
    for (int i = 0; i < 18; i++)
      tot += holeList.at(i)->getPar().toInt();

  if (what == TotalOut)
    for (int i = 0; i < 9; i++)
      tot += holeList.at(i)->getPar().toInt();
 
  if (what == TotalIn)
    for (int i = 9; i < 18; i++)
      tot += holeList.at(i)->getPar().toInt();

  return QString("%1").arg(tot);
}


void Course::dump() {
  qDebug() << " " << name;
  for (int i=0; i<holeList.size(); i++)
    holeList.at(i)->dump();
}

////////////////////////////////////////////////////////////////////////
// Club
////////////////////////////////////////////////////////////////////////

Club::Club(const QXmlAttributes &attrs) {
  name = attrs.value("name");
}

Club::Club(const QDomElement node) {
  name = node.attribute("name", "");
}

Club::Club(QString &name)
{
  this->name = name;
}

void Club::addCourse(Course *iCourse) {
  courseList << iCourse;
  iCourse->setParent(this);
}

void Club::delCourse(Course * course) {
    int index = courseList.indexOf(course);

    if (index != -1)
        courseList.removeAt(index);
}

bool Club::isEmpty()
{
    bool rc = false;
    if (courseList.count() == 0)
        rc = true;
    return rc;
}

QDomElement Club::toElement(QDomDocument doc)
{
  QDomElement node = doc.createElement("club");

  node.setAttribute("name", name);

  for (int i=0; i < courseList.size(); i++) {
    Course *course = courseList.at(i);
    node.appendChild(course->toElement(doc));
  }
  return node;
}
  
void Club::dump() {
  qDebug() << name;
  for (int i=0; i<courseList.size(); i++)
    courseList.at(i)->dump();
}

QString& Club::getName() { return name; }

Course *Club::getCourse(int pos) {
  return courseList.at(pos);
}

Course *Club::getCourse(const QString &courseName) 
{
  QListIterator<Course *> i(courseList);
  Course *c = 0;

  while (i.hasNext()) {
    c = i.next();
    if (c->getName() == courseName) {
      return c;
    }
  }
  return 0;
}
