/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QColor>
#include <QBrush>
#include <QFont>
#include "table-model.h"
#include "score-common.h"

QString empty("");

Qt::ItemFlags ScoreTableModel::flags (const QModelIndex &)
{
  return 0;
}

void ScoreTableModel::setMode(int m)
{
  currentMode = m;
}

int ScoreTableModel::mode(void)
{
  return currentMode;
}

// Assign the 'sList' to internal 'scoreList'. Set the current score
// to 'currentScore', or to 's'.
void ScoreTableModel::setScore(QList<Score *> &sList, Score *s)
{
  scoreList = sList;
  if (scoreList.size() > 0) {
    if (s) {
      currentScore = scoreList.indexOf(s);
      if (currentScore == -1)
	currentScore = 0;
    }
    score = scoreList.at(currentScore); // NOTE: assumes non-empty list
    course = findCourse(score->getClubName(), score->getCourseName());
  }
}

void ScoreTableModel::setClub(QList<Club *> &cList)
{
  clubList = cList;

  if (clubList.size() > 0)
    club = clubList.at(0);

  if (club)
    course = club->getCourse(0);
}

QString ScoreTableModel::getInfoText()
{
  QString str("");

  if (score)
    str = QString("%1, %2 / [%3/%4]").arg(score->getCourseName()).arg(score->getDate()).arg(currentScore+1).arg(scoreList.count());

  return str;
}

QString ScoreTableModel::getCountText()
{
  QString str = QString("%1/%2").arg(currentScore+1, 2).arg(scoreList.count(), 2);
  return str;
}

QString& ScoreTableModel::clubName(void)
{
  if (club)
    return club->getName();

  return empty;
}

QString& ScoreTableModel::courseName(void)
{
  if (course)
    return course->getName();

  return empty;
}

Course *ScoreTableModel::findCourse(const QString &clubName, 
				    const QString &courseName)
{
  QListIterator<Club *> i(clubList);
  Club *c;

  while (i.hasNext()) {
    c = i.next();
    if (c->getName() == clubName) {
      return c->getCourse(courseName);
    }
  }
  return 0;
}

Club *ScoreTableModel::getClub(void)
{
  return club;
}

Course *ScoreTableModel::getCourse(void)
{
  return course;
}

Score *ScoreTableModel::getScore(void)
{
  return score;
}

void ScoreTableModel::first()
{
  if (score && course) {
    currentScore = 0;
    score = scoreList.at(currentScore);
    course = findCourse(score->getClubName(), score->getCourseName());
    emit dataChanged(createIndex(0, 0), createIndex(ROW_COUNT-1, COL_COUNT-1));
  }
}

void ScoreTableModel::last()
{
  if (score && course) {
    currentScore = scoreList.size() - 1;
    score = scoreList.at(currentScore);
    course = findCourse(score->getClubName(), score->getCourseName());
    emit dataChanged(createIndex(0, 0), createIndex(ROW_COUNT-1, COL_COUNT-1));
  }
}

void ScoreTableModel::next()
{
  if (score && course) {
    if (currentScore < (scoreList.size() - 1)) {
      currentScore++;
      score = scoreList.at(currentScore);
      course = findCourse(score->getClubName(), score->getCourseName());
      emit dataChanged(createIndex(0, 0), createIndex(ROW_COUNT-1, COL_COUNT-1));
    }
  }
}

void ScoreTableModel::prev()
{
  if (score && course) {
    if (currentScore > 0) {
      currentScore--;
      score = scoreList.at(currentScore);
      course = findCourse(score->getClubName(), score->getCourseName());
      emit dataChanged(createIndex(0, 0), createIndex(ROW_COUNT-1, COL_COUNT-1));
    }
  }
}

int ScoreTableModel::rowCount(const QModelIndex &) const
{
  return 8;
}
 
int ScoreTableModel::columnCount(const QModelIndex &) const
{
  return 9 + 2; // 2 for in/out and tot columns
}

QModelIndex ScoreTableModel::index(int row, int column, const QModelIndex &parent) const
{
  if (hasIndex(row, column, parent)) {
    int flag = (parent.column() > 0) ? parent.column() : 0;
    return createIndex(row, column, flag);
  }
  else {
    return QModelIndex();
  }
}

QVariant ScoreTableModel::data(const QModelIndex &index, int role) const
{
  // TODO: move away from the stack

  if (!index.isValid())
    return QVariant();

  int row = index.row();
  int col = index.column();

  //
  // ALIGNMENT
  //
  if (role == Qt::TextAlignmentRole ) {
    return Qt::AlignCenter;
  }

  if (index.column() > 10)
    return QVariant();

  //
  // COLORS
  //
  if (role == Qt::BackgroundRole) {
    // Hole numbers 1-18. All hole nums, in, out and tot cell but not
    // the empty cell.
    if ((row == ROW_HOLE && col != 10) || row == ROW_HOLE_2) {
      QBrush brush(ScoreColor::holeBg());
      return brush;
    }
    if (score && course && (row == ROW_SCORE || row == ROW_SCORE_2)) {
      int par;
      int shots;
      if (row == ROW_SCORE) {
	par = course->getPar(col).toInt();
	shots = score->getScore(col).toInt();
      }
      else {
	par = course->getPar(col + 9).toInt();
	shots = score->getScore(col + 9).toInt();
      }

      if (col == (COLS+1) && row == ROW_SCORE_2) {
	// Total score
	QBrush brush(ScoreColor::total());
	return brush;
      }
      if (col == COLS) {
	// In and Out scores
	QBrush brush(ScoreColor::subTotal());
	return brush;
      }
      if (col < COLS) {
	if (shots == par) {
	  // Par
	  QBrush brush(ScoreColor::par());
	  return brush;
	}
	if (shots == (par-1)) {
	  // Birdie
	  QBrush brush(ScoreColor::birdie());
	  return brush;
	}
	if (shots == (par+1)) {
	  // Bogey
	  QBrush brush(ScoreColor::bogey());
	  return brush;
	}
	if (shots == (par+2)) {
	  // Double Bogey
	  QBrush brush(ScoreColor::doubleBogey());
	  return brush;
	}
	if (shots > (par+2)) {
	  // Very bad
	  QBrush brush(ScoreColor::bad());
	  return brush;
	}
      }
    }
    return QVariant();
  }
  //
  // FONT
  //
  if (role == Qt::FontRole) {
    if (row == ROW_SCORE_2 && col == (COLS+1)) {
	QFont font;
	font.setBold(true);
	return font;
    }
  }
  //
  // NUMBERS
  //
  if (role == Qt::DisplayRole) {

    if (col == COLS) {
      // In/out label
      if (row == ROW_HOLE)
	return QString("Out");
      if (row == ROW_HOLE_2)
	return QString("In");

      // In/Out for par
      if (score && course && row == ROW_PAR)
	return course->getTotal(TotalOut);
      if (score && course && row == ROW_PAR_2)
	return course->getTotal(TotalIn);

      // In/Out for score
      if (score && row == ROW_SCORE)
	return score->getTotal(TotalOut);
      if (score && row == ROW_SCORE_2)
	return score->getTotal(TotalIn);
      
    }
    else if (col == (COLS+1)) {
      // Total label
      if (row == ROW_HOLE_2)
	return QString("Tot");
      // Total score
      if (score && course && row == ROW_PAR_2)
	return course->getTotal(Total);
      if (score && row == ROW_SCORE_2)
	return score->getTotal(Total);
    }
    else {
      // data cells
      switch(row) {
      case ROW_HOLE:
	return col + 1;
      case ROW_HOLE_2:
	return col + 10;
      case ROW_PAR:
	if (score && course)
	  return course->getPar(col); 
      case ROW_PAR_2:
	if (score && course)
	  return course->getPar(col + 9); 
      case ROW_HCP: 
	if (score && course)
	  return course->getHcp(col); 
      case ROW_HCP_2:
	if (score && course)
	  return course->getHcp(col + 9);
      case ROW_SCORE:
	if (score)
	  return score->getScore(col);
      case ROW_SCORE_2: 
	if (score)
	  return score->getScore(col + 9);
      }
    }
  }
  return QVariant();
}

int ScoreTableModel::setItem(int row, int col, int)
{
  emit dataChanged(createIndex(row, col), createIndex(row, col));
  return 1;
}

QVariant ScoreTableModel::headerData(int section, Qt::Orientation orientation, int role) const
{
  // Only vertical header -- horizontal is hidden
  if (orientation == Qt::Horizontal)
    return QVariant();

  if (role == Qt::DisplayRole) {
    switch(section) {
    case ROW_PAR: 
    case ROW_PAR_2: 
      return QString("Par");
    case ROW_HCP: 
    case ROW_HCP_2: 
      return QString("HCP");
    case ROW_SCORE: 
    case ROW_SCORE_2: 
      return QString("Score");
    }
    return QVariant();
  }

  return QVariant();
}

