/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QVariant>

#include "stat-model.h"

StatModel::StatModel(QList<Club *> &cList, QList<Score *> &sList) : clubList(cList), scoreList(sList)
{
  update();
}

int StatModel::rowCount(const QModelIndex & parent) const
{
  return ROWS;
}

int StatModel::columnCount(const QModelIndex & parent) const
{
  return COLS;
}

QVariant StatModel::data(const QModelIndex & index, int role) const
{
  if (!index.isValid())
    return QVariant();

  int row = index.row();
  int col = index.column();

  if (col >= stat.size())
    return QVariant();
    
  //
  // ALIGNMENT
  //
  if (role == Qt::TextAlignmentRole ) {
    return Qt::AlignCenter;
  }

  //
  // NUMBERS
  //
  if (role == Qt::DisplayRole) {
    switch (row) {
    case ROW_ROUNDS: 
      return stat.at(col)->rounds();
    case ROW_AVERAGE: 
      return stat.at(col)->average();
    case ROW_MIN: 
      return stat.at(col)->min();
    case ROW_MAX: 
      return stat.at(col)->max();
    case ROW_BIRDIE: 
      return stat.at(col)->birdies();
    case ROW_PAR: 
      return stat.at(col)->pars();
    case ROW_BOGEY: 
      return stat.at(col)->bogeys();
    case ROW_MORE:
      return stat.at(col)->more();
    }
  }
  return QVariant();
}

QVariant StatModel::headerData(int section, Qt::Orientation orientation, int role) const
{
  // Only vertical header -- horizontal is hidden
  if (role != Qt::DisplayRole)
    return QVariant();

  if (orientation == Qt::Horizontal) {
    // TODO: check when no or less data than cols
    // HERE CRASH

    if (section < stat.size())
      return stat.at(section)->year();
  }

  if (orientation == Qt::Vertical) {
    switch(section) {
    case ROW_ROUNDS: 
      return QString("Rounds");
    case ROW_AVERAGE: 
      return QString("Average");
    case ROW_MIN: 
      return QString("Best");
    case ROW_MAX: 
      return QString("Worst");
    case ROW_BIRDIE: 
      return QString("Birdies");
    case ROW_PAR: 
      return QString("Pars");
    case ROW_BOGEY: 
      return QString("Bogeys");
    case ROW_MORE:
      return QString("Double+");
    }
  }

  return QVariant();
}

// TODO: dup code from table-model.cpp
Course *StatModel::findCourse(const QString &clubName, 
			      const QString &courseName)
{
  QListIterator<Club *> i(clubList);
  Club *c;

  while (i.hasNext()) {
    c = i.next();
    if (c->getName() == clubName) {
      return c->getCourse(courseName);
    }
  }
  return 0;
}

void StatModel::update(void)
{
  QListIterator<Score *> iScore(scoreList);
  QMultiMap<QString, Score *> yearMap;

  // Create multi map with years as keys, scores as values
  while (iScore.hasNext()) {
    Score *score = iScore.next();
    QString year = score->getDate().split("-").at(0);
    yearMap.insert(year, score);
  }
  // Create uniq list of years
  QList<QString> yearList = yearMap.uniqueKeys();

  // For each year collect the statistics
  QListIterator<QString> iYear(yearList);
  while (iYear.hasNext()) {
    QString year = iYear.next();

    StatItem *item = new StatItem;
    item->setYear(year);

    QList<Score *> scoresPerYear = yearMap.values(year);
    QListIterator<Score *> iScoresPerYear(scoresPerYear);
    
    item->setRounds(scoresPerYear.count());

    // for each year, add score
    int sum = 0;
    int min = 200;
    int max = 0;
    int pars = 0;
    int birdies = 0;
    int bogeys = 0;
    int more = 0;
    while (iScoresPerYear.hasNext()) {
      Score *s = iScoresPerYear.next();
      int tot = s->getTotal(Total).toInt();
      sum += tot;

      if (tot > max)
	max = tot;

      if (tot < min)
	min = tot;

      Course *c = findCourse(s->getClubName(), s->getCourseName());

      for (int i = 0; i < 18; i++) {
	int par = c->getPar(i).toInt();
	int shots = s->getScore(i).toInt();
	
	if (shots == (par - 1))
	  birdies++;
	else if (shots == par)
	  pars++;
	else if (shots == (par + 1))
	  bogeys++;
	else if (shots >= (par + 2))
	  more++;
      }
    }
    item->setBirdies(birdies);
    item->setPars(pars);
    item->setBogeys(bogeys);
    item->setMore(more);

    int avg = sum / scoresPerYear.count();
    item->setAverage(avg);
    item->setMin(min);
    item->setMax(max);

    stat << item;
  }
}
