/*
 * Copyright (C) 2009 Sakari Poussa
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 */

#include <QtGui>

#include "main-window.h"
#include "score-dialog.h"
#include "course-dialog.h"
#include "stat-model.h"
#include "xml-dom-parser.h"

QString appName("scorecard");
QString topDir("/opt");
QString mmcDir("/media/mmc1");
QString dataDirName("data");
QString dataDir;
QString imgDir(topDir + "/pixmaps");
QString scoreFileName("score.xml");
QString scoreFile;
QString clubFileName("club.xml");
QString clubFile;
QString logFile("/tmp/scorecard.log");

bool dateLessThan(const Score *s1, const Score *s2)
{
  return (*s1) < (*s2);
}

MainWindow::MainWindow(QMainWindow *parent): QMainWindow(parent)
{
  resize(800, 480);

  loadSettings();

  centralWidget = new QWidget(this);

  setCentralWidget(centralWidget);

  loadScoreFile(scoreFile, scoreList);
  loadClubFile(clubFile, clubList);

  // Sort the scores based on dates
  qSort(scoreList.begin(), scoreList.end(), dateLessThan); 
  createActions();
  createMenus();

  createTableView(scoreList, clubList);
  updateTitleBar();

  createLayout(centralWidget);
}

void MainWindow::loadSettings(void)
{
  bool external = false;

  QDir mmc(mmcDir);
  if (mmc.exists())
    external = true;

  // TODO: make via user option, automatic will never work
  external = false;

#ifndef Q_WS_HILDON
  dataDir = "./" + dataDirName;
#else
  if (external) {
    dataDir = mmcDir + "/" + appName + "/" + dataDirName;
  }
  else {
    dataDir = topDir + "/" + appName + "/" + dataDirName;
  }
#endif
  scoreFile = dataDir + "/" + scoreFileName;
  clubFile = dataDir + "/" + clubFileName;

  QDir dir(dataDir);
  if (!dir.exists())
    if (!dir.mkpath(dataDir)) {
      qWarning() << "Unable to create: " + dataDir;
      return;
    }
  qDebug() << "Data is at:" + dataDir;
}

void MainWindow::createLayout(QWidget *parent)
{

  buttonLayout = new QVBoxLayout;
  //labelLayout->addStretch();
  buttonLayout->addWidget(nextButton);
  buttonLayout->addWidget(prevButton);
  buttonLayout->addWidget(lastButton);
  buttonLayout->addWidget(firstButton);

  tableLayout = new QVBoxLayout;
  tableLayout->addWidget(table);

  QHBoxLayout *mainLayout = new QHBoxLayout(parent);
  mainLayout->addLayout(tableLayout);
  mainLayout->addLayout(buttonLayout);
  parent->setLayout(mainLayout);
}

// Setup 'score' tab view
void MainWindow::createTableView(QList<Score *> &scoreList, QList <Club *> &clubList)
{
  table = new QTableView;

  nextButton = new QPushButton(tr("Next"));
  prevButton = new QPushButton(tr("Prev"));
  firstButton = new QPushButton(tr("First"));
  lastButton = new QPushButton(tr("Last"));

  connect(nextButton, SIGNAL(clicked()), this, SLOT(nextButtonClicked()));
  connect(prevButton, SIGNAL(clicked()), this, SLOT(prevButtonClicked()));
  connect(firstButton, SIGNAL(clicked()), this, SLOT(firstButtonClicked()));
  connect(lastButton, SIGNAL(clicked()), this, SLOT(lastButtonClicked()));

  scoreTableModel = new ScoreTableModel();

  table->showGrid();

  table->setModel(scoreTableModel);
  table->setSelectionMode(QAbstractItemView::NoSelection);

  scoreTableModel->setScore(scoreList);
  scoreTableModel->setClub(clubList);

  // Fill out all the space with the tables
  table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setAutoFillBackground(true);
  table->horizontalHeader()->hide();
}

void MainWindow::createActions()
{
  newScoreAct = new QAction(tr("New Score"), this);
  connect(newScoreAct, SIGNAL(triggered()), this, SLOT(newScore()));

  newCourseAct = new QAction(tr("New Course"), this);
  connect(newCourseAct, SIGNAL(triggered()), this, SLOT(newCourse()));

  editScoreAct = new QAction(tr("Edit Score"), this);
  connect(editScoreAct, SIGNAL(triggered()), this, SLOT(editScore()));

  editCourseAct = new QAction(tr("Edit Course"), this);
  connect(editCourseAct, SIGNAL(triggered()), this, SLOT(editCourse()));

  statAct = new QAction(tr("Statistics"), this);
  connect(statAct, SIGNAL(triggered()), this, SLOT(viewStatistics()));

  nextAct = new QAction(tr( "   Next   "), this);
  connect(nextAct, SIGNAL(triggered()), this, SLOT(nextButtonClicked()));

  prevAct = new QAction("   Prev   ", this);
  connect(prevAct, SIGNAL(triggered()), this, SLOT(prevButtonClicked()));

  firstAct = new QAction(tr("   First  "), this);
  connect(firstAct, SIGNAL(triggered()), this, SLOT(firstButtonClicked()));

  lastAct = new QAction(tr( "   Last   "), this);
  connect(lastAct, SIGNAL(triggered()), this, SLOT(lastButtonClicked()));
}

void MainWindow::createMenus()
{
#ifdef Q_WS_HILDON
  menu = menuBar()->addMenu("");
#else
  menu = menuBar()->addMenu("Menu");
#endif

  menu->addAction(newScoreAct);
  menu->addAction(newCourseAct);
  menu->addAction(editScoreAct);
  menu->addAction(editCourseAct);
  menu->addAction(statAct);
}

void MainWindow::updateTitleBar()
{
  QString title = scoreTableModel->getInfoText();
  if (title.isEmpty())
    title = "ScoreCard - No Scores";

  setWindowTitle(title);
}

void MainWindow::firstButtonClicked()
{
  scoreTableModel->first();
  updateTitleBar();
}

void MainWindow::lastButtonClicked()
{
  scoreTableModel->last();
  updateTitleBar();
}

void MainWindow::nextButtonClicked()
{
  scoreTableModel->next();
  updateTitleBar();
}

void MainWindow::prevButtonClicked()
{
  scoreTableModel->prev();
  updateTitleBar();
}

// FIXME: dup code from table-model.cpp
Club *MainWindow::findClub(QString &name)
{
  QListIterator<Club *> i(clubList);
  Club *c;

  while (i.hasNext()) {
    c = i.next();
    if (c->getName() == name)
      return c;
  }
  return 0;
}

void MainWindow::newCourse()
{
  CourseSelectDialog *selectDialog = new CourseSelectDialog(this);

  int result = selectDialog->exec();
  if (result) {
    QString clubName;
    QString courseName;
    QString date;

    selectDialog->results(clubName, courseName);

    CourseDialog *courseDialog = new CourseDialog(this);
    courseDialog->init();

    QString title = "New Course: " + clubName + "," + courseName;
    courseDialog->setWindowTitle(title);

    int result = courseDialog->exec();
    if (result) {
      QVector<QString> par(18);
      QVector<QString> hcp(18);
      QVector<QString> len(18);

      courseDialog->results(par, hcp, len);

      Course *course = 0;
      Club *club = findClub(clubName);
      if (club) {
	course = club->getCourse(courseName);
	if (course) {
	  qDebug() << "Error: club/course already in the database";
	  return;
	}
	else {
	  course = new Course(courseName, par, hcp);
	  club->addCourse(course);
	}
      }
      else {
	// New club and course
	club = new Club(clubName);
	course = new Course(courseName, par, hcp);
	club->addCourse(course);
	clubList << club;
      }
      saveClubFile(clubFile, clubList);

      // TODO: does this really work? No mem leaks?
      scoreTableModel->setClub(clubList);
    }
  }
}

void MainWindow::editCourse()
{
  Course *course = scoreTableModel->getCourse();

  if (!course) {
    qWarning() << "No course on edit";
    return;
  }

  CourseDialog *courseDialog = new CourseDialog(this);
  courseDialog->init(course);

  QString title = "Edit Course: " + course->getName();
  courseDialog->setWindowTitle(title);
  
  int result = courseDialog->exec();
  if (result) {
    QVector<QString> par(18);
    QVector<QString> hcp(18);
    QVector<QString> len(18);
    
    courseDialog->results(par, hcp, len);
    
    course->update(par, hcp, len);
    saveClubFile(clubFile, clubList);
  }
}

void MainWindow::newScore()
{
  SelectDialog *selectDialog = new SelectDialog(this);

  selectDialog->init(clubList);

  int result = selectDialog->exec();
  if (result) {
    QString clubName;
    QString courseName;
    QString date;

    selectDialog->results(clubName, courseName, date);

    ScoreDialog *scoreDialog = new ScoreDialog(this);
    QString title = "New Score: " + courseName + ", " + date;
    scoreDialog->setWindowTitle(title);

    Club *club = findClub(clubName);
    if (!club) {
      qWarning() << "Error: no such club:" << clubName;
      return;
    }
    Course *course = club->getCourse(courseName);
    if (!course) {
      qWarning() << "Error: no such course:" << courseName;
      return;
    }
    scoreDialog->init(course);
    result = scoreDialog->exec();
    if (result) {
      QVector<QString> scores(18);

      scoreDialog->results(scores);
      Score *score = new Score(scores, clubName, courseName, date);
      scoreList << score;

      // Sort the scores based on dates
      qSort(scoreList.begin(), scoreList.end(), dateLessThan); 
      // Save it
      saveScoreFile(scoreFile, scoreList);

      // TODO: does this really work? No mem leaks?
      scoreTableModel->setScore(scoreList, score);
      updateTitleBar();
    }
  }
}

void MainWindow::editScore()
{
  Course *course = scoreTableModel->getCourse();
  Score *score = scoreTableModel->getScore();

  if (!course || !score) {
    qDebug() << "No score/course to edit";
    return;
  }

  QString date = score->getDate();

  ScoreDialog *scoreDialog = new ScoreDialog(this);
  
  QString title = "Edit Score: " + course->getName() + ", " + date;
  scoreDialog->setWindowTitle(title);

  scoreDialog->init(course, score);

  int result = scoreDialog->exec();

  if (result) {
    QVector<QString> scores(18);

    scoreDialog->results(scores);
    
    score->update(scores);

    // Sort the scores based on dates
    qSort(scoreList.begin(), scoreList.end(), dateLessThan); 
    // Save it
    saveScoreFile(scoreFile, scoreList);

    // TODO: does this really work? No mem leaks?
    scoreTableModel->setScore(scoreList, score);
    updateTitleBar();
  }
}

void MainWindow::viewStatistics()
{
  QMainWindow *win = new QMainWindow(this);
  QString title = "Statistics";
  win->setWindowTitle(title);

  StatModel *statModel = new StatModel(clubList, scoreList);

  QTableView *table = new QTableView;
  table->showGrid();
  table->setSelectionMode(QAbstractItemView::NoSelection);
  table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setAutoFillBackground(true);
  table->setModel(statModel);

  QWidget *central = new QWidget(win);
  win->setCentralWidget(central);

  QPushButton *b1 = new QPushButton("Graphs");

  QVBoxLayout *buttonLayout = new QVBoxLayout;
  buttonLayout->addWidget(b1);

  QTextEdit *textEdit = new QTextEdit;
  //getStat(textEdit);

  textEdit->setReadOnly(true);

  QVBoxLayout *infoLayout = new QVBoxLayout;
  infoLayout->addWidget(table);

  QHBoxLayout *mainLayout = new QHBoxLayout(central);
  mainLayout->addLayout(infoLayout);
  //mainLayout->addLayout(buttonLayout);

  central->setLayout(mainLayout);

  win->show();
}

void MainWindow::loadScoreFile(QString &fileName, QList<Score *> &list)
{
  ScoreXmlHandler handler(list);

  if (handler.parse(fileName))
    qDebug() << "File loaded:" << fileName << " entries:" << list.size();
}

void MainWindow::saveScoreFile(QString &fileName, QList<Score *> &list)
{
  ScoreXmlHandler handler(list);

  if (handler.save(fileName))
    // TODO: banner
    qDebug() << "File saved:" << fileName << " entries:" << list.size();
  else
    qWarning() << "Unable to save:" << fileName;
}

void MainWindow::loadClubFile(QString &fileName, QList<Club *> &list)
{
  ClubXmlHandler handler(list);

  if (handler.parse(fileName))
    qDebug() << "File loaded:" << fileName << " entries:" << list.size();
}

void MainWindow::saveClubFile(QString &fileName, QList<Club *> &list)
{
  ClubXmlHandler handler(list);

  if (handler.save(fileName))
    // TODO: banner
    qDebug() << "File saved:" << fileName << " entries:" << list.size();
  else
    qWarning() << "Unable to save:" << fileName;

}
