#include "tree-model.h"
#include <QDebug>

ScoreTreeModel::ScoreTreeModel(QList <Score *> &scoreList, QObject *parent) : QAbstractItemModel(parent) 
{
  rootItem = new TreeItem();
  setupModelData(scoreList, rootItem);
}

void ScoreTreeModel::setupModelData(QList<Score *> &scoreList, TreeItem *parent)
{
  QListIterator<Score *> iScore(scoreList);
  QMultiMap<QString, Score *> yearMap;
  QStringList yearList;
  QSet<QString> yearSet;

  // Create multi map with years as keys, scores as values
  // Create list of years
  while (iScore.hasNext()) {
    Score *score = iScore.next();
    QString year = score->getDate().split("-").at(0);
    yearMap.insert(year, score);
    yearList << year;
  }

  // Create uniq list of years
  yearSet = QSet<QString>::fromList(yearList);
  QList<QString> years = yearMap.keys();
  QSetIterator<QString> iYear(yearSet);

  // For each year, add scores as childs
  while (iYear.hasNext()) {
    QString year = iYear.next();

    qDebug() << "---" << year;

    TreeItem *newItem = new TreeItem(year);
    parent->appendChild(newItem);

    QList<Score *> scoresPerYear = yearMap.values(year);
    QListIterator<Score *> iScoresPerYear(scoresPerYear);

    // for each year, add score
    while (iScoresPerYear.hasNext()) {
      Score *s = iScoresPerYear.next();
      TreeItem *nextItem = new TreeItem(s);
      newItem->appendChild(nextItem);    
    }
  }
}

int ScoreTreeModel::rowCount(const QModelIndex & parent) const
{    
  TreeItem *parentItem;
  if (parent.column() > 0)
    return 0;

  if (!parent.isValid())
    parentItem = rootItem;
  else
    parentItem = static_cast<TreeItem*>(parent.internalPointer());
  
  return parentItem->childCount();
}
 
int ScoreTreeModel::columnCount(const QModelIndex & parent) const
{
  if (parent.isValid())
    return static_cast<TreeItem*>(parent.internalPointer())->columnCount();
  else
    return rootItem->columnCount();
}

TreeItem *ScoreTreeModel::itemFromIndex(const QModelIndex &index) const
{
  if (index.isValid())
    return static_cast<TreeItem*>(index.internalPointer());
  else
    return rootItem;
}

//Given a model index for a parent item, this function allows views
//and delegates to access children of that item
QModelIndex ScoreTreeModel::index(int row, int column, const QModelIndex &parent) const
{
  QModelIndex index;

  if (!hasIndex(row, column, parent))
    return QModelIndex();

  TreeItem *parentItem = itemFromIndex(parent);
  TreeItem *childItem = parentItem->children.value(row);

  if (childItem)
    index = createIndex(row, column, childItem);
  else
    index = QModelIndex();

  //qDebug() << "index: " << row << ":" << column << "parent" << parent << "index" << index;
  return index;
}

QModelIndex ScoreTreeModel::parent(const QModelIndex & index) const
{
  if (!index.isValid())
    return QModelIndex();

  TreeItem *item = itemFromIndex(index);
  if (!item)
    return QModelIndex();

  TreeItem *parent = item->parent;
  if (!parent)
    return QModelIndex();

  TreeItem *grandParent = parent->parent;
  if (!grandParent)
    return QModelIndex();
  
  int row = grandParent->children.indexOf(parent);
  return createIndex(row, 0, parent);
}

QVariant ScoreTreeModel::data(const QModelIndex &index, int role) const
{
  if (!index.isValid())
    return QVariant();

  if (role != Qt::DisplayRole && role < Qt::UserRole)
    return QVariant();

  TreeItem *item = itemFromIndex(index);
  if (!item)
    return QVariant();

  switch(role) {
  case Qt::DisplayRole:
    return item->str;

    // User defined roles
  case Type:
    return item->type;

  case Total:
    if (item->score)
      return item->score->getTotal(Total);
    break;

  case TotalOut:
    if (item->score)
      return item->score->getTotal(TotalOut);
    break;

  case TotalIn:
    if (item->score)
      return item->score->getTotal(TotalIn);
    break;

  default:
    qDebug() << "data: unknow role: " << role;
  }
  return QVariant();
}

QVariant ScoreTreeModel::headerData(int section, Qt::Orientation orientation, int role) const
{

    if (orientation == Qt::Horizontal && role == Qt::DisplayRole)
        return rootItem->data(section);

    return QVariant();

}

