#include <QtGui>
#include <QInputContext>

#include "score-dialog.h"

SelectDialog::SelectDialog(QWidget *parent) : QDialog(parent)
{
  QWidget *centralWidget = new QWidget(this);
  createLayout(centralWidget);

  setWindowTitle(tr("ScoreCard: Select Course and Date"));
}

void SelectDialog::createLayout(QWidget *parent)
{
  listClub = new QListWidget(parent);
  lineEditDate = new QLineEdit;
  pushButtonNext = new QPushButton(tr("Next"));

  QDate today(QDate::currentDate());
  lineEditDate->setText(today.toString("yyyy-MM-dd"));
  date = new QDateEdit;

  connect(pushButtonNext, SIGNAL(clicked()), this, SLOT(next()));

  leftLayout = new QVBoxLayout;
  leftLayout->addWidget(listClub);
  //leftLayout->addWidget(date);
  leftLayout->addWidget(lineEditDate);

  rightLayout = new QVBoxLayout;
  rightLayout->addStretch();
  rightLayout->addWidget(pushButtonNext);

  QHBoxLayout *mainLayout = new QHBoxLayout(parent);
  mainLayout->addLayout(leftLayout);
  mainLayout->addLayout(rightLayout);

  setLayout(mainLayout);
}

void SelectDialog::init(QList<Club *> &list)
{
  clubList = list;

  QListIterator<Club *> i(clubList);
  int index = 0;

  while (i.hasNext()) {
    Club *club = i.next();

    QList<Course *> courseList = club->getCourseList();

    QListIterator<Course *> j(courseList);
    while (j.hasNext()) {
      Course *course = j.next();

      QListWidgetItem *newItem = new QListWidgetItem;

      QString entry = club->getName() + "," + course->getName();

      newItem->setText(entry);
      listClub->insertItem(index, newItem);

      index++;
    }
  }
}

void SelectDialog::results(QString &club, 
			   QString &course, 
			   QString &date)
{  
  QListWidgetItem *current = listClub->currentItem();

  if (current) {
    QString tmp = current->text();

    QStringList stringList = tmp.split(",");
    club = stringList[0];
    course = stringList[1];
    date   = lineEditDate->text();
  }
}

bool SelectDialog::validate(void)
{
  return true;
}

void SelectDialog::next(void)
{
  if (validate())
    done(1);
  else {
    qDebug() << "SelectDialog: invalid data, cancel or correct";
  }
}

void SelectDialog::reject(void)
{
  done(0);
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

ScoreDialog::ScoreDialog(QWidget *parent) : QDialog(parent)
{
  resize(800, 400);

  QWidget *centralWidget = new QWidget(this);

  createTable();
  createButton();

  createLayout(centralWidget);
}

void ScoreDialog::createLayout(QWidget *parent)
{
  leftLayout = new QVBoxLayout;
  leftLayout->addWidget(table);

  rightLayout = new QVBoxLayout;
  rightLayout->addWidget(pushButtonUp);
  rightLayout->addWidget(pushButtonDown);
  rightLayout->addWidget(pushButtonNext);
  rightLayout->addStretch();
  rightLayout->addWidget(pushButtonFinish);

  QHBoxLayout *mainLayout = new QHBoxLayout(parent);
  mainLayout->addLayout(leftLayout);
  mainLayout->addLayout(rightLayout);
  setLayout(mainLayout);
}

void ScoreDialog::createTable(QWidget *parent)
{
  table = new QTableWidget(ROWS, COLS, parent);

  table->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
  table->verticalHeader()->setResizeMode(QHeaderView::Stretch);
  table->horizontalHeader()->hide();

  QStringList headers;
  headers << "" << "Par" << "HCP" << "Score" << "" << "Par" << "HCP" << "Score";
  table->setVerticalHeaderLabels(headers);
}

void ScoreDialog::createButton(QWidget *parent)
{
  pushButtonUp = new QPushButton(tr("+"));
  connect(pushButtonUp, SIGNAL(clicked()), this, SLOT(up()));

  pushButtonDown = new QPushButton(tr("-"));
  connect(pushButtonDown, SIGNAL(clicked()), this, SLOT(down()));

  pushButtonNext = new QPushButton(tr("Next"));
  connect(pushButtonNext, SIGNAL(clicked()), this, SLOT(next()));

  pushButtonFinish = new QPushButton(tr("Finish"));
  connect(pushButtonFinish, SIGNAL(clicked()), this, SLOT(finish()));
}

void ScoreDialog::init(Course *course, Score *score)
{
  QTableWidgetItem *par, *hcp, *scoreItem, *holeNum;
  QColor fgColor(Qt::white);
  QColor bgColor(Qt::darkGray);

  for (int i = 0; i < 18; i++) {
    par = new QTableWidgetItem(course->getPar(i));
    hcp = new QTableWidgetItem(course->getHcp(i));
    if (score)
      scoreItem = new QTableWidgetItem(score->getScore(i));
    else
      scoreItem = new QTableWidgetItem("");
    holeNum = new QTableWidgetItem(QString::number(i+1));

    holeNum->setTextColor(fgColor);
    holeNum->setBackgroundColor(bgColor);

    par->setTextAlignment(Qt::AlignCenter);
    hcp->setTextAlignment(Qt::AlignCenter);
    scoreItem->setTextAlignment(Qt::AlignCenter);
    holeNum->setTextAlignment(Qt::AlignCenter);

    if (i < 9) {
      table->setItem(ROW_HOLE, i, holeNum);
      table->setItem(ROW_PAR, i, par);
      table->setItem(ROW_HCP, i, hcp);
      table->setItem(ROW_SCORE, i, scoreItem);
    }
    else {
      table->setItem(ROW_HOLE_2, i-9, holeNum);
      table->setItem(ROW_PAR_2, i-9, par);
      table->setItem(ROW_HCP_2, i-9, hcp);
      table->setItem(ROW_SCORE_2, i-9, scoreItem);
    }
  }

  // Set focus to 1st cell
  table->setCurrentCell(ROW_SCORE, 0);
  if (!score)
    setDefaultScore(table);
}

// Set default score to par if not set
void ScoreDialog::setDefaultScore(QTableWidget *table)
{
  int row = table->currentRow();
  int col = table->currentColumn();
  
  if (row == ROW_SCORE)
    row = ROW_PAR;
  else if (row == ROW_SCORE_2)
    row = ROW_PAR_2;
  else {
    qDebug() << "ERROR: unknown row in default score";
    return;
  }
  QTableWidgetItem *par = table->item(row, col);
  QTableWidgetItem *score = table->item(row + 2, col);

  if (par && score && score->text() == "") {
    QVariant value(par->text());
    score->setData(Qt::DisplayRole, value);
  }
}

void ScoreDialog::up(void)
{
  QTableWidgetItem *item = table->currentItem();

  if (!item) {
    qWarning() << "ERROR: no current item";
    return;
  }

  int i = (item->text()).toInt();
  QVariant value(i+1);
  item->setData(Qt::DisplayRole, value);
}

void ScoreDialog::down(void)
{
  QTableWidgetItem *item = table->currentItem();

  if (!item)
    return;

  int i = (item->text()).toInt();
  QVariant value(i-1);
  item->setData(Qt::DisplayRole, value);
}

void ScoreDialog::next(void)
{
  if (table) {
    QTableWidgetItem *item = table->currentItem();
    moveToNextCell(item);
    setDefaultScore(table);
  }
}

void ScoreDialog::moveToNextCell(QTableWidgetItem *item)
{
  if (!item)
    return;

  QTableWidget *table = item->tableWidget();

  if (!table)
    return;

  int row = table->currentRow();
  int col = table->currentColumn();

  if (col < (COLS-1)) {
    col++;
  }
  else if (col == (COLS-1)) {
    col = 0;
    row = (row == ROW_SCORE_2) ? ROW_SCORE : ROW_SCORE_2;
  }
  table->setCurrentCell(row, col);
}

void ScoreDialog::results(QVector<QString> &scores)
{
  for (int i = 0; i < 9; i++) {
    QTableWidgetItem *frontItem = table->item(ROW_SCORE, i);
    QTableWidgetItem *backItem = table->item(ROW_SCORE_2, i);

    if (frontItem)
      scores[i] = frontItem->text();

    if (backItem)
      scores[i+9] = backItem->text();
  }
}

bool ScoreDialog::validate(void)
{
  for (int i=0; i<9; i++) {
    QTableWidgetItem *frontItem = table->item(ROW_SCORE, i);
    QTableWidgetItem *backItem = table->item(ROW_SCORE_2, i);
    
    if (!frontItem || !backItem)
      return false;
    
    QString str1 = frontItem->text();
    QString str2 = backItem->text();
    
    if (str1.isEmpty() || str2.isEmpty())
      return false;
  }
  return true;
}

void ScoreDialog::finish(void)
{
  if (validate())
    done(1);
  else {
    qDebug() << "ScoreDialog: invalid data, cancel or correct";
  }
}

void ScoreDialog::reject(void)
{
  done(0);
}
