/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "PlayerGreedy.h"

#include <cassert>
#include <cstdlib>

SolutionSet GreedyPlayer::PlayCard(const Table& table, size_t round)
{
  assert(!hand.empty());

  std::vector<SolutionSet> solutions;

  table.GetSolutionSets(hand,solutions);

  // 1. Do we have the change to make a scopa?

  for (int s=solutions.size()-1; s>=0; --s) {
    if (solutions[s].scopa) {
      return solutions[s];
    }
  }

  // 2. Can we get ours/somebody elses denare 7?

  for (int s=solutions.size()-1; s>=0; --s) {
    bool   found=false;
    size_t i=0;

    while (!found && i<solutions[s].table.size()) {
      found=solutions[s].IsTrick() &&
            ((solutions[s].yours.GetColor()==Card::denare && solutions[s].yours.GetValue()==7) ||
             (solutions[s].table[i].GetColor()==Card::denare && solutions[s].table[i].GetValue()==7));
      i++;
    }

    if (found) {
      return solutions[s];
    }
  }

  // 3. Get maximum denares...

  size_t denaresMax=0;
  int    denaresPos=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    size_t denares=0;

    for (size_t i=0; i<solutions[s].table.size(); i++) {
      if (solutions[s].table[i].GetColor()==Card::denare) {
        denares++;
      }
    }
    if (solutions[s].yours.GetColor()==Card::denare) {
      denares++;
    }

    if (solutions[s].IsTrick() && denares>denaresMax) {
      denaresMax=denares;
      denaresPos=s;
    }
  }

  if (denaresPos>=0) {
    return solutions[denaresPos];
  }

  // 4. Get maximum sevens...

  size_t sevensMax=0;
  int    sevensPos=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    size_t sevens=0;

    for (size_t i=0; i<solutions[s].table.size(); i++) {
      if (solutions[s].table[i].GetValue()==7) {
        sevens++;
      }
    }
    if (solutions[s].yours.GetValue()==7) {
      sevens++;
    }

    if (solutions[s].IsTrick() && sevens>sevensMax) {
      sevensMax=sevens;
      sevensPos=s;
    }
  }

  if (sevensPos>=0) {
    return solutions[sevensPos];
  }

  // 5. Get maximum points (primera)...
  size_t pointsMax=0;
  int    pointsPos=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    size_t points=0;

    for (size_t i=0; i<solutions[s].table.size(); i++) {
      points+=solutions[s].table[i].GetPoints();
    }
    points+=solutions[s].yours.GetPoints();

    if (solutions[s].IsTrick() && points>pointsMax) {
      pointsMax=points;
      pointsPos=s;
    }
  }

  if (pointsPos>=0) {
    return solutions[pointsPos];
  }

  // Finally, get the solution which gives us the most cards
  if (solutions.size()>0) {
    return solutions[solutions.size()-1];
  }

  // No chance to take a card we must play the best card possible...

  size_t valueOnTable=table.GetValueOnTable();

  //
  // Try to get mnore than 10 points on the table
  //

  // Play the smallest card, that more than 10 points are on the table - but no denare and no 7 ...

  int solution;

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick() && solutions[s].yours.GetColor()!=Card::denare && solutions[s].yours.GetValue()!=7) {
      if ((solution<0 && valueOnTable+solutions[s].yours.GetValue()>10) ||
          (solutions[solution].yours.GetValue()>solutions[s].yours.GetValue() &&
           valueOnTable+solutions[s].yours.GetValue()>10)) {
        solution=s;
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  // Play the smallest card, that more than 10 points are on the table - but no 7 ...

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick() && solutions[s].yours.GetValue()!=7) {
      if ((solution<0 && valueOnTable+ solutions[s].yours.GetValue()>10) ||
          ( solutions[solution].yours.GetValue()> solutions[s].yours.GetValue() &&
           valueOnTable+solutions[s].yours.GetValue()>10)) {
        solution=s;
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  // Play the smallest card, that more than 10 points are on the table - but no denare...

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick() && solutions[s].yours.GetColor()!=Card::denare) {
      if ((solution<0 && valueOnTable+solutions[s].yours.GetValue()>10) ||
          (solutions[solution].yours.GetValue()>solutions[s].yours.GetValue() &&
           valueOnTable+solutions[s].yours.GetValue()>10)) {
        solution=s;
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  // Play the smallest card, that more than 10 points are on the table

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick()) {
      if ((solution<0 && valueOnTable+solutions[s].yours.GetValue()>10) ||
          (solutions[solution].yours.GetValue()>solutions[s].yours.GetValue() &&
           valueOnTable+solutions[s].yours.GetValue()>10)) {
        solution=-1;
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  //
  // Cannot get above 10 points
  //

  // Play the smallest card - but no denare and no 7 ...

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick()) {
      if (solutions[s].yours.GetColor()!=Card::denare && solutions[s].yours.GetValue()!=7) {
        if (solution<0 || solutions[solution].yours.GetValue()>solutions[s].yours.GetValue()) {
          solution=s;
        }
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  // Play the smallest card - but no 7 ...

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick()) {
      if (solutions[s].yours.GetValue()!=7) {
        if (solution<0 || solutions[solution].yours.GetValue()>solutions[s].yours.GetValue()) {
          solution=s;
        }
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  // Play the smallest card - but no denare...

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick()) {
      if (solutions[s].yours.GetColor()!=Card::denare) {
        if (solution<0 || solutions[solution].yours.GetValue()>solutions[s].yours.GetValue()) {
          solution=s;
        }
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  // Play the smallest card

  solution=-1;

  for (size_t s=0; s<solutions.size(); s++) {
    if (!solutions[s].IsTrick()) {
      if (solution<0 || solutions[solution].yours.GetValue()>solutions[s].yours.GetValue()) {
        solution=s;
      }
    }
  }

  if (solution>=0) {
    return solutions[solution];
  }

  // Hmmm, no clever selection criteria, simply take the first one.
  return solutions[(size_t)(double(solutions.size())*rand()/RAND_MAX)];
}

