/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "PlayerGoal.h"

#include <cassert>
#include <cstdlib>

#include <Lum/Base/String.h>
#include <iostream>

GoalPlayer::GoalPlayer()
 : statScopa(0),
   statDenare7(0),
   statDenare(0),
   statPrimera(0),
   statMostCards(0),
   statAbove10(0),
   statBelow10(0),
   statRandom(0)
{
  // no code
}

void GoalPlayer::Initialize()
{
  Player::Initialize();

  goalDenare7=true;
  goalDenare=true;
  goalPrimera=true;
  goalMostCards=true;

  myDenare=0;
  yourDenare=0;

  myCards=0;
  yourCards=0;

  mySevens=0;
  yourSevens=0;
  mySixs=0;
  yourSixs=0;
}

SolutionSet GoalPlayer::PlayCard(const Table& table, size_t round)
{
  assert(!hand.empty());

  std::vector<SolutionSet> solutions;

  table.GetSolutionSets(hand,solutions);

  //
  // Always active goals
  //

  // Do we have the change to make a scopa?

  for (int s=solutions.size()-1; s>=0; --s) {
    if (solutions[s].scopa) {
      statScopa++;
      return solutions[s];
    }
  }

  //
  // Conditional goals
  //

  if (goalDenare7) {
    //
    // Denare 7 goal
    //

    for (int s=solutions.size()-1; s>=0; --s) {
      bool   found=false;
      size_t i=0;

      while (!found && i<solutions[s].table.size()) {
        found=solutions[s].IsTrick() &&
              ((solutions[s].yours.GetColor()==Card::denare && solutions[s].yours.GetValue()==7) ||
               (solutions[s].table[i].GetColor()==Card::denare && solutions[s].table[i].GetValue()==7));
        i++;
      }

      if (found) {
        statDenare7++;
        return solutions[s];
      }
    }
  }

  if (goalDenare) {
    //
    // Denare goal
    //

    size_t denaresMax=0;
    int    denaresPos=-1;

    for (size_t s=0; s<solutions.size(); s++) {
      size_t denares=0;

      for (size_t i=0; i<solutions[s].table.size(); i++) {
        if (solutions[s].table[i].GetColor()==Card::denare) {
          denares++;
        }
      }

      if (solutions[s].yours.GetColor()==Card::denare) {
        denares++;
      }

      if (solutions[s].IsTrick() && denares>denaresMax) {
        denaresMax=denares;
        denaresPos=s;
      }
    }

    if (denaresPos>=0) {
        statDenare++;
      return solutions[denaresPos];
    }
  }

  if (goalPrimera) {
    // Get maximum points (primeria)...
    size_t pointsMax=0;
    int    pointsPos=-1;

    for (size_t s=0; s<solutions.size(); s++) {
      size_t points=0;

      for (size_t i=0; i<solutions[s].table.size(); i++) {
        if (solutions[s].table[i].GetValue()==6 || solutions[s].table[i].GetValue()==7) {
          points+=solutions[s].table[i].GetPoints();
        }
      }

      if (solutions[s].yours.GetValue()==6 || solutions[s].yours.GetValue()==7) {
        points+=solutions[s].yours.GetPoints();
      }

      if (solutions[s].IsTrick() && points>pointsMax) {
        pointsMax=points;
        pointsPos=s;
      }
    }

    if (pointsPos>=0) {
      statPrimera++;
      return solutions[pointsPos];
    }
  }

  if (goalMostCards || !goalMostCards) {
    // Get maximum points (primeria)...
    size_t cardsMax=0;
    int    cardsPos=-1;

    for (size_t s=0; s<solutions.size(); s++) {
      if (solutions[s].IsTrick() && solutions[s].table.size()>cardsMax) {
        cardsMax=solutions[s].table.size();
        cardsPos=s;
      }
    }

    if (cardsPos>=0) {
      statMostCards++;
      return solutions[cardsPos];
    }
  }

  // No chance to take a card we must play the best card possible...

  size_t valueOnTable=table.GetValueOnTable();

  int solution;


  solution=-1;

  for (size_t i=0; i<=3; i++) {
    //
    // Play the smallest card, in that way that more than 10 points are on the table
    //

    for (size_t s=0; s<solutions.size(); s++) {
      if (solutions[s].IsTrick()) {
        continue;
      }

      if (i<3 && goalDenare7 && solutions[s].yours.GetColor()==Card::denare && solutions[s].yours.GetValue()==7) {
        continue;
      }

      if (i<2 && goalDenare && solutions[s].yours.GetColor()==Card::denare) {
        continue;
      }

      if (i<1 && goalPrimera && (solutions[s].yours.GetValue()==6 || solutions[s].yours.GetValue()==7)) {
        continue;
      }

      if ((solution<0 || solutions[solution].yours.GetValue()>solutions[s].yours.GetValue())
          && valueOnTable+solutions[s].yours.GetValue()>10) {
        solution=s;
      }
    }

    if (solution>=0) {
      statAbove10++;
      return solutions[solution];
    }

    //
    // Cannot get above 10 points - Play the smallest card
    //

    solution=-1;

    for (size_t s=0; s<solutions.size(); s++) {
      if (solutions[s].IsTrick()) {
        continue;
      }

      if (i<3 && goalDenare7 && solutions[s].yours.GetColor()==Card::denare && solutions[s].yours.GetValue()==7) {
        continue;
      }

      if (i<2 && goalDenare && solutions[s].yours.GetColor()==Card::denare) {
        continue;
      }

      if (i<1 && goalPrimera && (solutions[s].yours.GetValue()==6 || solutions[s].yours.GetValue()==7)) {
        continue;
      }

      if (solution<0 || solutions[solution].yours.GetValue()>solutions[s].yours.GetValue()) {
        solution=s;
      }
    }

    if (solution>=0) {
      statBelow10++;
      return solutions[solution];
    }
  }

  // Hmmm, no clever selection criteria, simply take the first one.
  statRandom++;

  return solutions[(size_t)(double(solutions.size())*rand()/RAND_MAX)];
}

void GoalPlayer::CardsPlayed(std::vector<Card>& cards,
                             bool ownCards,
                             size_t round)
{
  for (size_t i=0; i<cards.size(); i++) {
    // Probe Denare 7
    if (goalDenare7 && cards[i].GetColor()==Card::denare && cards[i].GetValue()==7) {
      //std::cout << "Goal 'Denare 7' droped!" << std::endl;
      goalDenare7=false;
    }

    // probe Denare
    if (cards[i].GetColor()==Card::denare) {
      if (ownCards) {
        myDenare++;
      }
      else {
        yourDenare++;
      }
    }

    // 6th and 7th for primera
    if (cards[i].GetValue()==7) {
      if (ownCards) {
        mySevens++;
      }
      else {
        yourSevens++;
      }
    }

    if (cards[i].GetValue()==6) {
      if (ownCards) {
        mySixs++;
      }
      else {
        yourSixs++;
      }
    }


    if (ownCards) {
      myCards++;
    }
    else {
      yourCards++;
    }
  }

  if (goalDenare && (myDenare>5 || yourDenare>5)) {
    goalDenare=false;
  }

  if (goalPrimera && ((mySevens>2 || (mySevens==2 && mySixs>2)) ||
                      (yourSevens>2 || (yourSevens==2 && yourSixs>2)))) {
    goalPrimera=false;
  }

  if (goalMostCards && (myCards>20 || yourCards>20)) {
    goalMostCards=false;
  }
}

void GoalPlayer::PrintStatistics()
{
  std::cout << "Scopa: " << statScopa << std::endl;
  std::cout << "Denare 7: " << statDenare7 << std::endl;
  std::cout << "Denare: " << statDenare << std::endl;
  std::cout << "Primera: " << statPrimera << std::endl;
  std::cout << "Most cards: " << statMostCards << std::endl;
  std::cout << "Above 10: " << statAbove10 << std::endl;
  std::cout << "Below 10: " << statBelow10 << std::endl;
  std::cout << "Random: " << statRandom << std::endl;
}

