/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "ChooseMoveDialog.h"

#include <Lum/Base/L10N.h>

#include <Lum/Model/Header.h>
#include <Lum/Model/Table.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/Table.h>
#include <Lum/Text.h>

#include "CardArea.h"
#include <iostream>
struct CardsEntry
{
  std::vector<Card> cards;
};

typedef Lum::Model::StdTable<CardsEntry> CardsModel;
typedef Lum::Base::Reference<CardsModel> CardsModelRef;

class CardsModelPainter : public Lum::TableCellPainter
{
private:
  Lum::OS::ColorRef                  backgroundColor;
  Lum::OS::FillRef                   background;
  std::vector<Lum::Images::ImageRef> images;

public:
  CardsModelPainter(const std::vector<Lum::Images::ImageRef>& images)
   : backgroundColor(0.0,0.5,0.0,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
     images(images)
  {
    if (!Lum::OS::display->GetTheme()->IsLightTheme()) {
      //goodFoodColor.Allocate(0.0,0.2,0.0,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor));
    }

    background=new Lum::OS::PlainFill(backgroundColor);
  }

  size_t GetProposedHeight() const
  {
    for (size_t i=0; i<images.size(); i++) {
      if (images[i].Valid()) {
        return images[i]->GetHeight()+
               Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceInterGroup);
      }
    }

    return 101+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceInterGroup);
  }

  Lum::OS::Fill* GetCellBackground() const
  {
    return background;
  }

  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    CardsEntry entry=dynamic_cast<const CardsModel*>(GetModel())->GetEntry(GetRow());

    for (size_t c=0; c<entry.cards.size(); c++) {
      Card                  card=entry.cards[c];
      Lum::Images::ImageRef image=images[card.GetIndex()];
      if (image.Valid()) {
        image->Draw(draw,
                    x+c*(image->GetWidth()+
                         Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceInterGroup)),
                    y+(height-image->GetHeight())/2);
      }
      else {
        size_t cardWidth=55+Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceInterGroup);
        size_t cardHeight=101;

        draw->PushForeground(Lum::OS::Display::whiteColor);
        draw->FillRectangle(x+c*cardWidth,
                            y+(height-cardHeight)/2,
                            cardWidth,cardHeight);
        draw->PopForeground();

        draw->PushForeground(Lum::OS::Display::blackColor);
        draw->DrawRectangle(x+c*cardWidth,
                            y+(height-cardHeight)/2,
                            cardWidth,cardHeight);
        draw->PopForeground();

        draw->PushForeground(Lum::OS::Display::blackColor);
        draw->PushFont(GetFont());
        draw->DrawString(x+c*cardWidth+(cardWidth-GetFont()->StringWidth(card.GetColorName()))/2,
                         y+(height-cardHeight)/2+GetFont()->ascent,
                         card.GetColorName());
        draw->DrawString(x+c*cardWidth+(cardWidth-GetFont()->StringWidth(card.GetValueName()))/2,
                         y+(height-cardHeight)/2+GetFont()->height+GetFont()->ascent,
                         card.GetValueName());
        draw->PopFont();
        draw->PopForeground();
      }
    }
  }
};

ChooseMoveDialog::ChooseMoveDialog(const Card& card,
                                   const std::vector <std::vector<Card> >& moves,
                                   const std::vector<Lum::Images::ImageRef>& images)
 : card(card),
   moves(moves),
   images(images),
   selection(new Lum::Model::SingleLineSelection())
{
  GetClosedAction()->Disable();

  Observe(GetClosedAction());
  Observe(selection);
}

void ChooseMoveDialog::PreInit()
{
  Lum::Panel            *vert;
  Lum::Text             *text;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;
  CardsModelRef         cardsModel;
  size_t                rowWidth=0;
  size_t                cardWidth=0;

  std::cout << "There are " << moves.size() << " possible moves!" << std::endl;

  for (size_t i=0; i<images.size(); i++) {
    if (images[i].Valid()) {
      cardWidth=std::max(cardWidth,images[i]->GetWidth()+
                         Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceInterGroup));
    }
    else {
      cardWidth=std::max(cardWidth,55+
                         Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceInterGroup));
    }
  }

  vert=Lum::VPanel::Create();

  text=new Lum::Text();
  text->SetFlex(true,false);
  text->SetText(_(L"CHOOSE_MOVE_CAPTION",
                  L"There are multiple possible card sets you could select\n"
                  L"with your card. Pleae choose the set you want to fetch:"));

  vert->Add(text);
  vert->AddSpace();

  cardsModel=new CardsModel();

  for (size_t i=0; i<moves.size(); i++) {
    CardsEntry entry;

    entry.cards=moves[i];

    rowWidth=std::max(rowWidth,moves[i].size()*cardWidth);

    cardsModel->Append(entry);
  }

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Cards",Lum::Base::Size::pixel,rowWidth,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetMinHeight(Lum::Base::Size::pixel,200);
  table->SetHeaderModel(headerModel);
  table->SetModel(cardsModel);
  table->SetSelection(selection);
  table->SetDefaultAction(GetClosedAction());
  table->SetPainter(new CardsModelPainter(images));
  table->SetAutoHSize(true);
  table->SetAutoFitColumns(true);

  vert->Add(table);
  vert->AddSpace();
  vert->Add(Lum::ButtonRow::Create(true,false)
            ->Add(Lum::Button::Create(_(L"CHOOSE_MOVE_CONTINUE",L"_Continue"),GetClosedAction(),Lum::Button::typeDefault,true,true)));

  SetMain(vert);

  Dialog::PreInit();
}

void ChooseMoveDialog::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==selection) {
    if (selection->HasSelection()) {
      GetClosedAction()->Enable();
    }
    else {
      GetClosedAction()->Disable();
    }
  }

  Dialog::Resync(model,msg);
}

size_t ChooseMoveDialog::GetMove() const
{
  return selection->GetLine()-1;
}

size_t GetMove(Lum::Dialog *parent,
               const Card& card,
               const std::vector <std::vector<Card> >& moves,
               const std::vector<Lum::Images::ImageRef>& images)
{
  size_t move;

  ChooseMoveDialog *dialog;

  dialog=new ChooseMoveDialog(card,moves,images);
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  move=dialog->GetMove();

  delete dialog;

  return move;
}
