/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "ChooseMoveDialog.h"

#include <Lum/Base/L10N.h>

#include <Lum/Model/Header.h>
#include <Lum/Model/Table.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/Table.h>
#include <Lum/Text.h>

#include "CardArea.h"

ChooseMoveDialog::ChooseMoveDialog(const Card& card,
                                   const std::vector <std::vector<Card> >& moves,
                                   const std::vector<Lum::Images::ImageRef>& images)
 : card(card),
   moves(moves),
   images(images),
   selection(new Lum::Model::SingleLineSelection())
{
  GetClosedAction()->Disable();

  Observe(GetClosedAction());
  Observe(selection);
}

void ChooseMoveDialog::PreInit()
{
  Lum::Panel                 *vert;
  Lum::Text                  *text;
  Lum::Table                 *table;

  Lum::Model::HeaderRef      headerModel;
  Lum::Model::ObjectTableRef movesModel;

  vert=Lum::VPanel::Create();

  text=new Lum::Text();
  text->SetFlex(true,false);
  text->SetText(_(L"CHOOSE_MOVE_CAPTION",
                  L"There are multiple possible card sets you could select\n"
                  L"with your card. Pleae choose the set you want to fetch:"));

  vert->Add(text);
  vert->AddSpace();

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Cards",Lum::Base::Size::pixel,0,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetHeaderModel(headerModel);
  table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->SetMinHeight(Lum::Base::Size::pixel,200);


  movesModel=new Lum::Model::ObjectTable();

  for (size_t i=0; i<moves.size(); i++) {
    CardArea *cardArea;

    cardArea=new CardArea();
    cardArea->SetSize(moves[i].size(),1);
    cardArea->SetImages(images);
    cardArea->SetCards(moves[i]);

    movesModel->Append(cardArea);
  }

  table->SetModel(movesModel);
  table->SetSelection(selection);

  vert->Add(table);
  vert->AddSpace();
  vert->Add(Lum::ButtonRow::Create(true,false)
            ->Add(Lum::Button::Create(_(L"CHOOSE_MOVE_CONTINUE",L"_Continue"),GetClosedAction(),Lum::Button::typeDefault,true,true)));

  SetMain(vert);

  Dialog::PreInit();
}

void ChooseMoveDialog::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==selection) {
    if (selection->HasSelection()) {
      GetClosedAction()->Enable();
    }
    else {
      GetClosedAction()->Disable();
    }
  }

  Dialog::Resync(model,msg);
}

size_t ChooseMoveDialog::GetMove() const
{
  return selection->GetLine()-1;
}

size_t GetMove(Lum::Dialog *parent,
               const Card& card,
               const std::vector <std::vector<Card> >& moves,
               const std::vector<Lum::Images::ImageRef>& images)
{
  size_t move;

  ChooseMoveDialog *dialog;

  dialog=new ChooseMoveDialog(card,moves,images);
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  move=dialog->GetMove();

  delete dialog;

  return move;
}
