/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Table.h"

SolutionSet::SolutionSet(const Card& yours)
 : yours(yours)
{
  // no code
}

bool SolutionSet::IsTrick() const
{
  return table.size()>0;
}

void Table::Calculate(const std::vector<Card>& table,
                      std::vector<std::vector<size_t> >& combinations,
                      std::map<size_t,std::vector<size_t> >& points)
{
  size_t last,start,stop,max;

  combinations.clear();
  points.clear();

  //
  // Calculation combinations
  //

  // Initial set (A list of single entry solution with value 1..table.size().
  for (size_t i=1; i<=table.size(); i++) {
    std::vector<size_t> entry;

    entry.push_back(i);

    combinations.push_back(entry);
  }

  // Now calculate possible combinations by taking initial combination and adding
  // one new entry if possible.
  max=table.size();
  start=0;
  last=0;
  while (combinations.size()!=last) {
    last=combinations.size();
    stop=combinations.size()-1;

    for (size_t i=start; i<=stop; i++) {
      for (size_t j=combinations[i].back()+1; j<=max; j++) {
        std::vector<size_t> entry;

        entry=combinations[i];
        entry.push_back(j);

        combinations.push_back(entry);
      }
    }

    start=stop+1;
  }

  //
  // Take all combinations, calculate points and fill points map.
  //

  for (size_t i=0; i<combinations.size(); i++) {
    size_t value=0;

    for (size_t j=0; j<combinations[i].size(); j++) {
      value+=table[combinations[i][j]-1].GetValue();
    }

    // Only add solutions if we don't have a solution or the size of the solution
    // set is not bigger than the existing solution.
    // Note, that the array combinations is sorted by increasing solution size, so
    // that we will not insert smaller solution if we already have found one!
    std::map<size_t,std::vector<size_t> >::const_iterator pv;

    pv=points.find(value);
    if (pv!=points.end() && combinations[i].size()>combinations[pv->second[0]].size()) {
      continue;
    }

    points[value].push_back(i);
  }
}

void Table::GetSolutionSets(const std::vector<Card>& hand,
                            const std::vector<Card>& table,
                            const std::vector<std::vector<size_t> >& combinations,
                            const std::map<size_t,std::vector<size_t> >& points,
                            std::vector<SolutionSet>& solutions)
{
  solutions.clear();

  // TODO: Use GetMoves()

  for (size_t c=0; c<hand.size(); c++) {
    std::map<size_t,std::vector<size_t> >::const_iterator iter;

    // Find matching combinations that can be fetched with this card
    iter=points.find(hand[c].GetValue());
    if (iter!=points.end()) {
      // Now iterator over all combinations and build the complete combination set
      for (size_t s=0; s<iter->second.size(); s++) {
        SolutionSet set(hand[c]);

        set.yoursPos=c;

        set.table.reserve(combinations[iter->second[s]].size());

        for (size_t i=0; i<combinations[iter->second[s]].size(); i++) {
          set.table.push_back(table[combinations[iter->second[s]][i]-1]);
        }

        set.solution=s;
        set.scopa=combinations[iter->second[s]].size()==table.size();

        solutions.push_back(set);
      }
    }
    else {
      SolutionSet set(hand[c]);

      set.yoursPos=c;
      set.solution=0;
      set.scopa=false;
      solutions.push_back(set);
    }
  }
}

void Table::Calculate()
{
  Calculate(table,combinations,points);
}

void Table::Initialize(const std::vector<Card>& initialCards)
{
  table=initialCards;

  Calculate();
}

bool Table::IsEmpty() const
{
  return table.empty();
}

bool Table::HasValueMultipleCollectionSets(size_t value) const
{
  std::map<size_t,std::vector<size_t> >::const_iterator exchange;

  exchange=points.find(value);
  if (exchange!=points.end()) {
    return exchange->second.size()>1;
  }
  else {
    return false;
  }
}

void Table::GetMoves(size_t value, std::vector<std::vector<Card> >& moves) const
{
  std::map<size_t,std::vector<size_t> >::const_iterator iter;

  // Find matching combinations that can be fetched with this card
  iter=points.find(value);
  if (iter!=points.end()) {
    // Now iterator over all combinations and build the complete combination set
    for (size_t s=0; s<iter->second.size(); s++) {
      std::vector<Card> move;

      move.reserve(combinations[iter->second[s]].size());

      for (size_t i=0; i<combinations[iter->second[s]].size(); i++) {
        move.push_back(table[combinations[iter->second[s]][i]-1]);
      }

      moves.push_back(move);
    }
  }
}

std::vector<Card> Table::PutCard(const Card& card, size_t solution)
{
  std::vector<Card>                                     result;
  std::map<size_t,std::vector<size_t> >::const_iterator exchange;

  exchange=points.find(card.GetValue());
  if (exchange!=points.end()) {
    size_t idx=exchange->second[solution];

    result.push_back(card);
    for (int i=combinations[idx].size()-1; i>=0; --i) {
      result.push_back(table[combinations[idx][i]-1]);
      table.erase(table.begin()+combinations[idx][i]-1);
    }
  }
  else {
    table.push_back(card);
  }

  Calculate();

  return result;
}

void Table::GetSolutionSets(const std::vector<Card>& hand, std::vector<SolutionSet>& solutions) const
{
  GetSolutionSets(hand,table,combinations,points,solutions);
}

size_t Table::GetValueOnTable() const
{
  size_t value=0;

  for (size_t i=0; i<table.size(); i++) {
    value+=table[i].GetValue();
  }

  return value;
}

