/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FadeText.h"

#include <Lum/Base/DateTime.h>

#define FADE_STEPS    20
#define TIME_FRACTION 20

FadeText::FadeText()
 : text(0),
   font(Lum::OS::display->GetFont(250)),
   fadeStep(0),
   timerAction(new Lum::Model::Action())
{
  colors.resize(FADE_STEPS);
  colors[0]=Lum::OS::display->GetColor(Lum::OS::Display::backgroundColor);
  for (size_t i=1; i<FADE_STEPS; i++) {
    colors[i]=Lum::OS::display->GetColor(Lum::OS::Display::textColor);
  }

  Observe(timerAction);
}

void FadeText::Fade()
{
  fadeStep--;

  if (fadeStep>0) {
    Lum::OS::display->AddTimer(0,1000000/TIME_FRACTION,timerAction);
  }

  Redraw();
}

void FadeText::AddText(const std::wstring& text)
{
  Lum::OS::FontExtent extent;

  font->StringExtent(text,extent,Lum::OS::Font::bold);

  texts.push_back(text);
  extents.push_back(extent);
}

void FadeText::SetText(size_t text)
{
  this->text=text;
  fadeStep=FADE_STEPS-1;

  Lum::OS::display->RemoveTimer(timerAction);
  Lum::OS::display->AddTimer(0,1000000/TIME_FRACTION,timerAction);

  Redraw();
}

void FadeText::SetFadeFinishedAction(Lum::Model::Action* action)
{
  fadeFinishedAction=action;
}

void FadeText::CalcSize()
{
  assert(texts.size()>0);

  minWidth=0;
  minHeight=0;

  for (size_t i=0; i<extents.size();i++) {
    minWidth=std::max(minWidth,(size_t)(extents[i].width-extents[i].left-extents[i].right));
    minHeight=std::max(minHeight,extents[i].height);
  }

  width=minWidth;
  height=minHeight;

  Object::CalcSize();
}

void FadeText::Draw(int x, int y, size_t w, size_t h)
{
  Object::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  if (fadeStep>0) {
    Lum::OS::DrawInfo *draw=GetDrawInfo();

    draw->PushFont(font,Lum::OS::Font::bold);
    draw->PushForeground(colors[fadeStep]);

    draw->DrawString(this->x+extents[text].left+
                     (width-extents[text].width-extents[text].left-extents[text].right)/2,
                     this->y+extents[text].ascent,texts[text]);

    draw->PopForeground();
    draw->PopFont();
  }
}

void FadeText::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==timerAction && timerAction->IsFinished()) {
    Fade();
  }

  Object::Resync(model,msg);
}


