/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/


#include <cstdlib>
#include <iostream>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Images/Loader.h>

#include <Lum/Model/Action.h>

#include <Lum/OS/Main.h>

#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Label.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Text.h>

// Game logic
#include "Heap.h"
#include "PlayerGoal.h"
#include "PlayerGreedy.h"
#include "PlayerHuman.h"
#include "PlayerJudge.h"
#include "PlayerRandom.h"
#include "Result.h"

// GUI
#include "CardArea.h"
#include "ChooseMoveDialog.h"
#include "FadeText.h"
#include "GameStartDialog.h"
#include "RoundResultDialog.h"
#include "ShowMoveDialog.h"

#include "config.h"

static Lum::Def::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  enum State {
    stateHumanMove,
    stateHumanFading,
    stateComputerMove,
    stateComputerCollect,
    stateFinished
  };

private:
  Lum::Model::ActionRef newGameAction;
  Lum::Model::ActionRef lastHumanMoveAction;
  Lum::Model::ActionRef lastComputerMoveAction;
  Lum::Model::ActionRef aboutAction;

  Lum::Model::ActionRef handSelectionAction;
  Lum::Model::ActionRef handFadedAction;
  Lum::Model::ActionRef tableFadedAction;

  Lum::Text             *game;
  Lum::Text             *round;
  Lum::Text             *points;
  Lum::Text             *scopas;
  CardArea              *handView;
  CardArea              *tableView;
  FadeText              *fadeText;
  std::vector<Lum::Images::ImageRef> images;

  State                 state;
  bool                  handFaded,tableFaded;

  std::vector<Player*>  players;
  size_t                globalPoints[2];
  std::vector<Result>   result;
  Heap                  *heap;
  Table                 *table;
  std::vector<Move>     lastHumanMove;
  std::vector<Move>     lastComputerMove;
  std::vector<Card>     currentComputerMove;
  size_t                currentComputerMoveMove;

  size_t                currentGame;
  size_t                currentRound;
  size_t                currentPlayer; //! Current player [0..players.size()-1]
  size_t                lastExchange;
  size_t                computerPlayer;
  size_t                winningPoints;

public:
  MainDialog()
  : newGameAction(new Lum::Model::Action()),
    lastHumanMoveAction(new Lum::Model::Action()),
    lastComputerMoveAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    handSelectionAction(new Lum::Model::Action()),
    handFadedAction(new Lum::Model::Action()),
    tableFadedAction(new Lum::Model::Action()),
    heap(new Heap()),
    table(new Table()),
    computerPlayer(1),
    winningPoints(11)
  {
    lastHumanMoveAction->Disable();
    lastComputerMoveAction->Disable();

    Observe(GetOpenedAction());

    Observe(newGameAction);
    Observe(lastHumanMoveAction);
    Observe(lastComputerMoveAction);
    Observe(aboutAction);

    Observe(handSelectionAction);
    Observe(handFadedAction);
    Observe(tableFadedAction);

    Lum::Base::Path path;

    path.SetNativeDir(L"cards");
    path.AppendDir(L"scopa");

    images.resize(40);

    for (Card card=Card::minCard; card<=Card::maxCard; ++card) {
      Lum::Base::Path       path;
      Lum::Images::ImageRef image;

      path.SetNativeDir(L"cards");
      path.AppendDir(L"standard");
      path.SetBaseName(card.GetFilename()+L".png");

      if (!path.Exists()) {
        path.SetNativeDir(L"cards");
        path.AppendDir(L"scopa");
        path.SetBaseName(card.GetFilename()+L".png");
      }

#if defined(APP_DATADIR)
      if (!path.Exists()) {
        path.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
        path.AppendDir(L"cards");
        path.AppendDir(L"standard");
        path.SetBaseName(card.GetFilename()+L".png");
      }

      if (!path.Exists()) {
        path.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
        path.AppendDir(L"cards");
        path.AppendDir(L"scopa");
        path.SetBaseName(card.GetFilename()+L".png");
      }
#endif

      image=Lum::Images::Factory::factory->CreateImage();

      if (Lum::Images::loader->Load(path.GetPath(),image)) {
        images[card.GetIndex()]=image;
      }
      else {
        std::cerr << "Cannot load image '" << Lum::Base::WStringToString(path.GetPath()) << "'!" << std::endl;
      }
    }

    SetState(stateFinished);
  }

  void PreInit()
  {
    Lum::Grid   *grid;
    Lum::Label  *label;
    Lum::Panel  *horiz,*vert;
    Lum::Text   *text;

    horiz=Lum::HPanel::Create(true,true);

    grid=new Lum::Grid();
    grid->SetFlex(true,true);
    grid->SetSize(2,2);
    grid->SetSpace(true,true);

    text=new Lum::Text();
    text->SetFlex(true,false);
    text->SetAlignment(Lum::Text::centered);
    text->SetText(_(L"HAND",L"Hand"));

    grid->SetObject(0,0,text);

    text=new Lum::Text();
    text->SetFlex(true,false);
    text->SetAlignment(Lum::Text::centered);
    text->SetText(_(L"TABLE",L"Table"));

    grid->SetObject(1,0,text);

    handView=new CardArea();
    handView->SetFlex(true,true);
    handView->SetSize(1,3);
    handView->SetSelectionAction(handSelectionAction);
    handView->SetFadeFinishedAction(handFadedAction);
    handView->SetImages(images);
    grid->SetObject(0,1,handView);


    tableView=new CardArea();
    tableView->SetFlex(true,true);
    tableView->SetSize(4,3);
    tableView->SetFadeFinishedAction(tableFadedAction);
    tableView->SetImages(images);
    grid->SetObject(1,1,tableView);

    horiz->Add(grid);

    horiz->AddSpace();

    vert=Lum::VPanel::Create(false,true);

    label=Lum::Label::Create(true,false);

    game=new Lum::Text();
    game->SetFlex(true,false);
    game->SetWidth(Lum::Base::Size::stdCharWidth,2);
    game->SetAlignment(Lum::Text::centered);
    game->SetText(L"1");
    label->AddLabel(_(L"LABEL_GAME",L"Game:"),game);

    round=new Lum::Text();
    round->SetFlex(true,false);
    round->SetWidth(Lum::Base::Size::stdCharWidth,2);
    round->SetAlignment(Lum::Text::centered);
    round->SetText(L"1");
    label->AddLabel(_(L"LABEL_ROUND",L"Round:"),round);

    points=new Lum::Text();
    points->SetFlex(true,false);
    points->SetWidth(Lum::Base::Size::stdCharWidth,5);
    points->SetAlignment(Lum::Text::centered);
    points->SetText(L"0:0");
    label->AddLabel(_(L"LABEL_POINTS",L"Points:"),points);

    scopas=new Lum::Text();
    scopas->SetFlex(true,false);
    scopas->SetWidth(Lum::Base::Size::stdCharWidth,5);
    scopas->SetAlignment(Lum::Text::centered);
    scopas->SetText(L"0:0");
    label->AddLabel(_(L"LABEL_SCOPAS",L"Scopas:"),scopas);

    vert->Add(label);

    vert->AddSpace();

    vert->Add(Lum::Button::Create(_(L"GAME_NEW",L"_New Game"),newGameAction,true,false));

    vert->AddSpace(true);

    fadeText=new FadeText();
    fadeText->SetFlex(true,false);
    fadeText->AddText(_(L"TEXT_SCOPA!",L"Scopa!"));
    vert->Add(fadeText);

    vert->AddSpace(true);
    vert->Add(Lum::Button::Create(_(L"GAME_LAST_COMPUTER",L"_Last computer move"),lastComputerMoveAction,true,false));
    vert->AddSpace();
    vert->Add(Lum::Button::Create(_(L"GAME_LAST_HUMAN",L"_Last human move"),lastHumanMoveAction,true,false));

    horiz->Add(vert);

    SetMain(horiz);

  Lum::Def::Menu *menu=Lum::Def::Menu::Create();

  menu
    ->GroupProject()
      ->ActionQuit(GetClosedAction())
    ->End()
    ->Group(_(L"MENU_GAME",L"_Game"))
      ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_NEW",L"_New game"))
                                .SetShortcut(Lum::OS::qualifierControl,L"g"),
                                newGameAction)
               .SetOpensDialog())
      ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_LAST_COMPUTER",L"_Show last _human move"))
                                .SetShortcut(Lum::OS::qualifierControl,L"h"),
                                lastHumanMoveAction)
               .SetOpensDialog())
      ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_LAST_HUMAN",L"Show last _computer move"))
                                .SetShortcut(Lum::OS::qualifierControl,L"c"),
                                lastComputerMoveAction)
               .SetOpensDialog())
    ->End()
    ->GroupHelp()
      //->ActionHelp()
      ->ActionAbout(aboutAction)
    ->End();


    SetMenu(menu);

    Dialog::PreInit();
  }

  void SetState(State state)
  {
    this->state=state;

#if defined(PRINT_DEBUG)
    switch (state) {
    case stateHumanMove:
      std::cout << "State human move" << std::endl;
      break;
    case stateHumanFading:
      std::cout << "State human fading" << std::endl;
      break;
    case stateComputerMove:
      std::cout << "State computer move" << std::endl;
      break;
    case stateComputerCollect:
      std::cout << "State computer collect" << std::endl;
      break;
    case stateFinished:
      std::cout << "State finished" << std::endl;
      break;
    }
#endif
    switch (state) {
    case stateHumanFading:
      handFaded=false;
      tableFaded=false;
      break;
    case stateComputerMove:
      handFaded=false;
      tableFaded=false;
      OnMoveComputer();
      break;
    case stateComputerCollect:
      OnCollectComputer();
      break;
    default:
      break;
    }
  }

  void UpdateCardsView()
  {
    for (size_t i=0; i<players.size(); i++) {
      if (dynamic_cast<HumanPlayer*>(players[i])!=NULL) {
        handView->SetCards(players[i]->hand,true);
        break;
      }
    }
    tableView->SetCards(table->table,true);
  }

  void UpdateDisplay()
  {
    game->SetText(Lum::Base::NumberToWString(currentGame));
    round->SetText(Lum::Base::NumberToWString(currentRound));
    points->SetText(Lum::Base::NumberToWString(globalPoints[0])+L":"+Lum::Base::NumberToWString(globalPoints[1]));
    if (currentGame%2==1) {
      scopas->SetText(Lum::Base::NumberToWString(result[0].GetScopas())+L":"+Lum::Base::NumberToWString(result[1].GetScopas()));
    }
    else {
      scopas->SetText(Lum::Base::NumberToWString(result[1].GetScopas())+L":"+Lum::Base::NumberToWString(result[0].GetScopas()));
    }

    UpdateCardsView();
  }

  void CalculatePoints(std::vector<Result>& result, std::vector<size_t>& points)
  {
    // Scopas
    points[0]=result[0].GetScopas();
    points[1]=result[1].GetScopas();

    // Who has more cards?
    if (result[0].GetCards()>result[1].GetCards()) {
      points[0]++;
    }
    else if (result[1].GetCards()>result[0].GetCards()) {
      points[1]++;
    }

    // Who has more denares?
    if (result[0].GetDenares()>result[1].GetDenares()) {
      points[0]++;
    }
    else if (result[1].GetDenares()>result[0].GetDenares()) {
      points[1]++;
    }

    // Who has denare 7?
    if (result[0].HasDenare7()) {
      points[0]++;
    }
    else {
      points[1]++;
    }

    // Primeria
    size_t pp[2];

    pp[0]=0;
    pp[1]=0;
    for (size_t c=Card::spade; c<=Card::denare; c++) {
      pp[0]+=result[0].GetHighestPointsOfColor((Card::Color)c);
      pp[1]+=result[1].GetHighestPointsOfColor((Card::Color)c);
    }

    result[0].SetPrimieraPoints(pp[0]);
    result[1].SetPrimieraPoints(pp[1]);

    if (pp[0]>pp[1]) {
      points[0]++;
    }
    else if (pp[1]>pp[0]) {
      points[1]++;
    }

#if defined(PRINT_DEBUG)
    std::cout << std::endl;
    std::cout << "Now calculating points:..." << std::endl;

    std::cout << "Scopas:   " << result[0].GetScopas() << " " << result[1].GetScopas() << std::endl;
    std::cout << "Cards:    " << result[0].GetCards() << " " << result[1].GetCards() << std::endl;
    std::cout << "Denares:  " << result[0].GetDenares() << " " << result[1].GetDenares() << std::endl;
    std::cout << "Denare7:  " << result[0].HasDenare7() << " " << result[1].HasDenare7() << std::endl;
    std::cout << "Primiera: " << result[0].GetPrimieraPoints() << " " << result[1].GetPrimieraPoints() << std::endl;

    std::cout << std::endl;
    for (size_t i=0; i<=1; i++) {
      std::cout << "Group " << i+1 << " has " << points[i] << " points." << std::endl;
    }
#endif
  }

  void OnFinished()
  {
#if defined(PRINT_DEBUG)
    std::cout << "Finished!" << std::endl;
#endif

    // The rest of the cards on the table belong to the player that collected the last cards

    if (!table->IsEmpty()) {
#if defined(PRINT_DEBUG)
      std::cout << "--- Final" << std::endl;
      std::cout << "Player " << lastExchange+1 << " gets the rest of the cards on the table:" << std::endl;

      for (size_t i=0; i<table->table.size(); i++) {
        std::cout << "* "<< Lum::Base::WStringToString(table->table[i].GetName()) << std::endl;
      }
#endif

      result[lastExchange%2].AddToCollected(table->table,false);

      if (dynamic_cast<HumanPlayer*>(players[lastExchange%2])!=NULL) {
        lastHumanMove[0]=Move(table->table,false);
      }
      else {
        lastComputerMove[0]=Move(table->table,false);
      }

      table->table.clear();

      UpdateDisplay();
    }

    // Now calculate final points

    std::vector<size_t> points(2);


    CalculatePoints(result,points);

    // Every 2nd game we have to switch the results, because after every game the
    // players rotate by one and thus the groups rotate, too.
    if (currentGame%2==0) {
      Result r;
      size_t p;

      r=result[0];
      result[0]=result[1];
      result[1]=r;

      p=points[0];
      points[0]=points[1];
      points[1]=p;
    }

    globalPoints[0]+=points[0];
    globalPoints[1]+=points[1];

    ShowRoundResult(this,result,points,globalPoints);

    UpdateDisplay();

    if ((globalPoints[0]>=winningPoints || globalPoints[1]>=winningPoints) &&
        globalPoints[0]!=globalPoints[1]) {

      size_t winner;

      winner=globalPoints[0]>globalPoints[1] ? 0 : 1;

      if (winner==0) {
        std::wstring body=_(L"GAME_OVER_HUMAN_BODY",
                            L"Congratulations, you have won the game\n"
                            L"against the computer with $(winner) - $(looser)!"
                            L"\n"
                            L"I'm impressed!");

        Lum::Base::L10N::Substitute(body,L"$(winner)",Lum::Base::NumberToWString(globalPoints[winner]));
        Lum::Base::L10N::Substitute(body,L"$(looser)",Lum::Base::NumberToWString(globalPoints[(winner+1)%2]));

        Lum::Dlg::Msg::ShowOk(this,_(L"GAME_OVER_HUMAN_TITLE",L"Congratulations, you have won!"),body);
      }
      else {
        std::wstring body=_(L"GAME_OVER_COMPUTER_BODY",
                            L"...but I have won the game with $(winner) - $(looser)!\n"
                            L"\n"
                            L"...more luck next time!");

        Lum::Base::L10N::Substitute(body,L"$(winner)",Lum::Base::NumberToWString(globalPoints[winner]));
        Lum::Base::L10N::Substitute(body,L"$(looser)",Lum::Base::NumberToWString(globalPoints[(winner+1)%2]));

        Lum::Dlg::Msg::ShowOk(this,_(L"GAME_OVER_COMPUTER_TITLE",L"I'm so sorry..."),body);
      }
    }
    else {
      // Finally rotate players so that everyone starts in turn...
      Player *p=players[0];

      for (size_t i=0; i<players.size()-1; i++) {
        players[i]=players[i+1];
      }
      players[players.size()-1]=p;

      OnNewRound();
    }
  }

  void OnMoveComputer()
  {
    currentComputerMoveMove=0;

    std::vector<Card> tmp;
    SolutionSet       solution=players[currentPlayer]->PlayCard(*table,currentRound);

    players[currentPlayer]->hand.erase(players[currentPlayer]->hand.begin()+solution.yoursPos);

#if defined(PRINT_DEBUG)
    std::cout << "Computer player plays " << Lum::Base::WStringToString(card.GetName()) <<std::endl;
#endif
    if (currentComputerMove.empty()) {
      currentComputerMove.push_back(solution.yours);
    }
    else {
      currentComputerMove[0]=solution.yours;
    }

    currentComputerMoveMove=solution.solution;

    tmp=table->table;
    tmp.push_back(solution.yours);

    tableView->SetCards(tmp,true);
  }

  void OnCollectComputer()
  {
    Card              card=currentComputerMove[0];
    std::vector<Card> collected;

    collected=table->PutCard(card,currentComputerMoveMove);

    if (lastComputerMove.empty()) {
      if (collected.empty()) {
        lastComputerMove.push_back(Move(card));
      }
      else {
        lastComputerMove.push_back(Move(collected));
      }
    }
    else {
      if (collected.empty()) {
        lastComputerMove[0]=Move(card);
      }
      else {
        lastComputerMove[0]=Move(collected);
      }
    }

    if (collected.size()>0) {
#if defined(PRINT_DEBUG)
      std::cout << "..and collected the following cards: " << std::endl;
      for (size_t card=0; card<collected.size(); card++) {
        std::cout << "* "<< Lum::Base::WStringToString(collected[card].GetName()) << std::endl;
      }
#endif

      for (size_t i=0; i<players.size(); i++) {
        players[i]->CardsPlayed(collected,
                                (i)==(currentPlayer%2),
                                currentRound);
      }

      if (table->IsEmpty()) {
#if defined(PRINT_DEBUG)
        std::cout << "=>>> SCOPA!" << std::endl;
#endif
        fadeText->SetText(0);
      }

      lastExchange=currentPlayer;
    }

    result[currentPlayer%2].AddToCollected(collected,table->IsEmpty());

    OnPlayerPlayed();
  }

  void OnPlayHuman(size_t cardIndex)
  {
    HumanPlayer *human=dynamic_cast<HumanPlayer*>(players[currentPlayer]);

    assert(human!=NULL);

    std::vector<Card> collected;
    Card              card=human->OnCardPlayed(cardIndex);
    size_t            move=0;

#if defined(PRINT_DEBUG)
    std::cout << "Human player plays " << Lum::Base::WStringToString(card.GetName()) <<std::endl;
#endif

    if (table->HasValueMultipleCollectionSets(card.GetValue())) {
      std::vector<std::vector<Card> > moves;

      table->GetMoves(card.GetValue(),moves);

#if defined(PRINT_DEBUG)
      std::cout << "Card played by human has multiple collection sets!" << std::endl;
#endif
      move=GetMove(this,card,moves,images);
    }

    collected=table->PutCard(card,move);

    if (lastHumanMove.empty()) {
      if (collected.empty()) {
        lastHumanMove.push_back(Move(card));
      }
      else {
        lastHumanMove.push_back(Move(collected));
      }
    }
    else {
      if (collected.empty()) {
        lastHumanMove[0]=Move(card);
      }
      else {
        lastHumanMove[0]=Move(collected);
      }
    }

    if (collected.size()>0) {
#if defined(PRINT_DEBUG)
      std::cout << "..and collected the following cards: " << std::endl;
      for (size_t card=0; card<collected.size(); card++) {
        std::cout << "* "<< Lum::Base::WStringToString(collected[card].GetName()) << std::endl;
      }
#endif

      for (size_t i=0; i<players.size(); i++) {
        players[i]->CardsPlayed(collected,
                                (i)==(currentPlayer%2),
                                currentRound);
      }

      if (table->IsEmpty()) {
#if defined(PRINT_DEBUG)
        std::cout << "=>>> SCOPA!" << std::endl;
#endif
        fadeText->SetText(0);
      }

      lastExchange=currentPlayer;
    }


    result[currentPlayer%2].AddToCollected(collected,table->IsEmpty());

    OnPlayerPlayed();
  }

  void OnPlayerPlayed()
  {
    currentPlayer++;

    if (currentPlayer==players.size()) {
      // New cards

      if (currentRound>1 && currentRound%3==0 && !heap->IsEmpty()) {
#if defined(PRINT_DEBUG)
        std::cout << "New cards..." << std::endl;
#endif
        for (size_t player=0; player<players.size(); player++) {
          for (size_t c=1; c<=3; c++) {
            players[player]->PutCard(heap->GetCard());
          }
        }
      }

      // Check if finished

      if (heap->IsEmpty()) {
        bool finished=true;

        for (size_t player=0; player<players.size(); player++) {
          if (players[player]->HasCardsInHand()) {
            finished=false;
            break;
          }
        }

        if (finished) {
          OnFinished();
          return;
        }
      }

      currentPlayer=0;
      currentRound++;
    }

    if (!lastHumanMove.empty()) {
      lastHumanMoveAction->Enable();
    }

    if (!lastComputerMove.empty()) {
      lastComputerMoveAction->Enable();
    }

    if (dynamic_cast<HumanPlayer*>(players[currentPlayer])==NULL) {
      SetState(stateHumanFading);
      UpdateDisplay();
    }
    else {
      SetState(stateHumanMove);
      UpdateDisplay();
    }
  }

  void OnNewRound()
  {
    std::vector<Card> initialCards;

    result.clear();
    result.resize(2);

    lastComputerMove.clear();
    lastHumanMove.clear();

    lastHumanMoveAction->Disable();
    lastComputerMoveAction->Disable();

    for (size_t player=0; player<players.size(); player++) {
      players[player]->Initialize();
    }

#if defined(PRINT_DEBUG)
    std::cout << "Mix cards and hand out initial set..." << std::endl;
#endif

    heap->InitializeAndMix();

    for (size_t player=0; player<players.size(); player++) {
      for (size_t c=1; c<=3; c++) {
        players[player]->PutCard(heap->GetCard());
      }
    }

    for (size_t h=1; h<=4; h++) {
      initialCards.push_back(heap->GetCard());
    }

    table->Initialize(initialCards);

    currentGame++;
    currentRound=1;
    currentPlayer=0;

    if (dynamic_cast<HumanPlayer*>(players[0])!=NULL) {
      SetState(stateHumanMove);
    }
    else {
      SetState(stateComputerMove);
    }

    UpdateDisplay();

    // TODO: Allow that the computer starts playing...
  }

  void OnNewGame()
  {
    // TODO: Support different playing orders

    if (!ShowGameStartDialog(this,winningPoints,computerPlayer)) {
      return;
    }

    // Cleanup old player settings

    for (size_t p=0; p<players.size(); p++) {
      delete players[p];
    }
    players.clear();

    // Initialize new players

    players.push_back(new HumanPlayer());
    switch (computerPlayer) {
    case 1:
      players.push_back(new JudgePlayer());
      break;
    case 2:
      players.push_back(new GoalPlayer());
      break;
    case 3:
      players.push_back(new GreedyPlayer());
      break;
    case 4:
      players.push_back(new RandomPlayer());
      break;
    default:
      assert(false);
    }

    currentGame=0;
    globalPoints[0]=0;
    globalPoints[1]=0;

    OnNewRound();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
    }
    else if (model==newGameAction && newGameAction->IsFinished()) {
      OnNewGame();
    }
    else if (model==lastComputerMoveAction && lastComputerMoveAction->IsFinished()) {
      ShowMove(this,true,lastComputerMove[0],images);
    }
    else if (model==lastHumanMoveAction && lastHumanMoveAction->IsFinished()) {
      ShowMove(this,false,lastHumanMove[0],images);
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==handSelectionAction && handSelectionAction->IsFinished() && state==stateHumanMove) {
      const CardArea::Selection &selection=dynamic_cast<const CardArea::Selection&>(msg);

      OnPlayHuman(selection.GetCardIndex());
    }
    else if (model==handFadedAction && handFadedAction->IsFinished()) {
      handFaded=true;
      if (state==stateHumanFading) {
        if (handFaded && tableFaded) {
          SetState(stateComputerMove);
        }
      }
    }
    else if (model==tableFadedAction && tableFadedAction->IsFinished()) {
      tableFaded=true;
      if (state==stateHumanFading) {
        if (handFaded && tableFaded) {
          SetState(stateComputerMove);
        }
      }
      else if (state==stateComputerMove) {
        if (tableFaded) {
          SetState(stateComputerCollect);
        }
      }
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    srand(time(NULL));

   return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Play a game of scopa with me..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");
  }
};

LUM_MAIN(Main,L"Scopa")
