/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "PlayerJudge.h"

#include <cassert>
#include <cstdlib>

#include <set>

#include <Lum/Base/String.h>

// 100 = 1 Point
//  25 = 1/4 Chance to make a point (1 of 4 necessary cards)
//   5 = 1/20 Chance to make a point (1 of 20 necessary cards)

#define RATING_SCOPA    100
#define RATING_DENARE7  100
#define RATING_DENARE    20
#define RATING_PRIMERA7  33
#define RATING_PRIMERA6   0
#define RATING_PRIMERA1   0
#define RATING_CARD       5

#include <iostream>

JudgePlayer::JudgePlayer()
{
  myCards.reserve(40);
  yourCards.reserve(40);
}

void JudgePlayer::Initialize()
{
  Player::Initialize();

  myCards.clear();
  yourCards.clear();
}

int JudgePlayer::GetRating(const SolutionSet& solution) const
{
  size_t rating=0;

  if (solution.IsTrick()) {
    if (solution.scopa) {
      rating+=RATING_SCOPA;
    }

    if (solution.yours.GetColor()==Card::denare && solution.yours.GetValue()==7) {
      rating+=RATING_DENARE7;
    }

    for (size_t i=0; i<solution.table.size(); i++) {
      if (solution.table[i].GetColor()==Card::denare && solution.table[i].GetValue()==7) {
        rating+=RATING_DENARE7;
      }
    }

    if (solution.yours.GetColor()==Card::denare) {
      rating+=RATING_DENARE;
    }

    for (size_t i=0; i<solution.table.size(); i++) {
      if (solution.table[i].GetColor()==Card::denare) {
        rating+=RATING_DENARE;
      }
    }

    if (solution.yours.GetValue()==7) {
      rating+=RATING_PRIMERA7;
    }
    else if (solution.yours.GetValue()==6) {
      rating+=RATING_PRIMERA6;
    }
    else if (solution.yours.GetValue()==1) {
      rating+=RATING_PRIMERA1;
    }

    for (size_t i=0; i<solution.table.size(); i++) {
      if (solution.table[i].GetValue()==7) {
        rating+=RATING_PRIMERA7;
      }
      else if (solution.table[i].GetValue()==6) {
        rating+=RATING_PRIMERA6;
      }
      else if (solution.table[i].GetValue()==1) {
        rating+=RATING_PRIMERA1;
      }
    }

    rating+=(solution.table.size()+1)*RATING_CARD;
  }

  return rating;
}

int JudgePlayer::GetRatingForSolution(const SolutionSet& solution,
                                      const std::vector<Card>& table,
                                      const std::vector<std::vector<size_t> >& combinations,
                                      const std::map<size_t,std::vector<size_t> >& points,
                                      const std::vector<Card>& hand,
                                      const std::vector<Card>& myCards,
                                      const std::vector<Card>& yourCards,
                                      size_t recursion,
                                      size_t maxRecursion,
                                      bool isme) const
{
  int tmpRating;

  // Calculate rating for this solution

  tmpRating=GetRating(solution);

  //std::cout << recursion << " rating for playing " << Lum::Base::WStringToString(solution.yours.GetName()) << ": " << tmpRating << std::endl;

  if (recursion==maxRecursion) {
    return tmpRating;
  }

  // Calculate new table as a result of playing this solution

  std::vector<Card>                                     newTable(table);
  std::map<size_t,std::vector<size_t> >::const_iterator exchange;

  exchange=points.find(solution.yours.GetValue());
  if (exchange!=points.end()) {
    size_t idx=exchange->second[solution.solution];

    for (int i=combinations[idx].size()-1; i>=0; --i) {
      newTable.erase(newTable.begin()+combinations[idx][i]-1);
    }
  }
  else {
    newTable.push_back(solution.yours);
  }

  /*
  std::cout << recursion << " cards on table:" << std::endl;
  for (size_t i=0; i<newTable.size(); i++) {
    std::cout << recursion << " * " << Lum::Base::WStringToString(newTable[i].GetName()) << std::endl;
  }*/

  std::vector<Card> newHand;

  if (isme) {
    // Now calculate the set of cards that the opponent possibly could have...
    // Take all cards and substract...
    // ...all cards already played
    // ...all cards the current player holds in his hand
    // ...all cards on the table

    std::set<Card> possibleCards;

    // All cards....
    for (Card card=Card::minCard; card<=Card::maxCard; ++card) {
      possibleCards.insert(card);
    }

    // ..all cards already played
    for (size_t i=0; i<myCards.size(); i++) {
      possibleCards.erase(myCards[i]);
    }
    for (size_t i=0; i<yourCards.size(); i++) {
      possibleCards.erase(yourCards[i]);
    }

    // ...all cards the current player holds in his hand
    for (size_t i=0; i<hand.size(); i++) {
      possibleCards.erase(hand[i]);
    }

    // ...all cards on the table
    for (size_t i=0; i<table.size(); i++) {
      possibleCards.erase(table[i]);
    }

    for (std::set<Card>::const_iterator iter=possibleCards.begin(); iter!=possibleCards.end(); ++iter) {
      newHand.push_back(*iter);
    }
  }
  else {
    // Calculate the cards we have in hand
    newHand=hand;
  }

  std::vector<std::vector<size_t> >     newCombinations;
  std::map<size_t,std::vector<size_t> > newPoints;
  std::vector<SolutionSet>              newSolutions;
  int                                   maxRating=0;
  int                                   averageRating=0;


  Table::Calculate(newTable,newCombinations,newPoints);
  Table::GetSolutionSets(newHand,newTable,newCombinations,newPoints,newSolutions);

  for (size_t t=0; t<newSolutions.size(); t++) {
    std::vector<Card> newCards;
    int               tmp2Rating;

    newCards=myCards;

    // Recalculate my new cards based on this solution and recalculate goals
    // if cards got gained.
    if (newSolutions[t].IsTrick()) {
      newCards.push_back(newSolutions[t].yours);

      for (size_t i=0; i<newSolutions[t].table.size(); i++) {
        newCards.push_back(newSolutions[t].table[i]);
      }
    }

    // Calculate rating for each possible solution

    if (isme) {
      tmp2Rating=GetRatingForSolution(newSolutions[t],
                                      newTable,
                                      newCombinations,
                                      newPoints,
                                      hand,
                                      newCards,
                                      yourCards,
                                      recursion+1,
                                      maxRecursion,
                                      !isme);
    }
    else {
      newHand=hand;

      newHand.erase(newHand.begin()+newSolutions[t].yoursPos);

      tmp2Rating=GetRatingForSolution(newSolutions[t],
                                      newTable,
                                      newCombinations,
                                      newPoints,
                                      hand,
                                      myCards,
                                      newCards,
                                      recursion+1,
                                      maxRecursion,
                                      !isme);
    }


    tmp2Rating=GetRating(newSolutions[t]);

    if (tmp2Rating>maxRating) {
      maxRating=tmp2Rating;
    }

    averageRating+=tmp2Rating;
  }

  if (newSolutions.size()>0) {
    averageRating=(averageRating+newSolutions.size()/2)/(int)newSolutions.size();
  }

  if (isme) {
    //std::cout << recursion << " average rating for other player: " << averageRating << ", " << tmpRating << " => " << tmpRating-averageRating<< std::endl;
    //std::cout << recursion << " max rating for other player: " << maxRating << ", " << tmpRating<< " => " << tmpRating-maxRating << std::endl;
    tmpRating-=averageRating;
  }
  else {
    //std::cout << recursion << " average rating for me: " << averageRating << ", " << tmpRating<< " => " << tmpRating+averageRating<< std::endl;
    //std::cout << recursion << " max rating for me: " << maxRating << ", " << tmpRating<< " => " << tmpRating+maxRating << std::endl;
    tmpRating+=averageRating;
  }

  return tmpRating;
}

SolutionSet JudgePlayer::PlayCard(const Table& table, size_t round)
{
  assert(!hand.empty());

  std::vector<SolutionSet> solutions;
  int                      solution=-1;
  int                      rating=0;

  table.GetSolutionSets(hand,solutions);

  for (size_t s=0; s<solutions.size(); s++) {
    std::vector<Card> newHand(hand);
    int               tmpRating;

    // Calculate rating for each possible solution

    newHand.erase(newHand.begin()+solutions[s].yoursPos);

    tmpRating=GetRatingForSolution(solutions[s],
                                   table.table,
                                   table.combinations,
                                   table.points,
                                   newHand,
                                   myCards,
                                   yourCards,
                                   0,
                                   2,
                                   true);

    //std::cout << "Playing " << Lum::Base::WStringToString(solutions[s].yours.GetName()) << " ==> " << tmpRating << std::endl;

    if (solution<0 || tmpRating>rating) {
      solution=s;
      rating=tmpRating;
    }
  }

  //std::cout << "====> " << rating << std::endl;

  if (solution>=0) {
    return solutions[solution];
  }

  // Hmmm, no clever selection criteria, simply take a random one.
  return solutions[(size_t)(double(solutions.size())*rand()/RAND_MAX)];
}

void JudgePlayer::CardsPlayed(std::vector<Card>& cards,
                              bool ownCards,
                              size_t round)
{
  for (size_t i=0; i<cards.size(); i++) {
    if (ownCards) {
      myCards.push_back(cards[i]);
    }
    else {
      yourCards.push_back(cards[i]);
    }
  }
}

void JudgePlayer::PrintStatistics()
{
  // TODO
}

