 #include "UnicodeString.h"


CUnicodeString::CUnicodeString()
{
	this->m_pCharacters = NULL;
	this->m_nLength = 0;
}

CUnicodeString::~CUnicodeString()
{
	Dispose();
}

void CUnicodeString::Dispose()
{
	if(this->m_pCharacters != NULL)
	{
		delete [] this->m_pCharacters;
		this->m_pCharacters = NULL;
	}
	this->m_nLength = 0;
}

bool CUnicodeString::SetString(wchar_t* in_characters)
{
	if(in_characters == NULL)
	{
		this->Dispose();

		return false;	
	}
	else
	{
		// if this CUnicodeString is not empty, clean it.
		this->Dispose();

		size_t length = ::GetLength(in_characters);

		if(length  < 0) return false;
		if(m_pCharacters != NULL)	return false;

		this->m_pCharacters = new wchar_t[length+1];
		if(this->m_pCharacters == NULL) return false;

		for(size_t i =0; i < length; i++)
		{
			this->m_pCharacters[i] = in_characters[i];
		}
		this->m_pCharacters[length] = 0x00;
		this->m_nLength = length;

		return true;
	}
}

bool CUnicodeString::SetString(CUnicodeString* in_string)
{
	bool result = false;

	if(in_string == NULL)
	{
		return false;
	}

	// if this CUnicodeString is not empty, clean it.
	this->Dispose();

	if(this->m_pCharacters == NULL)
	{
		size_t length = 0;
		size_t buffersize = in_string->GetLength() + 1;


		this->m_pCharacters = new wchar_t[buffersize];
		memset(this->m_pCharacters, 0, sizeof(wchar_t)*buffersize);

		result = in_string->GetString(this->m_pCharacters, buffersize, &length);

		this->m_nLength = length;
	}

	return result;
}

bool CUnicodeString::SetString(UTF32Bytes* in_UTF32String)
{
	if(in_UTF32String == NULL)
	{
		this->Dispose();

		return false;	
	}
	else
	{
		// if this CUnicodeString is not empty, clean it.
		this->Dispose();

		size_t length = ::GetLength(in_UTF32String);

		if(length  < 0) return false;
		if(m_pCharacters != NULL)	return false;

		this->m_pCharacters = new wchar_t[length+1];
		if(this->m_pCharacters == NULL) return false;

		for(size_t i =0; i<length; i++)
		{
			::UTF32BytesTowchar_t(&(this->m_pCharacters[i]), &(in_UTF32String[i]));
		}
		this->m_pCharacters[length] = 0x00;
		this->m_nLength = length;

		return true;
	}
}


// wchar_t* out_buffer : Output buffer of zero ending unicode string. 
//                       The buffer must be bigger than (out_length +1)*sizeof(wchar_t) bytes.
// size_t in_buffer_size : Input buffer size(unit: wchar_t).
//                         The buffer size must be bigger than (out_length +1).
// size_t* out_length : Return length of this string.
bool CUnicodeString::GetString(wchar_t* out_buffer, size_t in_buffer_size, size_t* out_length)
{
	if( (out_buffer == NULL) ||
		(in_buffer_size < (this->m_nLength+1)) )
	{
		if(out_length != NULL)
		{
			*out_length = this->m_nLength;
		}
		return false;
	}

	if(this->m_nLength > 0 && this->m_pCharacters != NULL)
	{
		for(size_t i=0; i<this->m_nLength; i++)
		{
			out_buffer[i] = this->m_pCharacters[i];
		}
		out_buffer[this->m_nLength] = 0x00;

		if(out_length != NULL)
		{
			*out_length = this->m_nLength;
		}
		return true;
	}
	else
	{
		if(in_buffer_size > 0 || out_buffer != NULL)
		{
			out_buffer[0] = 0x00;
		}
		if(out_length != NULL)
		{
			*out_length = 0;
		}
		return false;
	}
}


bool CUnicodeString::GetWChar(size_t in_index, wchar_t* out_character)
{
	if( (out_character == NULL) ||
		(in_index < 0 || in_index >= this->m_nLength) ||
		(this->m_pCharacters == NULL) )
	{
		return false;
	}

	*out_character = this->m_pCharacters[in_index];
	return true;
}


bool CUnicodeString::Equals(CUnicodeString* value, bool ignoreCase)
{
	if(value != NULL)
	{
		return ::Equals(this->m_pCharacters, value->m_pCharacters, ignoreCase);
	}
	else
	{
		return false;
	}
}

bool CUnicodeString::Equals(wchar_t* value, bool ignoreCase)
{
	return ::Equals(this->m_pCharacters, value, ignoreCase);
}


bool CUnicodeString::StartsWith(CUnicodeString* value, bool ignoreCase)
{
	if( value != NULL)
	{
		return ::StartsWith(this->m_pCharacters, value->m_pCharacters, ignoreCase);
	}
	else
	{
		return false;
	}
}

bool CUnicodeString::StartsWith(wchar_t* value, bool ignoreCase)
{
	return ::StartsWith(this->m_pCharacters, value, ignoreCase);
}

int CUnicodeString::CompareTo(CUnicodeString *value, bool ignoreCase)
{
	if( value != NULL)
	{
		return ::CompareTo(this->m_pCharacters, value->m_pCharacters, ignoreCase);
	}
	else
	{
		return 1;
	}
}

int CUnicodeString::CompareTo(wchar_t *value, bool ignoreCase)
{
	return ::CompareTo(this->m_pCharacters, value, ignoreCase);
}

bool CUnicodeString::ConvertToUTF32Bytes(UTF32Bytes* pUTF32String, size_t uBufferSize)
{
	if(pUTF32String == NULL) return false;
	if(uBufferSize <= this->m_nLength) return false;

	if(this->m_pCharacters != NULL && this->m_nLength > 0)
	{
		memset(pUTF32String, 0, sizeof(UTF32Bytes)*uBufferSize);

		for(size_t i=0; i<this->m_nLength; i++)
		{
			wchar_tToUTF32Bytes(&(pUTF32String[i]), &(this->m_pCharacters[i]));
		}

		return true;

	}
	else
	{
#ifdef OUTPUT_DEBUG_STRING
		OutputDebugString(L"CUnicodeString is empty.\r\n");
#endif
	}

	return false;

}


void CUnicodeString::ShowDebugString()
{
	if(this->m_pCharacters != NULL)
	{    
		OutputDebugString(this->m_pCharacters);
    }
}

