/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        mob.cpp ( Live Once Library, C++ )
 *
 * COMMENTS:
 */

#include "mob.h"

#include "map.h"
#include "msg.h"
#include "text.h"
#include "speed.h"
#include "item.h"
#include "gfxengine.h"
#include "score.h"
#include "avatar.h"

#include <stdio.h>
#include <stdlib.h>
#include <iostream>
using namespace std;

//
// MOB Implementation
//

MOB *MOB::theAvatar = 0;

DPDF MOB::theAvatarHP_DPDF(0);

MOB::MOB()
{
    myX = myY = -1;
    myTX = myTY = -1;
    myNext = 0;
    myMap = 0;
    myHX = myHY = -1;
    myFleeCount = 0;
    myBoredom = 0;
    myAIState = 0;
    myHP = 0;
    myInventory = 0;
    myIsSwallowed = false;
}

MOB::~MOB()
{
    if (this == theAvatar)
	setAvatar(0);
	
    delete myNext;
    delete myInventory;
}

MOB *
MOB::create(MOB_NAMES def)
{
    MOB		*mob;

    mob = new MOB();

    mob->myDefinition = def;

    mob->myHP = glb_mobdefs[def].max_hp;

    return mob;
}

MOB *
MOB::copy() const
{
    MOB		*mob;

    mob = new MOB();
    
    *mob = *this;

    // There are some exceptions...
    mob->myMap = 0;
    mob->myNext = 0;

    // Copy inventory one item at a time.
    // We are careful to maintain the same list order here so restoring
    // won't shuffle things unexpectadly
    ITEM	*item, *dup, *last = 0;

    mob->myInventory = 0;
    for (item = myInventory; item; item = item->getNext())
    {
	dup = item->copy();
	if (last)
	    last->setNext(dup);
	else
	    mob->myInventory = dup;
	last = dup;
    }
    
    return mob;
}

MOB *
MOB::createNPC(int depth)
{
    int		i;
    MOB_NAMES	mob = MOB_NONE;
    int		choice = 0;

    // Given the letter choice, choose a mob that matches it appropriately.
    choice = 0;
    for (i = 0; i < NUM_MOBS; i++)
    {
	// Stuff with 0 depth is never created manually.
	if (!glb_mobdefs[i].depth)
	    continue;

	// Use the baseletter to bias the creation.
	if (glb_mobdefs[i].depth <= depth)
	{
	    if (rand_choice(choice + glb_mobdefs[i].rarity) < glb_mobdefs[i].rarity)
		mob = (MOB_NAMES) i;
	    choice += glb_mobdefs[i].rarity;
	}
    }

    // Testing..
    // mob = MOB_FROG;

    return MOB::create(mob);
}

void
MOB::getLook(u8 &symbol, ATTR_NAMES &attr) const
{
    symbol = glb_mobdefs[myDefinition].symbol;
    attr = (ATTR_NAMES) glb_mobdefs[myDefinition].attr;
}

const char *
MOB::getName() const
{
    if (isAvatar())
	return "you";
    return glb_mobdefs[myDefinition].name;
}

const char *
MOB::getRawName() const
{
    return glb_mobdefs[myDefinition].name;
}

ITEM *
MOB::getRandomItem() const
{
    ITEM	*i, *result;
    int		choice = 0;

    result = 0;
    for (i = myInventory; i; i = i->getNext())
    {
	choice++;
	if (!rand_choice(choice))
	    result = i;
    }
    return result;
}

bool
MOB::hasItem(ITEM_NAMES itemname) const
{
    if (lookupItem(itemname))
	return true;

    return false;
}

ITEM *
MOB::lookupItem(ITEM_NAMES itemname) const
{
    ITEM	*i;

    for (i = myInventory; i; i = i->getNext())
    {
	if (i->getDefinition() == itemname)
	    return i;
    }
    return 0;
}

bool
MOB::canMove(int x, int y, bool checkmob) const
{
    MAP		*map = getMap();

    // Can move anywhere on non-maps.
    if (!map)
	return true;

    if (!map->isPassable(x, y))
    {
	TILE_NAMES	tile;
	tile = map->getTile(x, y);
	if (glb_tiledefs[tile].isphaseable && defn().passwall)
	{
	    // Can move through it...
	}
	else if (glb_tiledefs[tile].isdiggable && defn().candig)
	{
	    // Could dig through it.
	}
	else
	{
	    // Failed...
	    return false;
	}
    }

    if (checkmob && map->getMob(x, y))
	return false;

    return true;
}

void
MOB::move(int x, int y)
{
    // If we have swallowed something, move it too.
    if (getMap() && !isSwallowed())
    {
	PTRLIST<MOB *>	moblist;
	int		i;
	
	getMap()->getAllMobs(moblist, myX, myY);
	for (i = 0; i < moblist.entries(); i++)
	{
	    if (moblist(i)->isSwallowed())
	    {
		moblist(i)->move(x, y);
	    }
	}
    }
    myX = x;
    myY = y;
}

void
MOB::gainHP(int hp)
{
    int		maxhp;

    maxhp = defn().max_hp;
    if (isAvatar())
	maxhp = avatar_maxhp();

    myHP += hp;
    if (myHP > maxhp)
	myHP = maxhp;

    if (isAvatar())
    {
	// Update DPDF.
	// Constant add followed by min.
	getAvatarHP_DPDF() += hp;
	getAvatarHP_DPDF().min(getAvatarHP_DPDF(), DPDF(maxhp));
    }
}

bool
MOB::applyDamage(MOB *src, int hits, const DPDF &dpdf)
{
    // Being hit isn't boring.
    clearBoredom();

    if (hasItem(ITEM_INVULNERABLE))
	return false;

    // If we are the avatar, update our probability
    if (isAvatar())
    {
	MOB::getAvatarHP_DPDF() -= dpdf;
    }

    if (hits >= getHP())
    {
	// Death!
	if (src)
	    msg_format("%S <kill> %O!", src, this);
	else
	    msg_format("%S <be> killed!", this);

	// If there is a source, and they are swallowed,
	// they are released.
	if (src && src->isSwallowed())
	{
	    src->setSwallowed(false);
	}
	
	// If we are the avatar, special stuff happens.
	if (isAvatar())
	{
	    msg_update();

	    gfx_breakKeyRepeat();
	    gfx_clearKeyBuffer();

	    getMap()->fadeToBlack();
	}

	// Drop any stuff we have.
	ITEM		*c, *n;
	for (c = myInventory; c; c = n)
	{
	    n = c->getNext();

	    removeItem(c, true);
	    c->move(getX(), getY());
	    getMap()->addItem(c);
	}
	myInventory = 0;
	
	getMap()->removeMob(this);

	if (src->isAvatar())
	{
	    // Award our score.

	}

	if (getDefinition() == MOB_EVILDEMON)
	{
	    // We have won!
	    avatar_setvictory(true);
	}
	
	delete this;
	return true;
    }

    myHP -= hits;
    return false;
}

VERB_PERSON
MOB::getPerson() const
{
    if (isAvatar())
	return VERB_YOU;

    return VERB_IT;
}

bool
MOB::isFriends(MOB *other) const
{
    if (other == this)
	return true;
    
    if (isAvatar())
    {
	if (other->defn().isfriendly)
	    return true;
	else
	    return false;
    }

    // Only the avatar is evil!
    if (defn().isfriendly)
    {
	return true;
    }
    else
    {
	// Monsters hate the avtar!
	if (other->isAvatar())
	    return false;
    }
    return true;
}

void
MOB::viewDescription() const
{
    char	buf[100];
    int		min, q1, q2, q3, max;

    sprintf(buf, "Name: %s\n", getName());
    msg_report(buf);

    sprintf(buf, "Health: %d\n", getHP());
    msg_report(buf);

    getMeleeDPDF().getQuartile(min, q1, q2, q3, max);
    sprintf(buf, "Melee Weapon: %s (%d..%d..%d..%d..%d)\n",
		defn().melee_name,
		min, q1, q2, q3, max);
    msg_report(buf);

    if (defn().range_valid)
    {
	getRangedDPDF().getQuartile(min, q1, q2, q3, max);
	sprintf(buf, "Ranged Weapon: %s (%d..%d..%d..%d..%d), Range %d\n",
		    defn().range_name,
		    min, q1, q2, q3, max,
		    getRangedRange());
	msg_report(buf);
    }
    // Dump the text.txt if present.
    msg_report(text_lookup("mob", getRawName()));
}

AI_NAMES
MOB::getAI() const
{
    return (AI_NAMES) defn().ai;
}

bool
MOB::actionBump(int dx, int dy)
{
    int		 x, y;
    MOB		*mob;

    x = getX() + dx;
    y = getY() + dy;

    // Stand in place.
    if (!dx && !dy)
	return true;

    // If we are swallowed, we must attack.
    if (isSwallowed())
	return actionMelee(dx, dy);
    
    mob = getMap()->getMob(x, y);
    if (mob)
    {
	// Either kill or chat!
	if (mob->isFriends(this))
	{
	    return actionChat(dx, dy);
	}
	else
	{
	    return actionMelee(dx, dy);
	}
    }

    // No mob, see if we can move that way.
    // We let actionWalk deal with unable to move notification.
    return actionWalk(dx, dy);
}

bool
MOB::actionClimb()
{
    TILE_NAMES		tile;
    int			x, y;

    tile = getMap()->getTile(getX(), getY());

    if (tile == TILE_DOWNSTAIRS)
    {
	// Climbing is interesting.
	myFleeCount = 0;
	clearBoredom();
 
	MAP		*newmap;
	msg_format("%S <climb> down.", this);
	newmap = getMap()->getDownMap();
	getMap()->removeMob(this);

	newmap->findTile(TILE_UPSTAIRS, x, y);
	move(x, y);

	newmap->addMob(this);

	// Take no time as otherwise you can be insta-killed!
	return false;
    }
    else if (tile == TILE_UPSTAIRS)
    {
	if (!getMap()->peekUpMap())
	{
	    // No escapes from the Dungeon Of Death!  But each
	    // person has their own reason..
	    msg_report(text_lookup("exit", getRawName()));
	    return false;
	}
	else
	{
	    // Climbing is interesting.
	    myFleeCount = 0;
	    clearBoredom();
     
	    MAP *newmap;
	    msg_format("%S <climb> up.", this);

	    newmap = getMap()->getUpMap();
	    getMap()->removeMob(this);

	    newmap->findTile(TILE_DOWNSTAIRS, x, y);
	    move(x, y);
	
	    newmap->addMob(this);

	    // Take no time as otherwise you can be insta-killed!
	    return false;
	}
    }
    else if (tile == TILE_FOREST)
    {
	msg_format("%S <climb> a tree... and <fall> down!", this);
	return true;
    }

    msg_format("%S <see> nothing to climb here.", this);

    return false;
}

bool
MOB::actionChat(int dx, int dy)
{
    MOB		*victim;
    int		 x, y;

    x = getX() + dx;
    y = getY() + dy;

    // This should never occur, but to avoid
    // embarassaments...
    if (!isAvatar())
	return false;
    
    victim = getMap()->getMob(x, y);
    if (!victim)
    {
	// Talk to self
	msg_format("%S <talk> to %O!", this, this);
	return false;
    }

    msg_format("%S <chat> with %O:\n", this, victim);

    msg_quote(text_lookup("chat", getRawName(), victim->getRawName()));
    
    return true;
}

bool
MOB::actionMelee(int dx, int dy)
{
    MOB		*victim;
    int		 x, y;

    // If we are swallowed, attack the person on our square.
    if (isSwallowed())
	dx = dy = 0;

    x = getX() + dx;
    y = getY() + dy;
    
    victim = getMap()->getMob(x, y);
    if (!victim)
    {
	// Swing in air!
	msg_format("%S <swing> at empty air!", this);
	return false;
    }

    // Shooting is interesting.
    clearBoredom();
 
    int		damage;
    bool	victimdead;

    damage = getMeleeDamage();

    // attempt to kill victim
    if (damage)
    {
	msg_format("%S %v %O.", this, getMeleeVerb(),
				victim);
    }
    else
    {
	msg_format("%S <miss> %O.", this, victim);
    }

    victimdead = victim->applyDamage(this, damage, getMeleeDPDF());
    
    // Vampires regenerate!
    if (defn().isvampire)
    {
	myHP += damage;
	// If we are avatar, upgrade our dpdf
	// Note this doesn't clip!
	if (isAvatar())
	{
	    MOB::getAvatarHP_DPDF() += damage;
	}
    }

    // Grant our item...
    if (!victimdead)
    {
	ITEM_NAMES	item;

	item = (ITEM_NAMES) defn().melee_item;
	if (item != ITEM_NONE)
	{
	    ITEM	*i;

	    // Exclusive items you should only get one of so they
	    // don't stack.
	    if (!glb_itemdefs[item].exclusive ||
		!victim->hasItem(item))
	    {
		i = ITEM::create(item);
		if (i)
		    victim->addItem(i);
	    }
	}

	// Swallow the victim
	if (defn().swallows)
	{
	    victim->setSwallowed(true);
	    victim->move(getX(), getY());
	}

	// Steal something
	if (defn().isthief)
	{
	    ITEM		*item;
	    
	    // Get a random item from the victim
	    item = victim->getRandomItem();
	    if (item)
	    {
		msg_format("%S <steal> %O.", this, item);
		// Theft successful.
		myFleeCount += 30;
		victim->removeItem(item, true);
		addItem(item);
	    }
	    else
	    {
		// Steal a letter from the avatar.
		if (victim->isAvatar())
		{
		    // Well, there is nothing to steal anymore.
		}
	    }
	}
    }
	
    return true;
}

const char *
MOB::getMeleeVerb() const
{
    ITEM	*w;

    w = lookupItem(ITEM_WEAPON);
    if (!w)
	return defn().melee_verb;

    return glb_weapon_typedefs[w->getWeaponType()].verb;
}

int
MOB::getMeleeDamage() const
{
    DPDF	damage;

    damage = getMeleeDPDF();

    return damage.evaluate();
}

DPDF
MOB::getMeleeDPDF() const
{
    DPDF	dpdf(0);

    ITEM	*weapon;

    weapon = lookupItem(ITEM_WEAPON);
    if (weapon)
    {
	return weapon->getMeleeDPDF();
    }

    dpdf = defn().melee_damage.buildDPDF();

    dpdf *= (defn().melee_chance) / 100.0;

    return dpdf;
}

int
MOB::getRangedDamage() const
{
    DPDF	damage;

    damage = getRangedDPDF();

    return damage.evaluate();
}

DPDF
MOB::getRangedDPDF() const
{
    ITEM	*weapon;

    weapon = lookupItem(ITEM_SPELLBOOK);
    if (weapon)
    {
	return weapon->getRangeDPDF();
    }

    DPDF	dpdf(0);

    dpdf = defn().range_damage.buildDPDF();

    return dpdf;
}

int
MOB::getRangedRange() const
{
    int		range;

    ITEM	*weapon;

    weapon = lookupItem(ITEM_SPELLBOOK);
    if (weapon)
    {
	return weapon->getRangeRange();
    }

    range = defn().range_range;

    return range;
}

void
MOB::getRangedLook(u8 &symbol, ATTR_NAMES &attr) const
{
    symbol = defn().range_symbol;
    attr = (ATTR_NAMES) defn().range_attr;

    // Check out item, if any.
    ITEM		*i;

    i = lookupItem(ITEM_SPELLBOOK);

    if (i)
    {
	symbol = glb_book_typedefs[i->getBookType()].symbol;
	attr = (ATTR_NAMES) glb_book_elementdefs[i->getBookElement()].attr;
    }
}

bool
MOB::actionFire(int dx, int dy)
{
    MOB		*victim;
    
    // Check for no ranged weapon.
    if (!defn().range_valid && !hasItem(ITEM_SPELLBOOK))
    {
	msg_format("%S <have> no ranged weapon!", this);
	return false;
    }

    // No suicide.
    if (!dx && !dy)
    {
	msg_format("%S <decide> not to aim at %O.", this, this);
	return false;
    }
    
    // If swallowed, rather useless.
    if (isSwallowed())
    {
	msg_format("%S do not have enough room inside here.", this);
	return false;
    }

    // Check for sufficient mana.
    if (isAvatar())
    {
	ITEM		*r;

	r = lookupItem(ITEM_SPELLBOOK);
	if (r)
	{
	    if (r->getRangeMana() > avatar_mp())
	    {
		msg_format("%S <lack> sufficent mana.", this);
		return false;
	    }
	}
    }

    // Check for friendly kill.
    victim = getMap()->traceBullet(getX(), getY(), getRangedRange(), dx, dy);

    if (victim && victim->isFriends(this))
    {
	// Not a clear shot!
	if (isAvatar())
	{
	    msg_report(text_lookup("fire", getRawName()));
	    // We have special messages.
	    return false;
	}

	// Avoid friendly fire
	return false;
    }

    // Shooting is interesting.
    clearBoredom();
 
    u8		symbol;
    ATTR_NAMES	attr;
    int		area = 1;
    
    getRangedLook(symbol, attr);

    getMap()->displayBullet(getX(), getY(), 
			getRangedRange(),
			dx, dy,
			symbol, attr);

    if (!victim)
    {
	// Shoot at air!
	msg_format("%S <fire> at empty air!", this);
	return true;
    }

    // Attemp to kill victim
    msg_format("%S %v %O.", this, defn().range_verb,
			    victim);

    // Subtract the mana used.
    if (isAvatar())
    {
	ITEM		*r;

	r = lookupItem(ITEM_SPELLBOOK);
	if (r)
	{
	    avatar_setmp(avatar_mp() - r->getRangeMana());
	    area = r->getRangeArea();
	}
    }

    int		vx, vy;

    vx = victim->getX();
    vy = victim->getY();

    // Apply damage to everyone in range.
    getMap()->fireball(this, vx, vy, area, getRangedDPDF(), symbol, attr); 
    return true;
}

bool
MOB::actionWalk(int dx, int dy)
{
    int		 x, y;
    MOB		*victim;

    // If we are swallowed, we can't move.
    if (isSwallowed())
	return false;
    
    x = getX() + dx;
    y = getY() + dy;
    
    victim = getMap()->getMob(x, y);
    if (victim)
    {
	// If we are the avatar, screw being nice!
	if (isAvatar())
	    return actionMelee(dx, dy);
	// If it is thet avatar we are bumping into, screw being
	// nice.
	if (victim->isAvatar())
	    return actionMelee(dx, dy);

	msg_format("%S <bump> into %O.", this, victim);
	// Bump into the mob.
	return false;
    }

    // Check to see if we can move that way.
    if (canMove(x, y))
    {
	TILE_NAMES	tile;
	
	// Determine if it is a special move..
	tile = getMap()->getTile(x, y);
	if (glb_tiledefs[tile].isdiggable &&
	    !glb_tiledefs[tile].ispassable &&	
	    defn().candig)
	{
	    // Dig it!
	    if (tile == TILE_WALL)
		tile = TILE_BROKENWALL;
	    else
		tile = TILE_FLOOR;
	    
	    getMap()->setTile(x, y, tile);
	    return true;
	}

	int		ox, oy;
	ox = getX();
	oy = getY();
	
	// Move!
	move(x, y);

	// If we are a breeder, chance to reproduce!
	if (defn().breeder)
	{
	    static  int		lastbreedtime = -10;

	    // Only breed every 7 phases at most so one can
	    // always kill off the breeders.
	    if ((lastbreedtime < spd_gettime() - 7) &&
		!rand_choice(50 + getMap()->getNumMobs()))
	    {
		MOB		*child;
		
		msg_format("%S <reproduce>!", this);
		child = MOB::create(getDefinition());
		child->move(ox, oy);
		getMap()->addMob(child);
		lastbreedtime = spd_gettime();
	    }
	}
	
	// If we are the avatar and something is here, pick it up.
	if (isAvatar() && getMap()->getItem(x, y))
	    actionPickup();
	
	return true;
    }
    else
    {
	msg_format("%S <be> blocked by %O.", this, getMap()->getTerrainName(x, y));
	// Bump into a wall.
	return false;
    }
}

bool
MOB::actionPickup()
{
    ITEM		*item;

    assert(isAvatar());

    item = getMap()->getItem(getX(), getY());

    if (!item)
    {
	msg_format("%S <grope> the ground foolishly.", this);
	return false;
    }

    // Pick up the item!
    msg_format("%S <pick> up %O.", this, item);

    getMap()->removeItem(item);

    // Get our current weapon/armour/book
    ITEM		*olditem;

    olditem = lookupItem(item->getDefinition());

    // Some items are instantly used...
    switch (item->getDefinition())
    {
	case ITEM_HEALPOTION:
	{
	    addItem(item);

	    gainHP(5);
	    removeItem(item);
	    delete item;
	    break;
	}

	case ITEM_MANAPOTION:
	{
	    int		manaamt;

	    addItem(item);
    
	    manaamt = avatar_maxmp() - avatar_mp();
	    manaamt = MIN(15, manaamt);
	    avatar_setmp(avatar_mp() + manaamt);
	    removeItem(item);
	    delete item;
	    break;
	}

	case ITEM_WEAPON:
	case ITEM_SPELLBOOK:
	{
	    if (aiLikeMore(item, olditem) == item)
	    {
		// We want the new item...
		if (olditem)
		{
		    removeItem(olditem);
		    delete olditem;
		}
		addItem(item);
	    }
	    else
	    {
		// Keep the old item.
		msg_format("%S <discard> it as junk.", this);
		
		delete item;
	    }
	    break;
	}
	default:
	    addItem(item);
	    break;
    }

    return true;
}

void
MOB::addItem(ITEM *item)
{
    ITEM		*c;
    
    assert(!item->getNext());

    // Alert the world to our acquirement.
    if (isAvatar())
	if (item->defn().gaintxt)
	    msg_format(item->defn().gaintxt,
			this, item);

    // First, check to see if we can merge...
    for (c = myInventory; c; c = c->getNext())
    {
	if (item->canStackWith(c))
	{
	    c->combineItem(item);
	    delete item;
	    return;
	}
    }

    // Brand new item.
    item->setNext(myInventory);
    myInventory = item;
}

void
MOB::removeItem(ITEM *item, bool quiet)
{
    ITEM		*c, *l;

    // Alert the world to our acquirement.
    if (isAvatar() && !quiet)
	if (item->defn().losetxt)
	    msg_format(item->defn().losetxt,
			this, item);

    l = 0;
    for (c = myInventory; c; c = c->getNext())
    {
	if (c == item)
	{
	    if (l)
		l->setNext(c->getNext());
	    else
		myInventory = c->getNext();
	    c->setNext(0);
	    return;
	}
	l = c;
    }
    assert(0);
}

void
MOB::save(ostream &os) const
{
    int		val;

    val = myDefinition;
    os.write((const char *) &val, sizeof(int));

    os.write((const char *) &myX, sizeof(int));
    os.write((const char *) &myY, sizeof(int));

    os.write((const char *) &myTX, sizeof(int));
    os.write((const char *) &myTY, sizeof(int));

    os.write((const char *) &myHX, sizeof(int));
    os.write((const char *) &myHY, sizeof(int));

    os.write((const char *) &myHP, sizeof(int));
    os.write((const char *) &myAIState, sizeof(int));
    os.write((const char *) &myFleeCount, sizeof(int));
    os.write((const char *) &myBoredom, sizeof(int));

    val = isSwallowed();
    os.write((const char *) &val, sizeof(int));

    ITEM		*i;
    u8			 c;

    for (i = myInventory; i; i = i->getNext())
    {
	c = 1;
	os.write((const char *) &c, 1);

	i->save(os);
    }
    c = 0;
    os.write((const char *) &c, 1);
}

MOB *
MOB::load(istream &is)
{
    int		 val;
    MOB		*mob;
    u8		 c;

    mob = new MOB();

    is.read((char *)&val, sizeof(int));
    mob->myDefinition = (MOB_NAMES) val;

    is.read((char *)&mob->myX, sizeof(int));
    is.read((char *)&mob->myY, sizeof(int));

    is.read((char *)&mob->myTX, sizeof(int));
    is.read((char *)&mob->myTY, sizeof(int));

    is.read((char *)&mob->myHX, sizeof(int));
    is.read((char *)&mob->myHY, sizeof(int));

    is.read((char *)&mob->myHP, sizeof(int));
    is.read((char *)&mob->myAIState, sizeof(int));
    is.read((char *)&mob->myFleeCount, sizeof(int));
    is.read((char *)&mob->myBoredom, sizeof(int));

    is.read((char *)&val, sizeof(int));
    mob->setSwallowed(val ? true : false);

    ITEM		*i;

    while (1)
    {
	is.read((char *) &c, 1);
	if (!c)
	    break;

	i = ITEM::load(is);
	mob->addItem(i);
    }

    return mob;
}

bool
MOB::hasVisibleEnemies() const
{
    PTRLIST<MOB *> list;

    getVisibleEnemies(list);
    if (list.entries())
	return true;
    return false;
}

void
MOB::getVisibleEnemies(PTRLIST<MOB *> &list) const
{
    MAP		*map = getMap();
    MOB		*mob;

    // If we are blind, we can't see anyone.
    if (hasItem(ITEM_BLIND))
	return;

    for (mob = map->getMobHead(); mob; mob = mob->getNext())
    {
	// Ignore self
	if (mob == this)
	    continue;

	// Ignore out of fov
	if (!map->isFOV(mob->getX(), mob->getY()))
	    continue;

	// Ignore none hostile
	if (isFriends(mob))
	    continue;

	// Append.
	list.append(mob);
    }
}
