/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        map.cpp ( Live Once Library, C++ )
 *
 * COMMENTS:
 */

#include "map.h"
#include "gfxengine.h"
#include "mob.h"
#include "msg.h"
#include "item.h"
#include "text.h"

#include <assert.h>

#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <fstream>
using namespace std;

#define CODE_TOP 0
#define CODE_RIGHT 1
#define CODE_BOT 2
#define CODE_LEFT 3

#define SOLID_CODE 14

#define MAPFLAG_FOV 1
#define MAPFLAG_MAP 2

//
// Stores the legend information
//
class LEGEND
{
public:
	     LEGEND(u8 symbol, const char *descr);
    virtual ~LEGEND();

    const char	*getDescr() const { return myDescr; }
    u8		 getSymbol() const { return mySymbol; }

protected:
    u8		 mySymbol;
    char	*myDescr;
};

LEGEND::LEGEND(u8 symbol, const char *descr)
{
    // Strip leading spaces.
    while (ISSPACE(*descr))
	descr++;

    mySymbol = symbol;
    myDescr = strdup(descr);
}

LEGEND::~LEGEND()
{
    free(myDescr);
}

//
// Each of these is a 5x5 tile piece.
//
class PIECE
{
public:
    PIECE(u8 map[5][5], int sym);
    ~PIECE();

    bool	doesMatch(int top, int right, int bot, int left) const;
    int		getCode(int side) const;
    bool	codeSet(int x, int y) const;

    void	writeToMap(MAP *map, int mx, int my) const;

    PIECE	*getNext() const { return myNext; }
    void	setNext(PIECE *piece) { myNext = piece; }
    
protected:
    u8		myMap[5][5];

    PIECE	*myNext;
};

PIECE::PIECE(u8 map[5][5], int sym)
{
    int		x, y, rx, ry, t;

    for (x = 0; x < 5; x++)
    {
	for (y = 0; y < 5; y++)
	{
	    if (sym & 1)
		rx = x;
	    else
		rx = 4 - x;
	    if (sym & 2)
		ry = y;
	    else
		ry = 4 - y;
	    if (sym & 4)
	    {
		t = rx;
		rx = ry;
		ry = t;
	    }

	    myMap[rx][ry] = map[x][y];
	}
    }
	
    myNext = 0;
}

PIECE::~PIECE()
{
    delete myNext;
}

bool
PIECE::doesMatch(int top, int right, int bot, int left) const
{
    if (top != -1 && getCode(CODE_TOP) != top)
	return false;
    if (right != -1 && getCode(CODE_RIGHT) != right)
	return false;
    if (bot != -1 && getCode(CODE_BOT) != bot)
	return false;
    if (left != -1 && getCode(CODE_LEFT) != left)
	return false;

    return true;
}

int
PIECE::getCode(int side) const
{
    int		code = 0;
    int		x, y;

    switch (side)
    {
	case CODE_TOP:
	    y = 0;
	    for (x = 0; x < 5; x++)
	    {
		code <<= 1;
		if (codeSet(x, y))
		    code |= 1;
	    }
	    break;
	case CODE_RIGHT:
	    x = 4;
	    for (y = 0; y < 5; y++)
	    {
		code <<= 1;
		if (codeSet(x, y))
		    code |= 1;
	    }
	    break;
	case CODE_BOT:
	    y = 4;
	    for (x = 0; x < 5; x++)
	    {
		code <<= 1;
		if (codeSet(x, y))
		    code |= 1;
	    }
	    break;
	case CODE_LEFT:
	    x = 0;
	    for (y = 0; y < 5; y++)
	    {
		code <<= 1;
		if (codeSet(x, y))
		    code |= 1;
	    }
	    break;
	default:
	    assert(0);
	    break;
    }

    // We cannot care about corners!
    code &= SOLID_CODE;

    return code;
}

bool
PIECE::codeSet(int x, int y) const
{
    if (myMap[y][x] == '#')
	return true;

    return false;
}

void
PIECE::writeToMap(MAP *map, int mx, int my) const
{
    int		x, y;

    for (y = 0; y < 5; y++)
    {
	for (x = 0; x < 5; x++)
	{
	    if (myMap[y][x] == '#')
		map->setTile(mx+x, my+y, TILE_WALL);
	    else
		map->setTile(mx+x, my+y, TILE_FLOOR);
	}
    }
}

class PIECEDICT
{
public:
    PIECEDICT();
    ~PIECEDICT();
    
    PIECE	*getRandomPiece(int top, int right, int bot, int left);
    void	 append(PIECE *piece);

protected:
    PIECE	*myPieces;
};

PIECEDICT::PIECEDICT()
{
    char		buf[100];
    ifstream		is("../rooms/piecelist.map");
    u8			map[5][5];
    int			x, y, i;

    myPieces = 0;

    while (is.getline(buf, 100))
    {
	text_striplf(buf);
	if (buf[0] == '+')
	{
	    y = 0;
	    while (is.getline(buf, 100))
	    {
		text_striplf(buf);
		if (buf[0] == '+')
		    break;
		if (y < 5 && buf[0] == '|')
		{
		    for (x = 0; x < 5; x++)
		    {
			if (!buf[x+1])
			    break;
			map[y][x] = buf[x+1];
		    }
		    y++;
		}
	    }

	    // Got a new map.  Build the pieces.
	    for (i = 0; i < 8; i++)
	    {
		append(new PIECE(map, i));
	    }
	}
    }
}

PIECEDICT::~PIECEDICT()
{
    delete myPieces;
}

PIECE *
PIECEDICT::getRandomPiece(int top, int right, int bot, int left)
{
    int		 nummatch = 0;
    PIECE	*cur = 0, *match = 0;
    
    for (cur = myPieces; cur; cur = cur->getNext())
    {
	if (cur->doesMatch(top, right, bot, left))
	{
	    nummatch++;

	    if (!rand_choice(nummatch))
		match = cur;
	}	
    }

    return match;
}

void
PIECEDICT::append(PIECE *piece)
{
    piece->setNext(myPieces);
    myPieces = piece;
}

//
// MAP implementation
//
MAP::MAP()
{
    myWidth = myHeight = 0;
    myMobs = 0;
    myItems = 0;
    myDistMap = 0;
    myDistMapX = myDistMapY = -1;

    myDownMap = myUpMap = 0;
    myMobSafeNext = 0;
    myTotalRooms = 0;

    // Ensure our default lastx/lasty are reasonable.
    int		sx, sy, cx, cy, w, h;

    // Stolen from our basic redraw.
    sx = sy = 1;
    w = 35;
    h = 25;
    cx = 17;
    cy = 12;
    
    cx = cx - w / 2;
    cy = cy - h / 2;

    myLastX = sx;
    myLastY = sy;
    myLastMX = cx;
    myLastMY = cy;
    myLastW = w;
    myLastH = h;
}

MAP::~MAP()
{
    delete [] myDistMap;

    delete myMobs;
    delete myItems;

    // Delete any up and down linked maps.
    if (myDownMap)
    {
	myDownMap->myUpMap = 0;
	delete myDownMap;
    }
    
    if (myUpMap)
    {
	myUpMap->myDownMap = 0;
	delete myUpMap;
    }
}

int
MAP::dist(int x1, int y1, int x2, int y2)
{
    int		dx, dy;

    dx = x2 - x1;
    dy = y2 - y1;

    return MAX(ABS(dx), ABS(dy));
}

int
MAP::dist(const MOB *a, const MOB *b)
{
    // Less typing, more function calls :>
    return dist(a, b->getX(), b->getY());
}

int
MAP::dist(const MOB *a, int x, int y)
{
    return dist(a->getX(), a->getY(), x, y);
}

bool
MAP::isLinedUp(int x1, int y1, int x2, int y2)
{
    int		dx, dy;

    dx = x2 - x1;
    dy = y2 - y1;

    // Check ortho...
    if (!dx || !dy)
	return true;

    // Check diagonal.
    if (ABS(dx) == ABS(dy))
	return true;

    return false;
}

bool
MAP::isLinedUp(const MOB *a, const MOB *b)
{
    return isLinedUp(a, b->getX(), b->getY());
}

bool
MAP::isLinedUp(const MOB *a, int x, int y)
{
    return isLinedUp(a->getX(), a->getY(), x, y);
}

MAP *
MAP::copyAll() const
{
    MAP		*map, *peek, *dup, *lastdup;

    map = copy();
    lastdup = map;
    // Copy upwards.
    for (peek = peekUpMap(); peek; peek = peek->peekUpMap())
    {
	dup = peek->copy();
	dup->setDownMap(lastdup);
	lastdup->setUpMap(dup);
	lastdup = dup;
    }

    // Copy downwards
    lastdup = map;
    for (peek = peekDownMap(); peek; peek = peek->peekDownMap())
    {
	dup = peek->copy();
	dup->setUpMap(lastdup);
	lastdup->setDownMap(dup);
	lastdup = dup;
    }

    // REturn ourselves.
    return map;
}

MAP *
MAP::copy() const
{
    MAP		*map;

    map = new MAP();
    *map = *this;

    map->myUpMap = 0;
    map->myDownMap = 0;

    // Clear out volatile bits.
    map->myDistMap = 0;

    // Copy the mobs, being careful to keep the same order.
    MOB		*mob, *dupmob, *lastmob;

    map->myMobs = 0;
    map->myMobSafeNext = 0;
    lastmob = 0;
    for (mob = getMobHead(); mob; mob = mob->getNext())
    {
	dupmob = mob->copy();
	dupmob->setMap(map);
	if (lastmob)
	    lastmob->setNext(dupmob);
	else
	    map->myMobs = dupmob;
	lastmob = dupmob;
    }

    // Ditto for items on the ground.
    ITEM	*item, *dup, *last = 0;

    map->myItems = 0;
    for (item = myItems; item; item = item->getNext())
    {
	dup = item->copy();
	if (last)
	    last->setNext(dup);
	else
	    map->myItems = dup;
	last = dup;
    }

    return map;
}

void
MAP::locateAvatar(MOB_NAMES avatarname)
{
    MAP		*map;
    
    MOB::setAvatar(0);
    
    for (map = this; map; map = map->peekDownMap())
    {
	if (MOB::getAvatar())
	    break;
	map->locateAvatarLocal(avatarname);
    }

    for (map = peekUpMap(); map; map = map->peekUpMap())
    {
	if (MOB::getAvatar())
	    break;
	map->locateAvatarLocal(avatarname);
    }
}

void
MAP::locateAvatarLocal(MOB_NAMES avatarname)
{
    MOB		*mob;
    for (mob = getMobHead(); mob; mob = mob->getNext())
    {
	if (mob->getDefinition() == avatarname)
	{
	    MOB::setAvatar(mob);
	    break;
	}
    }
}

void
MAP::loadMap(const char *fname)
{
    char		line[100];

    ifstream		is(fname);
    PTRLIST<LEGEND *>	legendlist;
    PTRLIST<char *>	linelist;
    int			w = 0, h = 0, i;
    int			x, y;
    u8			symbol;

    while (is.getline(line, 100))
    {
	text_striplf(line);
	// Comments.
	if (line[0] == '#' && line[1] == ' ')
	    continue;

	// Legends:
	if (line[0] && line[1] == ':')
	{
	    legendlist.append(new LEGEND(line[0], &line[2]));
	}

	if (line[0] == '+' && line[1] == '-')
	{
	    // We have start of map proper.
	    for (w = 1; line[w] == '-'; w++);

	    if (line[w] != '+')
	    {
		cerr << "Map does not have proper bounded map." << endl;
	    }

	    w--;

	    // Load the actual map lines.
	    while (is.getline(line, 100))
	    {
		text_striplf(line);
		if (line[0] == '+')
		    break;

		if (line[0] != '|')
		{
		    cerr << "Map line missing | prefix." << endl;
		}
		if (line[w+1] != '|')
		{
		    cerr << "Map line too short!" << endl;
		}
		line[w+1] = '\0';
		linelist.append(strdup(&line[1]));
	    }

	    h = linelist.entries();
	}
    }

    if (!w)
    {
	// No map found, abort.
	cerr << "No map segment found!" << endl;
	gfx_shutdown();
	exit(0);
    }

    // Build our map.
    myWidth = w;
    myHeight = h;

    myTiles = MAPDATA(myWidth * myHeight);
    myTileFlags = MAPDATA(myWidth * myHeight);
    memset(myTileFlags.writeData(), 0, myWidth * myHeight);

    for (y = 0; y < myHeight; y++)
    {
	for (x = 0; x < myWidth; x++)
	{
	    symbol = linelist(y)[x];

	    parseLegend(x, y, legendlist, symbol);
	}
    }

    // Delete line list.
    for (y = 0; y < myHeight; y++)
    {
	free(linelist(y));
    }

    // Delete legend.
    for (i = 0; i < legendlist.entries(); i++)
	delete legendlist(i);
}

void
MAP::parseLegend(int x, int y, const PTRLIST<LEGEND *> &list,
		u8 symbol)
{
    // Find the legend.
    int			 i, j;
    const char		*s, *comma;
    char		*token;
    bool		 foundsymbol = false;

    for (i = 0; i < list.entries(); i++)
    {
	if (list(i)->getSymbol() == symbol)
	{
	    s = list(i)->getDescr();

	    foundsymbol = true;
	    
	    while (*s)
	    {
		for (comma = s; *comma && *comma != ','; comma++)
		{
		    if (*comma == '(')
		    {
			while (*comma && *comma != ')')
			{
			    comma++;
			}
		    }
		}
	    
		token = strdup(s);
		token[comma - s] = '\0';

		// We wish to apply token to this dude.
		if (token[0] == '(')
		{
		    // We have inheritance!
		    if (token[1] == symbol)
		    {
			cerr << "Naive infinite recursion: Try harder!" << endl;
			return;
		    }
		    parseLegend(x, y, list, token[1]);
		}
		else
		{
		    bool	madewish = false;
		    // Run through our wish engine...
		    for (j = 0; j < NUM_TILES; j++)
		    {
			if (!strcmp(glb_tiledefs[j].legend, token))
			{
			    madewish = true;
			    setTile(x, y, (TILE_NAMES) j);
			}
		    }

		    for (j = 0; j < NUM_MOBS; j++)
		    {
			if (!strcmp(glb_mobdefs[j].name, token))
			{
			    madewish = true;
			    MOB		*mob;

			    mob = MOB::create((MOB_NAMES) j);
			    mob->move(x, y);
			    addMob(mob);
			}
		    }

		    assert(madewish);
		}

		free(token);

		// Find next word.
		s = comma;
		while (*s == ',' || ISSPACE(*s))
		    s++;
	    }
	}
    }

    if (!foundsymbol)
    {
	cerr << "Did not find symbol " << symbol << endl;
    }
}

MAP *
MAP::getDownMap() const
{
    if (!myDownMap)
    {
	myDownMap = new MAP();
	myDownMap->myUpMap = (MAP *) this;
	myDownMap->build(myTotalRooms);
    }
    return myDownMap;
}

MAP *
MAP::getUpMap() const
{
    if (!myUpMap)
    {
	myUpMap = new MAP();
	myUpMap->myDownMap = (MAP *) this;
	myUpMap->build(1);
    }
    return myUpMap;
}

void
MAP::setDownMap(MAP *down)
{
    assert(!myDownMap);

    myDownMap = down;
}

void
MAP::setUpMap(MAP *up)
{
    assert(!myUpMap);

    myUpMap = up;
}

bool
MAP::findTile(TILE_NAMES tile, int &mx, int &my, bool avoidmob) const
{
    int		nummatch = 0;
    int		x, y;
    
    for (y = 0; y < myHeight; y++)
    {
	for (x = 0; x < myWidth; x++)
	{
	    if (getTile(x, y) == tile)
	    {
		if (avoidmob && getMob(x, y))
		    continue;

		nummatch++;
		if (!rand_choice(nummatch))
		{
		    mx = x;
		    my = y;
		}
	    }
	}
    }

    if (nummatch)
	return true;
    return false;
}
    
bool
MAP::findMoveable(int &mx, int &my, bool avoidmob) const
{
    int		nummatch = 0;
    int		x, y;
    
    for (y = 0; y < myHeight; y++)
    {
	for (x = 0; x < myWidth; x++)
	{
	    if (isPassable(x, y))
	    {
		if (avoidmob && getMob(x, y))
		    continue;

		nummatch++;
		if (!rand_choice(nummatch))
		{
		    mx = x;
		    my = y;
		}
	    }
	}
    }

    if (nummatch)
	return true;
    return false;
}
bool
MAP::findMoveableUnexplored(int &mx, int &my) const
{
    int		nummatch = 0;
    int		x, y;
    
    for (y = 0; y < myHeight; y++)
    {
	for (x = 0; x < myWidth; x++)
	{
	    if (isPassable(x, y) && !isMapped(x, y))
	    {
		nummatch++;
		if (!rand_choice(nummatch))
		{
		    mx = x;
		    my = y;
		}
	    }
	}
    }

    if (nummatch)
	return true;
    return false;
}

bool
MAP::findCloseMoveableUnexplored(int &mx, int &my, int cx, int cy) const
{
    int		nummatch = 0;
    int		x, y;
    int		mindist = 1000, d;
    
    for (y = 0; y < myHeight; y++)
    {
	for (x = 0; x < myWidth; x++)
	{
	    if (isPassable(x, y) && !isMapped(x, y))
	    {
		// Verify we are closer.
		// If we are equal, increase matches
		// If we are closer, reset matches.
		d = dist(x, y, cx, cy);
		if (d < mindist)
		{
		    nummatch = 1;
		    mindist = d;
		}
		else if (d == mindist)
		{
		    nummatch++;
		}
		else
		    continue;
			
		if (!rand_choice(nummatch))
		{
		    mx = x;
		    my = y;
		}
	    }
	}
    }

    if (nummatch)
	return true;
    return false;
}

bool
MAP::findNextSymbol(char symbol, int &mx, int &my) const
{
    int		nummatch = 0;
    int		x, y;
    TILE_NAMES	tile;
    MOB		*mob;
    ITEM	*item;
    bool	isblind;
    MOB		*avatar;

    avatar = MOB::getAvatar();
    isblind = false;
    if (avatar)
    {
	isblind = avatar->hasItem(ITEM_BLIND);
	if (avatar->isSwallowed())
	    isblind = true;
    }
    	

    x = mx;
    y = my;
    while (1)
    {
	// Increment position
	x++;
	if (x >= getWidth())
	{
	    x = 0;
	    y++;
	    if (y >= getHeight())
		y = 0;
	}
	
	// Return failure if back to start.
	if (x == mx && y == my)
	    return false;

	// See if we have the right symbol
	tile = getTile(x, y);
	if (glb_tiledefs[tile].symbol == symbol)
	{
	    mx = x;
	    my = y;
	    return true;
	}

	// See if the monster is correct
	mob = getMob(x, y);
	if (!isblind && mob && mob->defn().symbol == symbol)
	{
	    // Don't see things out of los if hide los is true.
	    if (!mob->defn().hidefromlos || isFOV(x, y))
	    {
		mx = x;
		my = y;
		return true;
	    }
	}    

	// See if the monster is correct
	item = getItem(x, y);
	if (!isblind && item && item->defn().symbol == symbol)
	{
	    mx = x;
	    my = y;
	    return true;
	}    
    }

    // Never reached.
    return false;
}

void
MAP::build(int roomssofar)
{
    // Build the world...
    int		rooms = 0;

    myWidth = 35;
    myHeight = 25;
    myTiles = MAPDATA(myWidth * myHeight);
    myTileFlags = MAPDATA(myWidth * myHeight);
    myDistMap = 0;

    // Clear ourselves to wall.
    clear(TILE_WALL);
    clearFlag(MAPFLAG_MAP, false);
    clearFlag(MAPFLAG_FOV, false);

    PIECE		*piece[5][7];
    int			x, y, i;
    bool		newpiece;
    
    for (y = 0; y < 5; y++)
	for (x = 0; x < 7; x++)
	    piece[y][x] = 0;

    PIECEDICT		piecedict;
    int			top, right, bot, left;

    // Initialize center & grow...
    piece[2][2] = piecedict.getRandomPiece(-1, -1, -1, -1);
    
    newpiece = true;
    while (newpiece)
    {
	newpiece = false;
	for (y = 0; y < 5; y++)
	{
	    for (x = 0; x < 7; x++)
	    {
		if (piece[y][x])
		    continue;

		// See if any point continuing.
		if ((x && piece[y][x-1] && piece[y][x-1]->getCode(CODE_RIGHT) != SOLID_CODE) ||
		    (x < 6 && piece[y][x+1] && piece[y][x+1]->getCode(CODE_LEFT) != SOLID_CODE) ||	
		    (y && piece[y-1][x] && piece[y-1][x]->getCode(CODE_BOT) != SOLID_CODE) ||	
		    (y < 4 && piece[y+1][x] && piece[y+1][x]->getCode(CODE_TOP) != SOLID_CODE))
		{
		    // Get our codes.
		    if (x)
		    {
			if (piece[y][x-1])
			    left = piece[y][x-1]->getCode(CODE_RIGHT);
			else
			    left = -1;
		    }
		    else
			left = SOLID_CODE;
		    if (x < 6)
		    {
			if (piece[y][x+1])
			    right = piece[y][x+1]->getCode(CODE_LEFT);
			else
			    right = -1;
		    }
		    else
			right = SOLID_CODE;
		    if (y)
		    {
			if (piece[y-1][x])
			    top = piece[y-1][x]->getCode(CODE_BOT);
			else
			    top = -1;
		    }
		    else
			top = SOLID_CODE;
		    if (y < 4)
		    {
			if (piece[y+1][x])
			    bot = piece[y+1][x]->getCode(CODE_TOP);
			else
			    bot = -1;
		    }
		    else
			bot = SOLID_CODE;

		    piece[y][x] = piecedict.getRandomPiece(top, right, bot, left);
		    // assert(piece[y][x]);
		    if (piece[y][x])
			newpiece = true;
		}
	    }
	}
    }

    // Write out all the pieces.
    for (y = 0; y < 5; y++)
    {
	for (x = 0; x < 7; x++)
	{
	    if (piece[y][x])
	    {
		rooms++;
		piece[y][x]->writeToMap(this, x*5, y*5);
	    }
	}
    }

    // Force the outer wall to be wall...
    for (x = 0; x < myWidth; x++)
    {
	setTile(x, 0, TILE_WALL);
	setTile(x, myHeight-1, TILE_WALL);
    }
    for (y = 0; y < myHeight; y++)
    {
	setTile(0, y, TILE_WALL);
	setTile(myWidth-1, y, TILE_WALL);
    }

    // Update total room count.
    myTotalRooms = roomssofar + rooms;

    // Make up and down stairs.
    findTile(TILE_FLOOR, x, y);
    setTile(x, y, TILE_UPSTAIRS);

    findTile(TILE_FLOOR, x, y);
    setTile(x, y, TILE_DOWNSTAIRS);

    // Add critters
    for (i = 0; i < (rooms/2); i++)
    {
	MOB		*mob;

	findMoveable(x, y, true);
#if 0
	if (!i)
	    mob = MOB::create(MOB_KLEPTOMANIAC);
	else
#endif
	    mob = MOB::createNPC(getDepth());
	mob->move(x, y);
	addMob(mob);
    }
    // Add the demon for end conditions!
    if (getDepth() >= 20)
    {
	MOB		*mob;

	mob = MOB::create(MOB_EVILDEMON);
	findMoveable(x, y, true);
	mob->move(x, y);
	addMob(mob);
    }

    // Add items.
    for (i = 0; i < (rooms/2); i++)
    {
	ITEM		*item;

	findMoveable(x, y, true);
	item = ITEM::createRandom(getDepth());

	item->move(x, y);
	addItem(item);
    }
}

void
MAP::clear(TILE_NAMES tile)
{
    int	    x, y;

    for (y = 0; y < myHeight; y++)
    {
	for (x = 0; x < myWidth; x++)
	{
	    setTile(x, y, tile);
	}
    }
}

void
MAP::clearFlag(int flag, bool val)
{
    int	    x, y;

    for (y = 0; y < myHeight; y++)
    {
	for (x = 0; x < myWidth; x++)
	{
	    setFlag(x, y, flag, val);
	}
    }
}

void
MAP::buildFOV(int cx, int cy)
{
    // Clear out FOV flag.
    clearFlag(MAPFLAG_FOV, false);

    // Starting with x, y, we spiral outwards so long
    // as we still have valid values.

    int		radius, maxrad;
    bool	hadfov;
    int		i, iminus;

    if (!isValidCoord(cx, cy))
	return;

    // Center is in FOV.
    setFlag(cx, cy, MAPFLAG_FOV|MAPFLAG_MAP, true);

    maxrad = MAX(myWidth, myHeight);
    for (radius = 1; radius < maxrad; radius++)
    {
	hadfov = false;
	for (i = -radius; i <= radius; i++)
	{
	    // This is i, but one step closer to the center.
	    iminus = i - SIGN(i);

	    if (isFOV(cx + i, cy + (radius-1)) ||
		isFOV(cx + iminus, cy + (radius-1)))
	    {
		if (hasLOSLoose(cx, cy, cx+i, cy+radius))
		{
		    setFlag(cx+i, cy+radius, MAPFLAG_FOV|MAPFLAG_MAP, true);
		    hadfov = true;
		}
	    }
	    if (isFOV(cx + i, cy - (radius-1)) ||
		isFOV(cx + iminus, cy - (radius-1)))
	    {
		if (hasLOSLoose(cx, cy, cx+i, cy-radius))
		{
		    setFlag(cx+i, cy-radius, MAPFLAG_FOV|MAPFLAG_MAP, true);
		    hadfov = true;
		}
	    }
	    if (isFOV(cx + (radius-1), cy + i) ||
		isFOV(cx + (radius-1), cy + iminus))
	    {
		if (hasLOSLoose(cx, cy, cx+radius, cy+i))
		{
		    setFlag(cx+radius, cy+i, MAPFLAG_FOV|MAPFLAG_MAP, true);
		    hadfov = true;
		}
	    }
	    if (isFOV(cx - (radius-1), cy + i) ||
		isFOV(cx - (radius-1), cy + iminus))
	    {
		if (hasLOSLoose(cx, cy, cx-radius, cy+i))
		{
		    setFlag(cx-radius, cy+i, MAPFLAG_FOV|MAPFLAG_MAP, true);
		    hadfov = true;
		}
	    }
	}

	// Early exit if everything in last circle was
	// out of fov.
	if (!hadfov)
	    break;
    }
}

bool
MAP::hasLOSLoose(int sx, int sy, int ex, int ey) const
{
    int		dx, dy;

    for (dy = -1; dy <= 1; dy++)
    {
	for (dx = -1; dx <= 1; dx++)
	{
	    if ((!dx && !dy) || isTransparent(sx+dx, sy+dy))
	    {
		if (hasLOS(sx+dx, sy+dy, ex, ey))
		    return true;
	    }
	}
    }

    return false;
}

bool
MAP::hasLOS(int sx, int sy, int ex, int ey) const
{
    int		cx, cy;
    int		acx, acy;
    int		x, y;

    cx = ex - sx;
    cy = ey - sy;
    acx = ABS(cx);
    acy = ABS(cy);

    // Invalid coords are never in LOS.
    if (!isValidCoord(sx, sy))
	return false;
    if (!isValidCoord(ex, ey))
	return false;
    
    // Check for trivial LOS.
    if (acx <= 1 && acy <= 1)
	return true;

    // We do a bresenham algorithm.
    // Note that we do *NOT* check the start or
    // the end coords!  Something is in LOS despite the value
    // of these coords.
    if (acx > acy)
    {
	// X major stepping.
	int		dx, dy, error;

	dx = SIGN(cx);
	dy = SIGN(cy);

	error = 0;
	
	// Take initial free step.
	error = acy;
	y = sy;
	for (x = sx + dx; x != ex; x += dx)
	{
	    if (error >= acx)
	    {
		error -= acx;
		y += dy;
	    }

	    // The current ray is between (x, y) and (x, y+dy).
	    // Thus, if error is not zero, we consider it a pass
	    // if *either* is transparent.
	    // Whoops. This gives tunnel vision!  You can tunnel
	    // through one thickness walls.
	    if (!isTransparent(x, y) ) // &&
//		(error == 0 || !isTransparent(x, y+dy)))    
		return false;

	    error += acy;
	}
    }
    else
    {
	// Y major stepping.
	int		dx, dy, error;

	dx = SIGN(cx);
	dy = SIGN(cy);

	error = 0;
	
	// Take initial free step.
	error = acx;
	x = sx;
	for (y = sy + dy; y != ey; y += dy)
	{
	    if (error >= acy)
	    {
		error -= acy;
		x += dx;
	    }

	    // The current ray is between (x, y) and (x+dx, y).
	    // Thus, if error is not zero, we consider it a pass
	    // if *either* is transparent.
	    if (!isTransparent(x, y) ) // &&
//		(error == 0 || !isTransparent(x + dx, y)))
		return false;

	    error += acx;
	}
    }

    // Nothing got in our way, so we're done.
    return true;
}

void
MAP::buildDistMap(int tx, int ty)
{
    int			x, y, dx, dy, dist;
    PTRLIST<int>	ystack, xstack;

    if (!myDistMap)
    {
	myDistMap = new int[myWidth * myHeight];
	myDistMapX = myDistMapY = -1;
    }
    
    // Check to see if already built.
    if (myDistMapX == tx && myDistMapY == ty)
    {
	return;
    }

    myDistMapX = tx;
    myDistMapY = ty;
    
    // First, clear everything out.
    for (y = 0; y < getHeight(); y++)
    {
	for (x = 0; x < getWidth(); x++)
	{
	    setDistance(x, y, -1);
	}
    }

    // No distance to root..
    setDistance(tx, ty, 0);
    xstack.append(tx);
    ystack.append(ty);
    while (ystack.entries())
    {
	x = xstack.removeFirst();
	y = ystack.removeFirst();

	dist = getDistance(x, y);
	dist++;

	for (dy = -1; dy <= 1; dy++)
	{
	    for (dx = -1; dx <= 1; dx++)
	    {
		if (!dx && !dy)
		    continue;

		if (!isValidCoord(x+dx, y+dy))
		    continue;

		// Ignore impassable
		if (!isPassable(x+dx, y+dy))
		    continue;

		// Ignore already set
		if (getDistance(x+dx, y+dy) >= 0)
		    continue;

		// Calculate distance
		setDistance(x+dx, y+dy, dist);

		// And add to stack.
		xstack.append(x+dx);
		ystack.append(y+dy);
	    }
	}
    }
}

void
MAP::clearDistMap()
{
    delete [] myDistMap;
    myDistMap = 0;
}

int
MAP::getDistance(int x, int y) const
{
    if (!isValidCoord(x, y))
	return -1;
    
    return myDistMap[y * myWidth + x];
}

void
MAP::setDistance(int x, int y, int dist)
{
    if (!isValidCoord(x, y))
	return;
    
    myDistMap[y * myWidth + x] = dist;
}

bool
MAP::isValidCoord(int x, int y) const
{
    if (x < 0 || x >= myWidth)
	return false;

    if (y < 0 || y >= myHeight)
	return false;

    return true;
}

TILE_NAMES
MAP::getTile(int x, int y) const
{
    if (!isValidCoord(x, y))
	return TILE_INVALID;
    
    return (TILE_NAMES) myTiles(y * myWidth + x);
}

void
MAP::setTile(int x, int y, TILE_NAMES tile)
{
    assert(isValidCoord(x, y));
    if (!isValidCoord(x, y))
	return;

    myTiles(y * myWidth + x) = tile;
}

bool
MAP::isPassable(int x, int y) const
{
    TILE_NAMES	tile;

    tile = getTile(x, y);

    return glb_tiledefs[tile].ispassable;
}

bool
MAP::isTransparent(int x, int y) const
{
    TILE_NAMES	tile;

    tile = getTile(x, y);

    return glb_tiledefs[tile].istransparent;
}

bool
MAP::isMapped(int x, int y) const
{
    return getFlag(x, y, MAPFLAG_MAP);
}

bool
MAP::isFOV(int x, int y) const
{
    return getFlag(x, y, MAPFLAG_FOV);
}

ITEM *
MAP::getItem(int x, int y) const
{
    ITEM		*item;

    for (item = myItems; item; item = item->getNext())
    {
	if (item->getX() == x && item->getY() == y)
	    return item;
    }
    return 0;
}

MOB *
MAP::getMob(int x, int y) const
{
    MOB		*mob;

    for (mob = myMobs; mob; mob = mob->getNext())
    {
	// Swallowed mobs not actually on the map.
	if (mob->isSwallowed())
	    continue;
	if (mob->getX() == x && mob->getY() == y)
	    return mob;
    }
    return 0;
}

int
MAP::getNumMobs() const
{
    int		num = 0;
    MOB		*mob;

    for (mob = myMobs; mob; mob = mob->getNext())
    {
	num++;
    }
    return num;
}
int
MAP::getAllMobs(PTRLIST<MOB *> &moblist, int x, int y) const
{
    MOB		*mob;

    for (mob = myMobs; mob; mob = mob->getNext())
    {
	if (mob->getX() == x && mob->getY() == y)
	{
	    moblist.append(mob);
	}
    }
    return moblist.entries();
}

MOB *
MAP::findMob(MOB_NAMES def) const
{
    MOB		*mob;

    for (mob = myMobs; mob; mob = mob->getNext())
    {
	if (mob->getDefinition() == def)
	    return mob;
    }
    return 0;
}

MOB *
MAP::traceBullet(int x, int y, int range, int dx, int dy) const
{
    MOB		*hit = 0;

    x += dx;
    y += dy;
    while (isValidCoord(x, y))
    {
	hit = getMob(x, y);
	if (hit)
	    break;

	if (!isTransparent(x, y))
	    break;
	
	// Move one step
	x += dx;
	y += dy;
	range--;
	if (range <= 0)
	    break;
    }

    return hit;
}

void
MAP::displayBullet(int x, int y, int range, int dx, int dy, u8 sym, ATTR_NAMES attr) const
{
    MOB		*hit = 0;
    int		 sx, sy;

    x += dx;
    y += dy;
    while (isValidCoord(x, y))
    {
	hit = getMob(x, y);

	if (!isTransparent(x, y))
	    break;

	// No delay for out of sight changes.
	if (isFOV(x, y))
	{
	    re_redraw();

	    // Plot our char.
	    sx = x + myLastX - myLastMX;
	    sy = y + myLastY - myLastMY;
	    // Ensure on screen...
	    if (sx >= myLastX && sx < myLastX + myLastW &&
		sy >= myLastY && sy < myLastY + myLastH)
	    {
		gfx_printchar(sx, sy, sym, attr);
	    }
	    
	    gfx_update();
	    gfx_delay(5);
	}
	
	// Move one step
	x += dx;
	y += dy;
	range--;
	if (range <= 0)
	    break;
	if (hit)
	    break;
    }

    // clear up display.
    re_redraw();
    gfx_update();
}

void
MAP::fireball(MOB *src, int tx, int ty, int rad, DPDF damage, 
	      u8 sym, ATTR_NAMES attr) 
{
    MOB		*hit = 0;
    int		 x, y, sx, sy;
    bool	 anyvis = false;

    re_redraw();
    for (y = ty-rad+1; y <= ty+rad-1; y++)
    {
	for (x = tx-rad+1; x <= tx+rad-1; x++)
	{
	    // No delay for out of sight changes.
	    if (!isValidCoord(x, y))
		continue;
	    if (!isFOV(x, y))
		continue;
	    
	    anyvis = true;

	    // Plot our char.
	    sx = x + myLastX - myLastMX;
	    sy = y + myLastY - myLastMY;
	    // Ensure on screen...
	    if (sx >= myLastX && sx < myLastX + myLastW &&
		sy >= myLastY && sy < myLastY + myLastH)
	    {
		gfx_printchar(sx, sy, sym, attr);
	    }
	}
    }
	    
    if (anyvis)
    {
	gfx_update();
    }
    for (y = ty-rad+1; y <= ty+rad-1; y++)
    {
	for (x = tx-rad+1; x <= tx+rad-1; x++)
	{
	    if (!isValidCoord(x, y))
		continue;

	    hit = getMob(x, y);
	    
	    // Ignore self hits.
	    if (hit == src)
		continue;

	    if (hit)
		hit->applyDamage(src, damage.evaluate(), damage);
	}
    }

    // Delay after damage effect
    if (anyvis)
    {
	// Only do extra delay if explosion
	if (rad > 1)
	    gfx_delay(8);
	re_redraw();
	gfx_update();
    }
}
void
MAP::describeSquare(int x, int y, bool isblind) const
{
    if (!isblind)
    {
	MOB		*mob;
	ITEM		*item;

	mob = getMob(x, y);
	msg_format("%S <see> %O.", MOB::getAvatar(), getTerrainName(x, y));
	if (mob)
	    msg_format("%S <see> %O.", MOB::getAvatar(), mob);
	item = getItem(x, y);
	if (item)
	    msg_format("%S <see> %O.", MOB::getAvatar(), item);
    }
    else if (1) // (isMapped(x, y))
    {
	msg_format("%S <recall> %O.", MOB::getAvatar(), getTerrainName(x, y));
    }
    else
    {
	msg_format("%S <know> nothing of that spot.", MOB::getAvatar());
    }
}

const char *
MAP::getTerrainName(int x, int y) const
{
    TILE_NAMES		tile;

    tile = getTile(x, y);

    return glb_tiledefs[tile].legend;
}

void
MAP::addItem(ITEM *item)
{
    assert(item->getNext() == 0);

    item->setNext(myItems);
    myItems = item;
}

void
MAP::removeItem(ITEM *item)
{
    ITEM	*c, *l;

    assert(item);
    if (!item)
	return;

    l = 0;
    for (c = myItems; c; c = c->getNext())
    {
	if (c == item)
	{
	    if (l)
		l->setNext(item->getNext());
	    else
		myItems = item->getNext();
	    item->setNext(0);
	    return;
	}

	l = c;
    }	    

    assert(0);
}

void
MAP::addMob(MOB *mob)
{
    assert(mob->getNext() == 0);

    mob->setNext(myMobs);
    myMobs = mob;

    mob->setMap(this);
}

void
MAP::removeMob(MOB *mob)
{
    MOB		*cur, *last;

    // Protect our safe next flag.
    if (mob && mob == myMobSafeNext)
    {
	myMobSafeNext = mob->getNext();
    }	    

    last = 0;
    for (cur = myMobs; cur; cur = cur->getNext())
    {
	if (cur == mob)
	{
	    if (last)
		last->setNext(cur->getNext());
	    else
		myMobs = cur->getNext();
	    cur->setNext(0);

	    cur->setMap(0);

	    return;
	}

	last = cur;
    }

    // Failure to remove a mob!
    assert(0);
}

bool
MAP::getFlag(int x, int y, int flag) const
{
    if (!isValidCoord(x, y))
	return false;

    if (myTileFlags(x + y * myWidth) & flag)
	return true;
    return false;
}

void
MAP::setFlag(int x, int y, int flag, bool val)
{
    assert(isValidCoord(x, y));
    if (!isValidCoord(x, y))
	return;

    if (val)
	myTileFlags(x + y * myWidth) |= flag;
    else
	myTileFlags(x + y * myWidth) &= ~flag;
}

void
MAP::redraw(int sx, int sy, int w, int h,
	    int cx, int cy) const
{
    // Adjust cx & cy so they refer to top left.
    cx = cx - w / 2;
    cy = cy - h / 2;

    myLastX = sx;
    myLastY = sy;
    myLastMX = cx;
    myLastMY = cy;
    myLastW = w;
    myLastH = h;

    re_redraw();
}

void
MAP::re_redraw() const
{
    int		sx, sy, w, h;
    int		x, y, cx, cy;
    TILE_NAMES	tile;
    u8		symbol;
    ATTR_NAMES	attr;
    bool	isblind;
    bool	isswallowed;

    sx = myLastX;
    sy = myLastY;
    w = myLastW;
    h = myLastH;
    cx = myLastMX;
    cy = myLastMY;

    isblind = false;
    if (MOB::getAvatar())
	isblind = MOB::getAvatar()->hasItem(ITEM_BLIND);

    isswallowed = false;
    if (MOB::getAvatar())
	isswallowed = MOB::getAvatar()->isSwallowed();

    // Plot all the points.
    for (y = sy; y < sy + h; y++)
    {
	for (x = sx; x < sx + w; x++)
	{
	    tile = getTile(x - sx + cx, y - sy + cy);

	    // Plot relevant character.
	    if (x >= 0 && x <= SCR_WIDTH && y >= 0 && y <= SCR_WIDTH)
	    {
		symbol = glb_tiledefs[tile].symbol;
		attr = (ATTR_NAMES) glb_tiledefs[tile].attr;

		// Override if out of FOV - shows up dim then.
		if (!isFOV(x - sx + cx, y - sy + cy))
		{
		    // We never draw the stairs dark as the user
		    // should be able to find them easily.
		    if (symbol != '>' && symbol != '<')
			attr = ATTR_OUTOFFOV;
		}
		if (isblind || isswallowed)
		    attr = ATTR_OUTOFFOV;

		// If it is not mapped, always ' '.
		if (!isMapped(x - sx + cx, y - sy + cy))
		    symbol = ' ';
		
		gfx_printchar(x, y, symbol, attr);
	    }
	}
    }

    // Print out all of our items.
    ITEM	*item;

    for (item = myItems; item; item = item->getNext())
    {
	x = item->getX() + sx - cx;
	y = item->getY() + sy - cy;

	// Verify it is in FOV.
	if (!isFOV(item->getX(), item->getY()))
	    continue;

	if (isblind || isswallowed)
	    continue;

	// Verify it is on screen.
	if (x >= sx && y >= sy && x < sx + w && y < sy + h)
	{
	    item->getLook(symbol, attr);
	    gfx_printchar(x, y, symbol, attr);
	}
    }

    // Print out all of our mobs.
    MOB		*mob;
    
    for (mob = myMobs; mob; mob = mob->getNext())
    {
	x = mob->getX() + sx - cx;
	y = mob->getY() + sy - cy;

	// Verify it is in FOV.
	if (!isFOV(mob->getX(), mob->getY()))
	{
	    continue;
	}

	if ((isblind || isswallowed) && !mob->isAvatar())
	    continue;

	// Mobs that are swallowed don't show up.
	if (mob->isSwallowed() && !mob->isAvatar())
	    continue;

	// Verify it is on screen.
	if (x >= sx && y >= sy && x < sx + w && y < sy + h)
	{
	    mob->getLook(symbol, attr);
	    gfx_printchar(x, y, symbol, attr);
	}
    }

    if (isswallowed)
    {
	// Print out the intestine walls.
	x = MOB::getAvatar()->getX() + sx - cx;
	y = MOB::getAvatar()->getY() + sy - cy;
	gfx_printchar(x-1, y-1, '/', ATTR_BROWN);
	gfx_printchar(x,   y-1, '-', ATTR_BROWN);
	gfx_printchar(x+1, y-1, '\\', ATTR_BROWN);
	gfx_printchar(x-1, y,   '|', ATTR_BROWN);
	gfx_printchar(x+1, y,   '|', ATTR_BROWN);
	gfx_printchar(x-1, y+1, '\\', ATTR_BROWN);
	gfx_printchar(x,   y+1, '-', ATTR_BROWN);
	gfx_printchar(x+1, y+1, '/', ATTR_BROWN);
    }
}

void
MAP::fadeToBlack() const
{
    int		r, x, y, d;
    u8		sym;
    ATTR_NAMES	attr;

    d = MAX(myLastW, myLastH) / 2;
    for (r = d+10; r > 0;  r--)
    {
	for (y = 0; y < myLastH; y++)
	{
	    for (x = 0; x < myLastW; x++)
	    {
		d = MAX(ABS(x - myLastW/2), ABS(y - myLastH/2));

		if (d == 0)
		    continue;
		if (d >= r)
		{
		    // pure black.
		    gfx_printchar(x+myLastX, y+myLastY, ' ', ATTR_NORMAL);
		}
		else if (d >= r-10)
		{
		    // Out of FOV.
		    gfx_getchar(x+myLastX, y+myLastY, sym, attr);
		    gfx_printchar(x+myLastX, y+myLastY, sym, ATTR_OUTOFFOV);
		}
	    }
	}
	// Refresh the screen
	gfx_update();

	// Abort on key press...
	if (gfx_isKeyWaiting())
	    break;

	// And delay a bit
	gfx_delay(3);
    }
}

void
MAP::doMoveNPC()
{
    MOB		*cur;

    for (cur = myMobs; cur; cur = myMobSafeNext)
    {
	myMobSafeNext = cur->getNext();

	// I forgot this line the first time.  That was embarrasing :>
	if (cur->isAvatar())
	    continue;
	
	if (!cur->aiForcedAction())
	{
	    cur->aiDoAI();
	}
    }
}

void
MAP::save(ostream &os) const
{
    os.write((const char *) &myWidth, sizeof(int));
    os.write((const char *) &myHeight, sizeof(int));

    os.write((const char *) myTiles.readData(), myWidth * myHeight);
    os.write((const char *) myTileFlags.readData(), myWidth * myHeight);
    
    os.write((const char *) &myTotalRooms, sizeof(int));

    u8		c;
    MOB		*mob;
    ITEM	*item;

    for (mob = myMobs; mob; mob = mob->getNext())
    {
	c = 1;
	os.write((const char *) &c, 1);
	mob->save(os);
    }
    c = 0;
    os.write((const char *) &c, 1);
    
    for (item = myItems; item; item = item->getNext())
    {
	c = 1;
	os.write((const char *) &c, 1);
	item->save(os);
    }
    c = 0;
    os.write((const char *) &c, 1);
}

void
MAP::load(istream &is)
{
    u8			c;
    
    is.read((char *) &myWidth, sizeof(int));
    is.read((char *) &myHeight, sizeof(int));

    delete [] myDistMap;

    myTiles = MAPDATA(myWidth * myHeight);
    myTileFlags = MAPDATA(myWidth * myHeight);
    myDistMap = 0;

    is.read((char *) myTiles.writeData(), myWidth * myHeight);
    is.read((char *) myTileFlags.writeData(), myWidth * myHeight);

    is.read((char *) &myTotalRooms, sizeof(int));

    // Load mobs...
    while (1)
    {
	is.read((char *) &c, 1);
	if (!c)
	    break;

	MOB		*mob;

	mob = MOB::load(is);

	addMob(mob);
    }

    // Load items
    while (1)
    {
	is.read((char *) &c, 1);
	if (!c)
	    break;

	ITEM		*item;

	item = ITEM::load(is);

	addItem(item);
    }
}
